//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders.vice;


import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Paint.Align;
import android.graphics.Paint.Style;
import android.graphics.Rect;
import android.graphics.Typeface;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.TypedValue;

import androidx.annotation.NonNull;

import de.rainerhock.eightbitwonders.EmulationUi;
import de.rainerhock.eightbitwonders.R;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;


/**
 * View for a virtual keyboard-key.
 */
@SuppressLint({"ClickableViewAccessibility"})
abstract class KeyButtonBase extends androidx.appcompat.widget.AppCompatButton {
    private boolean mHighlighted;
    private Drawable mOriginalColor = null;
    Drawable getOriginalColor() {
        return mOriginalColor;
    }

    public boolean isHighlighted() {
        return mHighlighted;
    }
    /**
     * Simple constructor to use when creating a KeyButton from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    KeyButtonBase(final Context context) {
        super(context);
        init(context, null);
    }
    KeyButtonBase(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init(context, attrs);
    }
    KeyButtonBase(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
        init(context, attrs);
    }

    private String mText;
    protected String getTextValue() {
        return mText;
    }

    abstract void setKeyboardController(ViceKeyboardFragment val);

    private String mShifttext;
    private String mCtrltext;
    private String mCbmText;
    private int mShiftpetscii;
    private int mCbmPetscii;
    private int mRow;
    private int mCol;
    private int mShiftstate;
    private int mModificatortype;
    private static final int MODIFICATOR_NONE = 0;
    private static final int MODIFICATOR_SHIFT = 1;
    private static final int MODIFICATOR_CBM = 2;
    private static final int MODIFICATOR_SHIFTLOCK = 4;

    boolean isShift() {
        return mModificatortype == MODIFICATOR_SHIFT;
    }

    boolean isCbm() {
        return mModificatortype == MODIFICATOR_CBM;
    }

    boolean isShiftLock() {
        return mModificatortype == MODIFICATOR_SHIFTLOCK;
    }

    boolean isModificator() {
        return mModificatortype != MODIFICATOR_NONE;
    }

    int getRow() {
        return mRow;
    }

    int getCol() {
        return mCol;
    }

    String getMainText() {
        return mText;
    }

    int getShiftState() {
        return mShiftstate;
    }

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    protected boolean isRemoteButton(final int keyCode) {
        return  EmulationUi.JOYSTICK_START_BUTTONS.contains(keyCode)
                || EmulationUi.JOYSTICK_DPAD_DIRECTION_BUTTONS.contains(keyCode)
                || EmulationUi.JOYSTICK_RC_FUNCTIONS.contains(keyCode)
                || EmulationUi.SYSTEM_KEYS.contains(keyCode);

    }


    private static final char CHARACTER_FOR_TEXTBOUNDS = (char) 166;
    private static final char CHARACTER_FOR_TEXTMARGINS = (char) 167;
    private static final int ALPHA_SECONDARY_SYMBOLS = 192;
    @SuppressLint("ClickableViewAccessibility")
    protected void init(final Context context, final AttributeSet attrs) {
        if (mPaintCbm.getTypeface() == mPaintDefault.getTypeface()) {
            for (Paint p : Arrays.asList(mPaintCbm, mPaintBorder)) {
                p.setTypeface(Typeface.createFromAsset(context.getAssets(), "fonts/cbm.ttf"));
                p.setStyle(Style.FILL);
                p.setTextAlign(Align.CENTER);
                p.setColor(getResources().getColor(R.color.button_cbmtext));

            }

            mPaintCbm.setStyle(Style.FILL);
            mPaintBorder.setStyle(Style.STROKE);
            mPaintIntl.setTextAlign(Align.LEFT);
            mPaintIntl.setColor(getResources().getColor(R.color.button_text));
            mPaintIntl.setAlpha(ALPHA_SECONDARY_SYMBOLS);
            mPaintDefault.setTextAlign(Align.CENTER);
            mPaintCbm.getTextBounds(Character.toString(CHARACTER_FOR_TEXTBOUNDS),
                    0, 1, mCbmBounds);
            mPaintCbm.getTextBounds(Character.toString(CHARACTER_FOR_TEXTMARGINS),
                    0, 1, mCbmMargin);

            mPaintSpecial.setAntiAlias(true);
            mPaintDefault.setAntiAlias(true);
            mPaintDefault.setColor(getResources().getColor(R.color.button_text));

            mPaintSpecial.setTextAlign(Align.CENTER);
            mPaintSpecial.setColor(getResources().getColor(R.color.button_cbmtext));

            mPaintCbm.setAlpha(ALPHA_SECONDARY_SYMBOLS);
            mPaintBorder.setAlpha(ALPHA_SECONDARY_SYMBOLS);
            mPaintSpecial.setAlpha(ALPHA_SECONDARY_SYMBOLS);

        }
        if (attrs != null) {
            //noinspection resource
            TypedArray a = context.getResources().obtainAttributes(attrs, R.styleable.vice_softkey);
            try {
                mText = a.getString(R.styleable.vice_softkey_text);
                mShiftstate = a.getInt(R.styleable.vice_softkey_shiftstate, 0);
                mShifttext = a.getString(R.styleable.vice_softkey_shift_text);
                mShiftpetscii = a.getInt(R.styleable.vice_softkey_shift_petscii, 0);
                mRow = a.getInt(R.styleable.vice_softkey_row, 0);
                mCol = a.getInt(R.styleable.vice_softkey_col, 0);
                mCtrltext = a.getString(R.styleable.vice_softkey_ctrl_text);
                mCbmText = a.getString(R.styleable.vice_softkey_cbm_text);
                mCbmPetscii = a.getInt(R.styleable.vice_softkey_cbm_petscii, 0);
                mModificatortype = a.getInt(R.styleable.vice_softkey_modificator_type, 0);
            } finally {
                a.recycle();
            }
            setContentDescription(mText);
        }
        mOriginalColor = getBackground();

    }


    private final Paint mPaintDefault = new Paint();
    private final Paint mPaintSpecial = new Paint();
    private final Paint mPaintCbm = new Paint();
    private final Paint mPaintIntl = new Paint();
    private final Paint mPaintBorder = new Paint();
    private final Rect mCbmBounds = new Rect();
    private final Rect mCbmMargin = new Rect();
    private final List<Float> mDipsizesLarge = Arrays.asList(10f, 12.5f, 15.7f, 20f);

    private final List<Float> mDipsizesSmall = Arrays.asList(8f, 10f, 12f);

    private final List<Float> mDipsizesCbm = Arrays.asList(5f, 8f, 10f, 12.5f, 15f);
    private final Rect mTextwidthBounds = new Rect();
    private final Rect mTextheightBounds = new Rect();

    private String petsciiString(final int petsciicode) {
        return Character.toString((char) petsciicode);
    }

    private void updateFontsize(final String textforwidth,
                                final Paint paint,
                                final List<Float> pixelsizes,
                                final float xmultiplier,
                                final float ymultiplier) {
        int sizeindex = pixelsizes.size();
        do {
            sizeindex--;
            paint.setTextSize(pixelsizes.get(sizeindex));
            paint.getTextBounds(textforwidth, 0, textforwidth.length(), mTextwidthBounds);
            paint.getTextBounds("|", 0, 1, mTextheightBounds);

        } while (sizeindex > 0
                && (Math.abs(mDrawingRect.width() * xmultiplier)
                < Math.abs(mTextwidthBounds.width())
                || Math.abs(mDrawingRect.height() * ymultiplier)
                < Math.abs(mTextheightBounds.height())));
    }

    private final Rect mDrawingRect = new Rect();

    private void drawTop(final Canvas canvas,
                         final String text,
                         final Paint paint,
                         final float xcenter,
                         final List<Float> pixelsizes,
                         final float xmultiplier,
                         final float ymultiplier) {
        updateFontsize(text, paint, pixelsizes, xmultiplier, ymultiplier);
        canvas.drawText(text, xcenter,
                //CHECKSTYLE DISABLE MagicNumber FOR 1 LINES
                (mDrawingRect.height() * 0.1f - paint.getFontMetricsInt().ascent), paint);
    }
    //CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
    private void drawTopBottom(final Canvas canvas,
                               final String text1,
                               final String text2,
                               final Paint paint,
                               final float xcenter,
                               final List<Float> pixelsizes,
                               final float xmultiplier,
                               final float ymultiplier) {
        updateFontsize(text1, paint, pixelsizes, xmultiplier, ymultiplier);
        float s = paint.getTextSize();
        if (text2 != null) {
            updateFontsize(text2, paint, pixelsizes, xmultiplier, ymultiplier);
        }
        paint.setTextSize(Math.min(s, paint.getTextSize()));
        canvas.drawText(text1, xcenter,
                (mDrawingRect.height() / 2f - paint.getFontMetricsInt().descent), paint);
        if (text2 != null) {
            canvas.drawText(text2, xcenter,
                    (mDrawingRect.height() / 2f - paint.getFontMetricsInt().ascent), paint);
        }
    }
    //CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
    private void drawTopBottom(final Canvas canvas,
                               final String text1,
                               final String text2,
                               final Paint paint1,
                               final Paint paint2,
                               final float xcenter,
                               final List<Float> pixelsizes1,
                               final List<Float> pixelsizes2,
                               final float xmultiplier) {

        if (text1 != null) {
            updateFontsize(text1, paint1, pixelsizes1, xmultiplier, YMULTIPLIER_HALFSIZE);
            if (paint1 == mPaintCbm) {
                canvas.drawText(text1, xcenter,
                        (mDrawingRect.height() / 2f - mPaintSpecial.getFontMetricsInt().descent),
                        paint1);
            } else {
                canvas.drawText(text1, xcenter,
                        (mDrawingRect.height() / 2f - paint1.getFontMetricsInt().descent), paint1);

            }
        }
        if (text2 != null) {
            updateFontsize(text2, paint2, pixelsizes2, xmultiplier, YMULTIPLIER_HALFSIZE);
            canvas.drawText(text2, xcenter,
                    (mDrawingRect.height() / 2f - paint2.getFontMetricsInt().ascent), paint2);
        }
    }

    private interface DisplayDependantValues {
        List<Float> getPixelsizesSmall();

        List<Float> getPixelsizesLarge();

        List<Float> getPixelsizesCbm();
    }

    private static final Map<Integer, DisplayDependantValues> DISPLAY_DEPENDANT_VALUES
            = new LinkedHashMap<>();

    private static DisplayDependantValues getDisplayDependantValues(final DisplayMetrics dm,
                                                                    final List<Float> dipsizesSmall,
                                                                    final List<Float> dipsizesLarge,
                                                                    final List<Float> dipsizesCbm) {
        if (!DISPLAY_DEPENDANT_VALUES.containsKey(dm.widthPixels)) {
            DISPLAY_DEPENDANT_VALUES.put(dm.widthPixels, new DisplayDependantValues() {
                private final List<Float> mPixelsizesSmall = new LinkedList<>();
                private final List<Float> mPixelsizesLarge = new LinkedList<>();
                private final List<Float> mPixelsizesCbm = new LinkedList<>();

                @Override
                public List<Float> getPixelsizesSmall() {
                    if (mPixelsizesSmall.isEmpty()) {
                        for (float dipsize : dipsizesSmall) {
                            mPixelsizesSmall.add(TypedValue.applyDimension(
                                    TypedValue.COMPLEX_UNIT_DIP, dipsize, dm));
                        }
                    }
                    return mPixelsizesSmall;
                }

                @Override
                public List<Float> getPixelsizesLarge() {
                    if (mPixelsizesLarge.isEmpty()) {
                        for (float dipsize : dipsizesLarge) {
                            mPixelsizesLarge.add(TypedValue.applyDimension(
                                    TypedValue.COMPLEX_UNIT_DIP, dipsize, dm));
                        }
                    }
                    return mPixelsizesLarge;
                }

                public List<Float> getPixelsizesCbm() {
                    if (mPixelsizesCbm.isEmpty()) {
                        for (float dipsize : dipsizesCbm) {
                            mPixelsizesCbm.add(TypedValue.applyDimension(
                                    TypedValue.COMPLEX_UNIT_DIP, dipsize, dm));
                        }
                    }
                    return mPixelsizesCbm;
                }

            });
        }
        return DISPLAY_DEPENDANT_VALUES.get(dm.widthPixels);
    }
    private static final float YMULTIPLIER_HALFSIZE = 0.5f;
    private static final float XMULTIPLIER_LEFT_SIDE = 0.5f;
    private static final float XMULTIPLIER_RIGHT_SIDE = 1.5f;

    private static final float YMULTIPLIER_SYMBOL = 0.25f;
    private static final float XMULTIPLIER_SYMBOL = 0.25f;
    //CHECKSTYLE DISABLE MethodLength FOR 1 LINES
    @Override
    protected void onDraw(final Canvas canvas) {
        DisplayDependantValues ddv = KeyButtonBase.getDisplayDependantValues(getResources()
                .getDisplayMetrics(), mDipsizesSmall, mDipsizesLarge, mDipsizesCbm);
        final List<Float> pixelsizessmall = ddv.getPixelsizesSmall();
        final List<Float> pixelsizesLarge = ddv.getPixelsizesLarge();
        final List<Float> pixelsizesCbm = ddv.getPixelsizesCbm();
        getDrawingRect(mDrawingRect);
        final float xcenter = mDrawingRect.width() / 2f;

        // Single Character, no shift, no CBM
        if (mText != null && mShifttext == null && mCtrltext == null && mShiftpetscii == 0
                && mCbmPetscii == 0 && mCbmText == null) {
            if (!mText.contains("\n")) {
                drawTop(canvas, mText, mPaintDefault,
                        xcenter, pixelsizesLarge,
                        1f, YMULTIPLIER_HALFSIZE);
            } else {
                String[] line = mText.split("\n");
                drawTopBottom(canvas, line[0], line[1], mPaintDefault,
                        xcenter, pixelsizesLarge, 1f, YMULTIPLIER_HALFSIZE);
            }
        } else if (mText != null && mShifttext != null && mCbmText == null && mCtrltext == null
                // 2 non-graphical Characters
                && mShiftpetscii == 0 && mCbmPetscii == 0) {
            drawTopBottom(canvas, mShifttext, mText, mPaintSpecial, mPaintDefault,
                    xcenter, pixelsizessmall, pixelsizesLarge, 1f);
        } else if (mText != null && mShifttext == null && mCbmPetscii != 0 && mShiftpetscii != 0) {
            // 1 Text and 2 graphical symbols
            drawTop(canvas, mText, mPaintDefault, xcenter,
                    pixelsizesLarge, 1f, YMULTIPLIER_HALFSIZE);
            updateFontsize(petsciiString(CHARACTER_FOR_TEXTBOUNDS), mPaintCbm, pixelsizesCbm,
                    XMULTIPLIER_SYMBOL, YMULTIPLIER_SYMBOL);
            mPaintCbm.setTextAlign(Align.LEFT);
            mPaintCbm.getTextBounds(Character.toString(CHARACTER_FOR_TEXTBOUNDS), 0, 1, mCbmBounds);
            //CHECKSTYLE DISABLE MagicNumber FOR 25 LINES
            canvas.drawText(petsciiString(mShiftpetscii), xcenter * 1.1f,
                    mDrawingRect.height() / 2f - mPaintDefault.getFontMetricsInt().ascent,
                    mPaintCbm);
            final float top = (float) mDrawingRect.height() / 2
                    - mPaintDefault.getFontMetricsInt().ascent - mCbmBounds.height() - 2;
            final float bottom = (float) mDrawingRect.height() / 2
                    - mPaintDefault.getFontMetricsInt().ascent + 1;
            canvas.drawRect(xcenter * 0.9f - 1 - mCbmBounds.width(), top,
                    2 + xcenter * 0.9f, bottom, mPaintBorder);
            canvas.drawRect(xcenter * 1.1f - 1, top,
                    xcenter * 1.1f + 2 + mCbmBounds.width(), bottom, mPaintBorder);

            if (mCbmPetscii == 173) {
                canvas.scale(-1, 1, xcenter, 0);
                canvas.drawText(petsciiString(189),
                        xcenter * 1.1f,
                        mDrawingRect.height() / 2f - mPaintDefault.getFontMetricsInt().ascent,
                        mPaintCbm);
                canvas.scale(-1, 1, xcenter, 0);
            } else {
                mPaintCbm.setTextAlign(Align.RIGHT);
                canvas.drawText(petsciiString(mCbmPetscii),
                        xcenter * 0.9f,
                        mDrawingRect.height() / 2f - mPaintDefault.getFontMetricsInt().ascent,
                        mPaintCbm);
            }
        } else if (mText != null && mShifttext != null && mCtrltext != null
                && mShiftpetscii == 0 && mCbmPetscii == 0) {
            // 2 Characters and one Color
            drawTopBottom(canvas, mShifttext, mText, mPaintSpecial, mPaintDefault,
                    xcenter * XMULTIPLIER_LEFT_SIDE,
                    pixelsizessmall, pixelsizesLarge,
                    YMULTIPLIER_HALFSIZE);
            String[] lines = mCtrltext.split(" ");
            mPaintSpecial.setTextAlign(Align.RIGHT);
            //CHECKSTYLE DISABLE MagicNumber FOR 6 LINES
            if (lines.length == 1) {
                drawTop(canvas, lines[0], mPaintSpecial,
                        getWidth() * 0.95f, pixelsizessmall, 0.2f, 0.2f);
            } else {
                drawTopBottom(canvas, lines[0], lines[1], mPaintSpecial,
                        getWidth() * 0.95f, pixelsizessmall, 0.2f, 0.2f);
            }
            mPaintDefault.setTextAlign(Align.CENTER);
            mPaintSpecial.setTextAlign(Align.CENTER);
        } else if (mText != null && mShifttext == null && mCbmPetscii == 0 && mShiftpetscii != 0
        && mCbmText == null) {
            drawTop(canvas, mText, mPaintDefault, xcenter,
                    pixelsizesLarge, 1f, YMULTIPLIER_HALFSIZE);
            updateFontsize(petsciiString(CHARACTER_FOR_TEXTBOUNDS), mPaintCbm, pixelsizesCbm,
                    XMULTIPLIER_SYMBOL, YMULTIPLIER_SYMBOL);
            mPaintCbm.setTextAlign(Align.CENTER);
            mPaintCbm.getTextBounds(Character.toString(CHARACTER_FOR_TEXTBOUNDS), 0, 1, mCbmBounds);
            //CHECKSTYLE DISABLE MagicNumber FOR 25 LINES
            canvas.drawText(petsciiString(mShiftpetscii), xcenter,
                    mDrawingRect.height() / 2f - mPaintDefault.getFontMetricsInt().ascent,
                    mPaintCbm);
            final float top = (float) mDrawingRect.height() / 2
                    - mPaintDefault.getFontMetricsInt().ascent - mCbmBounds.height() - 1;
            final float bottom = (float) mDrawingRect.height() / 2
                    - mPaintDefault.getFontMetricsInt().ascent + 1;
            canvas.drawRect(xcenter - (mCbmBounds.width() / 2f) - 1, top,
                    2 + xcenter + (mCbmBounds.width() / 2f), bottom, mPaintBorder);
        } else //noinspection ConstantValue
            if (mText != null && mShifttext == null && mCbmPetscii == 0 && mShiftpetscii != 0
                    && mCbmText != null) {
                drawTop(canvas, mText, mPaintDefault, xcenter,
                        pixelsizesLarge, 1f, YMULTIPLIER_HALFSIZE);
                updateFontsize(petsciiString(CHARACTER_FOR_TEXTBOUNDS), mPaintCbm, pixelsizesCbm,
                        XMULTIPLIER_SYMBOL, YMULTIPLIER_SYMBOL);
                mPaintCbm.setTextAlign(Align.LEFT);
                mPaintCbm.getTextBounds(Character.toString(CHARACTER_FOR_TEXTBOUNDS),
                        0, 1, mCbmBounds);
                //CHECKSTYLE DISABLE MagicNumber FOR 25 LINES
                canvas.drawText(petsciiString(mShiftpetscii), xcenter * 1.1f,
                        mDrawingRect.height() / 2f - mPaintDefault.getFontMetricsInt().ascent,
                        mPaintCbm);
                final float top = (float) mDrawingRect.height() / 2
                        - mPaintDefault.getFontMetricsInt().ascent - mCbmBounds.height() - 2;
                final float bottom = (float) mDrawingRect.height() / 2
                        - mPaintDefault.getFontMetricsInt().ascent + 1;
                canvas.drawRect(xcenter * 1.1f - 1, top,
                        xcenter * 1.1f + 2 + mCbmBounds.width(), bottom, mPaintBorder);
                mPaintCbm.setTextAlign(Align.RIGHT);
                canvas.drawText(mCbmText,
                        xcenter * 0.9f,
                        mDrawingRect.height() / 2f - mPaintDefault.getFontMetricsInt().ascent,
                        mPaintCbm);
                canvas.scale(-1, 1, xcenter, 0);
        } else if (mText != null && mShifttext != null && mCbmPetscii != 0 && mShiftpetscii == 0) {
                drawTopBottom(canvas,  mShifttext, null, mPaintSpecial, mPaintDefault,
                        xcenter  * XMULTIPLIER_RIGHT_SIDE,
                        pixelsizessmall, pixelsizesLarge,
                        YMULTIPLIER_HALFSIZE);
                drawTopBottom(canvas, null, mText,  mPaintSpecial, mPaintDefault,
                    xcenter,
                    pixelsizessmall, pixelsizesLarge,
                    YMULTIPLIER_HALFSIZE);
            String cbmtext;
                //CHECKSTYLE DISABLE MagicNumber FOR 5 LINES
            if (mCbmPetscii == 173) {
                cbmtext = petsciiString(189);
            } else {
                cbmtext = petsciiString(mCbmPetscii);
            }
            drawTopBottom(canvas, cbmtext, null,  mPaintCbm, mPaintDefault,
                    xcenter * XMULTIPLIER_LEFT_SIDE, pixelsizessmall, pixelsizesLarge,
                    XMULTIPLIER_SYMBOL);
        } else if (mText != null && mShifttext != null && mCbmText != null && mShiftpetscii == 0) {
            drawTopBottom(canvas,  mShifttext, null, mPaintSpecial, mPaintDefault,
                    xcenter  * XMULTIPLIER_RIGHT_SIDE,
                    pixelsizessmall, pixelsizesLarge,
                    YMULTIPLIER_HALFSIZE);
            drawTopBottom(canvas, null, mText,  mPaintSpecial, mPaintDefault,
                    xcenter,
                    pixelsizessmall, pixelsizesLarge,
                    YMULTIPLIER_HALFSIZE);

            drawTopBottom(canvas, mCbmText, null, mPaintSpecial,
                    xcenter * XMULTIPLIER_LEFT_SIDE,
                    pixelsizessmall, YMULTIPLIER_HALFSIZE, YMULTIPLIER_HALFSIZE);
        }
    }
    void setHighlighted(final boolean val) {
        mHighlighted = val;
    }

    @NonNull
    @Override
    public String toString() {
        return "'" + mText + "', " + super.toString();
    }
}
