//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.LayerDrawable;
import android.os.Looper;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.core.content.res.ResourcesCompat;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * A view with status information about drives and joysticks.
 */

public final class StatusbarView extends LinearLayout
        implements JoystickListener, DriveStatusListener {

    private final Map<Integer, Long> mLastmodificationTime = new HashMap<>();
    private Runnable mRunnable = null;

    /**
     * Simple constructor to use when creating a StatusbarView from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public StatusbarView(final Context context) {
        super(context);
        init(context, null, 0);
    }

    /**
     * Constructor that is called when inflating a StatusbarView from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public StatusbarView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init(context, attrs, 0);
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public StatusbarView(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
        init(context, attrs, defStyle);
    }
    private float mAlphaActive = 0;
    private float mAlphaInactive = 0;
    private void init(@SuppressWarnings("unused") final Context context,
                      @SuppressWarnings("unused") final AttributeSet attrs,
                      @SuppressWarnings("unused") final int defStyle) {
        if (context instanceof EmulationActivity) {
            mEmulationUi = ((EmulationActivity) context).getEmulationUi();
        }
        mAlphaActive = getResources().getInteger(R.integer.alpha_status_active) / 255f;
        mAlphaInactive = getResources().getInteger(R.integer.alpha_status_inactive) / 255f;

        mBorderPixels =
                (int) (2 * ((float) getContext().getResources().getDisplayMetrics().densityDpi
                        / DisplayMetrics.DENSITY_DEFAULT));
        LayoutInflater inflater = (LayoutInflater) getContext().getApplicationContext()
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        @SuppressLint("InflateParams") ViewGroup rootView =
                (LinearLayout) inflater.inflate(R.layout.view_statusbar, null);
        addView(rootView);
        setVisibleIfNotEmpty();
    }

    private int getActiveMonitor() {
        EmulationActivity ea = (EmulationActivity) getContext();
        return ea.getViewModel().getCurrentDisplayId();

    }
    void setMonitorList(final Emulation emulation) {
        Emulation.DualMonitorFunctions mf = emulation.getDualMonitorFunctions();
        if (mf != null) {
            final Map<Integer, String> displays = mf.getDisplays();
            TextView tv = findViewById(R.id.tv_monitor);
            if (!displays.isEmpty()) {
                tv.setText(displays.get(getActiveMonitor()));
                tv.setVisibility(View.VISIBLE);
            } else {
                tv.setVisibility(View.GONE);
            }
        }
        setVisibleIfNotEmpty();
    }

    @Override
    public void onJoystickChanged(final String joystickId, final Emulation.InputDeviceType type,
                                  final int portnumber,
                                  final boolean isAbsoluteMovement,
                                  final float x, final float y,
                                  final Set<Emulation.InputButton> buttons) {
        if (mActiveJoystickDrawable.containsKey(portnumber)) {
            Long time = mLastmodificationTime.get(portnumber);
            if (time != null && time > 0) {
                mLastmodificationTime.put(portnumber, System.currentTimeMillis());
            }
            ImageButton b = mActiveJoystickDrawable.get(portnumber);
            final LayerDrawable ld;
            if (b != null) {
                ld = (LayerDrawable) b.getDrawable();
            } else {
                ld = null;
            }
            if (ld != null) {
                final int active = getResources().getInteger(R.integer.alpha_status_active);
                final int inactive = getResources().getInteger(R.integer.alpha_status_inactive);
                Runnable r = () -> {
                    ld.findDrawableByLayerId(R.id.status_f).setAlpha(buttons.isEmpty() ? inactive : active);
                    ld.findDrawableByLayerId(R.id.status_n).setAlpha(y < 0 ? active : inactive);
                    ld.findDrawableByLayerId(R.id.status_s).setAlpha(y > 0 ? active : inactive);
                    ld.findDrawableByLayerId(R.id.status_e).setAlpha(x > 0 ? active : inactive);
                    ld.findDrawableByLayerId(R.id.status_w).setAlpha(x < 0 ? active : inactive);
                };
                if (Looper.getMainLooper().getThread() == Thread.currentThread()) {
                    r.run();
                } else {
                    post(r);
                }
            }
        }
    }

    private final Map<Object, TextView> mDiskDriveViews = new LinkedHashMap<>();
    private final LinkedList<Object> mDiskdrives = new LinkedList<>();
    private int mBorderPixels = 0;
    private final Map<Integer, ImageButton> mActiveJoystickDrawable = new LinkedHashMap<>();

    @Override
    public void setVisibility(final int visibility) {
        int newvisibility = visibility;
        if (getContext() instanceof EmulationActivity) {
            EmulationActivity ea = (EmulationActivity) getContext();
            newvisibility = Integer.parseInt(ea.getCurrentUseropts().getStringValue(
                    ea.getResources().getString(R.string.useropt_show_statusbar), "0"));

        }

        if (getVisibility() != newvisibility) {
            super.setVisibility(newvisibility);
        }
    }
    private static final int UPDATE_DELAY = 200;
    void setJoysticks(final List<Joystick> availableJoysticks) {
        mActiveJoystickDrawable.clear();
        ViewGroup joystickView = findViewById(R.id.joysticks);
        joystickView.removeAllViews();
        Drawable d = ResourcesCompat.getDrawable(getResources(), R.drawable.joystickinfo,
                getContext().getTheme());
        if (d != null) {
            int iconwidth = d.getIntrinsicWidth();
            int iconheight = d.getIntrinsicHeight();
            Joystick[] sortedList = new Joystick[availableJoysticks.size()];
            availableJoysticks.toArray(sortedList);
            Arrays.sort(sortedList);
            for (Joystick joy : sortedList) {

                if (joy.getPortnumber() != Joystick.PORT_NOT_CONNECTED
                        && !mActiveJoystickDrawable.containsKey(joy.getPortnumber())) {
                    ImageButton i = new ImageButton(getContext());
                    i.setEnabled(false);
                    i.setImageResource(R.drawable.joystickinfo);
                    i.setLayoutParams(new LayoutParams(
                            mBorderPixels + iconwidth, mBorderPixels + iconheight));
                    i.setBackgroundColor(getResources().getColor(android.R.color.transparent));
                    mActiveJoystickDrawable.put(joy.getPortnumber(), i);
                    joystickView.addView(i);
                    onJoystickChanged(joy.getId(), joy.getCurrentDeviceType(),
                            joy.getPortnumber(),
                            joy.getValueType() == Joystick.ValueType.POSITION,
                            0f, 0f,
                            Joystick.NO_PRESSED_BUTTON_KEYS);
                    if (joy instanceof MouseController) {
                        mLastmodificationTime.put(joy.getPortnumber(), System.currentTimeMillis());
                    } else {
                        mLastmodificationTime.put(joy.getPortnumber(), 0L);
                    }
                }
            }
        }

        mRunnable = new Runnable() {
            @Override
            public void run() {
                if (this == mRunnable) {
                    postDelayed(this, UPDATE_DELAY);
                    long now = System.currentTimeMillis();
                    int inactive = 0;
                    for (int port : mLastmodificationTime.keySet()) {
                        Long time = mLastmodificationTime.get(port);
                        if (time != null && time > 0 && time < now - UPDATE_DELAY) {
                            ImageButton b = mActiveJoystickDrawable.get(port);
                            LayerDrawable ld = null;
                            if (b != null) {
                                ld = (LayerDrawable) b.getDrawable();
                            }
                            if (ld != null) {
                                if (inactive == 0) {
                                    inactive = getResources().getInteger(
                                            R.integer.alpha_status_inactive);
                                }
                                ld.findDrawableByLayerId(R.id.status_n).setAlpha(inactive);
                                ld.findDrawableByLayerId(R.id.status_s).setAlpha(inactive);
                                ld.findDrawableByLayerId(R.id.status_e).setAlpha(inactive);
                                ld.findDrawableByLayerId(R.id.status_w).setAlpha(inactive);
                            }
                        }
                    }
                }
            }
        };
        mRunnable.run();

        findViewById(R.id.tv_joysticks).setVisibility(
                !mActiveJoystickDrawable.isEmpty() ? View.VISIBLE : View.GONE);
        setVisibleIfNotEmpty();
    }

    @Override
    public void updateDiskdriveList(final EmulationViewModel viewModel) {
        final List<?> diskdrives = new ArrayList<>(viewModel.getDiskdriveList());

        mEmulationUi.runOnUiThread(() -> {
            ViewGroup diskView = findViewById(R.id.drives);
            diskView.removeAllViews();
            mDiskDriveViews.clear();
            mDiskdrives.clear();
            LayoutParams lp = new LayoutParams(
                    LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
            lp.setMargins(mBorderPixels, 0, mBorderPixels, 0);
            for (Object drive : diskdrives) {

                TextView t = new TextView(getContext());
                try {
                    t.setText(String.valueOf(drive));
                } catch (Exception e) {
                    Log.v("updateDiskdriveList", "Exception", e);
                }


                t.setLayoutParams(lp);
                t.setGravity(Gravity.CENTER);
                t.setTextColor(getResources().getColor(R.color.status));
                mDiskDriveViews.put(drive, t);
                mDiskdrives.add(drive);
                updateDiskDriveState(drive, viewModel.getDiskdriveState(drive));
                diskView.addView(t);
            }
            findViewById(R.id.tv_drives).setVisibility(
                    mDiskDriveViews.isEmpty() ? View.GONE : View.VISIBLE);
            //updateDiskdriveList(mDiskdrives);
            setVisibleIfNotEmpty();
        });


    }

    void setVisibleIfNotEmpty() {
        if (mDiskdrives.isEmpty()
                && mTapedrives.isEmpty()
                && mActiveJoystickDrawable.isEmpty()
                && findViewById(R.id.tv_monitor).getVisibility() != View.VISIBLE) {
            setVisibility(View.GONE);
        } else {
            setVisibility(View.VISIBLE);
        }


    }
    @Override
    public void setDiskdriveState(final Object drive, final boolean active) {
        mEmulationUi.runOnUiThread(() -> updateDiskDriveState(drive, active));
    }

    private void updateDiskDriveState(final Object drive, final boolean active) {
        if (mDiskDriveViews.containsKey(drive)) {
            TextView tv = mDiskDriveViews.get(drive);
            if (tv != null) {
                tv.setAlpha(active ? mAlphaActive : mAlphaInactive);
            }
        }
    }

    void setOnTapeClickListener(final View.OnClickListener l) {
        findViewById(R.id.bn_tape).setOnClickListener(l);
    }
    private final List<Object> mTapedrives = new ArrayList<>();
    @Override
    public void updateTapedriveList(final List<?> tapedrives) {
        mTapedrives.clear();
        mTapedrives.addAll(tapedrives);
        findViewById(R.id.bn_tape).setVisibility(tapedrives.isEmpty() ? View.GONE : View.VISIBLE);
    }


    static final Map<TapedriveState, Integer> STATE_RESSOURCE_ID =
            new LinkedHashMap<TapedriveState, Integer>() {{
                put(TapedriveState.STOP, R.drawable.ic_media_stop);
                put(TapedriveState.START, R.drawable.ic_media_play);
                put(TapedriveState.FORWARD, R.drawable.ic_media_ff);
                put(TapedriveState.REWIND, R.drawable.ic_media_rew);
                put(TapedriveState.RECORD, R.drawable.ic_media_record);
            }};
    @Override
    public void setTapedriveState(final Object tape, final TapedriveState state) {
        //noinspection ConstantConditions
        ((ImageView) findViewById(R.id.iv_tape_status))
                .setImageResource(STATE_RESSOURCE_ID.get(state));
        findViewById(R.id.iv_tape_status).setTag(state);
    }

    @Override
    public void setTapedriveMotor(final Object tape, final boolean active) {
        final float alphaActive = getResources().getInteger(R.integer.alpha_status_active);
        final float alphaInactive = getResources().getInteger(R.integer.alpha_status_inactive);

        findViewById(R.id.tv_tape_counter).setAlpha((active ? alphaActive : alphaInactive) / 255f);
    }
    private static final int FAKE_TAPECOUNTER = 1000;
    @Override
    public void setTapedriveCounter(final Object tape, final int tapeCount) {
        ((TextView) findViewById(R.id.tv_tape_counter)).setText(String.valueOf(tapeCount));
        findViewById(R.id.tv_tape_counter).setVisibility(
                tapeCount < FAKE_TAPECOUNTER ? View.VISIBLE : View.GONE);
        setVisibleIfNotEmpty();
    }


    private EmulationUi mEmulationUi = null;

}



