//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.annotation.SuppressLint;
import android.content.Context;
import android.util.AttributeSet;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.LinearLayout;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

public class SoftkeysLinearLayout extends LinearLayout {
    private Runnable mRunOnVisibleChanged = null;
    private final Map<Object, Boolean> mHiddenWithKeyboard = new LinkedHashMap<>();

    /**
     * Simple constructor to use when creating a SoftkeysLinearLayout from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public SoftkeysLinearLayout(final Context context) {
        super(context);
        init();
    }
    /**
     * Constructor that is called when inflating a SoftkeysLinearLayout from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public SoftkeysLinearLayout(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init();
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public SoftkeysLinearLayout(final Context context, final AttributeSet attrs,
                                final int defStyle) {
        super(context, attrs, defStyle);
        init();
    }
    private final List<String> mPressedButtons = new LinkedList<>();
    /** @noinspection unused*/
    private void init() {
        //setOrientation(HORIZONTAL);
    }

    @Override
    public final void setVisibility(final int visibility) {
        if (getVisibility() != visibility) {
            super.setVisibility(visibility);
            if (mRunOnVisibleChanged != null) {
                mRunOnVisibleChanged.run();
            }
        }
    }

    private void update() {
        setVisibility(getChildCount() > 0 ? View.VISIBLE : View.GONE);
    }
    @SuppressLint("ClickableViewAccessibility")
    final void createSoftkey(final String id, final String text, final boolean enabled,
                       final int keyboardVisibility,
                       @NonNull final Emulation.SoftkeyFunctions functions,
                             @Nullable final Runnable runAfterPress,
                       @Nullable final Runnable runAfterRelease) {
        LayoutInflater inflater
                = (LayoutInflater) getContext().getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        @SuppressLint("InflateParams")
        Button b = (Button) inflater.inflate(R.layout.view_softkey_button, null);
        b.setText(text);
        b.setTag(id);
        b.setEnabled(enabled);
        b.setFocusable(b.isEnabled());
        b.setOnKeyListener((v, keycode, keyEvent) -> {
            if ((EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(keycode)
                    || EmulationUi.JOYSTICK_SELECT_BUTTONS.contains(keycode))) {
                if (keyEvent.getAction() == KeyEvent.ACTION_UP) {
                    mPressedButtons.remove(id);
                    functions.releaseKey(id);
                    if (runAfterRelease != null) {
                        runAfterRelease.run();
                    }

                } else {
                    mPressedButtons.add(id);
                    functions.pressKey(id);
                    if (runAfterPress != null) {
                        runAfterPress.run();
                    }

                }
                return true;
            }
            if (EmulationUi.JOYSTICK_BACK_BUTTONS.contains(keycode)) {
                if (keyEvent.getAction() == KeyEvent.ACTION_UP) {
                    if (runAfterRelease != null) {
                        runAfterRelease.run();
                    }
                }
            }
            return false;
        });
        b.setOnTouchListener((view, motionEvent) -> {
            switch (motionEvent.getAction()) {
                case MotionEvent.ACTION_DOWN:
                case MotionEvent.ACTION_POINTER_DOWN:
                    mPressedButtons.add(id);
                    functions.pressKey(id);
                    break;
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_POINTER_UP:
                    mPressedButtons.remove(id);
                    functions.releaseKey(id);
                    if (runAfterRelease != null) {
                        runAfterRelease.run();
                    }

                    break;
                default:
                    break;
            }
            if (view.isInTouchMode()) {
                view.post(view::clearFocus);
            }
            return false;
        });
        ViewGroup root = findViewById(R.id.softkeys);
        root.addView(b);
        if (keyboardVisibility != View.VISIBLE) {
            update();
        }

    }
    final void destroySoftkey(final String id, final Emulation.SoftkeyFunctions functions) {
        ViewGroup root = findViewById(R.id.softkeys);
        if (mPressedButtons.contains(id)) {
            functions.releaseKey(id);
            mPressedButtons.remove(id);
        }
        int toRemove = View.NO_ID;
        for (int i = 0; i < root.getChildCount(); i++) {
            View v = root.getChildAt(i);
            if (v.getTag() != null) {
                if (v.getTag().toString().equals(id)) {
                    toRemove = i;
                }
            }
        }
        if (toRemove != View.NO_ID) {
            root.removeViewAt(toRemove);
            root.requestLayout();
        }
        update();
    }
    final void enableSoftkey(final String key, final boolean enable) {
        ViewGroup root = findViewById(R.id.softkeys);
        for (int i = 0; i < root.getChildCount(); i++) {
            View v = root.getChildAt(i);
            if (v.getTag() != null) {
                if (v.getTag().toString().equals(key)) {
                    root.getChildAt(i).setEnabled(enable);
                    root.getChildAt(i).setFocusable(enable);
                }
            }
        }
    }
    final void setOnVisibilityChangedListener(final Runnable r) {
        mRunOnVisibleChanged = r;
    }
    final void onKeyboardVisibleChanged(final boolean keyboardVisible) {
        setVisibility(isSoftkeyVisible(keyboardVisible));

    }

    @SuppressLint({"ClickableViewAccessibility", "InflateParams"})
    final void createSoftkey(final SoftkeyConfigView.Softkey key,
                              final Emulation.SoftkeyFunctions functions,
                             @Nullable final Runnable runAfterPress,
                              @Nullable final Runnable runAfterRelease) {
        Button b;
        View existing = findViewWithTag(key.getId());
        if (existing instanceof Button) {
            b = (Button) existing;
        } else {
            b = (Button) ((LayoutInflater) getContext().getSystemService(
                    Context.LAYOUT_INFLATER_SERVICE))
                    .inflate(R.layout.view_softkey_button, null);
        }
        b.setText(key.getLabel());
        b.setTag(key.getId());
        b.setEnabled(true);
        mHiddenWithKeyboard.put(key.getId(), key.isHiddenWithKeyboard());
        b.setFocusable(b.isEnabled());
        b.setOnKeyListener((v, keycode, keyEvent) -> {
            if ((EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(keycode)
                    || EmulationUi.JOYSTICK_SELECT_BUTTONS.contains(keycode))) {
                if (keyEvent.getAction() == KeyEvent.ACTION_UP) {
                    mPressedButtons.remove(key.getId());
                    functions.releaseKey(key.getKeyId());
                    if (key.getModificatorKeyId() != null && !key.getModificatorKeyId().isEmpty()) {
                        functions.releaseKey(key.getModificatorKeyId());
                    }
                    if (runAfterRelease != null) {
                        runAfterRelease.run();
                    }
                } else {
                    mPressedButtons.add(key.getId());
                    if (key.getModificatorKeyId() != null && !key.getModificatorKeyId().isEmpty()) {
                        functions.pressKey(key.getModificatorKeyId());
                    }
                    functions.pressKey(key.getKeyId());
                    if (runAfterPress != null) {
                        runAfterPress.run();
                    }

                }
                return true;
            }
            if (EmulationUi.JOYSTICK_BACK_BUTTONS.contains(keycode)) {
                if (keyEvent.getAction() == KeyEvent.ACTION_UP) {
                    if (runAfterRelease != null) {
                        runAfterRelease.run();
                    }
                }
            }
            return false;
        });
        b.setOnTouchListener((view, motionEvent) -> {
            switch (motionEvent.getAction()) {
                case MotionEvent.ACTION_DOWN:
                case MotionEvent.ACTION_POINTER_DOWN:
                    if (key.getModificatorKeyId() != null && !key.getModificatorKeyId().isEmpty()) {
                        functions.pressKey(key.getModificatorKeyId());
                    }
                    mPressedButtons.add(key.getId());
                    functions.pressKey(key.getKeyId());
                    if (runAfterPress != null) {
                        runAfterPress.run();
                    }

                    break;
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_POINTER_UP:
                    mPressedButtons.remove(key.getId());
                    functions.releaseKey(key.getKeyId());
                    if (key.getModificatorKeyId() != null && !key.getModificatorKeyId().isEmpty()) {
                        functions.releaseKey(key.getModificatorKeyId());
                    }
                    if (runAfterRelease != null) {
                        runAfterRelease.run();
                    }
                    break;
                default:
                    break;
            }
            if (view.isInTouchMode()) {
                view.post(view::clearFocus);
            }
            return false;
        });
        ViewGroup root = findViewById(R.id.softkeys);
        root.addView(b);

    }
    private final Map<Object, Boolean> mHiddenByGamepad = new HashMap<>();
    final int isSoftkeyVisible(final boolean keyboardVisible) {
        int ret = View.GONE;
        for (int i = 0; i < getChildCount(); i++) {
            View v = getChildAt(i);
            boolean visible = !keyboardVisible
                    || (mHiddenWithKeyboard.containsKey(v.getTag())
                        && Boolean.FALSE.equals(mHiddenWithKeyboard.get(v.getTag())));
            if (visible) {
                if (mHiddenByGamepad.containsKey(v.getTag())) {
                    visible = Boolean.FALSE.equals(mHiddenByGamepad.get(v.getTag()));
                }
            }
            v.setVisibility(visible ? View.VISIBLE : View.GONE);
            if (visible) {
                ret = View.VISIBLE;
            }
        }
        return ret;
    }

    final void setHiddenByGamepad(final String id, final boolean hide) {
        mHiddenByGamepad.put(id, hide);
    }
}
