//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;


import static android.graphics.Bitmap.CompressFormat.PNG;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RadioButton;
import android.widget.RadioGroup;
import android.widget.SeekBar;
import android.widget.TableLayout;
import android.widget.TableRow;
import android.widget.TextView;

import androidx.activity.result.contract.ActivityResultContract;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.lifecycle.ViewModel;
import androidx.lifecycle.ViewModelProvider;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import de.rainerhock.eightbitwonders.EmulationConfiguration.PreferredJoystickType;

public final class ShareEmulationActivity extends BaseActivity {

    static final String VIRTUAL_WHEELJOYSTICK = "wheeljoystick";
    static final String CONNECTED_JOYSTICKS = "connectedjoysticks";

    public static final class ShareContract
            extends ActivityResultContract<Intent, Boolean> {

        @NonNull
        @Override
        public Intent createIntent(final @NonNull Context context, final Intent intent) {
            return intent;
        }

        @Override
        public Boolean parseResult(final int result, final @Nullable Intent intent) {
            return result == RESULT_OK;
        }
    }
    private static final String TAG = ShareEmulationActivity.class.getSimpleName();
    private static final int FULL_QUALITY = 100;

    public static final class ExportSettingsViewModel extends ViewModel {
        private final Map<Integer, String> mJoysticknames = new HashMap<>();
        private final ArrayList<Integer> mConnectedjoysticks = new ArrayList<>();
        private final ArrayList<Integer> mRequiredJoysticks = new ArrayList<>();
        private final Map<Integer, Emulation.InputDeviceType> mJoystickInputDevicetypes
                = new HashMap<>();
        private String mUseroptsYaml = null;
        private EmulationConfiguration mConfiguration = null;
        private EmulationConfiguration getConfiguration() {
            return mConfiguration;
        }
        private boolean mLockSettings = false;
        private EmulationConfiguration.Orientation mOrientation;
        private String mEmulatorId = "";
        private String mName = "";
        private String mURL = null;
        private byte[] mBitmapdata = null;
        private  PreferredJoystickType mJoysticktype = PreferredJoystickType.DIRECTIONAL;
        private List<Uri> mAttachedFiles = null;
        private List<String> mVisibleFiles;
        private final List<Uri> mAdditionalFiles = new ArrayList<>();
        private boolean mKeyboard = false;
        private final Map<String, String> mEmuProperties = new LinkedHashMap<>();
        private int mElapsedMiliseconds = 0;
        private Map<String, String> mInludeSaveStates = null;
    }
    private ExportSettingsViewModel mViewModel;
    private  ExportSettingsViewModel getViewModel() {
        if (mViewModel == null) {
            mViewModel = new ViewModelProvider(this).get(ExportSettingsViewModel.class);
            EmulationConfiguration conf = (EmulationConfiguration)
                    Objects.requireNonNull(
                            getIntent().getExtras()).getSerializable("configuration");
            mViewModel.mConfiguration = conf;
            if (conf != null) {

                mViewModel.mEmulatorId = conf.getEmulatorId();
                mViewModel.mOrientation = getEmulationScreenOrientation(this, conf);
                mViewModel.mKeyboard =
                        getIntent().getExtras().getBoolean("hardwarekeyboard", false)
                                || getIntent().getExtras().getBoolean("keyboardvisible", false);
                mViewModel.mBitmapdata = getIntent().getExtras().getByteArray("bitmap");
                if (getIntent().getBooleanExtra(VIRTUAL_WHEELJOYSTICK, false)) {
                    mViewModel.mJoysticktype = PreferredJoystickType.WHEEL;
                } else {
                    mViewModel.mJoysticktype = PreferredJoystickType.DIRECTIONAL;
                }
                @SuppressWarnings("unchecked")
                Map<String, String> emuProperties = (Map<String, String>)
                        getIntent().getSerializableExtra("emu_props");
                if (emuProperties != null) {
                    mViewModel.mEmuProperties.putAll(emuProperties);
                }
                if (getIntent().getSerializableExtra("attached_files") != null) {
                    mViewModel.mAttachedFiles = new ArrayList<>();
                    //noinspection unchecked
                    mViewModel.mAttachedFiles.addAll((List<Uri>) Objects.requireNonNull(
                            getIntent().getSerializableExtra("attached_files")));
                    mViewModel.mVisibleFiles = new ArrayList<>();
                    //noinspection unchecked
                    mViewModel.mVisibleFiles.addAll((List<String>) Objects.requireNonNull(
                            getIntent().getSerializableExtra("visible_files")));


                }
                @SuppressWarnings("unchecked")
                Map<Integer, String> joysticknames = (Map<Integer, String>)
                        getIntent().getSerializableExtra("joysticknames");
                if (joysticknames != null) {
                    mViewModel.mJoysticknames.putAll(joysticknames);
                }
                @SuppressWarnings("unchecked")
                Map<Integer, Emulation.InputDeviceType> connectedJoysticks =
                        (Map<Integer, Emulation.InputDeviceType>) getIntent()
                                .getSerializableExtra(CONNECTED_JOYSTICKS);
                if (connectedJoysticks != null) {
                    mViewModel.mConnectedjoysticks.addAll(connectedJoysticks.keySet());
                    mViewModel.mJoystickInputDevicetypes.putAll(connectedJoysticks);
                }


                for (Emulation.InputDeviceType idt : Arrays.asList(
                        Emulation.InputDeviceType.MOUSE, Emulation.InputDeviceType.PADDLE)) {
                    if (mViewModel.mJoystickInputDevicetypes.containsValue(idt)) {
                        findViewById(R.id.tv_warn_analog_input).setVisibility(View.VISIBLE);
                        break;
                    }
                }
            }
        }
        return mViewModel;
    }

    @Override
    protected void onSaveInstanceState(final @NonNull Bundle outState) {
        outState.putBoolean("saved", true);
        super.onSaveInstanceState(outState);
    }

    @Override
    public void onCreate(@Nullable final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_share_emulation);
        ExportSettingsViewModel viewModel = getViewModel();
        EditText etName = findViewById(R.id.et_name);
        etName.setText(viewModel.mName);
        etName.setSelection(etName.getText().length());
        findViewById(R.id.et_name).requestFocus();
        findViewById(R.id.bn_share).setEnabled(!viewModel.mName.isEmpty());
        etName.addTextChangedListener(createWatcher(s -> {
            findViewById(R.id.bn_share).setEnabled(!s.isEmpty());
            viewModel.mName = s;
        }));
        EditText etUrl = findViewById(R.id.et_url);
        if (viewModel.mURL != null) {
            etUrl.setText(viewModel.mURL);
        }
        etUrl.addTextChangedListener(createWatcher(s -> viewModel.mURL = s));
        Bitmap bmp = BitmapFactory.decodeByteArray(viewModel.mBitmapdata,
                0, viewModel.mBitmapdata.length);
        final Emulation.TimeMachineFunctions tmf;
        Emulation.PackCurrentStateFunctions pcsf = Emulationslist.getCurrentEmulation()
                .getPackCurrentStateFunctions();
        if (pcsf != null) {
            tmf = Emulationslist.getCurrentEmulation().getTimeMachineFunctions();
        } else {
            tmf = null;
        }
        ViewGroup vgMoment = findViewById(R.id.choose_moment);
        if (tmf != null && !getIntent().getBooleanExtra("no_timemachine", false)) {
            EmulationDialogFactory.initTimemachineSlider(vgMoment, bmp, tmf);
            ViewGroup.LayoutParams lp = vgMoment.getLayoutParams();
            lp.width = ViewGroup.LayoutParams.MATCH_PARENT;
            vgMoment.setLayoutParams(lp);
            findViewById(R.id.iv_screenshot).setVisibility(View.GONE);
        } else {
            findViewById(R.id.choose_moment).setVisibility(View.GONE);
            ((ImageView) findViewById(R.id.iv_screenshot)).setImageBitmap(
                    bmp);
        }

        ((CheckBox) findViewById(R.id.cb_kbd_required)).setOnCheckedChangeListener(
                (button, b) -> mViewModel.mKeyboard = b);
        ((CheckBox) findViewById(R.id.cb_landscape)).setOnCheckedChangeListener((button, b)
                -> mViewModel.mOrientation = b ? EmulationConfiguration.Orientation.LANDSCAPE
                : EmulationConfiguration.Orientation.DEFAULT);
        ((CheckBox) findViewById(R.id.cb_kbd_required)).setChecked(mViewModel.mKeyboard);
        LayoutInflater inflater = LayoutInflater.from(this);
        for (int port : mViewModel.mJoysticknames.keySet()) {
            @SuppressLint("InflateParams")
            View rowView = inflater.inflate(R.layout.view_joystick_requirement,
                    null);
            TextView tv = rowView.findViewById(R.id.tv_label);
            tv.setText(mViewModel.mJoysticknames.get(port));
            RadioGroup rg = rowView.findViewById(R.id.rg_option);
            int rbId;
            if (mViewModel.mConnectedjoysticks.contains(port)) {
                rbId = R.id.rb_optional;
            } else {
                rbId = R.id.rb_unused;
            }
            ((RadioButton) rg.findViewById(rbId)).setChecked(true);
            rg.setOnCheckedChangeListener((radioGroup, i) -> {
                final int unused = R.id.rb_unused;
                final int optional = R.id.rb_optional;
                final int required = R.id.rb_required;
                if (i == unused) {
                    if (mViewModel.mRequiredJoysticks.contains(port)) {
                        mViewModel.mRequiredJoysticks.remove(Integer.valueOf(port));
                    }
                    if (mViewModel.mConnectedjoysticks.contains(port)) {
                        mViewModel.mConnectedjoysticks.remove(Integer.valueOf(port));
                    }
                } else if (i == optional) {
                    if (!mViewModel.mConnectedjoysticks.contains(port)) {
                        mViewModel.mConnectedjoysticks.add(port);
                    }
                    if (mViewModel.mRequiredJoysticks.contains(port)) {
                        mViewModel.mRequiredJoysticks.remove(Integer.valueOf(port));
                    }
                } else if (i == required) {
                    if (!mViewModel.mConnectedjoysticks.contains(port)) {
                        mViewModel.mConnectedjoysticks.add(port);
                    }
                    if (!mViewModel.mRequiredJoysticks.contains(port)) {
                        mViewModel.mRequiredJoysticks.add(port);
                    }
                }
            });
            ((LinearLayout) findViewById(R.id.lv_joysticks)).addView(rowView);
        }
        updateAdditionalImages();
        CheckBox cbLock = findViewById(R.id.cb_lock_settings);
        cbLock.setChecked(viewModel.mLockSettings);
        cbLock.setOnCheckedChangeListener((view, checked) -> viewModel.mLockSettings = checked);
        String rootfolder = getIntent().getStringExtra("exportable_snapshotdata");
        if (rootfolder != null) {
            Map<String, String> m = new HashMap<>();
            addSavestatesForZip(new File(rootfolder), ".", m);
            if (!m.isEmpty()) {
                viewModel.mInludeSaveStates = m;
                CheckBox cbShareSavestates = findViewById(R.id.cb_share_savestates);
                cbShareSavestates.setVisibility(View.VISIBLE);
                cbShareSavestates.setOnCheckedChangeListener((view, checked)
                        -> viewModel.mInludeSaveStates = checked ? m : null);
            }
            Useropts opts = new Useropts();
            opts.setCurrentEmulation(this, mViewModel.mConfiguration.getEmulatorId(),
                    mViewModel.mConfiguration.getId());
            if (!opts.getStringKeys().isEmpty() || !opts.getIntegerKeys().isEmpty()
                    || !opts.getBooleanKeys().isEmpty()) {
                StringWriter s = new StringWriter();
                opts.yamlify(Useropts.Scope.CONFIGURATION, s);
                CheckBox cbShareUseropts = findViewById(R.id.cb_share_useropts);
                cbShareUseropts.setVisibility(View.VISIBLE);
                cbShareUseropts.setChecked(true);
                cbShareUseropts.setOnCheckedChangeListener((view, checked)
                        -> viewModel.mUseroptsYaml = checked ? s.toString() : null);
                mViewModel.mUseroptsYaml = s.toString();
            } else {
                mViewModel.mUseroptsYaml = null;
            }
        }
        findViewById(R.id.bn_share).setOnClickListener(view -> share(vgMoment, pcsf, tmf));
    }

    private void share(final ViewGroup vgMoment,
                       final Emulation.PackCurrentStateFunctions pcsf,
                       final Emulation.TimeMachineFunctions tmf) {
        if (findViewById(R.id.choose_moment).getVisibility() == View.VISIBLE) {
            SeekBar sb = vgMoment.findViewById(R.id.sb_seekbar);
            mViewModel.mElapsedMiliseconds = sb.getProgress();
            if (mViewModel.mElapsedMiliseconds > 0 && pcsf != null && tmf != null) {
                Bitmap b = tmf.getScreenPreview(mViewModel.mElapsedMiliseconds);
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                b.compress(PNG, FULL_QUALITY, baos);
                mViewModel.mBitmapdata = baos.toByteArray();
                Runnable r = pcsf.getDumpContentsRunnable(mViewModel.mElapsedMiliseconds,
                        getViewModel().getConfiguration());
                if (r != null) {
                    r.run();
                }

            }
        } else {
            mViewModel.mElapsedMiliseconds = 0;
        }

        if (getEditTexts()) {
            File f = null;
            try {
                f = createPackage(mViewModel);
            } catch (IOException e) {
                Log.e(TAG, "createPackage threw Exception", e);
                showErrorDialog(getResources().getString(R.string.share),
                        getResources().getString(R.string.exception_occured));
            }
            shareFile(f, mViewModel.mName);
        }
    }

    static void addSavestatesForZip(final File rootfolder, final String currentfolder,
                                    final Map<String, String> map) {
        File d = new File(rootfolder.getAbsolutePath() + "/" + currentfolder);
        File[] files = d.listFiles();
        if (files != null) {
            for (File f : files) {
                if (f.isDirectory()) {
                    addSavestatesForZip(rootfolder, currentfolder + "/" + f.getName(), map);
                }
                if (f.isFile()) {
                    map.put(currentfolder + "/" + f.getName(), f.getAbsolutePath());
                }
            }
        }

    }

    private void addImage(final String s, final View.OnClickListener listener) {
        TableLayout parent = findViewById(R.id.tl_additional_images);
        Uri uri = Uri.parse(s);
        LayoutInflater.from(this).inflate(R.layout.fragment_additional_file, parent);
        TableRow rowView = parent.findViewById(R.id.row_additional_file);
        TextView tv = rowView.findViewById(R.id.tv_filename);
        tv.setText(Uri.decode(FileUtil.getFileName(this, uri)));
        if (listener == null) {
            rowView.findViewById(R.id.bn_remove).setVisibility(View.GONE);
        } else {
            rowView.findViewById(R.id.bn_remove).setOnClickListener(listener);
        }
        rowView.setId(View.NO_ID);
    }
    private int mInitialChildcount = 0;
    private void updateAdditionalImages() {
        TableLayout parent = findViewById(R.id.tl_additional_images);
        parent.findViewById(R.id.bn_add).setOnClickListener(view
                -> openFile(false));
        if (mInitialChildcount == 0) {
            mInitialChildcount = parent.getChildCount();
        }
        for (int i = parent.getChildCount() - 1; i >= mInitialChildcount; i--) {
            parent.removeViewAt(i);
        }
        for (String s: mViewModel.mVisibleFiles) {
            addImage(s, null);
        }
        for (Uri uri:mViewModel.mAdditionalFiles) {
            addImage(Uri.decode(FileUtil.getFileName(this, uri)), view -> {
                mViewModel.mAdditionalFiles.remove(uri);

                findViewById(R.id.bn_add).requestFocus();
                updateAdditionalImages();
            });
        }
    }

    private boolean getEditTexts() {
        boolean nameOk;
        boolean urlOk;
        EditText etURL = findViewById(R.id.et_url);
        String s = etURL.getText().toString();
        try {
            if (s.isEmpty()) {
                mViewModel.mURL = null;
            } else {
                mViewModel.mURL = new URL(s).toString();
            }
            findViewById(R.id.tv_error_url).setVisibility(View.GONE);
            urlOk = true;
        } catch (MalformedURLException e) {
            findViewById(R.id.tv_error_url).setVisibility(View.VISIBLE);
            mViewModel.mURL = null;
            etURL.requestFocus();
            urlOk = false;
        }
        if (mViewModel.mName.isEmpty()) {
            findViewById(R.id.tv_error_name).setVisibility(View.VISIBLE);
            findViewById(R.id.et_name).requestFocus();
            nameOk = false;
        } else {
            findViewById(R.id.tv_error_name).setVisibility(View.GONE);
            nameOk = true;
        }
        findViewById(R.id.tv_all_errors)
                .setVisibility(!nameOk && !urlOk ? View.VISIBLE : View.GONE);

        return nameOk && urlOk;
    }
    @SuppressLint("DefaultLocale")
    private @NonNull  File createPackage(final ExportSettingsViewModel data) throws IOException {
        @SuppressWarnings("RegExpRedundantEscape")
        final List<String> entriesWritten = new LinkedList<>();
        //noinspection RegExpRedundantEscape
        File f = new File(getCacheDir() + File.separator
                + data.mName.trim().replaceAll("[^a-zA-Z\\d\\.\\-]", "_")
                + ".8bw");
        Log.v(TAG, String.format("File ist now %s", f));
        //try {
            @SuppressWarnings("IOStreamConstructor")
            ZipOutputStream out = new ZipOutputStream(new FileOutputStream(f));
            ZipEntry e;
            ByteArrayOutputStream baos;
            Properties props;

            e = new ZipEntry("screenshot.png");
            entriesWritten.add("screenshot.png");
            out.putNextEntry(e);
            out.write(mViewModel.mBitmapdata);
            out.closeEntry();

            props = new Properties();
            for (String key : mViewModel.mEmuProperties.keySet()) {
                if (key.equals("__openfiles__")) {
                    String[] values = Objects.requireNonNull(
                            mViewModel.mEmuProperties.get(key)).split(" ");
                    boolean isDrive = true;
                    StringBuilder sb = new StringBuilder();
                    for (String v : values) {
                        if (isDrive) {
                            sb.append(v).append(" ");
                        } else {
                            sb.append(Uri.encode(FileUtil.getFileName(this, Uri.parse(v))));
                        }
                        isDrive = !isDrive;
                    }
                    props.put(key, sb.toString().trim());
                } else {
                    props.put(key, mViewModel.mEmuProperties.get(key));
                }
            }
            baos = new ByteArrayOutputStream();
            props.store(baos, "");
            e = new ZipEntry("emu_properties");
            entriesWritten.add("emu_properties");
            out.putNextEntry(e);
            out.write(baos.toByteArray());
            out.closeEntry();

            props = new Properties();
            props.put("name", mViewModel.mName);
            props.put("emulation", mViewModel.mEmulatorId);
            props.put("id", UUID.randomUUID().toString());
            props.put("image", "screenshot.png");
            props.put("keyboard", mViewModel.mKeyboard ? "1" : "0");
            if (mViewModel.mURL != null) {
                props.put("url", mViewModel.mURL);
            } else {
                if (mViewModel.mConfiguration.getWebAdress() != null) {
                    props.put("url", mViewModel.mConfiguration.getWebAdress().toString());
                }

            }
            if (getIntent().getBooleanExtra("touch_diagonals_locked",  false)) {
                props.put("touchjoystick_diagonals_locked", "1");
            }
            props.put("orientation", mViewModel.mOrientation.toString());
            for (int joystickport : mViewModel.mConnectedjoysticks) {
                props.put(String.format(Locale.getDefault(), "joystick%d", joystickport),
                        mViewModel.mJoysticktype.toString());
                if (mViewModel.mRequiredJoysticks.contains(joystickport)) {
                    props.put(String.format("joystick%d_required", joystickport),
                            "1");
                }
                props.put(String.format("joystick%d_devicetype", joystickport),
                        String.valueOf(mViewModel.mJoystickInputDevicetypes.get(joystickport)));
            }
            for (Uri uri : mViewModel.mAdditionalFiles) {
                props.put(String.format("additional_file%d",
                        mViewModel.mAdditionalFiles.indexOf(uri)),
                        FileUtil.getFileName(this, uri));
            }
            baos = new ByteArrayOutputStream();
            props.store(baos, "");
            e = new ZipEntry("properties");
            entriesWritten.add("properties");
            out.putNextEntry(e);
            out.write(baos.toByteArray());
            out.closeEntry();

            List<Uri> allFiles = new LinkedList<>(mViewModel.mAttachedFiles);
            allFiles.addAll(mViewModel.mAdditionalFiles);
            for (Uri uri : allFiles) {
                InputStream is;
                if (uri.getScheme() != null) {
                    is = getContentResolver().openInputStream(uri);
                } else {
                    //noinspection IOStreamConstructor
                    is = new FileInputStream(uri.getPath());
                }
                if (is != null) {
                    byte[] filedata = new byte[is.available()];
                    //noinspection ResultOfMethodCallIgnored
                    is.read(filedata);
                    is.close();
                    out.putNextEntry(new ZipEntry(FileUtil.getFileName(this, uri)));
                    entriesWritten.add(FileUtil.getFileName(this, uri));
                    out.write(filedata);
                }
                out.closeEntry();
            }
            if (mViewModel.mInludeSaveStates != null) {
                writeSaveStates(mViewModel.mInludeSaveStates, out);
            }
            if (mViewModel.mUseroptsYaml != null) {
                out.putNextEntry(new ZipEntry("__settings__.yml"));
                out.write(mViewModel.mUseroptsYaml.getBytes(StandardCharsets.UTF_8));
                out.closeEntry();
            }
            File dir = mViewModel.mConfiguration.getLocalPath();
            if (dir != null) {
                File[] files = dir.listFiles();
                if (files != null) {
                    for (File file : files) {
                        if (file.isFile() && file.canRead()
                                && !entriesWritten.contains(file.getName())) {
                            //noinspection IOStreamConstructor
                            InputStream is = new FileInputStream(file);
                            byte[] filedata = new byte[is.available()];
                            //noinspection ResultOfMethodCallIgnored
                            is.read(filedata);
                            is.close();
                            out.putNextEntry(new ZipEntry(file.getName()));
                            out.write(filedata);
                            out.closeEntry();
                        }
                    }
                }
            }
            out.close();
            return f;

    }
    static void writeSaveStates(final Map<String, String> savestates, final ZipOutputStream out) {
        try {
            if (savestates != null) {
                for (String relpath : savestates.keySet()) {
                    if (savestates.get(relpath) != null) {
                        FileInputStream fis = new FileInputStream(savestates.get(relpath));
                        byte[] data = new byte[fis.available()];
                        if (fis.read(data) >= 0) {
                            out.putNextEntry(new ZipEntry(relpath
                                    .replaceFirst("./", "__savestates__/")));
                            out.write(data);
                            out.closeEntry();
                        }
                        fis.close();
                    }
                }
            }

        } catch (IOException | ClassCastException cce) {
            // should not happen
        }

    }

    @Override
    protected void onFileOpened(final Uri uri) {
        if (!mViewModel.mAdditionalFiles.contains(uri)) {
            mViewModel.mAdditionalFiles.add(uri);
            updateAdditionalImages();
        }
        findViewById(R.id.scroll).scrollTo(0, findViewById(R.id.bn_add).getBottom());
        findViewById(R.id.bn_add).getParent()
                .requestChildFocus(findViewById(R.id.bn_add), findViewById(R.id.bn_add));
        super.onFileOpened(uri);
    }
}
