package de.rainerhock.eightbitwonders;

import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.util.Log;

import androidx.annotation.NonNull;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

class Rp9Configuration {
    //private final ZipInputStream mZis;
    private final BaseActivity mCtx;
    private final Map<Integer, String> mFloppies = new LinkedHashMap<>();
    private final Map<Integer, String> mTapes = new LinkedHashMap<>();
    private final byte[] mData;
    private String mRamsize = null;
    private String mHomepage = null;
    private String mModel = null;
    private String mEmulatorId = null;
    private String mBitmap = null;
    private String mReadme = null;
    private String mId;
    private final Set<String> mJoysticks = new HashSet<>();
    private final Set<String> mMice = new HashSet<>();
    private static final int HASHPART_SIZE = 10;

    EmulationConfiguration createOneTimeConfiguration() throws Rp9ImportException {
        if (read()) {
            String targetpath = new File(mCtx.getCacheDir(), "streams").getAbsolutePath();
            Properties props = getProperties();
            byte[] data = mZippedData.get(mBitmap);
            Bitmap bmp;
            if (data != null) {
                bmp = BitmapFactory.decodeByteArray(data, 0, data.length);
            } else {
                bmp = null;
            }
            try {
                File f = new File(targetpath);
                if (!f.exists()) {
                    //noinspection ResultOfMethodCallIgnored
                    f.mkdirs();
                }
                writeDataToFolder(targetpath);
                return new ConfigurationFactory.PackagedConfiguration(mEmulatorId, mId, mTitle, bmp,
                        ConfigurationFactory.mapFromProperties(props),
                        ConfigurationFactory.mapFromProperties(
                                getEmuProperties(targetpath, props, mZippedData.keySet())),
                        targetpath,
                        null, false, true, false);
            } catch (IOException e) {
                throw new Rp9ImportException(
                        mCtx.getResources().getString(R.string.import_error), e);
            }

        }
        return null;
    }

    static class Rp9ImportException extends Exception {
        Rp9ImportException(final String message, final Throwable cause) {
            super(message, cause);
        }
    }
    Rp9Configuration(final BaseActivity context, final byte[] data) {
        mData = data;
        mCtx = context;
    }
    private String getXppText(final XmlPullParser xpp) throws XmlPullParserException, IOException {
        String ret = null;
        xpp.next();
        if (xpp.getEventType() == XmlPullParser.TEXT) {
            ret = xpp.getText();
        }

        if (ret == null) {
            Log.v(TAG, String.format("text requested for %s at line %d, but no text found.",
                    xpp.getName(), xpp.getLineNumber()));
        }
        return ret;
    }

    private static class Handlers extends LinkedHashMap<String, XmlElementHandler> { }
    private String mTitle = null;
    private boolean readManifest(final String data) throws Rp9ImportException {
        try {
            XmlPullParserFactory factory = XmlPullParserFactory.newInstance();
            factory.setNamespaceAware(true);
            XmlPullParser xpp = factory.newPullParser();
            xpp.setInput(new StringReader(data.substring(data.indexOf("<"))));

            final  Map<String, XmlElementHandler> applicationHandler
                    = new Handlers() {{

                put("description", p -> parse(p, new Handlers() {{
                    put("title", p -> mTitle = getXppText(p));
                    put("homepage", p -> mHomepage = getXppText(p));
                }}));
                put("extras", p -> parse(xpp, new Handlers() {{
                   put("image", p -> mBitmap = getXppText(p));
                   put("document", p -> mReadme = getXppText(p));
                }}));
                put("configuration", p -> parse(p, new Handlers() {{
                    put("system", p -> {
                        String text = getXppText(p);
                        if (SYSTEM_MAPPING.containsKey(text)) {
                            mEmulatorId = SYSTEM_MAPPING.get(text);
                        }
                        if (MODEL_MAPPING.containsKey(text)) {
                            mModel = MODEL_MAPPING.get(text);
                        }
                    });
                    put("ram", p -> mRamsize = getXppText(p));
                    put("peripheral", p -> {
                        String unit = p.getAttributeValue(null, "unit");
                        String text = getXppText(p);
                        if (unit != null && "input-joystick".equals(text)) {
                            mJoysticks.add(unit);
                        }
                        if (unit != null && "input-mouse".equals(text)) {
                            mMice.add(unit);
                        }

                    });

                }}));
                put("media", p -> parse(p, new Handlers() {{

                    put("floppy", p -> {
                        String priority = p.getAttributeValue(null, "priority");
                        mFloppies.put(Integer.valueOf(priority != null ? priority : "0"),
                                getXppText(p));
                    });
                    put("tape", p -> {
                        String priority = p.getAttributeValue(null, "priority");
                        mTapes.put(Integer.valueOf(priority != null ? priority : "0"),
                                getXppText(p));
                    });
                }}));
            }};
            parse(xpp, applicationHandler);
        } catch (IOException | XmlPullParserException e) {
            throw new Rp9ImportException(mCtx.getResources().getString(R.string.import_error), e);
        }
        return mEmulatorId != null && mTitle != null && mBitmap != null;
    }

    private static final Map<String, String> SYSTEM_MAPPING = new HashMap<String, String>() {{
        put("c-128", "C128");
        put("c-64", "C64");
        put("c-2001", "PET");
        put("c-3032", "PET");
        put("c-4032", "PET");
        put("c-8032", "PET");
        put("c-vic20", "VIC20");
        //put("c-vic1001", "VIC20");

    }};
    private static final Map<String, String> MODEL_MAPPING = new HashMap<String, String>() {{
        put("c-128", "0");
        put("c-64", "0");
        put("c-2001", "0");
        put("c-3032", "3");
        put("c-4032", "5");
        put("c-8032", "6");
        put("c-vic20", "0");
        //put("c-vic1001", "VIC20");

    }};
    private interface XmlElementHandler {
        void handle(XmlPullParser xpp)
                throws XmlPullParserException, Rp9ImportException, IOException;
    }
    private void parse(final XmlPullParser xpp, final Map<String, XmlElementHandler> handlers)
            throws XmlPullParserException, IOException, Rp9ImportException {
        int eventType = xpp.next();
        boolean done = false;
        int depth = xpp.getDepth();
        while (!done) {
            if (eventType == XmlPullParser.START_TAG) {
                for (String key : handlers.keySet()) {
                    if (key.equals(xpp.getName())) {
                        Objects.requireNonNull(handlers.get(key)).handle(xpp);
                    }
                }
            }
            if (eventType == XmlPullParser.END_TAG) {
                if (xpp.getDepth() == depth) {
                    done = true;
                }
            }
            eventType = xpp.next();
        }

    }

    private static final String TAG = Rp9Configuration.class.getSimpleName();
    private void storeAsset(final String targetpath, final String name) throws IOException {
        //noinspection IOStreamConstructor
        OutputStream os =  new FileOutputStream(new File(targetpath, name));
        os.write(mZippedData.get(name));
        os.close();
    }
    private boolean addFirstMedium(final Map<Integer, String> media, final Properties emuProps) {
        if (!media.isEmpty()) {
            int firstfloppy = Integer.MAX_VALUE;
            for (int index : media.keySet()) {
                if (index < firstfloppy) {
                    firstfloppy = index;
                }
            }
            emuProps.put("__start__", media.get(firstfloppy));
            return true;
        }
        return false;
    }
    private void storeConfiguration() throws IOException {
        final String targetpath = mCtx.getApplicationContext().getFilesDir()
                + "/packages/imported/" + mId;
        writeDataToFolder(targetpath);
    }
    private void writeDataToFolder(final String targetpath) throws IOException {

        if (new File(targetpath).exists() || new File(targetpath).mkdirs()) {
            if (mReadme != null) {
                storeAsset(targetpath, mReadme);
            }
            Properties props = getProperties();

            Properties emuProps = getEmuProperties(targetpath, props);
            emuProps.store(new FileWriter(new File(targetpath, "emu_properties")), "");
            props.store(new FileWriter(new File(targetpath, "properties")), "");
            storeAsset(targetpath, mBitmap);
            for (String floppy : mFloppies.values()) {
                storeAsset(targetpath, floppy);
            }
            for (String tape : mTapes.values()) {
                storeAsset(targetpath, tape);
            }
        } else {
            throw new IOException("Cannot create " + targetpath);
        }
    }
    private Properties getEmuProperties(final String targetpath,
                                        final Properties props) {
        return getEmuProperties(targetpath, props, null);
    }
    @NonNull
    private Properties getEmuProperties(final String targetpath,
                                        final Properties props,
                                        final Set<String> makeAbsolule) {
        Properties emuProps = new Properties();
        boolean hasMedia;
        if (addFirstMedium(mFloppies, emuProps)) {
            hasMedia = true;
        } else {
            hasMedia = addFirstMedium(mTapes, emuProps);
        }
        if (hasMedia) {
            InputStream is = mCtx.getResources()
                    .openRawResource(R.raw.speed_up_loading);
            try {
                byte[] data = new byte[is.available()];
                if (is.read(data) > 0) {
                    FileOutputStream os = new FileOutputStream(
                            new File(targetpath, "vice-actions.js"));
                    os.write(data);
                    os.close();
                }
                is.close();

            } catch (IOException e) {
                // That's not good, but okayish.
            }
        }
        int additionalIndex = 0;
        if (mFloppies.size() + mTapes.size() > 1) {
            for (String floppy : mFloppies.values()) {
                props.put(String.format(Locale.getDefault(),
                        "additional_file%d", additionalIndex), floppy);
                additionalIndex++;
            }
            for (String tape : mTapes.values()) {
                props.put(String.format(Locale.getDefault(),
                        "additional_file%d", additionalIndex), tape);
                additionalIndex++;
            }
        }

        emuProps.put("Model", mModel);
        if (mRamsize != null) {
            emuProps.put("ramsize", mRamsize);
        }
        if (makeAbsolule != null) {
            return ConfigurationFactory.createAbsolutePathProperties(
                    emuProps, targetpath, makeAbsolule);
        } else {
            return emuProps;
        }
    }

    @NonNull
    private Properties getProperties() {
        Properties props = new Properties();

        props.put("emulation", mEmulatorId);
        props.put("name", mTitle);
        if (mHomepage != null) {
            props.put("url", mHomepage);
        }
        if (mReadme != null) {
            props.put("readme", mReadme);
        }
        props.put("id", mId);
        props.put("image", mBitmap);
        props.put("orientation", "landscape");
        for (String joy : mJoysticks) {
            props.put(String.format("joystick%s", joy), "directional");
            props.put(String.format("joystick%s_required", joy), "true");

            // MOUSE
        }
        for (String mouse : mMice) {
            props.put(String.format("joystick%s", mouse), "directional");
            props.put(String.format("joystick%s_required", mouse), "true");
            props.put(String.format("joystick%s_devicetype", mouse), "MOUSE");
        }
        return props;
    }

    private boolean checkIntegrity() {
        boolean ret = mTitle != null && mEmulatorId != null && mZippedData.containsKey(mBitmap);
        if (ret) {
            byte[] data = mZippedData.get(mBitmap);
            if (data != null) {
                Bitmap bmp = BitmapFactory.decodeByteArray(data, 0, data.length);
                ret = bmp != null;
            } else {
                ret = false;
            }
        }
        for (String floppy: mFloppies.values()) {
            if (!mZippedData.containsKey(floppy)) {
                return false;
            }
        }
        return ret;
    }
    private final Map<String, byte[]> mZippedData = new HashMap<>();
    void add() throws Rp9ImportException {
        if (read() && checkIntegrity()) {
            try {
                storeConfiguration();
                Intent i = new Intent(
                        "de.rainerhock.eightbitwonders.update");
                i.setPackage("de.rainerhock.eightbitwonders");
                mCtx.sendBroadcast(i);
                mCtx.onContentImported(mTitle);
            } catch (IOException e) {
                throw new Rp9ImportException(
                        mCtx.getResources().getString(R.string.import_error), e);
            }
        }
    }
    boolean validate() {
        try {
            return read();
        } catch (Rp9ImportException e) {
            return false;
        }
    }
    private boolean read() throws Rp9ImportException {
        try {
            ZipInputStream zis = new ZipInputStream(new ByteArrayInputStream(mData));
            boolean hasManifest = false;
            ZipEntry entry = zis.getNextEntry();

            while (entry != null) {
                int count;
                byte[] buffer = new byte[(int) entry.getSize()];
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                while ((count = zis.read(buffer)) >= 0) {
                    baos.write(buffer, 0, count);
                }
                mZippedData.put(entry.getName(), baos.toByteArray());
                if (entry.getName().equals("rp9-manifest.xml")) {

                    hasManifest = readManifest(new String(baos.toByteArray(),
                            StandardCharsets.UTF_8));
                    MessageDigest digest = MessageDigest.getInstance("SHA-256");
                    byte[] hash = digest.digest(baos.toByteArray());
                    StringBuilder hexString = new StringBuilder(2 * hash.length
                            + "rp9-".length());
                    hexString.append("rp9-");
                    for (byte b : hash) {
                        String hex = Integer.toHexString(0xff & b);
                        if (hex.length() == 1) {
                            hexString.append('0');
                        }
                        hexString.append(hex);
                    }

                    String fullId = hexString.toString();
                    mId = String.join("_",
                            fullId.substring(0, HASHPART_SIZE + 4),
                            fullId.substring(fullId.length() - HASHPART_SIZE));
                    if (!hasManifest) {
                        break;
                    }
                }
                entry = zis.getNextEntry();
            }
            return hasManifest;
        } catch (IOException | NoSuchAlgorithmException e) {
            throw new Rp9ImportException(mCtx.getResources().getString(R.string.import_error), e);
        }

    }

}
