//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;


import android.content.res.Resources;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.PopupMenu;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.DialogFragment;

import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

final class MultiplexerJoystick extends AnalogController {
    private final String mDescription;
    private final String mUseroptsKey;
    private final String mUseroptsValueWheel;
    private final String mUseroptsValueStick;
    private final boolean mIsTv;
    private final List<Joystick> mOtherJoysticks;

    static final String DEVICE_SPECIFIC_FRAGMENT = "DEVICE_SPECIFIC_FRAGMENT";

    MultiplexerJoystick(final BaseActivity activity, final List<Joystick> joysticks) {
        Resources res = activity.getResources();
        mDescription = res.getString(R.string.joystick_multiplexer);
        mUseroptsKey = res.getString(R.string.key_multiplex_joystick_virtual_views);
        mUseroptsValueWheel = res.getString(R.string.virtual_views_wheel);
        mUseroptsValueStick = res.getString(R.string.virtual_views_stick);
        mIsTv = activity.isTv();
        mOtherJoysticks = new LinkedList<>(joysticks);
    }
    @Override
    String getId() {
        return "__MULTIPLEX__";
    }

    @Override
    protected boolean showMouseUsageHint() {
        return false;
    }

    @Override
    InputDeviceConfigFragment getPaddleConfigFragment() {
        return getConfigFragment();
    }

    @Override
    InputDeviceConfigFragment getMouseConfigFragment() {
        return getConfigFragment();
    }

    public static class SingleJoystickDialogFragment extends DialogFragment {
        SingleJoystickDialogFragment() {
            super();
        }
        private SettingsController mSettingsController = null;

        private SettingsController getSettingsController() {
            Serializable s = requireArguments().getSerializable(SETTINGSCONTROLLER);
            if (mSettingsController == null && s instanceof SettingsController) {
                mSettingsController = new SettingsController((SettingsController) s);
                mSettingsController.setModifiers(new HashMap<>());
            }
            return mSettingsController;
        }
        @Override
        public View onCreateView(final LayoutInflater inflater, final ViewGroup container,
                                 final Bundle savedInstanceState) {
            return inflater.inflate(R.layout.dialog_singlejoystick,
                    container, false);
        }

        @Override
        public void onViewCreated(final @NonNull View view,
                                  final @Nullable Bundle savedInstanceState) {
            super.onViewCreated(view, savedInstanceState);
            Joystick joy = (Joystick) requireArguments().getSerializable(JOYSTICK);
            if (joy != null) {
                Emulation.InputDeviceType idt = (Emulation.InputDeviceType) requireArguments()
                        .getSerializable(CURRENTDEVICETYPE);
                final InputDeviceConfigFragment f;
                if (idt != null) {
                    switch (idt) {
                        case DSTICK:
                            f = joy.getDstickConfigFragment();
                            break;
                        case PADDLE:
                            if (joy instanceof AnalogController) {
                                f = ((AnalogController) joy).getPaddleConfigFragment();
                            } else {
                                f = null;
                            }
                            break;
                        case MOUSE:
                            if (joy instanceof AnalogController) {
                                f = ((AnalogController) joy).getMouseConfigFragment();
                            } else {
                                f = null;
                            }
                            break;
                        default:
                            f = null;
                    }
                } else {
                    f = null;
                }
                if (f != null) {
                    Bundle b = new Bundle();
                    b.putSerializable(CURRENTDEVICETYPE, idt);
                    f.setArguments(b);
                    SettingsController sc = getSettingsController();
                    f.setSettingsController(sc);
                    getChildFragmentManager().beginTransaction().add(R.id.ll_container, f,
                                    DEVICE_SPECIFIC_FRAGMENT)
                            .commitNow();
                    view.findViewById(R.id.bn_apply).setOnClickListener(view1 -> {
                        sc.apply();
                        dismiss();
                    });

                }
            }
        }
    }
    private static final String SETTINGSCONTROLLER = "SETTINGSCONTROLLER";
    private static class MultiplexSettingsFragment extends InputDeviceConfigFragment {
        MultiplexSettingsFragment() {
            super();
        }
        private PopupMenu mPopupWithJoysticks = null;
        protected void createRealJoysticksPopup(final View v) {
            Button b = v.findViewById(R.id.bn_select);
            SettingsActivity activity = ((SettingsActivity) requireActivity());
            Map<Integer, Joystick> joysticks = new HashMap<>();
            Emulation.InputDeviceType idt = (Emulation.InputDeviceType)  requireArguments()
                    .getSerializable(CURRENTDEVICETYPE);
            b.setOnClickListener(view -> {
                 mPopupWithJoysticks = new PopupMenu(activity, b);
                for (Joystick joy: activity.getAvailableJoysticks()) {
                    if (joy.getSupportedDeviceTypes().contains(idt)) {
                        int id = View.generateViewId();
                        joysticks.put(id, joy);
                        mPopupWithJoysticks.getMenu().add(Menu.NONE, id, Menu.NONE,
                                joy.toString());
                    }
                }
                mPopupWithJoysticks.setOnMenuItemClickListener(menuItem -> {
                    Joystick joy = joysticks.get(menuItem.getItemId());

                    if (joy != null) {
                        DialogFragment dlg = new SingleJoystickDialogFragment();
                        Bundle dlgBundle = new Bundle();
                        dlgBundle.putSerializable(JOYSTICK, joy);
                        dlgBundle.putSerializable(CURRENTDEVICETYPE, idt);
                        dlgBundle.putSerializable(SETTINGSCONTROLLER,
                                activity.getSettingsController());
                        dlg.setArguments(dlgBundle);
                        dlg.showNow(activity.getSupportFragmentManager(), "");
                    }
                    mPopupWithJoysticks.dismiss();
                    return true;
                });
                mPopupWithJoysticks.show();
            });
        }
    }
    public static final class MultiplexAnalogDeviceSettingsFragment
            extends MultiplexSettingsFragment {

        public View onCreateView(final LayoutInflater inflater, final ViewGroup container,
                                 final Bundle savedInstanceState) {
            View ret = inflater.inflate(R.layout.fragment_config_multiplex_joystick_selector,
                    container, false);
            createRealJoysticksPopup(ret);
            return ret;
        }
    }

    private static final String JOYSTICK = "_JOYSTICK_";

    @NonNull
    @Override
    public String toString() {
        return mDescription;
    }

    @Override
    String getHardwareDescriptor() {
        return "__MULTIPLEX__";
    }

    @Override
    int getHardwareId() {
        return MULTIPLEXER_HARDWARE_ID;
    }

    @Override
    @NonNull List<Emulation.InputDeviceType> getSupportedDeviceTypes() {
        Set<Emulation.InputDeviceType> ret = new HashSet<>();
        for (Joystick joy : mOtherJoysticks) {
            ret.addAll(joy.getSupportedDeviceTypes());
        }
        return new LinkedList<>(ret);
    }

    @Override
    boolean canLockDiagonals() {
        return false;
    }

    @Override
    boolean hasSecondaryButton() {
        for (Joystick joy:mOtherJoysticks) {
            if (joy.hasSecondaryButton()) {
                return true;
            }
        }
        return false;
    }

    private static final String SHOW_VIRTUAL_JOYSTICK_STYLE = "_SHOW_VIRTUAL_JOYSTICK_STYLE_";
    public static class MulitplexDstickSettingsFragment extends MultiplexSettingsFragment {
        MulitplexDstickSettingsFragment() {
            super();
        }
        @Nullable
        @Override
        public View onCreateView(final @NonNull LayoutInflater inflater,
                                 final @Nullable ViewGroup container,
                                 final @Nullable Bundle savedInstanceState) {
            View ret = inflater.inflate(R.layout.fragment_config_multiplexer_joystick,
                    container, false);
            ret.findViewById(R.id.ll_virtual_joystick_style).setVisibility(
                    requireArguments().getBoolean(SHOW_VIRTUAL_JOYSTICK_STYLE)
                            ? View.VISIBLE : View.GONE);
            ((SettingsActivity) ret.getContext()).addView(ret);
            createRealJoysticksPopup(ret);
            return ret;
        }
    }

    @Override
    InputDeviceConfigFragment getDstickConfigFragment() {
        return getConfigFragment();
    }
    private InputDeviceConfigFragment getConfigFragment() {
        Bundle b = new Bundle();
        b.putBoolean(SHOW_VIRTUAL_JOYSTICK_STYLE, !mIsTv);
        b.putSerializable(CURRENTDEVICETYPE, getCurrentDeviceType());
        InputDeviceConfigFragment ret = new MulitplexDstickSettingsFragment();
        ret.setArguments(b);
        return ret;
    }

    @Override
    boolean deviceHasButtonWithKeycode(final int keycode) {
        return false;
    }
    EmulationConfiguration.PreferredJoystickType getPreferredVirtualType(
            final EmulationActivity ea) {
        String val = ea.getCurrentUseropts().getStringValue(mUseroptsKey, mUseroptsValueStick);
        if (mUseroptsValueStick.equals(val)) {
            return EmulationConfiguration.PreferredJoystickType.DIRECTIONAL;
        }
        if (mUseroptsValueWheel.equals(val)) {
            return EmulationConfiguration.PreferredJoystickType.WHEEL;
        }
        return null;
    }

}
