//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.content.ContextWrapper;
import android.graphics.Bitmap;
import android.graphics.Point;
import android.graphics.Rect;
import android.opengl.GLSurfaceView;
import android.opengl.GLUtils;
import android.os.Build;
import android.util.AttributeSet;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Display;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.PointerIcon;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ScrollView;

import androidx.annotation.NonNull;
import androidx.core.view.GestureDetectorCompat;


import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.microedition.khronos.egl.EGLConfig;
import javax.microedition.khronos.opengles.GL10;

/**
 * The View to be drawn on on. Emulations don't use it directly.
 */
public final class MonitorGlSurfaceView extends GLSurfaceView implements GLSurfaceView.Renderer,
        EmulationTestInterface {
    private static final String TAG = MonitorGlSurfaceView.class.getSimpleName();
    private int mCanvasWidth = 0;
    private float mCanvasAspectRatio = 0;
    private float mPixelAspectRatio = 0;
    private ScrollView mScrollview = null;
    private boolean mInLayout = true;

    /**
     * Simple constructor to use when creating a MonitorGlSurfaceView from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public MonitorGlSurfaceView(final Context context) {
        super(context);
        init();
    }
    /**
     * Constructor that is called when inflating a MappedSpinner from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public MonitorGlSurfaceView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init();
    }
    private int mViewHeight = 0;
    /** @noinspection unused*/
    private int mViewWidth = 0;
    private final Rect mOldScrollbarRect = new Rect(0, 0, 0, 0);
    private final Rect mNewScrollbarRect = new Rect(0, 0, 0, 0);
    void setScrollView(final @NonNull  ScrollView sv) {
        if (mScrollview != sv) {
            boolean cont = mScrollview == null;

            if (!cont) {
                sv.getGlobalVisibleRect(mNewScrollbarRect);
                mScrollview.getGlobalVisibleRect(mOldScrollbarRect);
                cont = !mNewScrollbarRect.equals(mOldScrollbarRect);
            }
            mScrollview = sv;
            if (cont) {
                mScrollview.getGlobalVisibleRect(mOldScrollbarRect);
                sv.addOnLayoutChangeListener(
                        (v, left, top, right, bottom, oldLeft, oldTop, oldRight, oldBottom) -> {
                            Log.v("LayoutChanged", "scrollview: " + left + ", " + top
                                    + ", " + right + ", " + bottom);
                            mViewWidth = right - left;
                            mViewHeight = bottom - top;
                            calcTransformation();
                        });
            }
        }
    }
    /** @noinspection FieldCanBeLocal*/
    private int[] mTextures;
    private static final int PIXELDPETH = 4;
    private static final float[] VERTEX_COORDINATES = new float[]{
        -1.0f, +1.0f, 0.0f,
        +1.0f, +1.0f, 0.0f,
        -1.0f, -1.0f, 0.0f,
        +1.0f, -1.0f, 0.0f
    };

    private static final float[] TEXTURE_COORDINATES = new float[]{
        0.0f, 0.0f,
        1.0f, 0.0f,
        0.0f, 1.0f,
        1.0f, 1.0f
    };

    private static final Buffer TEXCOORD_BUFFER =
            ByteBuffer.allocateDirect(TEXTURE_COORDINATES.length * PIXELDPETH)
            .order(ByteOrder.nativeOrder()).asFloatBuffer().put(TEXTURE_COORDINATES).rewind();
    private static final Buffer VERTEX_BUFFER =
            ByteBuffer.allocateDirect(VERTEX_COORDINATES.length * PIXELDPETH)
            .order(ByteOrder.nativeOrder()).asFloatBuffer().put(VERTEX_COORDINATES).rewind();
    private EmulationActivity getActivity() {
        Context context = getContext();
        while (context instanceof ContextWrapper) {
            if (context instanceof EmulationActivity) {
                return (EmulationActivity) context;
            }
            context = ((ContextWrapper) context).getBaseContext();
        }
        return null;
    }
    private void init() {

        setEGLContextClientVersion(1);
        setRenderer(this);
        setRenderMode(GLSurfaceView.RENDERMODE_WHEN_DIRTY);
        GestureDetectorCompat mGestureDetector = new GestureDetectorCompat(getContext(),
                new GestureDetector.SimpleOnGestureListener() {
                    @Override
                    public boolean onDown(@NonNull  final MotionEvent e) {
                        return true;
                    }

                    @Override
                    public boolean onDoubleTap(@NonNull final MotionEvent e) {
                        if ((e.getToolType(0) & MotionEvent.TOOL_TYPE_MOUSE) != 0) {
                            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N
                                    && PointerIcon.getSystemIcon(
                                            getContext(), PointerIcon.TYPE_NULL)
                                    .equals(Objects.requireNonNull(getActivity())
                                            .findViewById(R.id.screen).getPointerIcon())) {
                                return false;
                            }
                        }
                        if (mCanZoom) {
                            mZoomValue = !mZoomValue;
                            //noinspection ConstantConditions
                            updateViewModel(getActivity().getViewModel());
                        }
                        changeToNextFitting();
                        return true;
                    }
                });
        setOnTouchListener((view, event) -> {
            if (!mGestureDetector.onTouchEvent(event)) {
                return performClick();
            }
            return true;
        });


    }
    private void changeToNextFitting() {
        if (getActivity() != null) {
            getActivity().restoreCanvasSize();
            getActivity().onMonitorDoubleTap();
        }
    }
    static final int FULLVISIBLE_OR_SCROLL = 1;
    static final int ZOOM_OUT = 2;
    private int mTvMode = FULLVISIBLE_OR_SCROLL;
    void setTvMode(final int mode) {
        if (mode != mTvMode) {
            mTvMode = mode;
            calcTransformation();
        }
    }

    @Override
    public void onSurfaceCreated(final GL10 gl, final EGLConfig eglConfig) {
        mTextures = new int[1];
        gl.glTexEnvf(GL10.GL_TEXTURE_ENV, GL10.GL_TEXTURE_ENV_MODE, GL10.GL_REPLACE);
        //gl.glEnable(GL10.GL_BLEND);
        gl.glDisable(GL10.GL_BLEND);
        gl.glDisable(GL10.GL_ALPHA_TEST);
        gl.glDisable(GL10.GL_DEPTH_TEST);
        //
        gl.glEnable(GL10.GL_TEXTURE_2D);
        gl.glEnableClientState(GL10.GL_VERTEX_ARRAY);
        gl.glEnableClientState(GL10.GL_TEXTURE_COORD_ARRAY);
        gl.glGenTextures(mTextures.length, mTextures, 0);
        for (int mTexture : mTextures) {
            gl.glBindTexture(GL10.GL_TEXTURE_2D, mTexture);
            gl.glTexParameterf(GL10.GL_TEXTURE_2D, GL10.GL_TEXTURE_MAG_FILTER, GL10.GL_NEAREST);
            gl.glTexParameterf(GL10.GL_TEXTURE_2D, GL10.GL_TEXTURE_MIN_FILTER, GL10.GL_NEAREST);
            gl.glTexParameterf(GL10.GL_TEXTURE_2D, GL10.GL_TEXTURE_WRAP_S, GL10.GL_CLAMP_TO_EDGE);
            gl.glTexParameterf(GL10.GL_TEXTURE_2D, GL10.GL_TEXTURE_WRAP_T, GL10.GL_CLAMP_TO_EDGE);

        }
    }

    @Override
    public void onSurfaceChanged(final GL10 gl, final int width, final int height) {
        gl.glViewport(0, 0, width, height);
    }
    private final Point mBitmapSize = new Point(-1, -1);
    @Override
    public void onDrawFrame(final GL10 gl) {
        if (mVideoBitmap != null) {
            if (mBitmapSize.x != mVideoBitmap.getWidth()
                    || mBitmapSize.y != mVideoBitmap.getHeight()) {
                mBitmapSize.x = mVideoBitmap.getWidth();
                mBitmapSize.y = mVideoBitmap.getHeight();
                GLUtils.texImage2D(GL10.GL_TEXTURE_2D, 0, mVideoBitmap, 0);
            }
            GLUtils.texSubImage2D(GL10.GL_TEXTURE_2D, 0, 0, 0, mVideoBitmap);
            gl.glActiveTexture(GL10.GL_TEXTURE0);
            //gl.glBindTexture(GL10.GL_TEXTURE_2D, mTextures[0]);
            gl.glVertexPointer(3, GL10.GL_FLOAT, 0, VERTEX_BUFFER);
            gl.glTexCoordPointer(2, GL10.GL_FLOAT, 0, TEXCOORD_BUFFER);
            gl.glDrawArrays(GL10.GL_TRIANGLE_STRIP, 0, 4);
        }

    }

    private Bitmap mVideoBitmap = null;
    byte[] getLastRecordedBitmap() {
        if (mVideoBitmap != null) {
            Bitmap bmp = mVideoBitmap;
            int bytes = bmp.getAllocationByteCount();
            ByteBuffer buffer = ByteBuffer.allocate(bytes); //Create a new buffer
            bmp.copyPixelsToBuffer(buffer);
            return buffer.array();
        }
        return null;
    }
    static class StateCondition implements Condition {
        private final Condition mCondition;
        private boolean mWasSignalled = false;
        StateCondition(final Lock lock) {
            mCondition = lock.newCondition();
        }
        @Override
        public void await() throws InterruptedException {
            if (mWasSignalled) {
                return;
            }
            mCondition.await();
        }

        @Override
        public void awaitUninterruptibly() {
            if (mWasSignalled) {
                return;
            }
            mCondition.awaitUninterruptibly();
        }

        @Override
        public long awaitNanos(final long l) throws InterruptedException {
            if (mWasSignalled) {
                return l / 2;
            }
            return mCondition.awaitNanos(l);
        }

        @Override
        public boolean await(final long l, final TimeUnit timeUnit) throws InterruptedException {
            if (mWasSignalled) {
                return true;
            }
            return mCondition.await(l, timeUnit);
        }

        @Override
        public boolean awaitUntil(final Date date) throws InterruptedException {
            if (mWasSignalled) {
                return true;
            }
            return mCondition.awaitUntil(date);
        }

        @Override
        public void signal() {
            mWasSignalled = true;
            mCondition.signal();
        }

        @Override
        public void signalAll() {
            mWasSignalled = true;
            mCondition.signalAll();
        }
    }
    private static final int MINIMAL_WIDTH = 32;
    private ByteBuffer mWorkBuffer = ByteBuffer.allocate(1);
    void setCurrentBitmap(final Bitmap value) {
        if (value != null) {

            if (!mInLayout) {
                boolean waitingForScreenUpdate;
                synchronized (mScreenTestCondition) {
                    waitingForScreenUpdate = !mScreenTestCondition.isEmpty();
                }

                if (waitingForScreenUpdate) {
                    Log.v(TAG, "waitingForScreenUpdate");
                    mScreenTestLock.lock();
                    List<Condition> conditions;
                    synchronized (mScreenTestCondition) {
                        conditions = new LinkedList<>(mScreenTestCondition);
                    }
                    for (Condition c : conditions) {
                        c.signalAll();
                    }
                    mScreenTestCondition.clear();
                    mScreenTestLock.unlock();
                }

                if (mWaitingForBitmap != null) {
                    Log.v(TAG, "mWaitingForBitmap");
                    int bytes = value.getAllocationByteCount();
                    mWorkBuffer.clear();
                    if (mWorkBuffer.remaining() != bytes) {
                        mWorkBuffer = ByteBuffer.allocate(bytes); //Create a new buffer
                    }
                    value.copyPixelsToBuffer(mWorkBuffer);
                    if (getWidth() > MINIMAL_WIDTH) {
                        if (Arrays.equals(mWorkBuffer.array(), mWaitingForBitmap)) {
                            mBitmapTestLock.lock();
                            mWaitingForBitmap = null;
                            mBitmapCondition.signalAll();
                            mBitmapTestLock.unlock();
                        }
                    }
                }
                requestRender();
                if (mScrollview != null) {
                    boolean waitingForScrollUpdate;
                    synchronized (mScrollTestCondition) {
                        waitingForScrollUpdate = !mScrollTestCondition.isEmpty();
                    }

                    if (waitingForScrollUpdate) {
                        Log.v(TAG, "waitingForScrollUpdate");
                        List<ScrollTest> toDelete = new LinkedList<>();
                        int mTop = mScrollview.getScrollY();
                        int mBottom = getBottom()
                                - (mScrollview.getHeight() + mScrollview.getScrollY());
                        mScrollTestLock.lock();
                        for (ScrollTest test : mScrollTestCondition.keySet()) {
                            if (test.match(mTop, mBottom)) {
                                Condition c = Objects.requireNonNull(
                                        mScrollTestCondition.get(test));
                                c.signalAll();
                                toDelete.add(test);
                            }
                        }
                        mScrollTestLock.unlock();
                        synchronized (mScrollTestCondition) {
                            for (ScrollTest test : toDelete) {
                                mScrollTestCondition.remove(test);
                            }
                        }
                    }
                }
            }
            mVideoBitmap = value;
        }
    }

    void onGeometryChanged() {
        if (getActivity() != null) {
            getActivity().restoreCanvasSize();
        }
        requestRender();
    }
    void setCanvasGeometry(final int canvasWidth,
                           final int canvasHeight,
                           final float pixeAspectRatio,
                           final float canvasAspectRatio) {
        mCanvasWidth = canvasWidth;
        mCanvasAspectRatio = canvasAspectRatio;
        mPixelAspectRatio  = pixeAspectRatio;
        Log.v(getClass().getSimpleName(), String.format("setCanvasGeometry (%d, %d, %f, %f)",
        canvasWidth, canvasHeight, pixeAspectRatio, canvasAspectRatio));
        calcTransformation();

    }
    void setInLayout(final boolean value) {
        mInLayout = value;
    }
    void setBitmap(final Bitmap bitmap) {
        if (mInLayout) {
            Log.v(TAG, "bitmap set during layout.");
        }
        setCurrentBitmap(bitmap);
    }
    Bitmap getCurrentBitmap() {
        return mVideoBitmap;
    }
    private boolean mZoomInitialized = false;
    private boolean mZoomValue = false;

    void readFromViewModel(final boolean orientation,
                           final EmulationViewModel viewmodel) {
        mZoomInitialized = viewmodel.getZoomInitialized().get(orientation);
        mZoomValue = viewmodel.getZoomValue().get(orientation);

    }

    void updateViewModel(final EmulationViewModel viewmodel) {
        if (viewmodel != null) {
            @SuppressWarnings("ConstantConditions")
            boolean orientation = getActivity().getResources().getBoolean(R.bool.is_landscape);
            viewmodel.getZoomInitialized().put(orientation, mZoomInitialized);
            viewmodel.getZoomValue().put(orientation, mZoomValue);
        }
    }
    private boolean mCanZoom = false;


    private final DisplayMetrics mDisplayMetrics = new DisplayMetrics();

    private float getDeviceSpecificScreenAspectRatio(final float screenAspectRatio) {
        @SuppressWarnings("ConstantConditions")
        Display display = getActivity().getWindowManager().getDefaultDisplay();

        display.getMetrics(mDisplayMetrics);
        return screenAspectRatio * (mDisplayMetrics.ydpi / mDisplayMetrics.xdpi);

    }

    void calcTransformation() {
        //noinspection ConstantConditions
        Log.v(TAG, "calcTransformation called");
        if (getActivity() != null && getActivity().getViewModel() != null) {
            final int displayId = getActivity().getViewModel().getCurrentDisplayId();
            Objects.requireNonNull(getActivity())
                    .getViewModel().setPixelAspectRatio(
                            displayId, mPixelAspectRatio);
            getActivity().getViewModel().setScreenAspectRatio(displayId, mCanvasAspectRatio);
            if (mCanvasWidth != 0 && mCanvasAspectRatio != 0 && mPixelAspectRatio != 0) {
                Display display = getActivity().getWindowManager().getDefaultDisplay();
                Point displaySize = new Point();
                display.getSize(displaySize);

                if (!mZoomInitialized) {
                    mZoomValue = false;
                    mZoomInitialized = true;
                }
                float width = displaySize.x;
                float scaleX;
                float scaleY;
                float ratio = getDeviceSpecificScreenAspectRatio(mCanvasAspectRatio);
                int addheight;
                if (mZoomValue && mTvMode != ZOOM_OUT) {
                    addheight = 0;
                    scaleX = displaySize.x / width;
                    scaleY = displaySize.x / getDeviceSpecificScreenAspectRatio(mCanvasAspectRatio)
                            / mViewHeight;
                } else {
                    int height = mViewHeight;
                    View v = getActivity().findViewById(R.id.tvkeyboardspacer);
                    if (v.getVisibility() == VISIBLE && mTvMode != ZOOM_OUT) {
                        addheight = v.getHeight();
                    } else {
                        addheight = 0;
                    }
                    height += addheight;
                    if (width > height * ratio) {
                        scaleY = 1.0f;
                        scaleX = height * ratio / width;
                    } else {
                        scaleX = displaySize.x / width;
                        scaleY = displaySize.x / (ratio * height);

                    }
                }
                mCanZoom = width > mViewHeight * ratio;
                ViewGroup.LayoutParams lp = getLayoutParams();
                int newwidth = (int) (scaleX * width);
                int newheight = (int) (scaleY * mViewHeight);
                if (lp.width != newwidth || lp.height != newheight + addheight) {
                    //firstRun = false;
                    lp.width = newwidth;
                    lp.height = newheight + addheight;
                    Log.v(TAG, "width = " + width);
                    Log.v(TAG, "scalex = " + scaleX);
                    Log.v(TAG, "height = " + mViewHeight);
                    Log.v(TAG, "scaley = " + scaleY);
                    Log.v(TAG, "layout.width = " + lp.width);
                    Log.v(TAG, "layout.height = " + lp.height);
                    post(() -> {
                        mInLayout = true;
                        setLayoutParams(lp);
                        Log.v(TAG, "setLayoutParams finished");
                        addOnLayoutChangeListener(new OnLayoutChangeListener() {
                            //CHECKSTYLE DISABLE ParameterNumber FOR 2 LINES
                            @Override
                            public void onLayoutChange(final View view, final int l, final  int t,
                                                       final int r, final int b,
                                                       final int ot, final int ol,
                                                       final int or, final int ob) {
                                Log.v(TAG, "layout changed listener called");
                                removeOnLayoutChangeListener(this);
                                mInLayout = false;
                                requestRender();
                            }
                        });
                    });
                }
            }
        }
        Log.v(TAG, "calcTransformation finished");
    }
    private final Lock mScreenTestLock = new ReentrantLock();
    /** @noinspection MismatchedQueryAndUpdateOfCollection */
    private final List<Condition> mScreenTestCondition = new LinkedList<>();
    private final Lock mAudioTestLock = new ReentrantLock();
    private final Condition mAudioTestCondition = mAudioTestLock.newCondition();

    /**
     * Testing: Wait for the screen to be updated.
     * @param t wait for how many time units given in u.
     * @param u unit of time to wait for.
     * @return true if the screen has been redrawn within the given time.
     */
    @SuppressWarnings("SameParameterValue")
    public boolean waitForScreenRedraw(final long t, final TimeUnit u) {
        boolean ret = false;
        try {
            mScreenTestLock.lock();
            Log.v(getClass().getSimpleName() + "waitForScreenRedraw", "await called");
            Condition c = new StateCondition(mScreenTestLock);
            synchronized (mScreenTestCondition) {
                mScreenTestCondition.add(c);
            }
            ret = c.await(t, u);
            Log.v(getClass().getSimpleName() + "waitForScreenRedraw",
                    "await returned " + (ret ? "true" : "false"));

        } catch (InterruptedException e) {
            // nothing
        } finally {
            mScreenTestLock.unlock();
        }
        if (!ret) {
            Log.v("x", "y");
        }
        return ret;


    }
    private interface ScrollTest {
        boolean match(int top, int bottom);
    }
    private final Lock mScrollTestLock = new ReentrantLock();
    private final Map<ScrollTest, Condition> mScrollTestCondition = new HashMap<>();

    boolean waitForScroll(final int position, final long t, final TimeUnit u) {

        if (mScrollview != null) {
            int mTop = mScrollview.getScrollY();
            int mBottom = getHeight() - mScrollview.getHeight() + mScrollview.getScrollY();

            ScrollTest test;

            switch (position) {
                case EmulationUi.SCREENPART_TOP:
                    test = (top, bottom) -> top == 0;
                    break;
                case EmulationUi.SCREENPART_BOTTOM:
                    test = (top, bottom) -> bottom == 0;
                    break;
                case EmulationUi.SCREENPART_CENTER:
                    test = (top, bottom) -> Math.abs(top - bottom) < 2;
                    break;
                default:
                    return false;
            }
            if (test.match(mTop, mBottom)) {
                return true;
            } else {
                Condition condition = mScrollTestLock.newCondition();
                mScrollTestCondition.put(test, condition);
                try {
                    mScrollTestLock.lock();
                    return condition.await(t, u);

                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                } finally {
                    mScrollTestLock.unlock();

                }
            }
        }
        return false;
    }
    /**
     * Testing: Wait for the screen to be first drawn.
     * @param t wait for how many time units given in u.
     * @param u unit of time to wait for.
     * @return true if the screen has been drawn within the given time.
     */
    @SuppressWarnings("SameParameterValue")
    public boolean waitForScreenInit(final long t, final TimeUnit u) {
        if (mVideoBitmap != null) {
            return true;
        } else {
            return waitForScreenRedraw(t, u);
        }
    }
    private final Lock mBitmapTestLock = new ReentrantLock();
    private final Condition mBitmapCondition = mBitmapTestLock.newCondition();
    private byte[] mWaitingForBitmap = null;
    /**
     * Testing: Wait for the screen to match a specific value.
     * @param t wait for how many time units given in u.
     * @param u unit of time to wait for.
     * @param data data to match, to be determined with {@link Bitmap#copyPixelsToBuffer(Buffer)}
     * @return true if the screen's pixels match the given data within the given time.
     */
    @SuppressWarnings("SameParameterValue")
    public boolean waitForBitmap(final byte[] data, final long t, final TimeUnit u) {
        if (mWaitingForBitmap != null) {
            return false;
        }

        try {
            mBitmapTestLock.lock();
            mWaitingForBitmap = data;
            boolean ret = mBitmapCondition.await(t, u);
            mWaitingForBitmap = null;
            return ret;
        } catch (InterruptedException e) {
            // nothing
        } finally {
            mBitmapTestLock.unlock();
        }
        return false;
    }

    @Override
    public boolean waitForAudio(final long t, final TimeUnit u) {
        //noinspection ConstantConditions

        Emulation.SoundFunctions sf = getActivity().getViewModel()
                    .getEmulation().getSoundFunctions();
        if (sf != null) {
            boolean ret;
            mAudioTestLock.lock();
            sf.callOnAudioPlaying(() -> {
                mAudioTestLock.lock();
                mAudioTestCondition.signalAll();
                mAudioTestLock.unlock();
            });

            try {
                ret = mAudioTestCondition.await(t, u);
            } catch (InterruptedException e) {
                ret = false;
            } finally {
                mAudioTestLock.unlock();
            }
            return ret;
        }
        return false;
    }
}
