//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.app.PendingIntent;

import android.appwidget.AppWidgetManager;
import android.appwidget.AppWidgetProvider;
import android.content.Context;
import android.content.Intent;

import android.content.SharedPreferences;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.widget.RemoteViews;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Map;
import java.util.Properties;

/**
 * Implementation of App Widget functionality.
 * App Widget Configuration implemented in
 * {@link LaunchTileWidgetConfigureActivity LaunchTileWidgetConfigureActivity}
 */
public class LaunchTileWidget extends AppWidgetProvider {
    private static final String PREFS_NAME = "de.rainerhock.eightbitwonders.LaunchTileWidget";
    private static final String PREF_PREFIX_KEY = "appwidget_";
    private static String getSource(final Context context, final int appWidgetId) {
        SharedPreferences prefs = context.getApplicationContext()
                .getSharedPreferences(PREFS_NAME, 0);
        return prefs.getString(PREF_PREFIX_KEY + appWidgetId, null);
    }
    private static boolean populateRemoteView(final Context context, final RemoteViews views,
                                           final int appWidgetId) {
        String id = getSource(context, appWidgetId);
        boolean ret = false;
        if (id != null) {
            File dir = new File(id);
            if (dir.isDirectory()) {
                if (dir.isDirectory()) {
                    Properties props = new Properties();
                    try {
                        //noinspection IOStreamConstructor
                        props.load(new FileInputStream(new File(dir, "properties")));
                        String name = props.getProperty("name", null);
                        String image = props.getProperty("image", null);
                        if (name != null) {
                            views.setTextViewText(R.id.tv_name, name);
                            Bitmap bmp;
                            if (image != null && new File(dir, image).canRead()) {
                                //noinspection IOStreamConstructor
                                bmp = BitmapFactory.decodeStream(
                                        new FileInputStream(new File(dir, image)));
                                if (bmp != null) {
                                    views.setBitmap(R.id.iv_bitmap, "setImageBitmap", bmp);
                                    ret = true;
                                } else {
                                    String emulatorId = props.getProperty("emulation");
                                    Emulationslist.Description desc = Emulationslist
                                            .getDescription(emulatorId);
                                    if (desc != null) {
                                        views.setImageViewResource(R.id.iv_bitmap,
                                                desc.getImageResourceId());
                                        ret = true;
                                    }
                                }

                            } else {
                                String emulatorId = props.getProperty("emulation");
                                Emulationslist.Description desc = Emulationslist
                                        .getDescription(emulatorId);
                                if (desc != null) {
                                    views.setImageViewResource(R.id.iv_bitmap,
                                            desc.getImageResourceId());
                                    ret = true;
                                }
                            }
                        }
                    } catch (IOException e) {
                        //noinspection DataFlowIssue
                        ret = false;
                    }
                }

            } else {
                Emulationslist.Description desc = Emulationslist.getDescription(id);
                if (desc != null) {

                    views.setInt(R.id.iv_bitmap, "setBackgroundColor",
                            context.getResources().getColor(desc.getColorResourceId()));


                    views.setImageViewResource(R.id.iv_bitmap, desc.getImageResourceId());
                    views.setTextViewText(R.id.tv_name,
                            context.getString(desc.getNameResourceId()));
                    ret = true;
                }
            }
        }
        return ret;
    }
    static void updateAppWidget(final Context context, final AppWidgetManager appWidgetManager,
                                final int appWidgetId) {

        // Construct the RemoteViews object
        RemoteViews views = new RemoteViews(context.getPackageName(),
                R.layout.launch_tile_widget);
        //views.setTextViewText(R.id.appwidget_text, widgetText);
        Intent configIntent = new Intent(context, EmulationActivity.class);
        configIntent.setAction("WIDGET_CALL:" + getSource(context, appWidgetId));
        PendingIntent pIntent = PendingIntent.getActivity(context, 0, configIntent,
                PendingIntent.FLAG_IMMUTABLE);
        if (populateRemoteView(context, views, appWidgetId)) {
            views.setOnClickPendingIntent(R.id.root, pIntent);
        } else {
            views.setImageViewResource(R.id.iv_bitmap, android.R.drawable.ic_delete);
            views.setTextViewText(R.id.tv_name, context.getResources().getString(R.string.removed));
        }


        // Instruct the widget manager to update the widget
        appWidgetManager.updateAppWidget(appWidgetId, views);
    }

    static void removeWidget(final Context context, final EmulationConfiguration conf) {
        int id = getWidgetId(context, conf);
        if (id != NOT_FOUND) {
            context.getSharedPreferences(PREFS_NAME, 0).edit()
                    .remove(PREF_PREFIX_KEY + id).apply();
        }

    }

    private static final int NOT_FOUND = -1;
    static int getWidgetId(final Context context, final EmulationConfiguration conf) {
        SharedPreferences prefs = context.getSharedPreferences(PREFS_NAME, 0);

        String id;
        if (conf.getLocalPath() != null) {
            id = conf.getLocalPath().getAbsolutePath();
        } else {
            id = conf.getEmulatorId();
        }
        if (prefs.getAll() != null) {
            Map<String, ?> data = prefs.getAll();
            for (String key: data.keySet()) {
                Object value = data.get(key);
                if (id.equals(value)) {
                    return Integer.parseInt(key.replace(PREF_PREFIX_KEY, ""));
                }
            }
        }
        return NOT_FOUND;
    }

    @Override
    public final void onUpdate(final Context context, final AppWidgetManager appWidgetManager,
                               final int[] appWidgetIds) {
        for (int appWidgetId : appWidgetIds) {
            updateAppWidget(context, appWidgetManager, appWidgetId);
        }
    }

    @Override
    public final void onDeleted(final Context context, final int[] appWidgetIds) {
        // When the user deletes the widget, delete the preference associated with it.
        SharedPreferences.Editor prefs = context.getSharedPreferences(PREFS_NAME, 0).edit();
        for (int appWidgetId : appWidgetIds) {

            prefs = prefs.remove(PREF_PREFIX_KEY + appWidgetId);
        }
        prefs.apply();
    }
}
