//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;


import android.annotation.SuppressLint;
import android.os.Bundle;
import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewParent;
import android.widget.Button;
import android.widget.TextView;

import androidx.annotation.IdRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.DialogFragment;
import androidx.fragment.app.Fragment;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

class KeyboardJoystick extends Joystick implements View.OnKeyListener {
    private final List<Integer> mDeviceIds;

    public void showInitialAutofire(final JoystickSettingsView jsv) {
        jsv.setAutofireOptionsVisible(mKeyCode2ndButton != KEYCODE_UNDEFINED);
    }

    enum Keyset { KEYSET_A, KEYSET_B
    }
    private static final int KEYCODE_UNDEFINED = -1;
    private final Keyset mKeyset;
    private final String mDescriptor;
    private int mKeyCodeNorthWest = KEYCODE_UNDEFINED;
    private int mKeyCodeNorth = KEYCODE_UNDEFINED;
    private int mKeyCodeNorthEast = KEYCODE_UNDEFINED;
    private int mKeyCodeEast = KEYCODE_UNDEFINED;
    private int mKeyCodeWest = KEYCODE_UNDEFINED;
    private int mKeyCodeSouthWest = KEYCODE_UNDEFINED;
    private int mKeyCodeSouth = KEYCODE_UNDEFINED;
    private int mKeyCodeSouthEast = KEYCODE_UNDEFINED;
    private int mKeyCodeFire = KEYCODE_UNDEFINED;
    private int mKeyCode2ndButton = KEYCODE_UNDEFINED;
    public static class PressKeyFragment extends DialogFragment {
        PressKeyFragment() {
            super();
        }
        Fragment getHardwarekeyboardFragment(final @NonNull Keyset ks) {
            for (Fragment f : requireActivity().getSupportFragmentManager().getFragments()) {
                if (f instanceof KeyboardConfigFragment
                        && ks.equals(f.requireArguments().getSerializable("keyset"))) {
                    return f;
                }
            }
            return null;
        }
        Fragment getCurrentMultiplexedDeviceFragment() {
            for (Fragment f : requireActivity().getSupportFragmentManager().getFragments()) {
                if (f instanceof MultiplexerJoystick.SingleJoystickDialogFragment) {
                    for (Fragment f2 : f.getChildFragmentManager().getFragments()) {
                        if (MultiplexerJoystick.DEVICE_SPECIFIC_FRAGMENT.equals(f2.getTag())) {
                            return f2;
                        }
                    }
                }
            }
            return null;
        }

        private Button getButton(final Keyset keyset, final @IdRes int buttonId) {
            Fragment f = getCurrentMultiplexedDeviceFragment();
            if (f != null && f.getView() != null) {
                return f.getView().findViewById(buttonId);
            }
            f = getHardwarekeyboardFragment(keyset);
            if (f != null && f.getView() != null) {
                return f.getView().findViewById(buttonId);
            }
            return requireActivity().findViewById(buttonId);
        }
        @Override
        public void onResume() {
            super.onResume();
            int buttonId = requireArguments().getInt("button_id");
            Keyset keyset = (Keyset) requireArguments().getSerializable("keyset");
            Useropts useropt = ((SettingsActivity) requireActivity()).getCurrentUseropts();
            String useroptKey = requireArguments().getString("useropt_key");

            requireDialog().setOnKeyListener((dialog, keyCode, event) -> {
                if (event.getAction() == KeyEvent.ACTION_UP) {
                    if (Objects.requireNonNull(InputDevice.getDevice(
                            event.getDeviceId())).getKeyboardType()
                            == InputDevice.KEYBOARD_TYPE_ALPHABETIC) {
                        useropt.setValue(Useropts.Scope.GLOBAL, useroptKey,
                                event.getKeyCode());
                        Button b = getButton(keyset, buttonId);
                        b.setText(KeyEvent.keyCodeToString(
                                event.getKeyCode()).replace("KEYCODE_", ""));
                        View vg = b;
                        while (vg != null ) {
                            ViewParent parent = vg.getParent();
                            if (parent instanceof View) {
                                vg = (View) vg.getParent();
                                if (vg instanceof JoystickSettingsView) {
                                    JoystickSettingsView jsv = (JoystickSettingsView) vg;
                                    if (b.getId() == R.id.second) {
                                        jsv.setAutofireOptionsVisible(true);
                                        break;
                                    }
                                }
                            } else {
                                break;
                            }
                        }

                        dismiss();
                        return true;
                    }
                }
                return false;
            });
        }
        @Override
        public final View onCreateView(@NonNull final LayoutInflater inflater,
                                       @Nullable final ViewGroup parent,
                                       @Nullable final Bundle savedInstanceState) {
            ViewGroup root = (ViewGroup) inflater
                    .inflate(R.layout.dialog_select_key, parent, true);
            TextView prompt = root.findViewById(R.id.tv_hit_key_prompt);
            prompt.setText(getResources().getString(R.string.add_dot,
                    getResources().getString(requireArguments().getInt("promptId"))));
            TextView value = root.findViewById(R.id.tv_current_value);
            if (requireArguments().getString("current_value") != null) {
                value.setText(getResources().getString(R.string.current_value,
                        requireArguments().getString("current_value")));
            } else {
                value.setVisibility(View.GONE);
            }
            if (requireArguments().getString("current_value") != null) {
                root.findViewById(R.id.bn_delete).setOnClickListener(view -> {
                    Serializable s = requireArguments().getSerializable("keyset");
                    final Keyset keyset;
                    if (s != null) {
                        keyset = (Keyset) s;
                    } else {
                        keyset = Keyset.KEYSET_A;
                    }
                    int buttonId = requireArguments().getInt("button_id");
                    Useropts useropt = ((SettingsActivity) requireActivity()).getCurrentUseropts();
                    String useroptKey = requireArguments().getString("useropt_key");
                    //SettingsActivity s = (SettingsActivity) requireActivity();
                    Button b = getButton(keyset, buttonId);
                    b.setText(R.string.IDS_NONE);
                    View vg = b;
                    while (vg != null) {
                        if (vg.getParent() instanceof View) {
                            vg = (View) vg.getParent();
                            if (vg instanceof JoystickSettingsView) {
                                JoystickSettingsView jsv = (JoystickSettingsView) vg;
                                if (b.getId() == R.id.second) {
                                    jsv.setAutofireOptionsVisible(false);
                                    break;
                                }
                            }
                        } else {
                            break;
                        }
                    }
                    useropt.setValue(Useropts.Scope.GLOBAL, useroptKey, KEYCODE_UNDEFINED);
                    dismiss();
                });
            } else {
                root.findViewById(R.id.bn_delete).setVisibility(View.GONE);
            }
            root.findViewById(R.id.bn_cancel).setOnClickListener(view -> dismiss());

            return root;
        }

    }
    @Override
    void readUseropts(final BaseActivity activity) {
        String optsPrefix;
        switch (mKeyset) {
            case KEYSET_A:
                optsPrefix = "KEYSET_A_";
                break;
            case KEYSET_B:
                optsPrefix = "KEYSET_B_";
                break;
            default:
                return;
        }
        Useropts useropts = activity.getCurrentUseropts();
        mKeyCodeNorthWest = useropts.getIntegerValue(optsPrefix + "NW", KEYCODE_UNDEFINED);
        mKeyCodeNorth = useropts.getIntegerValue(optsPrefix + "N", KEYCODE_UNDEFINED);
        mKeyCodeNorthEast = useropts.getIntegerValue(optsPrefix + "NE", KEYCODE_UNDEFINED);
        mKeyCodeWest = useropts.getIntegerValue(optsPrefix + "W", KEYCODE_UNDEFINED);
        mKeyCodeEast = useropts.getIntegerValue(optsPrefix + "E", KEYCODE_UNDEFINED);
        mKeyCodeSouthWest = useropts.getIntegerValue(optsPrefix + "SW", KEYCODE_UNDEFINED);
        mKeyCodeSouth = useropts.getIntegerValue(optsPrefix + "S", KEYCODE_UNDEFINED);
        mKeyCodeSouthEast = useropts.getIntegerValue(optsPrefix + "SE", KEYCODE_UNDEFINED);
        mKeyCodeFire = useropts.getIntegerValue(optsPrefix + "FIRE", KEYCODE_UNDEFINED);
        mKeyCode2ndButton = useropts.getIntegerValue(optsPrefix + "AUTO", KEYCODE_UNDEFINED);
        if (mKeyCode2ndButton == KEYCODE_UNDEFINED) {
            setAutofiring(false);
            notifyListener();
        }
        super.readUseropts(activity);


    }

    @Override
    boolean canLockDiagonals() {
        return false;
    }

    @Override
    boolean hasSecondaryButton() {
        return false;
    }

    KeyboardJoystick(final Keyset keyset, final String descriptor, final List<Integer> deviceIds) {
        mKeyset = keyset;
        mDescriptor = descriptor;
        mDeviceIds = new ArrayList<>(deviceIds.size());
        mDeviceIds.addAll(deviceIds);
    }

    @NonNull
    public String toString() {
        return mDescriptor;
    }


    @Override
    String getId() {
        return "key-" + mKeyset.toString();
    }

    @Override
    String getHardwareDescriptor() {
        return "###keyboard###/" + mDescriptor;
    }

    @Override
    int getHardwareId() {
        return KEYBOARD_HARDWARE_ID;
    }


    @Override
    void connect(final EmulationActivity activity, final JoystickListener listener,
                 final Emulation.GamepadFunctions gamepadFunctions,
                 final MultiplexerJoystick muxer) {
        super.connect(activity, listener, gamepadFunctions, muxer);
        mPressedKeys.clear();
    }

    public static class KeyboardConfigFragment extends InputDeviceConfigFragment {
        KeyboardConfigFragment() {
            super();
        }
        @SuppressLint("UseSparseArrays")
        private  static final LinkedHashMapWithDefault<String, Integer> HIT_KEY_PROMPTS
                = new LinkedHashMapWithDefault<>(-1);

        static {

            HIT_KEY_PROMPTS.put("NW", R.string.IDS_PRESS_KEY_NORTHWEST);
            HIT_KEY_PROMPTS.put("N", R.string.IDS_PRESS_KEY_NORTH);
            HIT_KEY_PROMPTS.put("NE", R.string.IDS_PRESS_KEY_NORTHEAST);
            HIT_KEY_PROMPTS.put("W", R.string.IDS_PRESS_KEY_WEST);
            HIT_KEY_PROMPTS.put("E", R.string.IDS_PRESS_KEY_EAST);
            HIT_KEY_PROMPTS.put("SW", R.string.IDS_PRESS_KEY_SOUTHWEST);
            HIT_KEY_PROMPTS.put("S", R.string.IDS_PRESS_KEY_SOUTH);
            HIT_KEY_PROMPTS.put("SE", R.string.IDS_PRESS_KEY_SOUTHEAST);
            HIT_KEY_PROMPTS.put("FIRE", R.string.IDS_PRESS_KEY_FIRE);
            HIT_KEY_PROMPTS.put("AUTO", R.string.press_key_autofire_toggle);

        }
        @Override
        public View onCreateView(final LayoutInflater inflater, final ViewGroup container,
                                 final Bundle savedInstanceState) {
            View ret = inflater.inflate(R.layout.fragment_config_keyboard_joystick,
                    container, false);
            Serializable s = requireArguments().getSerializable("keyset");
            final Keyset keyset;
            if (s != null) {
                keyset = (Keyset) s;
            } else {
                keyset = Keyset.KEYSET_A;
            }
            String prefix;
            switch (keyset) {
                case KEYSET_A:
                    prefix = "KEYSET_A_";
                    break;
                case KEYSET_B:
                    prefix = "KEYSET_B_";
                    break;
                default:
                    prefix = "";
                    break;
            }
            Useropts useropt = ((SettingsActivity) requireActivity()).getCurrentUseropts();
            for (String key : HIT_KEY_PROMPTS.keySet()) {
                Button b = ret.findViewWithTag(key);
                int keycode = useropt.getIntegerValue(prefix + key, KEYCODE_UNDEFINED);
                if (keycode == KEYCODE_UNDEFINED) {
                    b.setText(R.string.IDS_NONE);
                } else {
                    b.setText(KeyEvent.keyCodeToString(keycode).replace("KEYCODE_", ""));
                }
                b.setOnClickListener(view -> {
                    DialogFragment f = new PressKeyFragment();
                    f.setStyle(DialogFragment.STYLE_NO_TITLE, R.style.AlertDialogTheme);
                    Bundle bundle = new Bundle();
                    bundle.putSerializable("keyset", keyset);
                    bundle.putBoolean("key_was_set", keycode != KEYCODE_UNDEFINED);
                    bundle.putString("useropt_key", prefix + key);
                    bundle.putInt("button_id", b.getId());
                    if (!b.getText().toString().equals(
                            getResources().getString(R.string.IDS_NONE))) {
                        bundle.putString("current_value", b.getText().toString());
                    }
                    bundle.putInt("promptId", HIT_KEY_PROMPTS.get(key));
                    f.setArguments(bundle);
                    f.showNow(requireActivity().getSupportFragmentManager(), null);
                });

            }
            return ret;

        }

    }

    @Override
    public InputDeviceConfigFragment getDstickConfigFragment() {
        Bundle b = new Bundle();
        b.putSerializable("keyset", mKeyset);
        InputDeviceConfigFragment f = new KeyboardConfigFragment();
        f.setArguments(b);
        return f;
    }

    @Override
    boolean deviceHasButtonWithKeycode(final int keycode) {
        for (int deviceId : mDeviceIds) {
            if (deviceId >= 0) {
                if (InputDevice.getDevice(deviceId) != null) {
                    return Objects.requireNonNull(InputDevice.getDevice(deviceId))
                            .hasKeys(keycode)[0];
                }
            }
        }
        return false;
    }
    @Override
    void disconnect() {
        mPressedKeys.clear();
        super.disconnect();
    }

    private final Set<Integer> mPressedKeys = new HashSet<>();
    @Override
    public boolean onKey(final View v, final int keyCode, final KeyEvent event) {

        if (!isListenerAttached()  || !mDeviceIds.contains(event.getDeviceId())) {
            return false;
        }
        if (event.getAction() == KeyEvent.ACTION_DOWN || event.getAction() == KeyEvent.ACTION_UP) {
            boolean val = event.getAction() == KeyEvent.ACTION_DOWN;
            if (val) {
                if (keyCode == mKeyCode2ndButton) {
                    setAutofiring(!isAutofiring());
                    return true;
                } else {
                    mPressedKeys.add(keyCode);
                }
            } else {
                if (keyCode != mKeyCode2ndButton) {
                    mPressedKeys.remove(keyCode);
                }
            }
            if (Arrays.asList(mKeyCodeNorthWest,
                    mKeyCodeNorth, mKeyCodeNorthEast, mKeyCodeWest, mKeyCodeEast, mKeyCodeSouthWest,
                    mKeyCodeSouth, mKeyCodeSouthEast, mKeyCodeFire)
                    .contains(keyCode)) {
                float y = (mPressedKeys.contains(mKeyCodeNorth)
                        || mPressedKeys.contains(mKeyCodeNorthEast)
                        || mPressedKeys.contains(mKeyCodeNorthWest)) ? -FULL_AMPLITUDE
                        : (mPressedKeys.contains(mKeyCodeSouth)
                        || mPressedKeys.contains(mKeyCodeSouthEast)
                        || mPressedKeys.contains(mKeyCodeSouthWest)) ? FULL_AMPLITUDE : 0;
                float x = (mPressedKeys.contains(mKeyCodeWest)
                        || mPressedKeys.contains(mKeyCodeSouthWest)
                        || mPressedKeys.contains(mKeyCodeNorthWest)) ? -FULL_AMPLITUDE
                        : (mPressedKeys.contains(mKeyCodeEast)
                        || mPressedKeys.contains(mKeyCodeNorthEast)
                        || mPressedKeys.contains(mKeyCodeSouthEast)) ? FULL_AMPLITUDE : 0;
                setXValue(x);
                setYValue(y);
                setRealButtonsPressed(mPressedKeys.contains(mKeyCodeFire)
                        ? Collections.singleton(mKeyCodeFire) : Collections.emptySet());
                return notifyListener();
            }
        }
        return false;
    }
}
