//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.util.AttributeSet;
import android.util.Log;
import android.view.MotionEvent;

import java.util.LinkedHashMap;
import java.util.Map;

public final class JoystickWheelView extends JoystickDirectionView<WheelJoystick> {
    /**
     * Simple constructor to use when creating a JoystickWheelView from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public JoystickWheelView(final Context context) {
        super(context);
    }
     /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public JoystickWheelView(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
    }
    /**
     * Constructor that is called when inflating a JoystickWheelView from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public JoystickWheelView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
    }
    protected float getDeadCenterSize() {
        return 0;
    }
    private static final Map<Directionvalue, Directionvalue> COUNTERCLOCKWISE_DIRECTION =
            new LinkedHashMap<Directionvalue, Directionvalue>() {{
                put(Directionvalue.CENTERED, Directionvalue.WEST);
                put(Directionvalue.WEST, Directionvalue.WEST);
                put(Directionvalue.EAST, Directionvalue.CENTERED);
            }};
    private static final Map<Directionvalue, Directionvalue> CLOCKWISE_DIRECTION =
            new LinkedHashMap<Directionvalue, Directionvalue>() {{
                put(Directionvalue.CENTERED, Directionvalue.EAST);
                put(Directionvalue.EAST, Directionvalue.EAST);
                put(Directionvalue.WEST, Directionvalue.CENTERED);
            }};
    private static final int THETA_UNDEFINED = 1000;
    private static final int FULL_ANGLE = 360;
    private int mTheta0 = THETA_UNDEFINED;
    private boolean angleDiffExceedsLimit(final int ang1, final int ang2) {
        int compare1 = ang1 + (ang1 < 0 ? FULL_ANGLE : 0);
        int compare2 = ang2 + (ang2 < 0 ? FULL_ANGLE : 0);
        int diff = compare2 - compare1;

        Log.v(getClass().getSimpleName(),
                String.format("angleDiffExceedsLimit, %d-%d=%d>%d? = %s",
                        compare2,
                        compare1,
                        diff,
                        ROTATION_TRIGGERING_ANGLE,
                        (compare2 - compare1) > ROTATION_TRIGGERING_ANGLE));
        return (compare2 - compare1) > ROTATION_TRIGGERING_ANGLE;
    }
    private int getCurrentTheta(final float x, final float y) {
        int ret = (int) (Math.atan2(x - getHalfwidth(), y - getHalfheight())
                * FULL_ANGLE / (2 * Math.PI));
        Log.v(getClass().getSimpleName(),
                String.format("getCurrentTheta (%f, %f) = %d", x, y, ret));
        return ret;
    }
    private static final int NEUTRALANGLE = 10;
    private static final int RIGHTANGLE = 90;
    private Directionvalue getDirectionValue(final int theta,
                                             final Directionvalue currentDirection,
                                             final Directionvalue upperAreaDirection,
                                             final Directionvalue lowerAreaDirection) {
        if (Math.abs(theta) > RIGHTANGLE + NEUTRALANGLE && theta < 2 * RIGHTANGLE - NEUTRALANGLE) {
            return upperAreaDirection;
        }
        if (Math.abs(theta) < RIGHTANGLE - NEUTRALANGLE && theta > NEUTRALANGLE) {
            return lowerAreaDirection;
        }
        return currentDirection;
    }


    @Override
    protected Directionvalue calcRegion(final int action, final float x0,
                                        final float y0,
                                        final Directionvalue previous,
                                        final boolean diagonalsEnabled) {
        mTheta0 = getCurrentTheta(x0, y0);
        if (mTheta0 > 0) {
            return getDirectionValue(mTheta0, previous, Directionvalue.WEST, Directionvalue.EAST);
        } else {
            return getDirectionValue(-mTheta0, previous, Directionvalue.EAST, Directionvalue.WEST);
        }
    }

    @Override
    void setWatching(final boolean val) {
        if (!val) {
            setTag(Directionvalue.CENTERED.toString());
        }
        super.setWatching(val);
    }

    @Override
    protected Directionvalue updateDirectionValue(final int action,
                                                  final float x, final float y,
                                                  final Directionvalue currentDirection) {
        Directionvalue newDirection;
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                mTheta0 = getCurrentTheta(x, y);
                newDirection = calcRegion(action, x, y, currentDirection, true);
                Log.v(getClass().getSimpleName(), String.format("x=%d,y=%d,dv=%s,newDirection=%s",
                        (int) x, (int) y, currentDirection.toString(), newDirection.toString()));
                break;
            case MotionEvent.ACTION_UP:
                mTheta0 = THETA_UNDEFINED;
                newDirection = Directionvalue.CENTERED;
                break;
            case MotionEvent.ACTION_MOVE:
                Log.v(getClass().getSimpleName(), String.format("ACTION_MOVE, x=%f,y=%f",
                x, y));
                if (mTheta0 != THETA_UNDEFINED) {
                    int theta = getCurrentTheta(x, y);
                    if (angleDiffExceedsLimit(theta, mTheta0)) {
                        newDirection = CLOCKWISE_DIRECTION.get(currentDirection);
                        Log.v(getClass().getSimpleName(), String.format(
                                ">> theta=%d, theta0=%d, current=%s, new=%s",
                                theta,
                                mTheta0,
                                currentDirection,
                                newDirection));

                    } else if (angleDiffExceedsLimit(mTheta0, theta)) {
                        newDirection = COUNTERCLOCKWISE_DIRECTION.get(currentDirection);
                        Log.v(getClass().getSimpleName(), String.format(
                                "<< theta=%d, theta0=%d, current=%s, new=%s",
                                theta,
                                mTheta0,
                                currentDirection,
                                newDirection));
                    } else {
                        newDirection = currentDirection;
                    }
                    if (newDirection != currentDirection) {
                        mTheta0 = theta;
                    }
                } else {
                    Log.v(getClass().getSimpleName(), "mTheta0 not defined.");
                    newDirection = currentDirection;
                }
                break;
            default:
                newDirection = Directionvalue.CENTERED;
                break;
        }
        setTag(newDirection != null ? newDirection.toString() : "CENTERED");
        if (newDirection != currentDirection) {
            showNewDirection(newDirection);
            if (getJoystick() != null) {
                getJoystick().onChanged();
            }
            return newDirection;
        } else {
            return currentDirection;
        }
    }
    private static final int ROTATION_CENTER = 10000 / 2;
    private static final int ROTATION_LEFT = 0;
    private static final int ROTATION_RIGHT = 10000;
    private static final int ROTATION_TRIGGERING_ANGLE = 15;
    @Override
    protected void showNewDirection(final Directionvalue dv) {
        if (dv != getLastDirection()) {
            setLastDirection(dv);
            getJoystick().onChanged();
            if (isPointingEast()) {
                getDrawable().setLevel(ROTATION_RIGHT);
            } else if (isPointingWest()) {
                getDrawable().setLevel(ROTATION_LEFT);
            } else {
                getDrawable().setLevel(ROTATION_CENTER);
            }
        }
    }

    @Override
    protected int getDragRadius(final int touchradius) {
        return touchradius;
    }

    @Override
    protected void initcontrol() {
        setTag(Directionvalue.CENTERED.toString());
        getDrawable().setLevel(ROTATION_CENTER);
        super.initcontrol();
    }
    boolean isPointingEast() {
        return DIRECTION_IS_RIGHT.contains(getLastDirection());
    }
    boolean isPointingWest() {
        return DIRECTION_IS_LEFT.contains(getLastDirection());
    }
}
