package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.CheckBox;
import android.widget.LinearLayout;
import android.widget.SeekBar;
import android.widget.Spinner;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;

import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

public final class JoystickSettingsView extends LinearLayout {
    private int mMaxAutofireFrequency = 0;
    private String mConfigFragmentTag = null;
    private int mPort = Joystick.PORT_NOT_CONNECTED;
    private SettingsController mSettingsController = null;
    private FragmentManager mFragmentManager = null;
    private ArrayAdapter<SettingsActivity.JoystickEntry> mSpinnerAdapter = null;
    private List<Emulation.InputDeviceType> mEmulatedDevicesTypes = null;
    private int mConfigFragmentContainer = 0;
    private int mDiagonalsEnabledId = View.NO_ID;
    private boolean mExcludeHardwareKeyboard = false;
    boolean isExcludingHardwarekeyboard() {
        return mExcludeHardwareKeyboard;
    }
    private void init(final boolean excludeHardwareKeyboard) {
        mExcludeHardwareKeyboard = excludeHardwareKeyboard;
        LayoutInflater inflater = (LayoutInflater) getContext()
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        inflater.inflate(R.layout.view_joystick, this, true);
        Spinner sp = findViewById(R.id.sp_joystick);
        sp.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(final AdapterView<?> adapterView, final View view,
                                       final int position, final long l) {
                if (mSpinnerListener != null) {
                    mSpinnerListener.onItemSelected(adapterView, view, position, l);
                }
                Joystick joy = ((SettingsActivity.JoystickEntry) adapterView
                        .getItemAtPosition(position)).getJoystick();
                if (joy != null) {
                    mSettingsController.onJoystickportChanged(joy, mPort);
                    joy.setPortnumber(mPort);
                } else {
                    mSettingsController.removeJoystick(mPort);
                }
                populateDeviceTypeSpinner(joy);
                if (joy != null) {
                    if (joy instanceof TouchJoystick) {
                        populateSecondaryButtonOptions(true);
                    } else {
                        findViewById(R.id.optional_seconadry_button).setVisibility(View.GONE);

                        if (joy.hasSecondaryButton()) {
                            populateSecondaryButtonOptions(false);
                            findViewById(R.id.secondary_buttons).setVisibility(View.VISIBLE);
                        } else {
                            findViewById(R.id.secondary_buttons).setVisibility(View.GONE);
                        }
                    }
                } else {
                    findViewById(R.id.autofire).setVisibility(View.GONE);
                    findViewById(R.id.optional_seconadry_button).setVisibility(View.GONE);
                }

            }
            @Override
            public void onNothingSelected(final AdapterView<?> adapterView) {
                if (mSpinnerListener != null) {
                    mSpinnerListener.onNothingSelected(adapterView);
                }

            }
        });
        mConfigFragmentContainer = View.generateViewId();
        mDiagonalsEnabledId = View.generateViewId();
        findViewById(R.id.digitalconfig).setId(mConfigFragmentContainer);
        initAutofireOptions();
    }

    JoystickSettingsView(final Context context, final FragmentManager fm,
                         final SettingsController settingsController,
                         final int port, final List<Emulation.InputDeviceType> types,
                         final int maxAutofireFrequency,
                         final boolean excludeHardwareKeyboard) {
        super(context);
        mFragmentManager = fm;
        mSettingsController = settingsController;
        mPort = port;
        mMaxAutofireFrequency = maxAutofireFrequency;
        mConfigFragmentTag = "joystick_configfragment_" + mPort;
        mEmulatedDevicesTypes = types;
        init(excludeHardwareKeyboard);


    }
    /**
     * Constructor that is called when inflating a JoystickSettingsView from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public JoystickSettingsView(final Context context, final @Nullable AttributeSet attrs) {
        super(context, attrs);
        init(false);
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyleAttr An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public JoystickSettingsView(final Context context, final @Nullable AttributeSet attrs,
                                final int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(false);
    }
    Joystick getCurrentJoystick() {
        Spinner sp = findViewById(R.id.sp_joystick);
        int pos = sp.getSelectedItemPosition();
        if (pos != View.NO_ID) {
            return ((SettingsActivity.JoystickEntry) sp.getAdapter()
                    .getItem(pos))
                    .getJoystick();
        }
        return null;
    }
    void setAdapter(final ArrayAdapter<SettingsActivity.JoystickEntry> adapter) {
        ((Spinner) findViewById(R.id.sp_joystick)).setAdapter(adapter);
        mSpinnerAdapter = adapter;
    }
    private void setInputDeviceType(final Joystick joy, final Emulation.InputDeviceType type) {
        if (type != null) {
            mSettingsController.storeValue(mConfigFragmentContainer,
                    "JOYSTICK_DEVICETYPE_" + mPort, String.valueOf(type));
        }
        FragmentTransaction fta = mFragmentManager.beginTransaction();
        Fragment oldFragment = mFragmentManager.findFragmentByTag(mConfigFragmentTag);
        if (oldFragment != null) {
            fta = fta.remove(oldFragment);
        }
        Fragment f = null;
        if (type != null) {
            switch (type) {

                case DSTICK:
                    f = joy.getDstickConfigFragment();
                    break;
                case PADDLE:
                    if (joy instanceof AnalogController) {
                        f = ((AnalogController) joy).getPaddleConfigFragment();
                    }

                    break;
                case MOUSE:
                    if (joy instanceof AnalogController) {
                        f = ((AnalogController) joy).getMouseConfigFragment();
                    }
                    break;
                default:
                    break;
            }
            if (f != null) {
                fta = fta.add(mConfigFragmentContainer, f, mConfigFragmentTag);
            }
        }
        if (joy != null && joy.canLockDiagonals() && type == Emulation.InputDeviceType.DSTICK) {
            CheckBox cb = findViewById(R.id.cb_lock_diagonals);
            cb.setVisibility(View.VISIBLE);
            String useroptKey = getResources()
                    .getString(R.string.key_joystick_lock_diagonals) + "_" + mPort;
            cb.setChecked(mSettingsController.getCurrentValue(useroptKey, false));
            cb.setOnCheckedChangeListener((compoundButton, checked)
                    -> mSettingsController.storeValue(mDiagonalsEnabledId, useroptKey, checked));
        }
        findViewById(R.id.cb_lock_diagonals).setVisibility(
                joy != null && joy.canLockDiagonals() && type == Emulation.InputDeviceType.DSTICK
                        ? View.VISIBLE : View.GONE);
        if (joy != null && joy.hasSecondaryButton()) {
            populateSecondaryButtonOptions(false);
        } else {
            findViewById(R.id.secondary_buttons).setVisibility(View.GONE);
            if (joy instanceof KeyboardJoystick) {
                KeyboardJoystick kbj = (KeyboardJoystick) joy;
                kbj.showInitialAutofire(this);
            }
        }
        fta.commitNow();
    }
    private static final Map<Joystick.SecondaryUsage, Integer> SECONDARY_FUNCTIONS
            = new LinkedHashMap<Joystick.SecondaryUsage, Integer>() {{
        put(Joystick.SecondaryUsage.FIRE, R.string.fire);
        put(Joystick.SecondaryUsage.AUTOFIRE, R.string.autofire_while_pressed);
        put(Joystick.SecondaryUsage.AUTOFIRE_TOGGLE, R.string.autofire_toggle);
        put(Joystick.SecondaryUsage.UP, R.string.up);
        put(Joystick.SecondaryUsage.DOWN, R.string.down);
        put(Joystick.SecondaryUsage.LEFT, R.string.left);
        put(Joystick.SecondaryUsage.RIGHT, R.string.right);
    }};
    void populateSecondaryButtonOptions(final boolean isTouchJoystick) {
        MappedSpinner sp = findViewById(R.id.sp_secondary_button);
        List<EmulationUi.SettingSpinnerElement> values = new LinkedList<>();
        String useroptKey = "JOYSTICK_SECONDARY_BUTTON_USAGE_" + mPort;
        String stored = mSettingsController.getCurrentValue(useroptKey,
                Joystick.SecondaryUsage.FIRE.name());
        int selected = ArrayAdapter.NO_SELECTION;
        int index = 0;

        for (Joystick.SecondaryUsage sf : SECONDARY_FUNCTIONS.keySet()) {
            if (sf != Joystick.SecondaryUsage.FIRE || !isTouchJoystick) {
                values.add(new EmulationUi.SettingSpinnerElement() {
                    @Override
                    public String getId() {
                        return sf.toString();
                    }

                    @Override
                    public String getText() {
                        //noinspection DataFlowIssue
                        return getResources().getString(
                                SECONDARY_FUNCTIONS.get(sf));
                    }

                    @NonNull
                    @Override
                    public String toString() {
                        return getText();
                    }
                });
                if (stored.equals(sf.toString())) {
                    selected = index;
                }
                index++;
            }
        }

        ArrayAdapter<EmulationUi.SettingSpinnerElement> adapter =
                new ArrayAdapter<>(getContext(),
                        android.R.layout.simple_spinner_item, values);
        adapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        sp.setAdapter(adapter);
        sp.setSelection(selected);
        sp.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(final AdapterView<?> parent, final View view,
                                       final int position, final long id) {
                EmulationUi.SettingSpinnerElement e =
                        (EmulationUi.SettingSpinnerElement) sp.getSelectedItem();
                mSettingsController.storeValue(sp.getId(), useroptKey, e.getId());
                setAutofireOptionsVisible(
                        mMaxAutofireFrequency > 1 && (
                        e.getId().equals(Joystick.SecondaryUsage.AUTOFIRE_TOGGLE.toString())
                        || e.getId().equals(Joystick.SecondaryUsage.AUTOFIRE.toString())));
            }

            @Override
            public void onNothingSelected(final AdapterView<?> parent) {

            }
        });
        if (isTouchJoystick) {
            LinearLayout ll = findViewById(R.id.optional_seconadry_button);
            CheckBox cb = ll.findViewById(R.id.cb_show_secondary_touch_button);
            ll.setVisibility(View.VISIBLE);
            if (Joystick.SecondaryUsage.FIRE.name().equals(stored)) {
                cb.setChecked(false);
                findViewById(R.id.secondary_buttons).setVisibility(View.GONE);
            } else {
                cb.setChecked(true);
                findViewById(R.id.secondary_buttons).setVisibility(View.VISIBLE);
            }
            cb.setOnCheckedChangeListener((buttonView, isChecked) -> {
                if (isChecked) {
                    findViewById(R.id.secondary_buttons).setVisibility(View.VISIBLE);
                    mSettingsController.storeValue(cb.getId(), useroptKey,
                            Joystick.SecondaryUsage.AUTOFIRE.name());
                    sp.setSelection(0);

                } else {
                    findViewById(R.id.secondary_buttons).setVisibility(View.GONE);
                    setAutofireOptionsVisible(false);
                    mSettingsController.storeValue(cb.getId(), useroptKey,
                            Joystick.SecondaryUsage.FIRE.name());
                }
            });
        }
    }
    private void initAutofireOptions() {
        ViewGroup vg = findViewById(R.id.autofire);
        String useroptKey = "JOYSTICK_AUTOFIRE_FREQ_" + mPort;
        int stored = Math.min(mSettingsController.getCurrentValue(useroptKey, 1),
                mMaxAutofireFrequency);
        SeekBar sb = vg.findViewById(R.id.sb_frequency);
        sb.setMax(mMaxAutofireFrequency - 1);
        sb.setProgress(stored);
        TextView tv = vg.findViewById(R.id.tv_frequency);
        tv.setText(getResources().getString(R.string.shots_per_second, stored));
        sb.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(final SeekBar seekBar, final int progress,
                                          final boolean fromUser) {
                mSettingsController.storeValue(sb.getId(), useroptKey, sb.getProgress() + 1);
                tv.setText(getResources().getString(R.string.shots_per_second,
                        sb.getProgress() + 1));
            }
            @Override
            public void onStartTrackingTouch(final SeekBar seekBar) {

            }

            @Override
            public void onStopTrackingTouch(final SeekBar seekBar) {

            }
        });
    }
    void setAutofireOptionsVisible(final boolean value) {
        findViewById(R.id.autofire).setVisibility(value ? View.VISIBLE : View.GONE);
    }
    ArrayAdapter<SettingsActivity.JoystickEntry> getAdapter() {
        return mSpinnerAdapter;
    }
    private static final Map<Emulation.InputDeviceType, Integer> DEVICETYPE_NAMES
            = new LinkedHashMap<Emulation.InputDeviceType, Integer>()  {{
        put(Emulation.InputDeviceType.DSTICK, R.string.joystick);
        put(Emulation.InputDeviceType.MOUSE, R.string.mouse);
        put(Emulation.InputDeviceType.PADDLE, R.string.paddle);
    }};

    private ArrayAdapter<MappedSpinner.MappedSpinnerElement> getMappedSpinnerElementArrayAdapter(
            final Joystick joy) {
        ArrayAdapter<MappedSpinner.MappedSpinnerElement> matchingDevicetypes
                = new ArrayAdapter<>(
                getContext(), android.R.layout.simple_spinner_item);
        matchingDevicetypes.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        for (Emulation.InputDeviceType idt : mEmulatedDevicesTypes) {
            if (joy.getSupportedDeviceTypes().contains(idt)) {
                Integer nameId = DEVICETYPE_NAMES.get(idt);
                if (nameId != null) {
                    matchingDevicetypes.add(new MappedSpinner.MappedSpinnerElement() {
                        @Override
                        public String getId() {
                            return idt.name();
                        }
                        @Override
                        public String getText() {
                            return getResources().getString(nameId);
                        }

                        @NonNull
                        @Override
                        public String toString() {
                            return getText();
                        }
                    });
                }

            }
        }
        return matchingDevicetypes;
    }
    void setSelection(final Joystick joy) {
        if (joy != null) {
            for (int i = 0; i < mSpinnerAdapter.getCount(); i++) {
                SettingsActivity.JoystickEntry entry = mSpinnerAdapter.getItem(i);
                if (entry != null && joy.equals(entry.getJoystick())) {
                    setSelection(i);
                    break;
                }
            }
        } else {
            setSelection(0);
        }
    }
    void setSelection(final int pos) {
        ((Spinner) findViewById(R.id.sp_joystick)).setSelection(pos);
        populateDeviceTypeSpinner(getCurrentJoystick());
    }
    void populateDeviceTypeSpinner(final Joystick joy) {
        if (joy != null) {
            ArrayAdapter<MappedSpinner.MappedSpinnerElement> matchingDevicetypes
                    = getMappedSpinnerElementArrayAdapter(joy);
            switch (matchingDevicetypes.getCount()) {
                case 0:
                    break;
                case 1:
                    setInputDeviceType(joy, Emulation.InputDeviceType.valueOf(
                            Objects.requireNonNull(matchingDevicetypes.getItem(0)).getId()));
                    findViewById(R.id.typeselection).setVisibility(View.GONE);
                    break;
                default:
                    findViewById(R.id.typeselection).setVisibility(View.VISIBLE);
                    MappedSpinner sp = findViewById(R.id.sp_typeselector);
                    sp.setAdapter(matchingDevicetypes);

                    sp.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
                        @Override
                        public void onItemSelected(final AdapterView<?> adapterView,
                                                   final View view, final int pos, final long l) {
                            Emulation.InputDeviceType t = Emulation.InputDeviceType.valueOf(
                                    ((MappedSpinner.MappedSpinnerElement) sp.getAdapter()
                                            .getItem(pos)).getId());
                            setInputDeviceType(joy, t);
                        }

                        @Override
                        public void onNothingSelected(final AdapterView<?> adapterView) {

                        }
                    });
                    int selected = 0;
                    String stored = mSettingsController.getCurrentValue(
                            "JOYSTICK_DEVICETYPE_" + mPort,
                            Emulation.InputDeviceType.DSTICK.name());
                    for (int i = 0; i < matchingDevicetypes.getCount(); i++) {
                        if (Objects.requireNonNull(matchingDevicetypes.getItem(i)).getId()
                                .equals(stored)) {
                            selected = i;
                            break;
                        }
                    }
                    sp.setSelection(selected);
                    break;
            }
        } else {
            setInputDeviceType(null, null);
            findViewById(R.id.typeselection).setVisibility(View.GONE);
        }

    }
    private AdapterView.OnItemSelectedListener mSpinnerListener = null;
    void setOnItemSelectedListener(final AdapterView.OnItemSelectedListener listener) {
        mSpinnerListener = listener;
    }
    void setEmulatedDevicetypes(final List<Emulation.InputDeviceType> types) {
        mEmulatedDevicesTypes = types;
    }
    void setPort(final int port, final String text) {
        String tag = String.format(Locale.getDefault(), "port#%d", port);
        findViewById(R.id.sp_joystick).setTag(tag);
        ((TextView) findViewById(R.id.tv_textview)).setText(text);
    }
    void setAsSingleview() {
        for (int i = 0; i < mSpinnerAdapter.getCount(); i++) {
            SettingsActivity.JoystickEntry entry = mSpinnerAdapter.getItem(i);
            if (entry != null && entry.getJoystick() instanceof MultiplexerJoystick) {
                setSelection(i);
                findViewById(R.id.sp_joystick).setEnabled(false);
                break;
            }
        }
    }
}
