//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.annotation.SuppressLint;
import android.content.Context;
import android.util.AttributeSet;
import android.util.Log;
import android.view.MotionEvent;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

abstract class JoystickDirectionView<T extends Joystick>
        extends JoystickElementView<T> {
    protected enum Directionvalue { CENTERED, NORTH, NORTHWEST, WEST,
        SOUTHWEST, SOUTH, SOUTHEAST, EAST, NORTHEAST
    }
    protected static final Map<Integer, List<Directionvalue>> STICKIMAGES = new LinkedHashMap<>();
    protected static final List<Directionvalue> DIRECTION_IS_LEFT = Arrays.asList(
            Directionvalue.NORTHWEST,
            Directionvalue.WEST,
            Directionvalue.SOUTHWEST);
    protected static final List<Directionvalue> DIRECTION_IS_RIGHT = Arrays.asList(
            Directionvalue.NORTHEAST,
            Directionvalue.EAST,
            Directionvalue.SOUTHEAST);
    protected static final List<Directionvalue> DIRECTION_IS_UP = Arrays.asList(
            Directionvalue.NORTHWEST,
            Directionvalue.NORTH,
            Directionvalue.NORTHEAST);
    protected static final List<Directionvalue> DIRECTION_IS_DOWN = Arrays.asList(
            Directionvalue.SOUTHWEST,
            Directionvalue.SOUTH,
            Directionvalue.SOUTHEAST);
    private Directionvalue mLastDirection = Directionvalue.CENTERED;
    protected Directionvalue getLastDirection() {
        return mLastDirection;
    }
    protected void setLastDirection(final Directionvalue value)  {
        mLastDirection = value;
    }
    JoystickDirectionView(final Context context) {
        super(context);
    }

    JoystickDirectionView(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
    }

    JoystickDirectionView(final Context context, final AttributeSet attrs) {
        super(context, attrs);
    }
    private boolean mIsWatching = false;
    private JoystickStickView.Directionvalue mCurrentDirectionValue =
            JoystickStickView.Directionvalue.CENTERED;
    void setWatching(final boolean val) {
        Log.v(getClass().getSimpleName(), "setWatching " + (val ? "(true)" : "(false"));
        if (val) {
            if (!mIsWatching) {
                showNewDirection(JoystickStickView.Directionvalue.CENTERED);
                mCurrentDirectionValue = JoystickStickView.Directionvalue.CENTERED;
                if (getJoystick() != null) {
                    getJoystick().onChanged();
                }
            }
        } else {
            if (mIsWatching) {
                showNewDirection(JoystickStickView.Directionvalue.CENTERED);
                mCurrentDirectionValue = JoystickStickView.Directionvalue.CENTERED;
                if (getJoystick() != null) {
                    getJoystick().onChanged();
                }
            }
        }
        mIsWatching = val;
    }
    private static final Double ANGLE = 20d;
    private static final Double X_CENTER_RANGE = Math.sin(ANGLE * Math.PI / 180);
    private static final Double Y_CENTER_RANGE = Math.cos(ANGLE * Math.PI / 180);
    private static final Double X_DIAGONAL_RANGE = Math.sin((90 - ANGLE) * Math.PI / 180);
    private static final Double Y_DIAGONAL_RANGE = Math.cos((90 - ANGLE) * Math.PI / 180);
    private static final float DEAD_CENTER_SIZE = 0.125f;
    protected float getDeadCenterSize() {
        return DEAD_CENTER_SIZE;
    }
    private float maxOfThree(final float i1, final float i2, final float i3) {
        return Math.max(i1, Math.max(i2, i3));
    }
    protected Directionvalue calcRegion(final int action,
                                        final float x0,
                                        final float y0,
                                        final Directionvalue previous,
                                        final boolean diagonalsEnabled) {
        float x = x0;
        float y = y0;
        double d = Math.sqrt((x - mHalfwidth) * (x - mHalfwidth)
                + (y - mHalfheight) * (y - mHalfheight));

        if (d < getDeadCenterSize() * mRadius) {
            Log.v(getClass().getSimpleName(), "d zu klein");
            if (action == MotionEvent.ACTION_MOVE) {
                return previous;
            } else {
                return Directionvalue.CENTERED;
            }
        }
        if (diagonalsEnabled) {
            if (Math.abs(x - mHalfwidth) < getDeadCenterSize() * mRadius) {
                Log.v(getClass().getSimpleName(), "x zu klein");
                x = mHalfwidth;
            }
            if (Math.abs(y - mHalfheight) < getDeadCenterSize() * mRadius) {
                Log.v(getClass().getSimpleName(), "y zu klein");
                y = mHalfheight;
            }

            double xCenterRange =
                    X_CENTER_RANGE * d;
            double xDiagonalRange =
                    X_DIAGONAL_RANGE * d;
            double yCenterRange =
                    Y_CENTER_RANGE * d;
            double yDiagonalRange =
                    Y_DIAGONAL_RANGE * d;

            if (x - mHalfwidth >= -xCenterRange && x - mHalfwidth < xCenterRange
                    && y - mHalfheight < -yCenterRange) {
                return Directionvalue.NORTH;
            }
            if (x - mHalfwidth >= xCenterRange && x - mHalfwidth < xDiagonalRange
                    && y - mHalfheight < -yDiagonalRange) {
                return Directionvalue.NORTHEAST;
            }
            if (x - mHalfwidth >= xDiagonalRange && y - mHalfheight >= -yDiagonalRange
                    && y - mHalfheight < yDiagonalRange) {
                return Directionvalue.EAST;
            }
            if (x - mHalfwidth >= xCenterRange && x - mHalfwidth < xDiagonalRange
                    && y - mHalfheight >= yDiagonalRange) {
                return Directionvalue.SOUTHEAST;
            }
            if (x - mHalfwidth >= -xCenterRange && x - mHalfwidth < xCenterRange
                    && y - mHalfheight >= yCenterRange) {
                return Directionvalue.SOUTH;
            }
            if (x - mHalfwidth < -xCenterRange && x - mHalfwidth >= -xDiagonalRange
                    && y - mHalfheight >= yDiagonalRange) {
                return Directionvalue.SOUTHWEST;
            }
            if (x - mHalfwidth < -xDiagonalRange && y - mHalfheight >= -yDiagonalRange
                    && y - mHalfheight < yDiagonalRange) {
                return Directionvalue.WEST;
            }
            if (x - mHalfwidth < -xCenterRange && x - mHalfwidth >= -xDiagonalRange
                    && y - mHalfheight < -yDiagonalRange) {
                return Directionvalue.NORTHWEST;
            }
        } else {
            float dleft = mHalfwidth - x;
            float dright = x - mHalfwidth;
            float dtop = mHalfheight - y;
            float dbottom = y - mHalfheight;
            if (dleft > maxOfThree(dtop, dright, dbottom)) {
                return Directionvalue.WEST;
            }
            if (dright > maxOfThree(dleft, dtop, dbottom)) {
                return Directionvalue.EAST;
            }
            if (dtop > maxOfThree(dbottom, dleft, dright)) {
                return Directionvalue.NORTH;
            }
            if (dbottom > maxOfThree(dtop, dleft, dright)) {
                return Directionvalue.SOUTH;
            }
        }
        return previous;
    }
    private int mHalfwidth = 0;
    protected int getHalfwidth() {
        return mHalfwidth;
    }
    private int mHalfheight = 0;
    protected int getHalfheight() {
        return mHalfheight;
    }
    private int mRadius = 0;

    protected abstract Directionvalue updateDirectionValue(int action, float x, float y,
                                                           Directionvalue currentDirection);
    protected boolean isWatching() {
        return mIsWatching;
    }
    protected abstract void showNewDirection(Directionvalue dv);
    protected static final int UNLIMITED_MOVE_RADIUS = -1;
    protected abstract int getDragRadius(int touchradius);

    @Override
    public boolean onTouchEvent(final MotionEvent event) {
        int d;
        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                d = (int) (Math.sqrt((event.getX() - mHalfwidth) * (event.getX() - mHalfwidth)
                        + (event.getY() - mHalfheight) * (event.getY() - mHalfheight)));
                if (d < mRadius) {
                    setWatching(true);
                    mCurrentDirectionValue = updateDirectionValue(event.getAction(),
                            event.getX(), event.getY(), mCurrentDirectionValue);
                    return true;
                }
                break;
            case MotionEvent.ACTION_UP:
                if (isWatching()) {
                    setWatching(false);
                    return true;
                }
                break;
            case MotionEvent.ACTION_MOVE:
                    if (getDragRadius(mRadius) == UNLIMITED_MOVE_RADIUS
                            || Math.sqrt(((event.getX() - mHalfwidth)
                            * (event.getX() - mHalfwidth)
                            + (event.getY() - mHalfheight)
                            * (event.getY() - mHalfheight)))
                            <= getDragRadius(mRadius)) {

                        float x;
                        float y;
                        if (Math.abs(event.getX() - mHalfwidth)
                                < getDeadCenterSize() * mRadius) {
                            x = mHalfwidth;
                        } else {
                            x = event.getX();
                        }
                        if (Math.abs(event.getY() - mHalfheight)
                                < getDeadCenterSize() * mRadius) {
                            y = mHalfheight;
                        } else {
                            y = event.getY();
                        }
                        float lastEventRawx = event.getRawX();
                        float lastEventRawy = event.getRawY();
                        Log.v(getClass().getSimpleName(),
                                String.format("Position inside: [%d,%d]",
                                        (int) lastEventRawx, (int) lastEventRawy));
                        mCurrentDirectionValue = updateDirectionValue(
                                event.getAction(), x, y, mCurrentDirectionValue);
                        setWatching(true);
                        return true;
                    }
                break;
            default:
                Log.v(getClass().getSimpleName(), event.toString());

        }

        return super.onTouchEvent(event);
    }

    @SuppressLint("ClickableViewAccessibility")
    protected void initcontrol() {
        setLongClickable(false);
        setTag("CENTERED");

        addOnLayoutChangeListener(
                (v, left, top, right, bottom, oldLeft, oldTop, oldRight, oldBottom) -> {
                    if (getRotation() == 0) {
                        mHalfwidth = (right - left) / 2;
                        mHalfheight = (bottom - top) / 2;
                        mRadius = (int) (Math.sqrt(
                                Math.pow(mHalfwidth - (v.getPaddingLeft() + v.getPaddingRight()), 2)
                                        + Math.pow(mHalfheight - (v.getPaddingTop()
                                        + v.getPaddingBottom()), 2f)));
                    }
                });
    }
}
