//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.app.Activity;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.res.TypedArray;
import android.util.AttributeSet;
import android.view.KeyEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewStub;
import android.widget.LinearLayout;
import android.widget.TextView;

import java.util.LinkedList;
import java.util.List;


public final class GroupHeader extends LinearLayout {
    private int mTarget;
    private boolean mOpen;
    int getTargetId() {
        return mTarget;
    }
    private StateChangedListener mStateChangedListener = null;

    /**
     * Simple constructor to use when creating a GroupHeader from code.
     * @param context The Context the view is running in, through which it can access the
     *                current theme, resources, etc.
     */
    public GroupHeader(final Context context) {
        super(context);
        init(null, 0);
    }

    /**
     * Constructor that is called when inflating a GroupHeader from XML.
     * This is called when a view is being constructed from an XML file,
     * supplying attributes that were specified in the XML file.
     * his version uses a default style of 0, so the only attribute values applied are
     * those in the Context's Theme and the given AttributeSet.
     * @param context The Context the view is running in, through which it can access the current
     *                theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     */
    public GroupHeader(final Context context, final AttributeSet attrs) {
        super(context, attrs);
        init(attrs, 0);
    }
    /**
     * Perform inflation from XML and apply a class-specific base style from a theme attribute.
     * This constructor of View allows subclasses to use their own base style
     * when they are inflating. For example, a Button class's constructor would call
     * this version of the super class constructor and supply R.attr.buttonStyle for defStyleAttr;
     * this allows the theme's button style to modify all of the base view attributes
     * (in particular its background) as well as the Button class's attributes.
     * @param context The Context the view is running in, through which it can access
     *                the current theme, resources, etc.
     * @param attrs The attributes of the XML tag that is inflating the view.
     *              This value may be null.
     * @param defStyle An attribute in the current theme that contains a reference to
     *                 a style resource that supplies default values for the view.
     *                 Can be 0 to not look for defaults.
     */
    public GroupHeader(final Context context, final AttributeSet attrs, final int defStyle) {
        super(context, attrs, defStyle);
        init(attrs, defStyle);
    }

    /**
     * Set the the button state and make the corresponding View visible/gone.
     * @param value true to show the correspondign view, false to hide it.
     */
    void setViewVisible(final boolean value) {
        updateState(value);
    }

    /**
     * Check if corresponding view was set to visible or invisible by the button.
     * @return true if the view was set to visible, false if set to invisible.
     */
    boolean isViewVisible() {
        return mOpen;
    }
    private Activity getActivity() {
        Context context = getContext();
        while (context != null) {
            if (context instanceof Activity) {
                return (Activity) context;
            }
            if (context instanceof ContextWrapper) {
                context = ((ContextWrapper) context).getBaseContext();
            } else {
                return null;
            }
        }
        return null;
    }
    private final static float ROTATE_TIP_RIGHT = 0f;
    private final static float ROTATE_TIP_BOTTOM = 90f;
    private void inflateTarget(final Activity activity, final ViewGroup vg) {
        for (int i = 0; i < vg.getChildCount(); i++) {
            View child = vg.getChildAt(i);
            if (child instanceof ViewGroup) {
                inflateTarget(activity, (ViewGroup) child);
            } else if (child instanceof ViewStub) {
                child.setVisibility(View.VISIBLE);
                int id = ((ViewStub) child).getInflatedId();
                ViewGroup target = activity.findViewById(id);
                inflateTarget(activity, target);

            }
        }
        if (activity instanceof SettingsActivity) {
            ((SettingsActivity) activity).addView(vg);
        }
    }
    private void updateState(final boolean newOpen) {
        View v = getRootView().findViewById(mTarget);
        if (v != null) {
            if (newOpen) {
                Activity activity = getActivity();
                if (activity != null) {
                    int id;
                    if (v instanceof ViewStub) {
                        ((ViewStub) v).inflate();
                        id = ((ViewStub) v).getInflatedId();
                    } else {
                        id = v.getId();
                    }
                    if (v instanceof ViewStub) {
                        v = activity.findViewById(((ViewStub) v).getInflatedId());
                    }

                    if (v instanceof ViewGroup) {
                        inflateTarget(activity, (ViewGroup) v);
                    }
                    v.setVisibility(View.VISIBLE);
                    post(() -> setNextFocusDownId(id));
                }
            } else {
                v.setVisibility(View.GONE);
            }
        }
        mOpen = newOpen;
        findViewById(R.id.tv_state).setRotation(mOpen ? ROTATE_TIP_BOTTOM : ROTATE_TIP_RIGHT);
        if (mStateChangedListener != null) {
            mStateChangedListener.onStateChanged(this, newOpen);
        }

    }
    private final List<Integer> mKeysPressed = new LinkedList<>();
    private void init(final AttributeSet attrs, final int defStyle) {
        // Load attributes
        @SuppressWarnings("resource")
        final TypedArray a = getContext().obtainStyledAttributes(
                attrs, R.styleable.GroupHeader, defStyle, 0);
        mTarget = a.getResourceId(R.styleable.GroupHeader_target, View.NO_ID);
        inflate(getContext(), R.layout.view_groupheader, this);
        TextView tvTitle = findViewById(R.id.tv_title);
        tvTitle.setText(a.getResourceId(R.styleable.GroupHeader_title,
                R.string.set_at_runtime));
        updateState(a.getBoolean(R.styleable.GroupHeader_open, false));
        a.recycle();
        setOnKeyListener((v, keycode, keyEvent) -> {
            if (EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains(keycode)) {
                if (keyEvent.getAction() == KeyEvent.ACTION_DOWN) {
                    mKeysPressed.add(keycode);
                    return true;
                }
                if (mKeysPressed.contains(keycode) && keyEvent.getAction() == KeyEvent.ACTION_UP) {
                    if (!mOpen) {
                        requestFocus();
                        getChildAt(0).requestFocus();
                    }
                    updateState(!mOpen);
                    mKeysPressed.remove(Integer.valueOf(keycode));
                    return true;

                }
            }
            return false;
        });
        findViewById(R.id.root).setOnClickListener(view -> updateState(!mOpen));

    }
    interface StateChangedListener {
        void onStateChanged(GroupHeader view, boolean isOpen);
    }

    /**
     * Listener to be called when section is opened or closed by user.
     * @param listener a {@link StateChangedListener} or null
     */
    void setStateChangedListener(final StateChangedListener listener) {
        mStateChangedListener = listener;
    }
}
