//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.Context;
import android.graphics.Point;
import android.util.Log;
import android.view.View;

import androidx.appcompat.widget.PopupMenu;
import androidx.lifecycle.ViewModel;

import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

public final class EmulationViewModel extends ViewModel implements Serializable {
    private final Map<String, String> mSoftkeyText = new LinkedHashMap<>();
    private boolean mKeyboardRequesteByEmulation = false;
    private boolean mKeyboardForced = false;
    private final Set<Integer> mMouseDevices = new HashSet<>();
    private final SoftkeyConfigView.SoftkeyList mUiSoftkeys
            = new SoftkeyConfigView.SoftkeyList();


    void addJsSoftkey(final String id, final String value) {
        mSoftkeyText.put(id, value);
    }

    void removeJsSoftkey(final String id) {
        mSoftkeyText.remove(id);
    }

    boolean hasJsSoftkey(final String id) {
        return mSoftkeyText.containsKey(id);
    }

    String getJsSoftkey(final String id) {
        if (mSoftkeyText.containsKey(id)) {
            return mSoftkeyText.get(id);
        }
        return null;
    }
    void setUiSoftkeys(final SoftkeyConfigView.SoftkeyList keys) {
        mUiSoftkeys.clear();
        mUiSoftkeys.addAll(keys);
    }
    SoftkeyConfigView.SoftkeyList getUiSoftkeys() {
        return mUiSoftkeys;
    }
    Set<String> getSoftkeyIds() {
        return mSoftkeyText.keySet();
    }

    private final Map<String, Boolean> mSoftkeyEnabled = new LinkedHashMap<>();

    boolean isSoftkeyEnabled(final String key) {
        if (mSoftkeyEnabled.containsKey(key)) {
            return Boolean.TRUE.equals(mSoftkeyEnabled.get(key));
        }
        return false;
    }

    void setSoftkeyEnabled(final String key, final boolean value) {
        mSoftkeyEnabled.put(key, value);
    }
    private boolean mPopupMenuVisible = false;

    boolean isPopupMenuVisible() {
        return mPopupMenu != null && mPopupMenuVisible;
    }

    void setPopupMenuVisible(final boolean value) {
        mPopupMenuVisible = value;
    }

    private PopupMenu mPopupMenu = null;

    PopupMenu getPopupMenu() {
        return mPopupMenu;
    }

    void setPopupMenu(final PopupMenu m) {
        mPopupMenu = m;
    }

    private boolean mAskingForPermissions = false;

    boolean isAskingForPermissions() {
        return mAskingForPermissions;
    }

    void setAskingForPermissions(final boolean value) {
        mAskingForPermissions = value;
    }

    private boolean mKeyboardUsed = false;
    private final Map<Integer, Float> mScreenAspectRatio = new HashMap<>();
    private transient Emulation mEmulation = null;
    private boolean mSoftwareKeyboardAvailable = false;
    private boolean mHardwareKeyboardAvailable = false;
    private final LinkedHashMapWithDefault<Boolean, Boolean> mZoomInitialized =
            new LinkedHashMapWithDefault<>(false);
    private final LinkedHashMapWithDefault<Boolean, Boolean> mZoomValue =
            new LinkedHashMapWithDefault<>(false);
    private final LinkedHashMap<Boolean, Integer> mKeyboardVisible =
            new LinkedHashMap<Boolean, Integer>() {{
                put(true, View.GONE);
                put(false, View.VISIBLE);
            }};
    private boolean mPausedByUser = false;

    void setNotPausedByUser() {
        mPausedByUser = false;
    }

    boolean isPausedByUser() {
        return mPausedByUser;
    }

    private boolean mStopped = false;
    private final Map<Integer, Float> mPixelAspectRatio = new LinkedHashMap<>();
    private String mSnapshotPath = null;
    private boolean mResumed = false;


    Float getScreenAspectRatio(final int displayId) {
        if (mScreenAspectRatio.containsKey(displayId)) {
            return mScreenAspectRatio.get(displayId);
        }
        return 0f;
    }

    void setScreenAspectRatio(final int displayId, final float screenAspectRatio) {
        mScreenAspectRatio.put(displayId, screenAspectRatio);
    }

    void setKeyboardUsed() {
        mKeyboardUsed = true;
    }

    boolean isKeyboardUsed() {
        return mKeyboardUsed;
    }

    Emulation getEmulation() {
        return mEmulation;
    }

    void setEmulation(final Emulation emulation) {
        if (emulation == null) {
            Log.v(getClass().getSimpleName(), "setEmulation (null");
        } else {
            Log.v(getClass().getSimpleName(), "setEmulation (" + emulation + ")");
        }
        mEmulation = emulation;
    }

    boolean isSoftwareKeyboardAvailable() {
        return mSoftwareKeyboardAvailable;
    }

    void setSoftwareKeyboardAvailable(final boolean softwareKeyboardAvailable) {
        mSoftwareKeyboardAvailable = softwareKeyboardAvailable;
    }

    @SuppressWarnings("unused")
    boolean isHardwareKeyboardAvailable() {
        return mHardwareKeyboardAvailable;
    }

    void setHardwareKeyboardAvailable(final boolean hardwareKeyboardAvailable) {
        mHardwareKeyboardAvailable = hardwareKeyboardAvailable;
    }

    LinkedHashMapWithDefault<Boolean, Boolean> getZoomInitialized() {
        return mZoomInitialized;
    }

    LinkedHashMapWithDefault<Boolean, Boolean> getZoomValue() {
        return mZoomValue;
    }

    Integer getKeyboardVisible(final boolean isInLandscape) {
        return mKeyboardVisible.get(isInLandscape);
    }
    private final LinkedHashMap<Boolean, Integer> mKeyboardInitialVisible =
            new LinkedHashMap<Boolean, Integer>() {{
                put(true, View.GONE);
                put(false, View.VISIBLE);
            }};
    boolean isKeyboardVisibibleFromStart(final boolean isInLandscape) {
        return Integer.valueOf(View.VISIBLE).equals(mKeyboardInitialVisible.get(isInLandscape));

    }
    void initializeKeyboardDefaultVisibility(final Context ctx, final EmulationConfiguration conf) {
        updateKeyboardDefaultVisibility(ctx, conf);
        mKeyboardInitialVisible.clear();
        mKeyboardInitialVisible.putAll(mKeyboardVisible);
    }
    void updateKeyboardDefaultVisibility(final Context ctx, final EmulationConfiguration conf) {
        Useropts opts = new Useropts();
        opts.setCurrentEmulation(ctx, conf.getEmulatorId(), conf.getId());
        String val = opts.getStringValue(
                ctx.getString(R.string.useropt_keyboard_default_visibility), "");
        if (val.isEmpty()) {
            mKeyboardVisible.put(false, conf.isKeyboardUsed() ? View.VISIBLE : View.GONE);
            mKeyboardVisible.put(true, View.GONE);
        } else {
            String always = ctx.getString(R.string.value_keyboard_visible_always);
            String portrait = ctx.getString(R.string.value_keyboard_visible_portrait);
            mKeyboardVisible.put(false, (val.equals(always) || val.equals(portrait)
                    ? View.VISIBLE : View.GONE));
            mKeyboardVisible.put(true, val.equals(always) ? View.VISIBLE : View.GONE);

        }
    }
    void setKeyboardVisible(final boolean isInLandscape, final int visibility) {
        mKeyboardVisible.put(isInLandscape, visibility);
    }

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    boolean isStopped() {
        return mStopped;
    }

    void setStopped(final boolean screenlock) {
        mStopped = screenlock;
    }

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    Float getPixelAspectRatio(final int displayId) {
        if (mPixelAspectRatio.containsKey(displayId)) {
            return mPixelAspectRatio.get(displayId);
        }
        return 0f;
    }

    void setPixelAspectRatio(final int displayId, final float pixelAspectRatio) {
        mPixelAspectRatio.put(displayId, pixelAspectRatio);
    }

    String getSnapshotPath() {
        return mSnapshotPath;
    }

    void setSnapshotPath(final File snapshotPath) {
        mSnapshotPath = snapshotPath.getAbsolutePath();
    }

    private final Map<Integer, Integer> mCanvasX = new HashMap<>();
    private final Map<Integer, Integer> mCanvasY = new HashMap<>();


    void setCanvasSize(final int displayId, final Point p) {
        mCanvasX.put(displayId, p.x);
        mCanvasY.put(displayId, p.y);
    }

    Point getCanvasSize(final int displayId) {
        Integer x = 1;
        Integer y = 1;
        if (mCanvasX.containsKey(displayId) && mCanvasY.containsKey(displayId)) {
            x = mCanvasX.get(displayId);
            y = mCanvasY.get(displayId);
        }
        if (x != null && y != null) {
            return new Point(x, y);
        } else {
            return new Point(1, 1);
        }
    }

    private EmulationConfiguration mConfiguration = null;

    void setConfiguration(final EmulationConfiguration conf) {
        mConfiguration = conf;
    }

    EmulationConfiguration getConfiguration() {
        return mConfiguration;
    }

    private final ArrayList<Integer> mOpenSettingsSections = new ArrayList<>();

    void setOpenSettingsSections(final List<Integer> open) {
        mOpenSettingsSections.clear();
        if (open != null) {
            mOpenSettingsSections.addAll(open);
        }
        Log.v("mOpenSettingsSections", "size set to " + mOpenSettingsSections.size());
    }

    private Map<Integer, EmulationConfiguration.PreferredJoystickType> mPreferredJoysticktypes
            = null;

    Map<Integer, EmulationConfiguration.PreferredJoystickType> getPreferredJoystickTypes() {
        return mPreferredJoysticktypes;
    }

    void setPreferredJoystickTypes(
            final Map<Integer, EmulationConfiguration.PreferredJoystickType> val) {
        if (val != null) {
            mPreferredJoysticktypes = new HashMap<>(val);
        } else {
            mPreferredJoysticktypes = new HashMap<>();
        }

    }

    void setPreferredJoystickType(final int port,
                                  final EmulationConfiguration.PreferredJoystickType val) {
        mPreferredJoysticktypes.put(port, val);
    }

    ArrayList<Integer> getOpenSettingsSections() {
        return mOpenSettingsSections;
    }


    private final List<Object> mDiskdrives = new LinkedList<>();
    private final List<Object> mTapedrives = new LinkedList<>();

    void updateDiskdriveList(final List<?> diskdrives) {
        synchronized (mDiskdrives) {
            mDiskdrives.clear();
            mDiskdrives.addAll(diskdrives);
        }
    }

    private final Set<Object> mActiveDiskdrives = new HashSet<>();

    void setDiskdriveState(final Object drive, final boolean active) {
        synchronized (mActiveDiskdrives) {
            if (active) {
                mActiveDiskdrives.add(drive);
            } else {
                mActiveDiskdrives.remove(drive);
            }
        }
    }

    void updateTapedriveList(final List<?> tapedrives) {
        synchronized (mTapedrives) {
            mTapedrives.clear();
            if (!tapedrives.isEmpty()) {
                mTapedrives.addAll(tapedrives);
            }
        }
    }

    private final List<Object> mTapedriveMotors = new LinkedList<>();

    void setTapedriveMotor(final Object tape, final boolean active) {
        if (active && mTapedriveMotors.contains(tape)) {
            mTapedriveMotors.add(tape);
        } else {
            mTapedriveMotors.remove(tape);
        }
    }


    List<?> getDiskdriveList() {
        synchronized (mDiskdrives) {
            return mDiskdrives;
        }
    }

    boolean getDiskdriveState(final Object drive) {
        synchronized (mActiveDiskdrives) {
            return mActiveDiskdrives.contains(drive);
        }
    }

    List<?> getTapedriveList() {
        synchronized (mTapedrives) {
            return mTapedrives;
        }
    }


    private final Set<Object> mPauseReasons = new HashSet<>();

    void addPauseReason(final Object reason) {
        mPauseReasons.add(reason);
    }

    boolean removePauseReason(final Object reason) {
        return mPauseReasons.remove(reason);
    }

    boolean hasNoPauseReasons() {
        return mPauseReasons.isEmpty();
    }

    void clearPauseReasons() {
        mPauseReasons.clear();
    }

    void addActiveMouse(final int deviceId) {
        mMouseDevices.add(deviceId);
    }

    void removeActiveMouse(final int deviceId) {
        mMouseDevices.remove(deviceId);
    }

    boolean containsActiveMouse(final int deviceId) {
        return mMouseDevices.contains(deviceId);
    }
    private final Set<Integer> mJoysticksWithToggledAutofire = new HashSet<>();
    boolean isJoystickAutofireToggled(final int portnumber) {
        return mJoysticksWithToggledAutofire.contains(portnumber);

    }
    void setJoystickAutofireToggled(final int portnumber, final boolean value) {
        if (value) {
            mJoysticksWithToggledAutofire.add(portnumber);
        } else {
            mJoysticksWithToggledAutofire.remove(portnumber);
        }
    }
    private static class PauseReason implements Serializable {

    }
    private PauseReason mSubActivityPauseReason = new PauseReason();
    void addSubActivityPauseReason() {
        mSubActivityPauseReason = new PauseReason();
        addPauseReason(mSubActivityPauseReason);
    }
    void removeSubActivityPauseReason() {
        if (mSubActivityPauseReason != null) {
            removePauseReason(mSubActivityPauseReason);
            mSubActivityPauseReason = null;
        }
    }
    void setResumed(final boolean value) {
        mResumed = value;
    }

    boolean isResumed() {
        return mResumed;
    }

    private Serializable mLifecyclePauseReason = null;

    void setLifecyclePauseReason(final Serializable pauseReason) {
        mLifecyclePauseReason = pauseReason;
    }

    Serializable getLifecyclePauseReason() {
        return mLifecyclePauseReason;
    }

    void setKeyboardRequesteByEmulation(final boolean keyboardRequesteByEmulation) {
        mKeyboardRequesteByEmulation = keyboardRequesteByEmulation;
    }

    boolean isKeyboardRequesteByEmulation() {
        return mKeyboardRequesteByEmulation;
    }

    void setKeyboardForced(final boolean keyboardForced) {
        mKeyboardForced = keyboardForced;
    }

    boolean isKeyboardForced() {
        return mKeyboardForced;
    }
    static final int NO_DISPLAY_SELECTED = -1;
    private int mDisplayId = NO_DISPLAY_SELECTED;
    void setCurrentDisplayId(final int displayId) {
        mDisplayId = displayId;
    }
    int getCurrentDisplayId() {
        return mDisplayId;
    }
}
