package de.rainerhock.eightbitwonders;

import static android.graphics.Bitmap.CompressFormat.PNG;

import android.content.Context;
import android.content.Intent;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Point;
import android.net.Uri;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ScrollView;
import android.widget.Toast;

import androidx.annotation.IdRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.os.ConfigurationCompat;
import androidx.fragment.app.Fragment;

import com.jakewharton.processphoenix.ProcessPhoenix;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

class EmulationUiImpl implements EmulationUi {
    private final EmulationActivity mActivity;
    private final List<Joystick> mAvailableJoysticks;

    EmulationUiImpl(final EmulationActivity activity) {
        mActivity = activity;
        mAvailableJoysticks = activity.getAvailableJoysticks();
    }
    EmulationActivity getActivity() {
        return mActivity;
    }
    private EmulationViewModel getViewModel() {
        return mActivity.getViewModel();
    }
    private boolean isTv() {
        return mActivity.isTv();
    }
    int getKeyboardVisibility() {
        return findViewById(R.id.keyboardview).getVisibility();
    }
    @NonNull
    @Override
    public Context getContext() {
        return mActivity;
    }
    @Override
    public void onEmulatorException(final Emulation emu, final NativeSignalException exception) {
        Intent i = new Intent(mActivity, MainActivity.class);
        i.putExtra("configuration", getViewModel().getConfiguration());
        i.putExtra("recovery", true);
        i.putExtra("id", UUID.randomUUID().toString());
        i.putExtra("recovery-written", exception.isRecoveryWritten());
        ProcessPhoenix.triggerRebirth(mActivity, i);
    }
    @Override
    public void onEmulatorException(final Emulation emu, final Exception exception) {
        Intent i = new Intent();
        i.putExtra("recovery", false);
        i.putExtra("configuration", mActivity.getViewModel().getConfiguration());
        i.putExtra("error-message", exception.getLocalizedMessage());
        mActivity.finishWithResult(i);
    }
    @Override
    public void onEmulatorInitialized(final Emulation emu) {
        mActivity.onEmulatorInitialized(emu);
    }
    @Override
    public boolean isInRecovery() {
        return mActivity.getIntent().getBooleanExtra("recovery", false);
    }
    @Override
    public Intent newIntentForUi() {
        return new Intent(mActivity, EmulationActivity.class);
    }

    @Override
    public void runOnUiThread(final Runnable r) {
        mActivity.runOnUiThread(r);
    }
    private static final List<String> NTSC_COUNTRIES = Arrays.asList("CA", "MX", "US", "AG",
            "AW", "BS", "BB", "BZ", "BM", "VG", "KY", "CR", "CU", "DM", "DO", "SV", "GT", "GD",
            "HT", "HN", "JM", "MS", "AN", "NI", "PA", "PR", "KN", "LC", "TT", "VI", "BO", "CL",
            "CO", "EC", "GY", "PY", "PE", "SR", "VE", "JP", "PH", "KR", "TW", "BU", "KH", "VN",
            "TH", "AS", "GU", "MP", "MH", "PW", "FJ", "DG");

    @Override
    public boolean isInNtscRegion() {
        Locale l;
        try {
            //noinspection unchecked
            Class<EmulationActivity.TestInterface> clz =
                    (Class<EmulationActivity.TestInterface>)
                            Objects.requireNonNull(EmulationActivity.class.getClassLoader())
                    .loadClass("de/rainerhock/eightbitwonders/TestBase");
            Method method = clz.getMethod("getTestLocale");
            l = (Locale) method.invoke(clz.newInstance());
            if (l == null) {
                l = Locale.getDefault();
            }
        } catch (ClassNotFoundException e) {
            l = Locale.getDefault();
        } catch (NoSuchMethodException | InvocationTargetException
                 | IllegalAccessException | InstantiationException e) {
            throw new RuntimeException(e);
        }
        return NTSC_COUNTRIES.contains(l.getCountry());
    }

    @Override
    public boolean isInPalRegion() {
        return !isInNtscRegion();
    }

    @Override
    public Useropts getCurrentUseropts() {
        return mActivity.getCurrentUseropts();
    }
    private <T extends View> T findViewById(final @IdRes int id) {
        return mActivity.findViewById(id);
    }
    private StatusbarView getDriveStatusListener() {
        return findViewById(R.id.st_statusbar);
    }

    @Override
    public void updateDiskdriveList(final EmulationViewModel viewModel) {
        DriveStatusListener l = getDriveStatusListener();
        if (l != null) {
            getDriveStatusListener().updateDiskdriveList(viewModel);
            getDriveStatusListener().setVisibleIfNotEmpty();

        }
    }
    private boolean mToastAboutSoftkeysSent = false;
    @Override
    public void createSoftkey(final String id, final String text) {
        if (getViewModel().hasJsSoftkey(id)) {
            destroySoftkey(id);
        }
        getViewModel().addJsSoftkey(id, text);
        getViewModel().setSoftkeyEnabled(id, true);
        if (!isTv()) {
            SoftkeysLinearLayout root = findViewById(R.id.softkeys);
            root.createSoftkey(id, text, true, getKeyboardVisibility(),
                    getViewModel().getEmulation().getSoftkeyFunctions(), null, null);
        } else {
            if (!mToastAboutSoftkeysSent) {
                findViewById(R.id.screen).post(() ->
                        Toast.makeText(getContext().getApplicationContext(),
                        R.string.softkeys_created, Toast.LENGTH_SHORT).show());
                mToastAboutSoftkeysSent = true;
            }
        }
    }
    @Override
    public void destroySoftkey(final String id) {
        SoftkeysLinearLayout root = findViewById(R.id.softkeys);
        getViewModel().removeJsSoftkey(id);
        root.destroySoftkey(id, getViewModel().getEmulation().getSoftkeyFunctions());
    }

    @Override
    public void enableSoftkey(final String key, final boolean enable) {
        getViewModel().setSoftkeyEnabled(key, enable);
        SoftkeysLinearLayout root = findViewById(R.id.softkeys);
        root.enableSoftkey(key, enable);
    }

    @Override
    public void setDiskdriveState(final Object drive, final boolean active) {
        getViewModel().setDiskdriveState(drive, active);
        getDriveStatusListener().setDiskdriveState(drive, active);
    }

    @Override
    public void updateTapedriveList(final List<?> tapedrives) {
        getViewModel().updateTapedriveList(tapedrives);
        getDriveStatusListener().updateTapedriveList(tapedrives);
    }

    @Override
    public void setTapedriveState(final Object tape, final TapedriveState state) {
        getDriveStatusListener().setTapedriveState(tape, state);
    }

    @Override
    public void setTapedriveMotor(final Object tape, final boolean active) {
        getViewModel().setTapedriveMotor(tape, active);
        getDriveStatusListener().setTapedriveMotor(tape, active);
    }

    @Override
    public void setTapedriveCounter(final Object tape, final int tapeCount) {
        getDriveStatusListener().setTapedriveCounter(tape, tapeCount);
    }

    @Override
    public void onCanvasSizeChanged(final int monitor, final int canvasWidth,
                                    final int canvasHeight,
                                    final float pixelAspectRatio,
                                    final float screenAspectRatio) {

        int display = getViewModel().getCurrentDisplayId();
        if (display == monitor || display == EmulationViewModel.NO_DISPLAY_SELECTED) {
            getViewModel().setCanvasSize(monitor, new Point(canvasWidth, canvasHeight));
            getViewModel().setPixelAspectRatio(monitor, pixelAspectRatio);
            getViewModel().setScreenAspectRatio(monitor, screenAspectRatio);
            MonitorGlSurfaceView gl = mActivity.findViewById(R.id.gv_monitor);
            mActivity.runOnUiThread(() -> gl.setCanvasGeometry(
                    canvasWidth, canvasHeight, pixelAspectRatio, screenAspectRatio));
        }
    }

    @Override
    public void showFliplistDialog() {
        mActivity.showFliplistDialog();
    }

    @Override
    public void setBitmap(final int monitor, final Bitmap bitmap) {
        mActivity.setBitmap(monitor, bitmap);
    }

    @Override
    public void updateDiskdriveList(final List<?> diskdrives) {
        getViewModel().updateDiskdriveList(diskdrives);
        DriveStatusListener l = getDriveStatusListener();
        if (l != null) {
            getDriveStatusListener().updateDiskdriveList(getViewModel());
        }

    }

    @Override
    public Bitmap getCurrentBitmap() {
        return mActivity.getCurrentBitmap();
    }

    @Override
    public boolean isKeycodeAvailable(final int keycode) {
        for (Joystick joy: mAvailableJoysticks) {
            if (joy.deviceHasButtonWithKeycode(keycode)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void showVirtualKeyboard(final int visibleScreenpart) {
        if (!getViewModel().isHardwareKeyboardAvailable()) {
            getViewModel().setKeyboardRequesteByEmulation(true);
            final MonitorGlSurfaceView gl = findViewById(R.id.gv_monitor);
            final ScrollView sv = findViewById(R.id.scroll);
            Runnable r;
            switch (visibleScreenpart) {
                case (EmulationUi.SCREENPART_TOP):
                    r = () -> sv.fullScroll(ScrollView.FOCUS_UP);
                    break;
                case (EmulationUi.SCREENPART_BOTTOM):
                    r = () -> sv.fullScroll(ScrollView.FOCUS_DOWN);
                    break;
                case (EmulationUi.SCREENPART_CENTER):
                    r = () -> sv.scrollTo(0, (gl.getHeight() - sv.getHeight()) / 2);
                    break;
                default:
                    return;
            }
            getViewModel().setKeyboardForced(true);
            runOnUiThread(() -> {
                mActivity.setKeyboardVisibility(View.VISIBLE);
                sv.addOnLayoutChangeListener(new View.OnLayoutChangeListener() {
                    @Override
                    //CHECKSTYLE DISABLE ParameterNumber FOR 1 LINES
                    public void onLayoutChange(final View view, final int i, final int i1,
                                               final int i2, final int i3, final int i4,
                                               final int i5, final int i6, final int i7) {
                        r.run();
                        sv.removeOnLayoutChangeListener(this);
                    }
                });
            });
        }
    }
    @Override
    public void restoreVirtualKeyboardVisibility() {
        if (getViewModel().isKeyboardRequesteByEmulation()) {
            getViewModel().setKeyboardRequesteByEmulation(false);
            getViewModel().setKeyboardForced(false);
            runOnUiThread(() -> mActivity.setKeyboardVisibility(View.GONE));
        }
    }

    @Override
    public boolean getDeviceFeature(final int feature) {
        switch (feature) {
            case DEVICEFEATURE_HARDWARE_KEYBOARD:
                return getViewModel().isHardwareKeyboardAvailable();
            case DEVICEFEATURE_TOUCH:
                return mActivity.getPackageManager()
                        .hasSystemFeature("android.hardware.touchscreen");
            case DEVICEFEATURE_TV:
                return isTv();
            default:
                return false;
        }
    }
    private List<Field> getStaticInts(@SuppressWarnings("SameParameterValue") final Class<?> cls) {
        List<Field> ret = new LinkedList<>();
        Field[] declaredFields = cls.getDeclaredFields();
        for (Field field : declaredFields) {
            if (java.lang.reflect.Modifier.isStatic(field.getModifiers())) {
                if (field.getType() == int.class) {
                    ret.add(field);
                }
            }
        }
        return ret;
    }
    @Override
    public void getConstants(final EmulationUi.SetConstantsCallback callback) {
        //noinspection rawtypes
        for (Class clz: Arrays.asList(EmulationUi.class, Joystick.class)) {
            for (Field field : getStaticInts(clz)) {
                try {
                    callback.setFeature(field.getName(), field.getInt(null));
                } catch (IllegalAccessException e) {
                    if (BuildConfig.DEBUG) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
        for (int keycode : Joystick.MAPPED_BUTTONS.values()) {
            callback.setFeature(KeyEvent.keyCodeToString(keycode), keycode);
        }
    }

    @Override
    public String getSystemLanguage() {
        Locale l;
        try {
            //noinspection unchecked
            Class<EmulationActivity.TestInterface> clz =
                    (Class<EmulationActivity.TestInterface>) Objects.requireNonNull(
                                    EmulationActivity.class.getClassLoader())
                            .loadClass("de/rainerhock/eightbitwonders/TestBase");
            Method method = clz.getMethod("getTestLocale");
            l = (Locale) method.invoke(clz.newInstance());
        } catch (ClassNotFoundException e) {
            l = ConfigurationCompat.getLocales(Resources.getSystem().getConfiguration()).get(0);
        } catch (NoSuchMethodException | IllegalAccessException | InstantiationException
                 | InvocationTargetException e) {
            throw new RuntimeException(e);
        }
        if (l == null) {
            l = Locale.getDefault();
        }
        return l.getLanguage();
    }

    @Override
    public byte[] getContentData(final String address) {
        return mActivity.getContentData(getViewModel().getConfiguration(), address);
    }
    @Override
    public void putContentData(final String address,
                               final byte[] data) {
        mActivity.putContentData(getViewModel().getConfiguration(), address, data);

    }
    @Override
    public int getContentAccess(final String adress) {
        return FileUtil.getContentAccess(mActivity, adress);
    }
    @Override
    public String getFileName(final Uri uri) {
        return FileUtil.getFileName(mActivity, uri);
    }
    @Override
    public File tryToGetAsFile(final Uri uri) {
        return FileUtil.tryToGetAsFile(mActivity, uri);
    }

    @Override
    public File getInitialSnapshotPath(final EmulationConfiguration conf, final int level) {
        return mActivity.getInitialSnapshotPathImpl(conf, level);
    }
    private static final int FULLQUALITY = 100;
    @Override
    public void storeJunitScreenshot(final Bitmap bmp, final String filename) {
        try {
            Class<?> clz = Objects.requireNonNull(getClass().getClassLoader())
                    .loadClass("org.junit.Test");
            if (clz != null) {
                File target = new File(getContext().getCacheDir(),
                        filename.endsWith(".png") ? filename : filename + ".png");
                try {
                    bmp.compress(PNG, FULLQUALITY, new FileOutputStream(target));
                } catch (FileNotFoundException e) {
                    // should not happen
                }
            }
        } catch (ClassNotFoundException e) {
            // nothing;
        }
    }

    @Override
    public EmulationUi.JoystickToKeysHelper
    createJoystickToKeysHelper(final int port, final Emulation.SoftkeyFunctions mapper) {
        return new JoystickToKeysHelperImpl(mActivity, port, mapper);
    }

    @Nullable
    @Override
    public File getDirtyClosedSaveState(final EmulationConfiguration conf) {
        return mActivity.getInstanceStatePath(conf);
    }
    private final Map<View, Fragment> mTemporaryFragments = new HashMap<>();
    @Override
    public void setVisibleMonitor(final int active) {
        mActivity.setVisibleMonitor(active);
        mActivity.restoreStatusbar(findViewById(R.id.st_statusbar));
    }
    @Override
    public void removeInternalKeyboardFragmentView(final View v) {
        if (mTemporaryFragments.containsKey(v)) {
            Fragment f = mTemporaryFragments.get(v);
            if (f != null && mActivity.getSupportFragmentManager().getFragments().contains(f)) {
                mActivity.getSupportFragmentManager().beginTransaction().remove(f).commitNow();
                mTemporaryFragments.remove(v);
            }
        }
    }
    @Override
    public View createInternalKeyboardFragmentView(final KeyboardFragment f) {
        String tag = new Object().toString();
        mActivity.getSupportFragmentManager().beginTransaction().add(f, tag).commitNow();
        LayoutInflater inflater = (LayoutInflater) mActivity
                .getSystemService(Context.LAYOUT_INFLATER_SERVICE);
        ViewGroup vg = new ViewGroup(mActivity) {
            @Override
            protected void onLayout(final boolean changed, final int l, final int t,
                                    final int r, final int b) {

            }
        };
        mTemporaryFragments.put(vg, f);
        return f.onCreateView(inflater, vg, f.getArguments());
    }

}
