//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ProgressBar;
import android.widget.TextView;

import static de.rainerhock.eightbitwonders.DownloaderFactory.AdditionalDownloads.DownloadException;
import static de.rainerhock.eightbitwonders.SettingsActivity.CONFIGURATION;

import androidx.activity.result.ActivityResultLauncher;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.DialogFragment;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.lifecycle.ViewModelProvider;

import java.io.Serializable;
import java.util.Objects;

/**
 * This class implements launching an EmulationActivity.
 */
abstract class EmulationLauncherActivity extends BaseActivity {
    private static boolean mEmulationRunning = false;

    abstract ActivityResultLauncher<Intent> getEmulationContract();
    private final ActivityResultLauncher<Intent> mRunEmulationContract = getEmulationContract();

    protected static boolean isEmulationRunning() {
        return mEmulationRunning;
    }

    protected static void setEmulationRunning(final boolean emulationRunning) {
        EmulationLauncherActivity.mEmulationRunning = emulationRunning;
    }
    protected final Intent createStartEmulationIntent(final EmulationConfiguration conf) {
        Intent intent = new Intent(this, EmulationActivity.class);
        intent.putExtra(CONFIGURATION, conf);
        return intent;

    }
    protected final void launchEmulation(final Intent intent) {

        mRunEmulationContract.launch(intent);
    }
    protected final void openLocalFile() {
        if (!openFile(false)) {
            showErrorDialog(getString(R.string.could_not_access_storage),
                    getString(R.string.check_content_provider),
                    () -> {
                    });
        }
    }
    @Override
    protected void onResume() {

        FragmentManager fm = getSupportFragmentManager();
        Fragment f = fm.findFragmentByTag(STARTED_EMULATION);
        if (f != null) {
            fm.beginTransaction().remove(f).commitNow();
        }
        super.onResume();
    }

    protected final void startEmulation(final EmulationConfiguration conf) {
        FragmentManager fm = getSupportFragmentManager();
        final Serializable monitor = "";
        final String name = conf.getName();
        EmulationStartingDialogFragment f = EmulationStartingDialogFragment
                .newInstance(name, monitor);
        new Thread(() -> runOnUiThread(() -> {
            if (name != null) {
                fm.beginTransaction().add(f, STARTED_EMULATION).commitNow();
            }
            executeEmulationStart(conf, () -> downloadFiles(conf, () -> {
                Intent intent = createStartEmulationIntent(conf);

                if (fm.findFragmentByTag(DOWNLOAD_DIALOG) == null) {
                    EmulationLauncherActivity.setEmulationRunning(true);
                }
                try {
                    mRunEmulationContract.launch(intent);
                } catch (IllegalStateException e) {
                    // should not happen
                }

            }, de -> handleDownloadError(conf, de, () -> startEmulation(conf))));
        })).start();
    }

    protected final void handleDownloadError(final EmulationConfiguration conf,
                                             final DownloadException de,
                                             final Runnable runOnRetry) {
        AlertDialogBuilder builder = new AlertDialogBuilder(this);
        builder.setIcon(android.R.drawable.stat_notify_error);
        builder.setTitle(getResources().getString(R.string.download_error));
        int openFileResId;
        if (de.getZipURL() != null) {
            builder.setMessage(getResources().getString(
                    de.isSslError()
                            ? R.string.download_zip_required_ssl
                            : R.string.download_zip_required, de.getZipURL().toString()));
            openFileResId = R.string.open_file;
            builder.setNegativeButton(getResources().getString(R.string.share_link),
                    (dialogInterface, i) -> {

                        Intent intent = new Intent(Intent.ACTION_SEND);
                        intent.setType("text/plain");
                        intent.putExtra(Intent.EXTRA_TITLE, getResources().getString(
                                R.string.app_name));
                        intent.putExtra(Intent.EXTRA_TEXT, getResources().getString(
                                R.string.download_zip_text,
                                de.getZipURL().toString(),
                                getResources().getString(R.string.app_name)));
                        startActivity(BaseActivity.createChooserIntent(intent,
                                de.getZipURL().toString()));
                        dialogInterface.dismiss();
                        }


                    );
        } else {
            builder.setMessage(getResources()
                    .getString(
                            de.isSslError()
                                    ? R.string.could_not_download_file_retry_ssl
                                    : R.string.could_not_download_file_retry,
                            de.getUserfriendlyPath()));
            openFileResId = R.string.local_file;
        }
        builder.setPositiveButton(R.string.retry,
                (dialogInterface, i) -> {
                    runOnRetry.run();
                    dialogInterface.dismiss();
                });
        if (de.getHashes() != null) {
            builder.setNeutralButton(openFileResId,
                    (dialogInterface, i) -> {
                        MainActivity.MainViewModel viewModel = new ViewModelProvider(this)
                                .get(MainActivity.MainViewModel.class);
                        viewModel.setDownloadException(de);
                        viewModel.setConfiguration(conf);
                        executeWithFilesystemAccess(null, this::openLocalFile,
                                null, this::openLocalFile);
                        dialogInterface.dismiss();
                    });
        }
        builder.setOnDismissListener(dialogInterface -> {
            FragmentManager fm = getSupportFragmentManager();
            Fragment fStart = fm.findFragmentByTag(STARTED_EMULATION);
            if (fStart != null) {
                fm.beginTransaction().remove(fStart).commitNow();
            }
            dialogInterface.dismiss();

        });
        builder.create().show();
    }
    abstract static class PleaseWaitDialogFragment extends DialogFragment {
        @SuppressLint("InflateParams")
        @Nullable
        @Override
        public View onCreateView(final @NonNull LayoutInflater inflater,
                                 final @Nullable ViewGroup container,
                                 final @Nullable Bundle savedInstanceState) {

            return inflater.inflate(R.layout.view_progressdialog, null);
        }
        abstract @StringRes int getTextId();
        @Override
        public void onCreate(final @Nullable Bundle savedInstanceState) {
            super.onCreate(savedInstanceState);
            setStyle(DialogFragment.STYLE_NO_TITLE, getTheme());
        }
        @Override
        public void onViewCreated(final @NonNull View view,
                                  final @Nullable Bundle savedInstanceState) {
            ViewGroup vg = (ViewGroup) view;
            try {
                Class<?> clz = Objects.requireNonNull(getClass().getClassLoader())
                        .loadClass("org.junit.Test");
                if (clz != null) {
                    ProgressBar b = vg.findViewById(R.id.pb_indeterminate);
                    b.setIndeterminateDrawable(ContextCompat.getDrawable(
                            view.getContext(), R.drawable.ic_media_record));
                }
            } catch (ClassNotFoundException e) {
                // nothing;
            }
            ((TextView) vg.findViewById(R.id.tv_text)).setText(getTextId());
            super.onViewCreated(view, savedInstanceState);
        }

    }

    public static final class DownloadDialogFragment extends PleaseWaitDialogFragment {
        @Override
        int getTextId() {
            return R.string.downloading;
        }
    }

    public static final class EmulationStartingDialogFragment extends DialogFragment {
        public EmulationStartingDialogFragment() {
            super();
        }
        static EmulationStartingDialogFragment newInstance(
                final String name, final Serializable monitor) {
            EmulationStartingDialogFragment f = new EmulationStartingDialogFragment();
            Bundle b = new Bundle();
            b.putString(STARTED_EMULATION, name);
            b.putSerializable("MONITOR", monitor);
            f.setArguments(b);
            return f;
        }
        @SuppressLint("InflateParams")
        @Override
        public View onCreateView(final @NonNull LayoutInflater inflater,
                                 final @Nullable ViewGroup container,
                                 final @Nullable Bundle savedInstanceState) {
            String text = getResources().getString(
                    R.string.starting, requireArguments().getString(STARTED_EMULATION));
            View ret = inflater.inflate(R.layout.view_starting_emulation, null);
            ((TextView) ret.findViewById(R.id.tv_starting_emulation))
                    .setText(text);
            return ret;
        }

        @Override
        public void onCreate(final @Nullable Bundle savedInstanceState) {
            super.onCreate(savedInstanceState);
            setStyle(DialogFragment.STYLE_NO_TITLE, getTheme());
        }

        @Override
        public void onAttach(final @NonNull Context context) {
            super.onAttach(context);
            Serializable s = requireArguments().getSerializable("MONITOR");
            if (s != null) {
                synchronized (s) {
                    s.notifyAll();
                }
            }
        }
    }

}
