//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------
package de.rainerhock.eightbitwonders;

import android.annotation.SuppressLint;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.ShapeDrawable;
import android.graphics.drawable.shapes.OvalShape;
import android.os.Bundle;
import android.util.Log;
import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.content.res.ResourcesCompat;
import androidx.fragment.app.DialogFragment;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * Adds lifecycle features to Dialogfragment.
 * <ul>
 *     <li>
 *          As long as an EmulationDialogFragment is started, no Pause dialog will be shown
 *          when the activity is resumed.
 *     </li>
 *     <li>
 *         a reason for pausing the mulation is added and removed, so the emulation will not run
 *         as long as an EmulationDialogFragment is active.
 *     </li>
 * </ul>
 */
public class EmulationDialogFragment extends DialogFragment {
    /**
     * Create an instance and set pausereason, so no other one will be created.
     * As long as one or more pausereasons exist the emulation will be paused.
     * @param pauseReason a pausereason that will be removed in onDestroy
     */
    public EmulationDialogFragment(final Object pauseReason) {
        super();
        mPauseReason = pauseReason;

    }
    /**
     * Create and instance that will create a pausereason in onStart and destroy it in onStop.
     * As long as one or more pausereasons exist the emulation will be paused.
     */
    public EmulationDialogFragment() {
        super();
    }
    private List<MenuItem> mMenuItems;
    private Menu mMenu = null;
    private Integer mTitleId = View.NO_ID;

    /**
     * Set default style.
     */
    protected void setStyle() {
        setStyle(DialogFragment.STYLE_NO_TITLE, getTheme());
    }
    private boolean mDelayAfterDialog = false;
    protected final void delayAfterDismiss() {
        mDelayAfterDialog = true;
    }
    @Override
    public final void onCreate(@Nullable final Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setStyle();

    }
    private Object mPauseReason = null;

    @Override
    public final void dismiss() {
        if (getActivity() instanceof EmulationActivity) {
            EmulationActivity ea = (EmulationActivity) getActivity();
            if (ea.noActionPending()) {
                ea.unmuteEmulation();
            }
        }
        super.dismiss();
    }
    final void continueEmulation() {
        if (getActivity() instanceof EmulationActivity) {
            EmulationActivity ea = (EmulationActivity) getActivity();
            Emulation emu = ea.getViewModel().getEmulation();
            ea.unmuteEmulation();
            emu.setPaused(false);
        }
    }

    @Override
    public final void onStart() {
        super.onStart();
        if (BuildConfig.DEBUG) {
            String tag = getClass().getSimpleName();
            Exception e = new Exception();
            for (int i = 0; i < e.getStackTrace().length; i++) {
                Log.v(tag, String.format(this + ".onStart traceback [%d] %s", i,
                        e.getStackTrace()[i].toString()));
            }
        }

        if (mPauseReason == null) {
            if (getActivity() instanceof EmulationActivity) {
                mPauseReason = ((EmulationActivity) getActivity()).addPauseReason();
            }
        }
    }
    @Override
    public final void onStop() {
        super.onStop();
        //super.onStart();
        if (BuildConfig.DEBUG) {
            String tag = getClass().getSimpleName();
            Exception e = new Exception();
            for (int i = 0; i < e.getStackTrace().length; i++) {
                Log.v(tag, String.format(this + ".onStop traceback [%d] %s", i,
                        e.getStackTrace()[i].toString()));
            }
        }

        if (mPauseReason != null) {
            if (getActivity() instanceof EmulationActivity) {
                if (mDelayAfterDialog) {
                    ((EmulationActivity) getActivity()).removePauseReasonAfterDelay(mPauseReason);
                } else {
                    ((EmulationActivity) getActivity()).removePauseReason(mPauseReason);
                }
            }
            mPauseReason = null;
        }

    }

    static void addCloseMenuItem(final Menu menu) {
        boolean drawIcon = false;
        for (int i = 0; i < menu.size(); i++) {
            if (menu.getItem(0).getIcon() != null) {
                drawIcon = true;
                break;
            }
        }
        MenuItem mi = menu.add(R.string.leave_menu);
        if (drawIcon) {
            mi.setIcon(R.drawable.ic_close);
        }
        mi.setOnMenuItemClickListener(menuItem -> true);

    }
    final void setMenu(final Menu menu, final boolean showCloseMenuitem) {
        mMenu = menu;
        mMenuItems = new LinkedList<>();
        for (int i = 0; i < menu.size(); i++) {
            mMenuItems.add(menu.getItem(i));
        }
        if (showCloseMenuitem) {
            addCloseMenuItem(menu);
        }
    }
    private final Set<Integer> mDisableForJoystick = new HashSet<>();
    final void setTitle(final int resStringId) {
        mTitleId = resStringId;
    }
    final void setMenuEnableByJoystick(final int resId,
                                 @SuppressWarnings("SameParameterValue") final boolean value) {
        if (value) {
            mDisableForJoystick.remove(resId);
        } else {
            mDisableForJoystick.add(resId);
        }
    }

    private final Set<Integer> mHideDisabled = new HashSet<>();
    final void setDisabledMenuHidden(final int resId,
                               @SuppressWarnings("SameParameterValue") final boolean value) {
        if (value) {
            mHideDisabled.add(resId);
        } else {
            mHideDisabled.remove(resId);
        }

    }
    private Drawable createEmptyDrawable(final Drawable sameSize) {
        ShapeDrawable ret = new ShapeDrawable(new OvalShape());
        ret.getPaint().setColor(
                requireContext().getResources().getColor(android.R.color.transparent));
                //getContext().getResources().getColor(android.R.color.transparent));
        ret.setIntrinsicHeight(sameSize.getIntrinsicHeight());
        ret.setIntrinsicWidth(sameSize.getIntrinsicWidth());
        return ret;
    }

    interface MenuItemUpdater {
        void update(MenuItem mi);
    }
    interface CurrentInputDeviceSetter {
        void setInputDevice(InputDevice device);
    }
    private CurrentInputDeviceSetter mCurrentInputDeviceSetter = null;
    final void setCurrentInputDeviceSetter(final CurrentInputDeviceSetter setter) {
        mCurrentInputDeviceSetter = setter;
    }
    private void setInputDevice(final InputDevice device) {
        if (mCurrentInputDeviceSetter != null) {
            mCurrentInputDeviceSetter.setInputDevice(device);
        }
    }
    private void setAction(final Button b, final Runnable r) {
        b.setOnClickListener(view -> {
            setInputDevice(null);
            r.run();
        });
        b.setOnKeyListener((view, i, event) -> {

            if (event.getAction() == KeyEvent.ACTION_DOWN) {
                setInputDevice(event.getDevice());
                if (EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains((event.getKeyCode()))) {
                    return true;
                }
                if (EmulationUi.JOYSTICK_BACK_BUTTONS.contains(event.getKeyCode())) {
                    return true;
                }
            }
            if (event.getAction() == KeyEvent.ACTION_UP) {
                if (EmulationUi.JOYSTICK_MAXIMAL_ACTIONS_BUTTONS.contains((event.getKeyCode()))) {
                    boolean isRemote;
                     isRemote = event.getDevice().hasKeys(KeyEvent.KEYCODE_DPAD_CENTER)[0]
                             && event.getDevice().hasKeys(KeyEvent.KEYCODE_BACK)[0];
                     if (!isRemote) {
                         isRemote = event.getDevice().hasKeys(KeyEvent.KEYCODE_BUTTON_A)[0]
                                 && event.getDevice().hasKeys(KeyEvent.KEYCODE_BUTTON_B)[0];

                     }
                    if (isRemote || !mDisableForJoystick.contains(b.getId())) {
                        r.run();
                    } else {
                        Toast.makeText(getActivity(), R.string.use_remote_control,
                                        Toast.LENGTH_LONG)
                                .show();
                    }
                    return true;
                } else {
                    if (EmulationUi.JOYSTICK_BACK_BUTTONS.contains(event.getKeyCode())) {
                        dismiss();
                        return true;
                    }
                }
                setInputDevice(null);
            }

            return false;
        });

    }
    private List<View> getMenuViews(final Menu menu, final List<MenuItem> menuitems,
                                    final ViewGroup menuitemsView, final int resId) {
        List<View> menuViews = new LinkedList<>();
        boolean landscapeMode = useLandscapeMode();
        boolean first = true;
        for (MenuItem mi:menuitems) {
            ViewGroup entryView = (ViewGroup) LayoutInflater.from(getContext()).inflate(
                    resId, menuitemsView, false);
            LayoutInflater.from(getContext()).inflate(first
                    ? R.layout.view_dialog_first_button : R.layout.view_dialog_button, entryView);
            Button b = entryView.findViewById(R.id.bn_text);
            b.setText(mi.getTitle());
            b.setId(mi.getItemId());
            Drawable[] drawables = b.getCompoundDrawables();
            if (mi.isCheckable()) {
                if (mi.isChecked()) {
                    drawables[2] = ResourcesCompat.getDrawable(
                        requireContext().getResources(),
                        android.R.drawable.checkbox_on_background, requireContext().getTheme());
                } else {
                    drawables[2] = ResourcesCompat.getDrawable(
                        requireContext().getResources(),
                        android.R.drawable.checkbox_off_background, requireContext().getTheme());
                }
            } else {
                drawables[2] = createEmptyDrawable(
                        Objects.requireNonNull(ResourcesCompat.getDrawable(
                                requireContext().getResources(),
                                android.R.drawable.checkbox_off_background,
                                requireContext().getTheme())));
            }
            if (mi.getIcon() != null) {
                drawables[0] = mi.getIcon();
            } else {
                drawables[0] = createEmptyDrawable(
                        Objects.requireNonNull(
                                ResourcesCompat.getDrawable(requireContext().getResources(),
                                android.R.drawable.ic_media_play, requireContext().getTheme())));

            }

            b.setCompoundDrawablesRelativeWithIntrinsicBounds(drawables[0],
                    drawables[1],
                    drawables[2],
                    drawables[3]);


            if (mi.getSubMenu() == null) {
                setAction(b, () -> {
                    menu.performIdentifierAction(mi.getItemId(), 0);
                    dismiss();
                });
            } else {
                setAction(b, () -> {
                    setMenu(mi.getSubMenu(), ((BaseActivity) requireActivity()).isTv());
                    redraw(null);

                });
            }
            if (mi.isVisible()) {
                menuViews.add(entryView);
            }
            if (landscapeMode) {
                if (mHideDisabled.contains(b.getId()) && !mi.isEnabled()) {
                    entryView.setVisibility(View.GONE);
                }
                b.setEnabled(mi.isEnabled());
                b.setFocusable(mi.isEnabled());

            } else {
                entryView.setVisibility(mi.isEnabled() ? View.VISIBLE : View.GONE);
            }
            if (mi.isVisible() && mi.isEnabled()) {
                first = false;
            }
        }
        return menuViews;
    }
    final void redraw(final MenuItemUpdater updater) {
        if (mMenu != null) {
            if (updater != null) {
                for (int i = 0; i < mMenu.size(); i++) {
                    updater.update(mMenu.getItem(i));
                }
            }
            ((ViewGroup) requireView()).removeAllViews();
            ((ViewGroup) requireView()).addView(draw());
        }
    }

    /**
     * Create content view.
     * @param inflater The LayoutInflater object that can be used to inflate
     * any views in the fragment,
     * @param container If non-null, this is the parent view that the fragment's
     * UI should be attached to.  The fragment should not add the view itself,
     * but this can be used to generate the LayoutParams of the view.
     * @param savedInstanceState If non-null, this fragment is being re-constructed
     * from a previous saved state as given here.
     *
     * @return populated view.
     */
    @Nullable
    @Override
    public View onCreateView(@NonNull final LayoutInflater inflater,
                             @Nullable final ViewGroup container,
                             @Nullable final Bundle savedInstanceState) {
        if (mMenu != null) {
            return draw();

        }
        return null;

    }
    private boolean useLandscapeMode() {
        if (!requireContext().getResources().getBoolean(R.bool.is_landscape)) {
            return false;
        }
        int visibleItems = 0;
        if (mMenu != null) {
            for (int i = 0; i < mMenu.size(); i++) {
                if (mMenu.getItem(i).isVisible()) {
                    visibleItems++;
                }
            }
        }
        return visibleItems > MAX_SINGLECOL_MENUS;
    }
    private static final  int MAX_SINGLECOL_MENUS = 7;
    @SuppressLint("InflateParams")
    private View draw() {
        ViewGroup rootView;
        int singlecolId = R.id.singlecol;
        int leftcolId = R.id.leftcol;
        int rightcol = R.id.rightcol;
        int currentViewId;
        if (useLandscapeMode()) {
            rootView = (LinearLayout) LayoutInflater.from(getContext()).
                    inflate(R.layout.dialog_menu_doublerow, null, false);
            currentViewId = leftcolId;
        } else {
            rootView = (LinearLayout) LayoutInflater.from(getContext()).
                    inflate(R.layout.dialog_menu_singlerow, null, false);
            currentViewId = singlecolId;
        }
        TextView tv = rootView.findViewById(R.id.tv_title);
        if (mTitleId != View.NO_ID) {
            tv.setVisibility(View.VISIBLE);
            tv.setText(getResources().getString(mTitleId));
        } else {
            tv.setVisibility(View.GONE);
        }

        List<View> menuViews = getMenuViews(mMenu, mMenuItems, rootView,
                R.layout.fragment_popup_dialogitem);
        for (View v : menuViews) {
            ((ViewGroup) rootView.findViewById(currentViewId)).addView(v);
            if (v.getVisibility() == View.VISIBLE) {
                if (currentViewId == leftcolId) {
                    currentViewId = rightcol;
                } else {
                    if (currentViewId == rightcol) {
                        currentViewId = leftcolId;
                    }
                }
            }
        }
        setStyle();
        return rootView;
    }
}
