//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------

package de.rainerhock.eightbitwonders;

import static de.rainerhock.eightbitwonders.Joystick.PORT_NOT_CONNECTED;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.ActivityManager;
import android.app.Dialog;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.IBinder;
import android.os.PowerManager;
import android.util.Log;
import android.view.Gravity;
import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;

import androidx.activity.OnBackPressedCallback;
import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContract;
import androidx.appcompat.widget.PopupMenu;

import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.view.OneShotPreDrawListener;
import androidx.fragment.app.DialogFragment;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;
import androidx.lifecycle.LifecycleOwner;
import androidx.lifecycle.ViewModelProvider;

import java.io.File;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

/**
 * This is the activity the emulations run in. Emulations communicate with it via the
 * interface {@link EmulationUi}.
 */
public final class EmulationActivity extends DeviceSupportActivity {

    private static final String TAG = EmulationActivity.class.getSimpleName();
    static final String FRAGMENT_DIALOG = "FRAGMENT_DIALOG";
    private static final String REMOTE_KEYBOARD = "FRAGMENT_REMOTE_KEYBOARD";
    private EmulationViewModel mViewModel = null;
    private final List<Joystick> mAvailableJoysticks = new LinkedList<>();
    static final String KEYBOARDFRAGMENT = "KEYBOARDFRAGMENT";
    private InputDevice mActiveInputDevice = null;
    private boolean mMonitorSet = false;
    private SoftkeyConfigView getSoftkeyConfigView() {
        return findViewById(R.id.softkeyconfig);
    }
    private void applySettings(final Intent result) {
        evaluateSettings(result);
        getViewModel().updateKeyboardDefaultVisibility(this,
                getViewModel().getConfiguration());
        applyKeyboardSettings();
        getSoftkeyConfigView().applyUseropts(getCurrentUseropts());
    }
    private final ActivityResultLauncher<SettingsActivity.SettingsConfiguration> mSettingsLauncher
            = registerForActivityResult(new SettingsActivity.ShowSettingsContract(),
            result -> {
                applySettings(result);
                handleCommonLaunchresult(result);
            });
    private final ActivityResultLauncher<Object> mNewFilePropertiesLauncher
            = registerForActivityResult(new ActivityResultContract<Object, Intent>() {
        @Override
        public Intent parseResult(final int result, @Nullable final Intent intent) {
            handleCommonLaunchresult(intent);
            return intent;

        }

        @NonNull
        @Override
        public Intent createIntent(@NonNull final Context context, final Object dummy) {
            if (getViewModel().getEmulation().getFileCreationFunction() != null) {
                return getViewModel().getEmulation().getFileCreationFunction()
                        .getNewFilePropertiesIntent(EmulationActivity.this);
            }
            return new Intent();
        }
    }, result -> {
        if (result != null) {
            SubActivityBlockFragment.add(this);
            mCreateFileData = result.getSerializableExtra("createdata");
            createFile(result.getStringExtra("filename"));
        }
    });
    private final ActivityResultLauncher<Intent> mShareLauncher
            = registerForActivityResult(new ShareEmulationActivity.ShareContract(), result -> {
        if (result) {
            unmuteEmulation();
        } else {
            showSnapshotsDialog();
        }
    });
    private final ActivityResultLauncher<SettingsActivity.SettingsConfiguration>
            mInitialSettingsLauncher = registerForActivityResult(
            new SettingsActivity.ShowSettingsContract(), result -> {
                applySettings(result);
                getViewModel().initializeKeyboardDefaultVisibility(this,
                        getViewModel().getConfiguration());
                updateOrientation(getViewModel().getConfiguration());
                runEmulation();
                handleCommonLaunchresult(result);

            });
    private boolean mSettingsEvaluated = false;

    EmulationViewModel getViewModel() {
        if (mViewModel == null) {
            mViewModel = new ViewModelProvider(this).get(EmulationViewModel.class);
        }
        return mViewModel;
    }

    void onEmulatorInitialized(final Emulation emu) {
        synchronized ((mUpdateUiMonitor)) {
            if (!mUpdateUiDone) {
                try {
                    mUpdateUiMonitor.wait();
                } catch (InterruptedException e) {
                    Log.e(TAG, e.getMessage(), e);
                }
            }
        }
        if (mPauseEmulationImmediately) {
            emu.setPaused(true);
            mPauseEmulationImmediately = false;
        }

        ((MonitorGlSurfaceView) findViewById(R.id.gv_monitor))
                .onGeometryChanged();
    }
    void showSoftkeyDialog() {
        DialogFragment f = EmulationDialogFactory.showSoftkeyDialog();
        f.showNow(getSupportFragmentManager(), FRAGMENT_DIALOG);
    }

    protected void onFileCreated(@NonNull final Uri uri) {
        if (mCreateFileData != null && getViewModel().getEmulation().getFileCreationFunction()
                .createFile(getContentResolver(), uri, mCreateFileData)) {
            onFileOpened(uri);
        } else {
            Toast.makeText(this, getResources().getString(R.string.could_not_open_file,
                            uri.getPath() != null ? new File(uri.getPath()).getName() : "???"),
                            Toast.LENGTH_LONG).show();
            unmuteEmulation();
        }
        super.onFileCreated(uri);
    }
    static final  int PAUSE_TICK_DURATION = 100;
    static final  int PAUSE_TICK_COUNT = 10;
    private void pauseAfterTimeMachine(final @NonNull  Runnable runWhenEnding) {
        for (Joystick joy:mAvailableJoysticks) {
            joy.startStateBuffering();
        }
        Object pauseReason = addPauseReason();
        DialogFragment f = EmulationDialogFactory.createBePreparedDialog(
                () -> {
                    for (Joystick joy:mAvailableJoysticks) {
                        if (joy.isBuffered()) {
                            joy.readFromStateBuffer();
                        }
                    }
                    removePauseReason(pauseReason);
                    runWhenEnding.run();
                }
        );
        runOnUiThread(() -> f.show(getSupportFragmentManager(), FRAGMENT_DIALOG));

    }

    public static class SubActivityBlockFragment extends Fragment {
        /**
         * Needed by android, do not use.
         */
        public SubActivityBlockFragment() {
            super();
        }

        static final String TAG = "SUBACTIVITY_BLOCK";

        static void add(final EmulationActivity a) {
            a.getSupportFragmentManager().beginTransaction()
                    .add(new SubActivityBlockFragment(), TAG).commit();
        }

        @SuppressWarnings("BooleanMethodIsAlwaysInverted")
        static boolean exists(final EmulationActivity a) {
            for (Fragment f : a.getSupportFragmentManager().getFragments()) {
                if (f instanceof SubActivityBlockFragment) {
                    return true;
                }
            }
            return false;
        }

        static void remove(final EmulationActivity a) {
            Fragment toDelete = null;
            for (Fragment f : a.getSupportFragmentManager().getFragments()) {
                if (f instanceof SubActivityBlockFragment) {
                    toDelete = f;
                    break;
                }
            }
            if (toDelete != null) {
                a.getSupportFragmentManager().beginTransaction().remove(toDelete).commit();
            }
        }

    }
    void shareContent(final Intent i) {
        SubActivityBlockFragment.add(this);
        muteEmulation();
        mShareLauncher.launch(i);
    }

    private Serializable mCreateFileData = null;

    @Override
    protected boolean openFile(final boolean isRetry) {
        SubActivityBlockFragment.add(this);
        getViewModel().addSubActivityPauseReason();
        return super.openFile(isRetry);
    }
    void muteEmulation() {
        if (getViewModel().getEmulation().getSoundFunctions() != null) {
            getViewModel().getEmulation().getSoundFunctions().mute();
        }
    }
    void unmuteEmulation() {
        if (getViewModel().getEmulation().getSoundFunctions() != null) {
            getViewModel().getEmulation().getSoundFunctions().unmute();
        }
    }
    @Override
    protected void onFileOpened(final Uri uri) {

        final Runnable r = () -> {
            boolean followUpAction = false;
            Intent emulationIntent = getViewModel().getEmulation().getFileFunctions()
                    .getIntent(uri, FileUtil.getFileName(this, uri));
            if (emulationIntent != null && emulationIntent.getComponent() != null) {
                applyScreenOrientation(emulationIntent);
                if (!emulationIntent.getComponent().getClassName()
                        .equals(getClass().getName())) {
                    SubActivityBlockFragment.add(this);
                    followUpAction = true;
                    if (!getViewModel().getEmulation().isPaused()) {
                        getViewModel().getEmulation().setPaused(true,
                                () -> runOnUiThread(() ->
                                        mHandleImageLauncher.launch(emulationIntent)));
                    } else {
                        mHandleImageLauncher.launch(emulationIntent);
                    }
                }
                String errormessage = emulationIntent.getStringExtra("errormessage");
                if (errormessage != null) {
                    showErrorMessage(errormessage);
                    if (emulationIntent.getBooleanExtra("retry", false)) {
                        getViewModel().removeSubActivityPauseReason();
                        //openFile(true);
                        followUpAction = true;
                        if (!getViewModel().getEmulation().isPaused()) {
                            getViewModel().getEmulation().setPaused(true, () ->
                                    runOnUiThread(() -> openFile(true)));
                        } else {
                            openFile(true);
                        }
                    } else {
                        unmuteEmulation();
                    }
                }
            }
            if (!followUpAction) {
                unmuteEmulation();
                invalidateOptionsMenu();
                getViewModel().removeSubActivityPauseReason();
            }
        };
        if (getViewModel().getEmulation().isPaused()) {
            r.run();
        } else {
            getViewModel().getEmulation().setPaused(true, () -> runOnUiThread(r));
        }
        super.onFileOpened(uri);
    }

    @Override
    protected void onFileOpenCancelled(final boolean wasRetry) {
        unmuteEmulation();
        getViewModel().removeSubActivityPauseReason();
    }
    protected void handleCommonLaunchresult(final Intent resultData) {

        disposeInstanceState(getViewModel().getConfiguration());
        getViewModel().setStopped(false);
        if (resultData != null) {
            String message = resultData.getStringExtra(getPackageName() + ".SuccessMessage");
            if (message != null) {
                Toast.makeText(getApplicationContext(), message, Toast.LENGTH_LONG).show();
            }
            message = resultData.getStringExtra(getPackageName() + ".ErrorMessage");
            if (message != null) {
                showErrorMessage(message);
            }
        }
        invalidateOptionsMenu();
        unmuteEmulation();
        getViewModel().removeSubActivityPauseReason();
        restoreStatusbar(findViewById(R.id.st_statusbar));
    }

    void applyScreenOrientation(final @NonNull Intent i) {
        int orientation = ActivityInfo.SCREEN_ORIENTATION_UNSPECIFIED;
        switch (getEmulationScreenOrientation(getViewModel().getConfiguration())) {
            case PORTRAIT:
                orientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT;
                break;
            case LANDSCAPE:
                orientation = ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE;
                break;
            default:
                break;
        }
        i.putExtra("screen_orientation", orientation);
    }

    private EmulationConfiguration getInitialConfiguration() {
        EmulationConfiguration conf =
                getViewModel().getConfiguration();
        if (conf == null && getIntent().getAction() != null
                && getIntent().getAction().startsWith("WIDGET_CALL:")) {
            String[] parts = getIntent().getAction().split(":", 2);
            if (parts.length == 2) {
                String id = parts[1];
                conf = getWidgetConfiguration(id);
            }
        }
        if (conf == null) {
            conf = (EmulationConfiguration) getIntent()
                    .getSerializableExtra(SettingsActivity.CONFIGURATION);
        }
        return conf;
    }
    @Nullable
    private EmulationConfiguration getWidgetConfiguration(final String widgetId) {
        EmulationConfiguration conf = null;
        if (Emulationslist.getEmulationIds().contains(widgetId)) {
            conf = new ConfigurationFactory.MachineConfiguration(this, widgetId);
        } else {
            for (EmulationConfiguration c : getAssetConfigurations(
                    getResources().getAssets(), "packages", null)) {
                if (c.getLocalPath() != null
                        && c.getLocalPath().getAbsolutePath().equals(widgetId)) {
                    conf = c;
                    break;
                }
            }
            if (conf == null) {
                for (EmulationConfiguration c : getLocalConfigurations()) {
                    if (c.getLocalPath() != null
                            && c.getLocalPath().getAbsolutePath().equals(widgetId)) {
                        conf = c;
                        break;
                    }
                }
            }
        }
        return conf;
    }

    private boolean mPauseEmulationImmediately = false;
    private void evaluateSettings(final Intent result) {
        if (!mSettingsEvaluated) {
            Runnable runAfter = () -> {
                if (!isTv()) {
                    runOnUiThread(this::createTouchKeyboard);
                }
            };
            Emulation.MachineSettingsFunctions msf =
                    Emulationslist.getCurrentEmulation() != null
                            ? Emulationslist.getCurrentEmulation().getMachineSettingsFunction()
                            : null;
            if (msf != null) {
                msf.applyUseropts(getCurrentUseropts(), runAfter);
            } else {
                runAfter.run();
            }
            if (result != null) {
                if (result.getSerializableExtra(SettingsActivity.OPENSECTIONS) != null) {
                    //noinspection unchecked
                    getViewModel().setOpenSettingsSections((List<Integer>)
                            result.getSerializableExtra(SettingsActivity.OPENSECTIONS));
                }
            } else {
                runAfter.run();
            }
            mMonitorSet = false;
            updateUi(isInLandscape());
            EmulationConfiguration conf = getViewModel().getConfiguration();
            updateOrientation(conf);
            mSettingsEvaluated = true;
        }
    }

    void addOnBackPressedCallback(final LifecycleOwner owner) {
        getOnBackPressedDispatcher().addCallback(owner, new OnBackPressedCallback(true) {
            @Override
            public void handleOnBackPressed() {
                // Do custom work here
                if (isEnabled()) {
                    if (!isTv() && !getViewModel().isKeyboardVisibibleFromStart(isInLandscape())
                            && getKeyboardVisibility() == View.VISIBLE) {
                        setKeyboardVisibility(View.GONE);
                        Fragment f = getSupportFragmentManager()
                                .findFragmentByTag(KEYBOARDFRAGMENT);
                        if (getViewModel().getEmulation().getSoftwareKeyboardFunctions() != null
                                && f instanceof KeyboardFragment) {
                            getViewModel().getEmulation().getSoftwareKeyboardFunctions()
                                    .onKeyboardFragmentDismiss((KeyboardFragment) f);
                        }
                        getViewModel().setKeyboardVisible(isInLandscape(), View.GONE);
                    } else {
                        quitEmulation();
                    }
                }
            }
        });
    }
    private void setImmersiveMode() {
        View decorView = getWindow().getDecorView();
        int uiOptions = (View.SYSTEM_UI_FLAG_LAYOUT_HIDE_NAVIGATION
                | View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                | View.SYSTEM_UI_FLAG_FULLSCREEN);
        if (!BuildConfig.DEBUG || android.os.Build.VERSION.SDK_INT > Build.VERSION_CODES.KITKAT) {
            uiOptions |= View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY | View.SYSTEM_UI_FLAG_HIDE_NAVIGATION;
        }
        decorView.setSystemUiVisibility(uiOptions);
    }

    @Override
    public void onWindowFocusChanged(final boolean hasFocus) {
        super.onWindowFocusChanged(hasFocus);
        if (hasFocus) {
            setImmersiveMode();
        }
    }

    @SuppressLint("SourceLockedOrientationActivity")
    @Override
    protected void onCreate(final Bundle savedInstanceState) {
        Log.v(TAG, "onCreate");
        System.loadLibrary("jni");
        super.onCreate(savedInstanceState);
        addOnBackPressedCallback(this);
        setImmersiveMode();

        if (savedInstanceState != null) {
            mPauseEmulationImmediately = savedInstanceState.getBoolean("saved", false);
        } else {
            mPauseEmulationImmediately = false;
        }
        if (android.os.Debug.isDebuggerConnected()) {
            android.os.Debug.waitForDebugger();
        }
        setContentView(R.layout.activity_emulation);
        mSurfaceView = findViewById(R.id.gv_monitor);
        SoftkeysLinearLayout softkeys = findViewById(R.id.softkeys);
        softkeys.setOnVisibilityChangedListener(() -> {
            Log.v("softkeys", "setOnVisibilityChangedListener "
                    + softkeys.getVisibility());
            ((MonitorGlSurfaceView) findViewById(R.id.gv_monitor))
                    .onGeometryChanged();
        });
        String pkg = getApplication().getPackageName();
        String errormessage = getIntent().getStringExtra("errormessage");
        if (getIntent().getSerializableExtra(pkg + ".ViewModel") != null) {
            mViewModel = (EmulationViewModel)
                    getIntent().getSerializableExtra(pkg + ".ViewModel");
            if (mViewModel != null) {
                mViewModel.setEmulation(Emulationslist.getCurrentEmulation());
            }
        }


        EmulationConfiguration conf = getInitialConfiguration();
        getViewModel().initializeKeyboardDefaultVisibility(this, conf);
        updateOrientation(conf);
        getViewModel().setConfiguration(conf);

        prepareEmulationStart(conf);
        if (!isTv()) {
            createTouchKeyboard();
        }
        if (errormessage != null) {
            Toast.makeText(this, errormessage, Toast.LENGTH_LONG).show();
        }
        if (!EmulationWatchdogService.isRunning(this)) {
            Intent i = new Intent(this, EmulationWatchdogService.class);
            File f = mEmulationUi.getDirtyClosedSaveState(getViewModel().getConfiguration());
            if (f != null) {
                i.putExtra("path", f.getAbsolutePath());
            }
            startService(i);
        }
    }
    private boolean launchInitialSetting(final EmulationConfiguration conf) {
        Useropts opts = new Useropts();
        opts.setCurrentEmulation(this, conf.getEmulatorId(), conf.getId());
        if (conf instanceof ConfigurationFactory.StreamConfiguration
                && opts.getBooleanValue("__show_settings__", true)) {
            opts.setValue(Useropts.Scope.CONFIGURATION, "__show_settings__", false);
            SubActivityBlockFragment.add(this);
            ArrayList<Integer> opensections = new ArrayList<>(
                    getViewModel().getOpenSettingsSections());
            for (int section : Arrays.asList(R.id.joystickgroup, R.id.mobile_options)) {
                if (!opensections.contains(section)) {
                    opensections.add(section);
                }
            }
            getViewModel().setOpenSettingsSections(opensections);
            mSettingsEvaluated = false;
            mInitialSettingsLauncher.launch(createSettingsConfiguration(false));
            return true;
        }
        return false;
    }
    private void prepareEmulationStart(final @NonNull  EmulationConfiguration conf) {
        String pkg = getApplication().getPackageName();
        getViewModel().setConfiguration(conf);
        if (getViewModel().getEmulation() == null) {
            if (getIntent().getBooleanExtra(pkg + ".Relaunch",
                    false)) {
                getViewModel().setEmulation(Emulationslist.getCurrentEmulation());
            } else {
                getViewModel().setEmulation(Emulationslist.createEmulation(
                        getEmulationUi(), conf));
            }
            getViewModel().getEmulation().setEmulationUI(getEmulationUi());
            if (getViewModel().getEmulation() != null) {
                getViewModel().setSnapshotPath(getSnapshotPath(conf));
                if (getViewModel().getPreferredJoystickTypes() == null) {
                    getViewModel().setPreferredJoystickTypes(conf.getVirtualJoystickTypes());
                }
            }
        }
        if (getViewModel().getEmulation() != null) {
            conf.apply(this, getViewModel().getEmulation());
        }
        if (!launchInitialSetting(conf)) {
            findViewById(R.id.screen).addOnLayoutChangeListener(new View.OnLayoutChangeListener() {
                //CHECKSTYLE DISABLE ParameterNumber FOR 2 LINES
                @Override
                public void onLayoutChange(final View view, final int i, final int i1, final int i2,
                                           final int i3, final int i4, final int i5, final int i6,
                                           final int i7) {
                    Runnable r = () -> {
                        if (getViewModel().getEmulation() != null) {
                            if (!getViewModel().getEmulation().isRunning()) {
                                new Thread(() -> runOnUiThread(() -> {
                                    getViewModel().getEmulation().startThread();
                                    removePreparingDialogs();
                                })).start();
                            }
                        }
                    };
                    if (getIntent().getAction() != null && getIntent().getAction()
                            .startsWith("WIDGET_CALL:")) {

                        executeEmulationStart(conf, () -> downloadFiles(conf, r,
                                de -> showErrorMessage(de.getLocalizedMessage())));

                    } else {
                        executeEmulationStart(conf, r);
                    }
                    view.removeOnLayoutChangeListener(this);
                }
            });

        }
    }

    private  SettingsActivity.SettingsConfiguration createSettingsConfiguration(
            final boolean isModelFix) {
        return new SettingsActivity.SettingsConfiguration(getViewModel(), isModelFix,
                new SettingsActivity.CurrentEmulationCallbacks(getViewModel().getEmulation(),
                        getViewModel().getConfiguration(), getViewModel()));
    }

    @SuppressLint("SourceLockedOrientationActivity")
    private void updateOrientation(final EmulationConfiguration conf) {
        switch (getEmulationScreenOrientation(conf)) {
            case PORTRAIT:
                setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_USER_PORTRAIT);
                break;
            case LANDSCAPE:
                setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_USER_LANDSCAPE);
                break;
            default:
                setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_FULL_USER);
                break;
        }
    }

    @SuppressLint("MissingSuperCall")
    @Override
    protected void onNewIntent(final Intent intent) {
        if (intent.getAction() != null) {
            String[] parts = intent.getAction().split(":", 2);
            if (parts.length == 2) {
                String id = parts[1];
                EmulationConfiguration newConf = getWidgetConfiguration(id);
                EmulationConfiguration oldConf = getViewModel().getConfiguration();
                if (newConf != null && oldConf != null
                        && newConf.getId().equals(oldConf.getId())
                        && newConf.getEmulatorId().equals(oldConf.getEmulatorId())) {
                    return;
                }
            }
            if (Emulationslist.getCurrentEmulation() != null
                    && getViewModel().getConfiguration() != null
                    && intent.getAction() != null
                    && intent.getAction().startsWith("WIDGET_CALL:")) {

                for (Fragment f : getSupportFragmentManager().getFragments()) {
                    if (f instanceof EmulationDialogFactory.PauseDialogFragment) {
                        getSupportFragmentManager().beginTransaction().remove(f).commitNow();
                    }
                }
                getViewModel().setLifecyclePauseReason(null);
                getViewModel().getEmulation().setPaused(true);
                if (parts.length == 2) {
                    String id = parts[1];
                    EmulationConfiguration newConf = getWidgetConfiguration(id);
                    if (newConf != null) {
                        SubActivityBlockFragment.add(this);
                        cancelRunningEmulation(getViewModel().getConfiguration().getName(),
                                newConf.getName(), () -> {
                                    getViewModel().setEmulation(Emulationslist
                                            .createEmulation(getEmulationUi(), newConf));
                                    getViewModel().setConfiguration(newConf);
                                    //triggerEmulationStart(getViewModel().getConfiguration());
                                    Runnable r = () -> {
                                        if (getViewModel().getEmulation() != null) {
                                            if (!getViewModel().getEmulation().isRunning()) {
                                                getViewModel().getEmulation().startThread();
                                            }
                                        }
                                    };
                                    if (!launchInitialSetting(newConf)) {
                                        Runnable rStart = () -> downloadFiles(newConf, r,
                                                de -> showErrorMessage(de.getLocalizedMessage()));
                                        executeEmulationStart(newConf, rStart);
                                        evaluateSettings(intent);
                                        getViewModel().updateKeyboardDefaultVisibility(
                                                this, getViewModel().getConfiguration());
                                        applyKeyboardSettings();
                                        getSoftkeyConfigView().applyUseropts(getCurrentUseropts());
                                        updateUi(isInLandscape());
                                    }

                                }, () -> {
                                    if (!SubActivityBlockFragment.exists(this)) {
                                        showPauseDialog(false);
                                    }
                                });
                    }
                }
            } else {
                if (!SubActivityBlockFragment.exists(this)) {
                    showPauseDialog(false);
                }
            }
        } else {
            super.onNewIntent(intent);
        }
    }

    private final Set<Integer> mPressedButtons = new LinkedHashSet<>();
    private Bitmap mBitmap = null;

    @Override
    protected void onResume() {
        super.onResume();
        applyKeyboardSettings();
        getSoftkeyConfigView().applyUseropts(getCurrentUseropts());
        getViewModel().getEmulation().setEmulationUI(getEmulationUi());
        MonitorGlSurfaceView gl = findViewById(R.id.gv_monitor);
        gl.setScrollView(findViewById(R.id.scroll));
        mPressedButtons.clear();
        boolean blockebydialog = false;
        List<Fragment> deadPauseDialogs = new LinkedList<>();
        if (getViewModel().isResumed()) {
            if (!SubActivityBlockFragment.exists(this)) {
                FragmentManager fm = getSupportFragmentManager();
                for (Fragment f : fm.getFragments()) {
                    if (f instanceof EmulationDialogFragment) {
                        if (f instanceof EmulationDialogFactory.PauseDialogFragment) {
                            deadPauseDialogs.add(f);
                        } else {
                            blockebydialog = true;
                            break;
                        }
                    }
                }
                if (mViewModel.isPopupMenuVisible()) {
                    blockebydialog = true;
                }
                for (Fragment f:deadPauseDialogs) {
                    fm.beginTransaction().remove(f).commit();
                }
                if (!blockebydialog) {
                    findViewById(android.R.id.content).post(() -> new Thread(
                            () -> runOnUiThread(() -> showPauseDialog(false))).start());
                } else {
                    getViewModel().getEmulation().setPaused(true);
                }
            } else {
                if (mViewModel.isAskingForPermissions()) {
                    mViewModel.setAskingForPermissions(false);
                } else {
                    SubActivityBlockFragment.remove(this);
                }

            }
        }
        getViewModel().setResumed(true);
        updateOrientation(getViewModel().getConfiguration());
        updateUi(isInLandscape());
        if (getViewModel().getLifecyclePauseReason() != null) {
            removePauseReason(getViewModel().getLifecyclePauseReason());
        }
    }

    private void applyKeyboardSettings() {
        if (getViewModel().getKeyboardVisible(isInLandscape()) == View.VISIBLE
                || getViewModel().isKeyboardForced()) {
                setKeyboardVisibility(autoShowKeyboard() ? View.VISIBLE : View.GONE);
        } else {
            setKeyboardVisibility(View.GONE);
        }
    }
    private Configuration mCurrentConfig = new Configuration();
    @Override
    public void onConfigurationChanged(@NonNull final Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        int orientation = mCurrentConfig.orientation;
        if (orientation == 0) {
            View root = findViewById(android.R.id.content);
            if (root.getWidth() < root.getHeight()) {
                orientation = Configuration.ORIENTATION_PORTRAIT;
            } else {
                orientation = Configuration.ORIENTATION_LANDSCAPE;
            }
        }
        if (newConfig.orientation != orientation) {
            ((MonitorGlSurfaceView) findViewById(R.id.gv_monitor)).setInLayout(true);
        }
        SoftkeysLinearLayout softkeys = findViewById(R.id.softkeys);
        softkeys.setOnVisibilityChangedListener(() -> {
            Log.v("softkeys", "setOnVisibilityChangedListener "
                    + softkeys.getVisibility());
            ((MonitorGlSurfaceView) findViewById(R.id.gv_monitor))
                    .onGeometryChanged();
        });
        if (!isTv()) {
            createTouchKeyboard();
        }
        for (Joystick joy : mAvailableJoysticks) {
            joy.onConfigurationChanged(newConfig);
        }
        updateUi(isInLandscape());
        getViewModel().setResumed(true);
        MonitorGlSurfaceView v = findViewById(R.id.gv_monitor);
        v.setVisibility(View.VISIBLE);
        EmulationDialogFragment f = (EmulationDialogFragment) getSupportFragmentManager()
                .findFragmentByTag(FRAGMENT_DIALOG);
        if (f != null) {
            if (f instanceof MainMenuDialogFragment) {
                if (!useDialogAsMenu()) {
                    f.dismiss();
                    getSupportFragmentManager().beginTransaction().remove(f).commitNow();
                    getSupportFragmentManager().popBackStack();
                    updateMainMenu();
                } else {
                    f.redraw(mi -> {
                        if (mi.getItemId() == R.id.mi_enablekeyboard) {
                            mi.setChecked(getKeyboardVisibility() == View.VISIBLE);
                        }
                    });
                }
            } else {
                getSupportFragmentManager()
                        .beginTransaction()
                        .detach(f)
                        .commitNowAllowingStateLoss();
                getSupportFragmentManager()
                        .beginTransaction()
                        .attach(f)
                        .commitAllowingStateLoss();
            }
        } else {
            if (mViewModel.isPopupMenuVisible()) {
                mViewModel.getPopupMenu().dismiss();
                mViewModel.setPopupMenu(null);
                mViewModel.setPopupMenuVisible(false);
                updateMainMenu();
            }
        }
        mCurrentConfig = new Configuration(newConfig);
    }

    @Override
    public void onPause() {
        super.onPause();
        if (getViewModel().getEmulation() != null) {
            if (!getViewModel().isPausedByUser()) {
                PowerManager pm = (PowerManager) getSystemService(Context.POWER_SERVICE);
                getViewModel().setStopped(getViewModel().isStopped() || !pm.isScreenOn());
            }
        }
        ((MonitorGlSurfaceView) findViewById(R.id.gv_monitor)).updateViewModel(getViewModel());
        disconnectJoysticks();
        getViewModel().setLifecyclePauseReason(addPauseReason());
    }

    @Override
    public boolean dispatchKeyEvent(final KeyEvent event) {
        return onKey(event.getKeyCode(), event) || super.dispatchKeyEvent(event);
    }
    @Override
    protected void onAvailableKeyboardsChanged(final boolean hardware, final boolean software) {
        getViewModel().setHardwareKeyboardAvailable(hardware);
        getViewModel().setSoftwareKeyboardAvailable(software);
        super.onAvailableKeyboardsChanged(hardware, software);
        if (isJoystickListChanged(mAvailableJoysticks)) {
            checkRequiredJoysticks(mAvailableJoysticks);
        }
        updateTouchKeyboard();
    }

    List<Joystick> getAvailableJoysticks() {
        return mAvailableJoysticks;
    }

    private boolean isJoystickListChanged(final List<Joystick> newList) {
        final List<Joystick> oldList = mAvailableJoysticks;
        if (oldList.size() != newList.size()) {
            return true;
        }
        for (Joystick o : oldList) {
            boolean found = false;
            for (Joystick n : newList) {
                if (o.getId().equals(n.getId())) {
                    found = true;
                }
            }
            if (!found) {
                return true;
            }
        }
        return false;
    }
    private void updateJoystickSettings() {
        int requiredcount;
        int requiredport = PORT_NOT_CONNECTED;
        if (Emulationslist.getCurrentEmulation() != null) {
            Emulation.JoystickFunctions jf = Emulationslist.getCurrentEmulation()
                    .getJoystickFunctions();
            if (jf != null) {
                Emulation.InputDeviceType type = null;

                List<Integer> allPorts = new ArrayList<>(jf.getJoystickports().keySet());
                List<Integer> requiredPorts = new ArrayList<>(getViewModel().getConfiguration()
                        .getRequiredJoystickports(allPorts));
                requiredcount = requiredPorts.size();
                if (requiredcount == 1) {
                    requiredport = requiredPorts.get(0);
                    type = getViewModel().getConfiguration().getInputDeviceTypes()
                            .get(requiredcount);
                }
                Map<Integer, Emulation.InputDeviceType> types = new HashMap<>();
                for (int port : allPorts) {
                    types.put(port, null);
                }
                for (Joystick joy : mAvailableJoysticks) {
                    if (requiredcount == 1) {
                        if (joy.getSupportedDeviceTypes().contains(type)) {
                            joy.setPortnumber(requiredport);
                            types.put(requiredport, type);
                        } else {
                            joy.setPortnumber(PORT_NOT_CONNECTED);
                        }
                    } else {
                        joy.readUseropts(this);
                        types.put(joy.getPortnumber(), joy.getCurrentDeviceType());
                    }
                    //types.put(joy.getPortnumber(), null);
                    if (joy.getPortnumber() != Joystick.PORT_NOT_CONNECTED) {
                        if (joy instanceof TouchJoystick) {
                            getViewModel().setPreferredJoystickType(joy.getPortnumber(),
                                    EmulationConfiguration.PreferredJoystickType.DIRECTIONAL);
                        }
                        if (joy instanceof WheelJoystick) {
                            getViewModel().setPreferredJoystickType(joy.getPortnumber(),
                                    EmulationConfiguration.PreferredJoystickType.WHEEL);
                        }
                        if (getViewModel().isJoystickAutofireToggled(joy.getPortnumber())) {
                            joy.setAutofiring(true);
                        }
                    }
                }
                for (int port : types.keySet()) {
                    jf.setActiveInputDeviceType(port, types.get(port));
                }
                checkRequiredJoysticks(mAvailableJoysticks);
            }
        }
        getSoftkeyConfigView().setJoysticks(mAvailableJoysticks, findViewById(R.id.softkeys));
        restoreSoftkeys();
    }
    private final ActivityResultLauncher<Intent> mCreateDocumentLauncher
            = registerForActivityResult(new FileCreateionContract(), result -> {
        if (result != null && result.getData() != null) {
            onFileCreated(result.getData());
        } else {
            unmuteEmulation();
        }
    });
    private static final String RESULT = "____RESULT__";
    private final ActivityResultLauncher<Intent> mHandleImageLauncher
            = registerForActivityResult(new ActivityResultContract<Intent, Intent>() {
                @Override
                public Intent parseResult(final int result, final @Nullable Intent intent) {
                    if (intent != null) {
                        intent.putExtra(RESULT, result);
                    }
                    return intent;
                }

                @NonNull
                @Override
                public Intent createIntent(final @NonNull Context context, final Intent intent) {
                    return intent;
                }
            }, result -> {
                if (result != null && result.getBooleanExtra(RESULT, false) && !openFile(false)) {
                    showErrorDialog(getString(R.string.could_not_access_storage),
                            getString(R.string.check_content_provider));
                }
                handleCommonLaunchresult(result);
            });
    void createFile(final String defaultFilename) {
        getViewModel().addSubActivityPauseReason();
        Intent intent = new Intent(Intent.ACTION_CREATE_DOCUMENT);
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        intent.putExtra(Intent.EXTRA_TITLE, defaultFilename);
        intent.setType("application/*");
        muteEmulation();
        mCreateDocumentLauncher.launch(BaseActivity.createChooserIntent(intent, null));
    }
    @Override
    protected void onAvailableJoysticksChanged(final @NonNull  List<Joystick> joysticks) {
        if (!joysticks.isEmpty()) {
            joysticks.add(new MultiplexerJoystick(this, joysticks));
        }
        super.onAvailableJoysticksChanged(joysticks);
        boolean changed = isJoystickListChanged(joysticks);
        Log.v(getClass().getSimpleName(), "onAvailableJoysticksChanged: " + changed);
        if (changed) {
            mAvailableJoysticks.clear();
            mAvailableJoysticks.addAll(joysticks);
            updateJoystickSettings();
            setDynamicUiElements();
            Emulation.JoystickFunctions jf = getViewModel().getEmulation().getJoystickFunctions();
            if (jf != null) {
                List<Integer> allPorts = new ArrayList<>(jf.getJoystickports().keySet());
                List<Integer> requiredPorts = new ArrayList<>(getViewModel().getConfiguration()
                        .getRequiredJoystickports(allPorts));
                int requiredcount = requiredPorts.size();
                for (Joystick joy : joysticks) {
                    Log.v("requiredCount", joy + " -> " + joy.getPortnumber());
                    if (joy.getPortnumber() != Joystick.PORT_NOT_CONNECTED) {
                        requiredcount--;
                    }
                }
                if (requiredcount == 1) {
                    for (Joystick joy : joysticks) {
                        joy.setPortnumber(requiredPorts.get(0));

                    }
                }
                Runnable removeOldFragment = () -> {
                    Fragment f = getSupportFragmentManager().findFragmentByTag(FRAGMENT_DIALOG);
                    if (f instanceof EmulationDialogFactory.RequiredJoysticksFragment) {
                        getSupportFragmentManager().beginTransaction().remove(f).commit();
                    }
                };
                if (requiredcount > 0) {
                    Object additionalPauseReason = addPauseReason();
                    removeOldFragment.run();
                    removePauseReason(additionalPauseReason);
                    EmulationDialogFactory.showRequiredJoysticksFragment(
                            getViewModel().getConfiguration().getVirtualJoystickTypes(),
                            getViewModel().getConfiguration().getInputDeviceTypes(),
                            requiredPorts).showNow(getSupportFragmentManager(), FRAGMENT_DIALOG);
                } else {
                    removeOldFragment.run();
                    List<String> joysticknames = new ArrayList<>();
                    List<Integer> configuredJoysticks = getViewModel().getConfiguration()
                            .getUsedJoystickports(allPorts);
                    if (configuredJoysticks != null) {
                        for (int i : configuredJoysticks) {
                            boolean found = false;
                            for (Joystick joy : mAvailableJoysticks) {
                                if (joy.getPortnumber() == i) {
                                    found = true;
                                    break;
                                }
                            }
                            if (!found) {
                                joysticknames.add(jf.getJoystickports().get(i));
                            }
                        }
                    }
                    if (!joysticknames.isEmpty()) {
                        int length = isInUnitTest() ? Toast.LENGTH_LONG : Toast.LENGTH_SHORT;
                        Toast.makeText(this,
                                getString(R.string.not_connected,
                                        String.join(", ", joysticknames)),
                                length).show();
                    }
                }
            }
        }
    }
    void updateHardwareForTest() {
        updateUi(isInLandscape());
    }
    @Override
    void updateHardware() {
        super.updateHardware();
        if (getViewModel().getEmulation().getJoystickFunctions() != null) {
            autosetJoystickports(getViewModel().getEmulation()
                            .getJoystickFunctions(), getAvailableJoysticks());
        }
    }

    private void autosetJoystickports(final Emulation.JoystickFunctions jf,
                                      final List<Joystick> joysticks) {
        Map<Integer, EmulationConfiguration.PreferredJoystickType> preferred =
                getViewModel().getPreferredJoystickTypes();
        if (preferred.isEmpty()) {
            preferred = getViewModel().getConfiguration().getVirtualJoystickTypes();
        }
        getViewModel().getConfiguration().adaptJoysticks(this, jf, joysticks,
                preferred);
    }

    private void checkRequiredJoysticks(final List<Joystick> joysticks) {
        if (getViewModel().getEmulation() != null) {
            Emulation.JoystickFunctions jf = getViewModel().getEmulation().getJoystickFunctions();
            if (jf != null && !joysticks.isEmpty()) {
                autosetJoystickports(jf, joysticks);
            }
        }
    }

    @SuppressLint("SourceLockedOrientationActivity")
    void setKeyboardVisibility(final int visibility) {
        Log.v("LayoutChanged", "setkeyboardvisibility (" + visibility + ")");
        if (visibility != findViewById(R.id.keyboardview).getVisibility()) {
            if (isTv()) {
                Emulation.SoftwareKeyboardFunctions skf = mViewModel.getEmulation()
                        .getSoftwareKeyboardFunctions();
                if (skf != null) {
                    if (visibility == View.VISIBLE) {
                        Fragment f = getSupportFragmentManager().findFragmentByTag(REMOTE_KEYBOARD);
                        if (f != null) {
                            getSupportFragmentManager().beginTransaction().remove(f).commit();
                        }
                        ViewGroup screen = findViewById(R.id.screen);
                        screen.setDescendantFocusability(ViewGroup.FOCUS_AFTER_DESCENDANTS);
                        TvKeyboardFragment dlg = new TvKeyboardFragment(
                                mViewModel.getEmulation().getSoftwareKeyboardFunctions());
                        dlg.setIsShownListener(new TvKeyboardFragment.TvKeyboardShownListener(
                                EmulationActivity.this, () -> getSupportFragmentManager()
                                .beginTransaction().remove(dlg).commitNow()));
                        dlg.setInputDevice(EmulationActivity.this.mActiveInputDevice);
                        dlg.show(getSupportFragmentManager(), REMOTE_KEYBOARD);
                    } else {
                        Fragment f = getSupportFragmentManager().findFragmentByTag(REMOTE_KEYBOARD);
                        if (f != null) {
                            getSupportFragmentManager().beginTransaction().remove(f).commit();
                        }
                    }
                }
            } else {
                View kv = findViewById(R.id.keyboardview);
                kv.setVisibility(visibility);
                ((SoftkeysLinearLayout) findViewById(R.id.softkeys))
                        .onKeyboardVisibleChanged(kv.getVisibility() == View.VISIBLE);
                ((MonitorGlSurfaceView) findViewById(R.id.gv_monitor))
                        .onGeometryChanged();
            }
        }
    }

    @Override
    public boolean onGenericMotionEvent(final MotionEvent event) {
        for (Joystick joy: mAvailableJoysticks) {
            if (joy instanceof View.OnGenericMotionListener) {
                if (((View.OnGenericMotionListener) joy).onGenericMotion(null, event)) {
                    return true;
                }
            }
        }
        return false;
    }

    private boolean autoShowKeyboard() {
        if (getViewModel().isKeyboardRequesteByEmulation()) {
            return true;
        }
        if (getViewModel().getKeyboardVisible(isInLandscape()) == View.VISIBLE) {
            return true;
        }
        return getViewModel().isKeyboardVisibibleFromStart(isInLandscape()) && !isTv();
    }

    private boolean isMenuitemChecked(final int itemId) {
        if (itemId == R.id.mi_enablekeyboard) {
            return findViewById(R.id.keyboardview).getVisibility() == View.VISIBLE
                    || getSupportFragmentManager().findFragmentByTag(REMOTE_KEYBOARD) != null;
        }
        if (itemId == R.id.mi_warp) {
            if (getViewModel().getEmulation().getSpeedUpFunctions() != null) {
                return getViewModel().getEmulation().getSpeedUpFunctions().getMaximumSpeedMode();
            }
            return false;
        }
        return false;
    }

    void showFliplistDialog() {
        DialogFragment f = EmulationDialogFactory.showFliplistDialog();
        f.showNow(getSupportFragmentManager(), FRAGMENT_DIALOG);
    }

    private MonitorGlSurfaceView mSurfaceView = null;

    void setBitmap(final int displayId, final Bitmap bitmap) {
        if (mSurfaceView != null && displayId == mViewModel.getCurrentDisplayId()) {
            mSurfaceView.setBitmap(bitmap);
        }
        if (bitmap != null && displayId == mVisibleMonitor) {
            mBitmap = bitmap;
        }
    }
    Bitmap getCurrentBitmap() {
        return mBitmap;
    }

    void removePauseReasonAfterDelay(final Object pauseReason) {
        pauseAfterTimeMachine(() -> removePauseReason(pauseReason));
    }
    private View.OnClickListener createListener(final int menuid) {
        if (menuid == R.id.mi_enablekeyboard
                && getViewModel().getConfiguration().isKeyboardUsed()
                && getViewModel().getEmulation().getSoftwareKeyboardFunctions() != null) {
            return v -> {
                View keyboardview = findViewById(R.id.keyboardview);
                int newvisibility = keyboardview.getVisibility() == View.VISIBLE
                        ? View.GONE : View.VISIBLE;

                getViewModel().setKeyboardVisible(isInLandscape(), newvisibility);
                setKeyboardVisibility(newvisibility);
                ((TouchDisplayRelativeLayout) findViewById(R.id.crowded_area))
                        .setBottomMargin(newvisibility == View.GONE);
            };
        }
        EmulationViewModel vm = getViewModel();
        if (menuid == R.id.mi_softkeys   && isTv()
                && !(vm.getSoftkeyIds().isEmpty() && vm.getUiSoftkeys().isEmpty())) {
            return v -> showSoftkeyDialog();
        }
        if (menuid == R.id.mi_pause) {
            return v -> showPauseDialog(false);
        }
        if (menuid == R.id.mi_rewind) {
            if (getViewModel().getEmulation().getTimeMachineFunctions() != null) {
                return v -> {
                    DialogFragment f = EmulationDialogFactory.showTimemachineDialog();
                    f.showNow(getSupportFragmentManager(), FRAGMENT_DIALOG);
                };
            }
        }
        if (menuid == R.id.mi_tape_functions) {
            if (getViewModel().getEmulation().getTapeDeviceFunctions() != null
            && getViewModel().getEmulation().getTapeDeviceFunctions().isAvailableNow()) {
                return v -> showTapeFunctions();
            }
        }
        if (menuid == R.id.mi_reset) {
            if (getViewModel().getEmulation().getResetFunctions() != null) {
                return v -> {
                    DialogFragment f = EmulationDialogFactory.showResetFunctionsDialog();
                    f.showNow(getSupportFragmentManager(), FRAGMENT_DIALOG);
                };
            }
        }
        if (menuid == R.id.mi_detach) {
            if (getViewModel().getEmulation().getFileFunctions().getDetachFunctions() != null) {
                if (!getViewModel().getEmulation().getFileFunctions().getDetachFunctions()
                        .getFunctions().isEmpty()) {
                    return v -> {
                        DialogFragment f = EmulationDialogFactory.showDetachFunctionsDialog();
                        f.showNow(getSupportFragmentManager(), FRAGMENT_DIALOG);
                    };
                }
            }
        }
        if (menuid == R.id.mi_settings) {
            return v -> {
                SubActivityBlockFragment.add(this);
                mSettingsEvaluated = false;
                mSettingsLauncher.launch(createSettingsConfiguration(
                        !getViewModel().getConfiguration().isBareMachine()));
            };
        }
        if (menuid == R.id.mi_create
                && getViewModel().getEmulation().getFileCreationFunction() != null) {
            return v -> {
                SubActivityBlockFragment.add(this);
                mNewFilePropertiesLauncher.launch(null);
            };

        }
        if (menuid == R.id.mi_warp && getViewModel().getEmulation().getSpeedUpFunctions() != null) {
            return v -> {
                Emulation e = getViewModel().getEmulation();
                e.getSpeedUpFunctions().setMaximumSpeedMode(
                        !e.getSpeedUpFunctions().getMaximumSpeedMode());
            };
        }
        if (menuid == R.id.mi_storage_access
                && getViewModel().getEmulation().getFileFunctions() != null) {
            return v -> {
                Runnable runAfter = () -> {
                    if (!openFile(false)) {
                        Object pauseReason = addPauseReason();
                        showErrorDialog(getString(R.string.could_not_access_storage),
                                getString(R.string.check_content_provider),
                                () -> removePauseReason(pauseReason));
                    }
                };
                Runnable stopPause = () -> {
                    getViewModel().setResumed(false);
                    getViewModel().getEmulation().setPaused(false);
                };
                executeWithFilesystemAccess(() -> mViewModel.setAskingForPermissions(true),
                        runAfter, stopPause, () -> {
                            runAfter.run();
                            stopPause.run();
                        });

            };
        }
        if (menuid == R.id.mi_store_snaphots
                && getViewModel().getEmulation().getPackCurrentStateFunctions() != null) {
            return v -> showSnapshotsDialog();
        }
        if (menuid == R.id.mi_fliplist) {
            Set<Uri> fl = getViewModel().getConfiguration().getFliplist();
            Emulation.FliplistFunctions ff = getViewModel().getEmulation()
                    .getFliplistFunctions(fl);
            if (ff != null) {
                if (fl.isEmpty()) {
                    fl = ff.getFilesInList();
                }
                if (!fl.isEmpty() && !ff.getTargetDevices().isEmpty()) {
                    return v -> showFliplistDialog();
                }
            }
        }
        if (menuid == R.id.quit && isTv()) {
            return v -> quitEmulation();
        }
        if (menuid == R.id.mi_switch_monitor) {
            Emulation.DualMonitorFunctions mf = getViewModel().getEmulation()
                    .getDualMonitorFunctions();
            if (mf != null && mf.getDisplays().size() == 2) {
                return v -> {
                    for (int key : mf.getDisplays().keySet()) {
                        if (key != getViewModel().getCurrentDisplayId()) {
                            getViewModel().setCurrentDisplayId(key);
                            setVisibleMonitor(key);
                            getCurrentUseropts().setValue(
                                    Useropts.Scope.CONFIGURATION, "active_monitor", key);
                            restoreStatusbar(findViewById(R.id.st_statusbar));
                            Toast.makeText(this,
                                    getResources().getString(R.string.toast_switched_monitor,
                                            mf.getDisplays().get(key)), Toast.LENGTH_SHORT).show();
                            break;
                        }
                    }
                };
            }
            if (mf != null && mf.getDisplays().size() > 2) {
                return v -> EmulationDialogFactory.showSwitchMonitorFunctionsDialog().showNow(
                        getSupportFragmentManager(), FRAGMENT_DIALOG);
            }
        }
        return null;
    }

    void showTapeFunctions() {
        EmulationDialogFactory.showTapeFunctionsDialog()
                .showNow(getSupportFragmentManager(), FRAGMENT_DIALOG);
    }

    private PopupMenu createMenus() {
        @SuppressLint("RtlHardcoded")
        View v = findViewById(R.id.bottomAnchor);
        @SuppressLint("RtlHardcoded")
        PopupMenu pm = new PopupMenu(this, v, Gravity.RIGHT);
        pm.inflate(R.menu.menu_emulation);
        return pm;
    }
    int getSoftkeyboardType() {
        String key = getResources().getString(R.string.useropt_keyboard_variants);
        String valCompact = getResources().getString(R.string.value_keyboard_variant_compact);
        String valExact = getResources().getString(R.string.value_keyboard_variant_exact);

        if (getCurrentUseropts().getStringValue(key, valCompact).equals(valExact)) {
            return Emulation.SOFTKEYBOARD_EXACT_IF_POSSIBLE;
        } else {
            return Emulation.SOFTKEYBOARD_ALWAYS_COMPACT;
        }
    }
    private void createTouchKeyboard() {
        Emulation.SoftwareKeyboardFunctions skf =
                getViewModel().getEmulation().getSoftwareKeyboardFunctions();

        if (skf != null) {
            int type = getSoftkeyboardType();
            Fragment f = skf.createKeyboardFragment(type);
            try {
                FragmentManager fm = getSupportFragmentManager();
                if (f != null) {
                    Fragment toDelete = fm.findFragmentByTag(KEYBOARDFRAGMENT);
                    FragmentTransaction fta = fm.beginTransaction();
                    if (toDelete != null) {
                        fta = fta.remove(toDelete);
                    }
                    fta.add(R.id.keyboardview, f, KEYBOARDFRAGMENT).commitNow();
                }
            } catch (IllegalStateException e) {
                // dirty
            }
        }
    }

    int getKeyboardVisibility() {
        return findViewById(R.id.keyboardview).getVisibility();
    }

    private void updateTouchKeyboard() {
        Integer keyboardVisibility;
        if (getViewModel().isSoftwareKeyboardAvailable()) {
            keyboardVisibility = getViewModel().getKeyboardVisible(isInLandscape());
        } else {
            if (getViewModel().isKeyboardVisibibleFromStart(isInLandscape())) {
                keyboardVisibility = View.VISIBLE;
            } else {
                keyboardVisibility = View.GONE;
            }
        }
        if (keyboardVisibility != null) {
            if (getKeyboardVisibility() != View.INVISIBLE
                    && getKeyboardVisibility() != keyboardVisibility) {
                setKeyboardVisibility(keyboardVisibility);
            }
        }
    }

    void onMonitorDoubleTap() {
        getCurrentUseropts().setValue(
                Useropts.Scope.GLOBAL, "show_double_tap_hint", false);
    }

    private void showPauseDialog(final boolean extended) {
        try {
            EmulationDialogFactory.showPauseDialog(extended).
                    showNow(getSupportFragmentManager(), FRAGMENT_DIALOG);
        } catch (IllegalStateException e) {
            // this is ok, may happen during test.
        }
    }

    private void runEmulation() {
        Runnable starter = () -> {
            hidePreparingDialogs();
            if (getViewModel().getEmulation() != null) {
                if (!getViewModel().getEmulation().isRunning()) {
                    getViewModel().getEmulation().startThread();
                }
            }
        };
        if (getIntent().getAction() != null
                && getIntent().getAction().startsWith("WIDGET_CALL:")) {
            downloadFiles(getViewModel().getConfiguration(), starter, de -> {
                AlertDialogBuilder builder = new AlertDialogBuilder(this);
                builder.setIcon(android.R.drawable.ic_dialog_alert)
                        .setTitle(R.string.download_error)
                        .setMessage(getResources().getString(R.string.restart_tile_from_main,
                                de.getMessage(),
                                getViewModel().getConfiguration().getName(),
                                getResources().getString(R.string.app_name)))
                        .setPositiveButton(getResources().getString(R.string.start_app,
                                        getResources().getString(R.string.app_name)),
                                (dialogInterface, i) -> {
                                    Intent intent = new Intent(this, MainActivity.class);
                                    startActivity(intent);
                                    dialogInterface.dismiss();
                                })
                        .setOnDismissListener(dialogInterface -> finish())
                        .create().show();
            });
        } else {
            prepareEmulationStart(getViewModel().getConfiguration());
        }
    }

    void restoreStatusbar(final StatusbarView sb) {
        sb.setJoysticks(mAvailableJoysticks);
        sb.updateDiskdriveList(getViewModel());
        sb.updateTapedriveList(getViewModel().getTapedriveList());
        if (!isTv()) {
            sb.setMonitorList(mViewModel.getEmulation());
        }
        if (sb.getVisibility() == View.VISIBLE) {
            if (getViewModel().getTapedriveList().isEmpty()) {
                sb.setOnTapeClickListener(null);
            } else {
                sb.setOnTapeClickListener(
                        view -> showTapeFunctions());
            }
        }
        Log.v("displayId", "restoreStatusbar");
        setVisibleMonitor(getViewModel().getCurrentDisplayId());
    }
    private EmulationUiImpl mEmulationUi = null;
    EmulationUi getEmulationUi() {
        if (mEmulationUi == null || mEmulationUi.getActivity() != this) {
            mEmulationUi = new EmulationUiImpl(this);
        }
        return mEmulationUi;
    }
    private void updateUi(final boolean orientation) {

        if (getViewModel().getEmulation() != null) {
            if (!isTv()) {
                updateTouchKeyboard();
            }
        }
        updateHardware();
        updateJoystickSettings();
        setDynamicUiElements();
        Emulation.DualMonitorFunctions dmf = getViewModel().getEmulation()
                .getDualMonitorFunctions();
        if (!mMonitorSet) {
            if (dmf != null) {
                Resources res = getResources();
                String machineSettingsVal = res.getString(R.string.value_initial_monitor_machine);
                String monitorKey = res.getString(R.string.useropt_initial_monitor);
                int monitor = dmf.getDefaultMonitor();
                if (!machineSettingsVal.equals(
                        getCurrentUseropts().getStringValue(monitorKey, null))) {
                    monitor = getCurrentUseropts().getIntegerValue("active_monitor", monitor);
                    dmf.setActiveMonitor(monitor);
                }
                Log.v("displayId", "dmf != null");
                getViewModel().setCurrentDisplayId(monitor);
            } else {
                Log.v("displayId", "dmf == null");
                getViewModel().setCurrentDisplayId(0);
            }
            mMonitorSet = true;
        }
        restoreStatusbar(findViewById(R.id.st_statusbar));
        restoreSoftkeys();
        ((MonitorGlSurfaceView) findViewById(R.id.gv_monitor))
                .readFromViewModel(orientation, getViewModel());

        synchronized ((mUpdateUiMonitor)) {
            mUpdateUiDone = true;
            mUpdateUiMonitor.notifyAll();
        }
    }
    private int mVisibleMonitor = 0;
    void setVisibleMonitor(final int active) {
        Log.v("displayId", "setVisibleMonitor (" + active + ")");
        getViewModel().setCurrentDisplayId(active);
        if (getViewModel().getEmulation().getDualMonitorFunctions() != null) {
            getViewModel().getEmulation().getDualMonitorFunctions().setActiveMonitor(active);
        }
        mVisibleMonitor = active;
    }

    private void restoreSoftkeys() {

        SoftkeysLinearLayout root = findViewById(R.id.softkeys);
        root.removeAllViews();
        addSoftkeyViews(root, null, null);
        root.setVisibility(isTv() ? View.GONE : root.isSoftkeyVisible(
                getViewModel().getKeyboardVisible(isInLandscape()) == View.VISIBLE));

    }

    void addSoftkeyViews(final SoftkeysLinearLayout vg, @Nullable  final Runnable runAfterPress,
                         @Nullable  final Runnable runAfterRelease) {
        if (getViewModel().getEmulation() != null
                && getViewModel().getEmulation().getSoftkeyFunctions() != null) {
            Emulation.SoftkeyFunctions functions = getViewModel().getEmulation()
                    .getSoftkeyFunctions();
            for (SoftkeyConfigView.Softkey key : getViewModel().getUiSoftkeys()) {
                vg.createSoftkey(key, functions, runAfterPress, runAfterRelease);
            }
        }
        for (String key : getViewModel().getSoftkeyIds()) {
            vg.createSoftkey(key, getViewModel().getJsSoftkey(key),
                    getViewModel().isSoftkeyEnabled(key),
                    getKeyboardVisibility(),
                    getViewModel().getEmulation().getSoftkeyFunctions(),
                    runAfterPress, runAfterRelease);
        }
    }

    private final Object mUpdateUiMonitor = new Object();
    private boolean mUpdateUiDone = false;

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    private boolean isTvKeyboardVisible() {
        return getSupportFragmentManager().findFragmentByTag(REMOTE_KEYBOARD) != null;
    }

    private void recenterTriggeringJoystick(final KeyEvent event) {
        for (Joystick joy : mAvailableJoysticks) {
            if (joy.getHardwareId() == event.getDeviceId()) {
                joy.reset();
                joy.notifyListener();
            }
        }
    }
    boolean onKey(final int keyCode, final KeyEvent event) {
        Log.v(getClass().getSimpleName(), "KeyEvent: " + event.toString());

        if (event.getAction() == KeyEvent.ACTION_DOWN) {
            mPressedButtons.add(keyCode);
            if (keyCode == KeyEvent.KEYCODE_MEDIA_PAUSE
                    //|| keyCode == KeyEvent.KEYCODE_BACK
                    || keyCode == KeyEvent.KEYCODE_MEDIA_PLAY_PAUSE
                    || keyCode == KeyEvent.KEYCODE_SETTINGS
                    || keyCode == KeyEvent.KEYCODE_MEDIA_FAST_FORWARD
                    || keyCode == KeyEvent.KEYCODE_MENU
                    || keyCode == KeyEvent.KEYCODE_DPAD_CENTER
                    || EmulationUi.JOYSTICK_MINIMAL_ACTION_BUTTONS.contains(keyCode)
                    || EmulationUi.JOYSTICK_START_BUTTONS.contains(keyCode)
            ) {
                return true;
            }
        }
        boolean wasHitHere = false;
        if (event.getAction() == KeyEvent.ACTION_UP) {
            wasHitHere = mPressedButtons.contains(keyCode);
            mPressedButtons.remove(keyCode);
            if (wasHitHere) {
                if (keyCode == KeyEvent.KEYCODE_MEDIA_PAUSE
                        || keyCode == KeyEvent.KEYCODE_MEDIA_PLAY_PAUSE) {
                    showPauseDialog(false);
                    return true;
                }
                if (EmulationUi.JOYSTICK_START_BUTTONS.contains(keyCode)) {
                    recenterTriggeringJoystick(event);
                    showPauseDialog(!isTvKeyboardVisible());
                    return true;
                }

                if (keyCode == KeyEvent.KEYCODE_SETTINGS) {
                    SubActivityBlockFragment.add(this);
                    mSettingsLauncher.launch(createSettingsConfiguration(
                            !getViewModel().getConfiguration().isBareMachine()));
                    return true;
                }
                if (keyCode == KeyEvent.KEYCODE_MEDIA_FAST_FORWARD) {
                    if (getViewModel().getEmulation().getSpeedUpFunctions() != null) {
                        View.OnClickListener l = createListener(R.id.mi_warp);
                        if (l != null) {
                            l.onClick(null);
                        }
                    }
                }
                if (keyCode == KeyEvent.KEYCODE_MENU
                        || keyCode == KeyEvent.KEYCODE_DPAD_CENTER
                        || EmulationUi.JOYSTICK_MINIMAL_ACTION_BUTTONS.contains(keyCode)) {
                    recenterTriggeringJoystick(event);
                    if (!isTvKeyboardVisible()) {
                        showMainMenu();
                    }
                    return true;
                }
            }
        }
        if (delegateToEmulatedInputDevices(keyCode, event)) {
            return true;
        }
        if (event.getAction() == KeyEvent.ACTION_DOWN) {
            if (keyCode == KeyEvent.KEYCODE_BACK) {
                return true;
            }
        }
        if (event.getAction() == KeyEvent.ACTION_UP) {
            if (keyCode == KeyEvent.KEYCODE_BACK
                    && event.getRepeatCount() == 0) {
                if (!getViewModel().containsActiveMouse(event.getDeviceId())) {
                    onBackPressed();
                }
                return true;
            }
        }
        if (wasHitHere && keyCode == KeyEvent.KEYCODE_F12) {
            showMainMenu();
            return true;
        }
        return false;
    }
    private boolean delegateToGamepadFunctions(final SoftkeyConfigView scv,
                                               final KeyEvent event) {
        if (event.getAction() == KeyEvent.ACTION_DOWN && scv.onButtonPressed(event.getKeyCode(),
                 getViewModel().getEmulation().getSoftkeyFunctions())) {
            return true;
        }
        //noinspection RedundantIfStatement
        if (event.getAction() == KeyEvent.ACTION_UP && scv.onButtonReleased(event.getKeyCode(),
                        getViewModel().getEmulation().getSoftkeyFunctions())) {
            return true;
        }
        return false;
    }

    private boolean delegateToGamepadFunctions(final Emulation.GamepadFunctions gf,
                                               final KeyEvent event) {
        if (gf != null) {
            if (event.getAction() == KeyEvent.ACTION_DOWN
                    && gf.onButtonPressed(event.getKeyCode())) {
                return true;
            }
            //noinspection RedundantIfStatement
            if (event.getAction() == KeyEvent.ACTION_UP
                    && gf.onButtonReleased(event.getKeyCode())) {
                return true;
            }
        }
        return false;

    }
    boolean delegateToEmulatedInputDevices(final int keyCode, final KeyEvent event) {
        if (delegateToGamepadFunctions(getViewModel()
                .getEmulation().getGamepadFunctions(), event)) {
            return true;
        }
        if (delegateToGamepadFunctions(getSoftkeyConfigView(), event)) {
            return true;
        }
        for (Joystick joy : mAvailableJoysticks) {
            if (joy instanceof View.OnKeyListener) {
                View.OnKeyListener kbj = (View.OnKeyListener) joy;
                boolean handled = kbj.onKey(null, keyCode, event);
                if (handled) {
                    return true;
                }
            }
        }
        if (getViewModel().getEmulation().getHardwareKeyboardFunctions() != null
                && mViewModel.isHardwareKeyboardAvailable()) {
            if (keyCode != KeyEvent.KEYCODE_BACK) {
                boolean handled = KeycodeHelper.translateKeyEvent(event,
                        getViewModel().getEmulation().getHardwareKeyboardFunctions());
                if (handled) {
                    getViewModel().setKeyboardUsed();
                }
                return handled;
            }
        }
        return false;
    }

    void setDynamicUiElements() {
        Emulation.JoystickFunctions jf = getViewModel().getEmulation().getJoystickFunctions();
        EmulationConfiguration.PreferredJoystickType joystickType = null;
        if (jf != null) {
            boolean multiplexer = false;
            for (Joystick j : mAvailableJoysticks) {
                if (j instanceof MultiplexerJoystick && j.getPortnumber() != PORT_NOT_CONNECTED) {
                    joystickType = ((MultiplexerJoystick) j).getPreferredVirtualType(this);
                    multiplexer = true;
                }
            }
            if (!multiplexer) {
                for (Joystick joy : mAvailableJoysticks) {
                    if (joy.getPortnumber() != Joystick.PORT_NOT_CONNECTED) {
                        if (joy instanceof TouchJoystick) {
                            joystickType = EmulationConfiguration.PreferredJoystickType.DIRECTIONAL;
                            getViewModel().setPreferredJoystickType(joy.getPortnumber(),
                                    EmulationConfiguration.PreferredJoystickType.DIRECTIONAL);
                        }
                        if (joy instanceof WheelJoystick) {
                            joystickType = EmulationConfiguration.PreferredJoystickType.WHEEL;
                            getViewModel().setPreferredJoystickType(joy.getPortnumber(),
                                    EmulationConfiguration.PreferredJoystickType.WHEEL);
                        }
                    }
                }
            }
            MultiplexerJoystick muxer = new MultiplexerJoystick(this, mAvailableJoysticks);
            List<Integer> ports = getViewModel().getConfiguration().getUsedJoystickports(
                    new LinkedList<>(jf.getJoystickports().keySet()));
            muxer.readUseropts(this);

            if (muxer.getPortnumber() == Joystick.PORT_NOT_CONNECTED
                    && ports != null && ports.size() == 1) {
                muxer.setPortnumber(ports.get(0));
            }

            for (Joystick joystick : mAvailableJoysticks) {
                if (!(joystick instanceof VirtualJoystick)) {
                    joystickType = null;
                }

                joystick.connect(this, (joy, devicetype, portnumber,
                                        isAbsolutePosition, xvalue, yvalue, fire) -> {
                    getViewModel().getEmulation().getJoystickFunctions()
                            .onJoystickChanged(joy, devicetype, portnumber,
                                    isAbsolutePosition, xvalue, yvalue, fire);
                    StatusbarView sb = findViewById(R.id.st_statusbar);
                    sb.onJoystickChanged(joy, devicetype, portnumber,
                            isAbsolutePosition, xvalue, yvalue, fire);
                }, getViewModel().getEmulation().getGamepadFunctions(), muxer);
            }
        }


        ((TouchDisplayRelativeLayout) findViewById(R.id.crowded_area))
                .updateAllViews(this, mAvailableJoysticks, this::showMainMenu,
                        !autoShowKeyboard(), joystickType);
    }

    private void disconnectJoysticks() {
        for (Joystick j : mAvailableJoysticks) {
            j.disconnect();
        }
    }

    void showErrorMessage(final String text) {
        Toast.makeText(this, text, Toast.LENGTH_LONG).show();
    }
    void finishWithResult(final Intent i) {
        setResult(Activity.RESULT_OK, i);
        removeStreamData(mViewModel.getConfiguration());
        finish();
    }
    void onEmulatorFinished() {
        Emulationslist.disposeCurrentEmulation();
        setResult(RESULT_OK, new Intent());
        try {
            //noinspection unchecked
            Class<TestInterface> clz = (Class<TestInterface>) Objects.requireNonNull(
                    EmulationActivity.class.getClassLoader())
                    .loadClass("de/rainerhock/eightbitwonders/TestBase");
            Method method = clz.getMethod("isIsolatedEmulationActivityTest");
            if (Boolean.FALSE.equals(method.invoke(clz.newInstance()))) {
                finish();
            }

        } catch (ClassNotFoundException e) {
            removeStreamData(mViewModel.getConfiguration());
            setResult(RESULT_OK);
            finish();
        } catch (NoSuchMethodException | InvocationTargetException
                 | IllegalAccessException | InstantiationException e) {
            throw new RuntimeException(e);
        }
    }
    public static class MainMenuDialogFragment extends EmulationDialogFragment {
        /**
         * Needed by android, do not use.
         */
        public MainMenuDialogFragment() {
            super();
        }
        MainMenuDialogFragment(final Serializable addPauseReason) {
            super(addPauseReason);
        }
    }

    private void showMenu(final Menu menu) {
        EmulationDialogFragment f = new MainMenuDialogFragment(addPauseReason());
        if (getViewModel().getSoftkeyIds().isEmpty() || !isTv()) {
            f.setDisabledMenuHidden(R.id.mi_softkeys, true);
        } else {
            f.setDisabledMenuHidden(R.id.mi_enablekeyboard, true);
        }
        f.setDisabledMenuHidden(R.id.quit, true);
        f.setDisabledMenuHidden(R.id.mi_switch_monitor, true);
        f.setMenu(menu, isTv());
        f.setCurrentInputDeviceSetter(device -> mActiveInputDevice = device);
        getSupportFragmentManager().beginTransaction().add(f, FRAGMENT_DIALOG).commit();
        for (int i : Arrays.asList(R.id.mi_storage_access, R.id.mi_create)) {
            f.setMenuEnableByJoystick(i, false);
        }
    }

    private void quitEmulation() {
        EmulationDialogFactory.showQuitDialog()
                .showNow(getSupportFragmentManager(), FRAGMENT_DIALOG);
    }
    private boolean useDialogAsMenu() {
        return isTv() || !getResources().getBoolean(R.bool.enough_space_for_menu);
    }
    // return !null when a menu is required, null if not (either no menu is to be displayed
    // or there is a dialog
    PopupMenu createRequiredMainMenu() {
        Fragment f = getSupportFragmentManager().findFragmentByTag(FRAGMENT_DIALOG);
        boolean doIt = f == null;
        if (doIt) {
            List<MenuItem> menuItems = new LinkedList<>();
            boolean useDialog = useDialogAsMenu();

            PopupMenu pm = createMenus();
            Menu m = pm.getMenu();
            for (int i = 0; i < m.size(); i++) {
                menuItems.add(m.getItem(i));
            }

            pm.setOnMenuItemClickListener(item -> {
                Objects.requireNonNull(createListener(item.getItemId())).onClick(null);
                return true;
            });

            for (MenuItem mi : menuItems) {
                if (createListener(mi.getItemId()) != null) {
                    if (mi.isCheckable()) {
                        mi.setChecked(isMenuitemChecked(mi.getItemId()));
                    }
                } else {
                    if (!mi.hasSubMenu()) {
                        if (useDialog) {
                            mi.setEnabled(false);
                        } else {
                            mi.setVisible(false);
                        }
                    }
                }
                if (!useDialog && !mi.isEnabled()) {
                    mi.setVisible(false);
                }
            }
            if (useDialog) {
                showMenu(m);
            } else {
                Serializable pr = addPauseReason();
                pm.setOnDismissListener(popupMenu -> {
                    mViewModel.setPopupMenuVisible(false);
                    if (noActionPending()) {
                        unmuteEmulation();
                    }
                    removePauseReason(pr);
                });
                mViewModel.setPopupMenuVisible(true);
                mViewModel.setPopupMenu(pm);
                return pm;
            }
        }
        return null;
    }
    boolean noActionPending() {
        return getSupportFragmentManager().findFragmentByTag(FRAGMENT_DIALOG) != null
                || SubActivityBlockFragment.exists(this);
    }
    void updateMainMenu() {
        PopupMenu pm = createRequiredMainMenu();
        if (pm != null) {
            OneShotPreDrawListener.add(findViewById(R.id.bottomAnchor), pm::show);
        }
        mViewModel.setPopupMenu(pm);

    }
    void showMainMenu() {
        Fragment f = getSupportFragmentManager().findFragmentByTag(FRAGMENT_DIALOG);
        if (f != null) {
            getSupportFragmentManager().beginTransaction().remove(f).commitNow();
        }
        muteEmulation();
        PopupMenu pm = createRequiredMainMenu();
        if (pm != null) {
            //hideSystemUI();
            pm.show();
        }
    }
    interface IsShownListener {
        void isShown(Dialog d);
    }

    private void showSnapshotsDialog() {
        EmulationDialogFactory.showSaveStatesDialog(this, EmulationDialogFactory.NO_TIMEMACHINE,
                getCurrentBitmap()).showNow(getSupportFragmentManager(), FRAGMENT_DIALOG);
    }


    void restoreCanvasSize() {
        Log.v(TAG, "restoreCanvasSize");
        MonitorGlSurfaceView gl = findViewById(R.id.gv_monitor);
        gl.setCanvasGeometry(
        Objects.requireNonNull(getViewModel()
                        .getCanvasSize(getViewModel().getCurrentDisplayId())).x,
                Objects.requireNonNull(getViewModel()
                                .getCanvasSize(getViewModel().getCurrentDisplayId())).y,
                getViewModel().getPixelAspectRatio(getViewModel().getCurrentDisplayId()),
                getViewModel().getScreenAspectRatio(getViewModel().getCurrentDisplayId()));


    }

    @Override
    protected void onStop() {
        super.onStop();
        if (!isFinishing()) {
            getViewModel().setStopped(true);
            PopupMenu pm = getViewModel().getPopupMenu();
            if (pm != null) {
                pm.dismiss();
                getViewModel().setPopupMenu(null);
            }
            MainActivity.setDestroyedData(getViewModel());
        }
    }

    @Override
    protected void onRestoreInstanceState(@NonNull final Bundle savedInstanceState) {
        if (savedInstanceState.getBoolean("saved", false)) {
            if (!isFinishing()) {
                showPauseDialog(false);
            }
        }
        super.onRestoreInstanceState(savedInstanceState);
    }
    File getInstanceStatePath(final EmulationConfiguration conf) {
        if (conf != null) {
            return new File(getCacheDir(),
                    File.separator + "instancestate_" + conf.getEmulatorId()
                            + "_" + conf.getId());
        }
        return null;

    }
    private void disposeInstanceState(final EmulationConfiguration conf) {
        File f = getInstanceStatePath(conf);
        if (f != null) {
            if (!f.delete()) {
                f.deleteOnExit();
            }
        }
    }

    @Override
    protected void onSaveInstanceState(@NonNull final Bundle outState) {
        if (!isFinishing() && getViewModel().getEmulation().createStateForNextRestart(
                getInstanceStatePath(getViewModel().getConfiguration()))) {
            outState.putBoolean("saved", true);
        }
        super.onSaveInstanceState(outState);
    }

    private static long counter = 0;
    Serializable addPauseReason() {
        Serializable lock = counter;
        counter = counter + 1;
        mViewModel.addPauseReason(lock);
        if (BuildConfig.DEBUG) {
            Log.v(TAG, String.format("pausereason %d added from %s", lock,
                    new Exception().getStackTrace()[1].toString()));
        }
        if (!mViewModel.getEmulation().isPaused()) {
            mViewModel.getEmulation().setPaused(true);
        }
        return lock;
    }
    void clearPauseReasons() {
        mViewModel.clearPauseReasons();
        mViewModel.getEmulation().setPaused(false);
        findViewById(R.id.gv_monitor).invalidate();
    }
    void removePauseReason(final Object lock) {
        if (BuildConfig.DEBUG) {
            Log.v(TAG, String.format("pausereason %s to be removed from %s", lock.toString(),
                    new Exception().getStackTrace()[1].toString()));
        }
        if (mViewModel.removePauseReason(lock)) {
            if (mViewModel.hasNoPauseReasons()) {
                Log.v(TAG, "list is now empty, set pause = false");
                mViewModel.getEmulation().setEmulationUI(getEmulationUi());
                mViewModel.getEmulation().setPaused(false);
            }
        } else if (mViewModel.hasNoPauseReasons()) {
            Log.v(TAG, "list was empty, set pause = false");
            mViewModel.getEmulation().setPaused(false);
            Log.w(getClass().getSimpleName(), String.format(
                    "%s to be removed was never added, may result in a hanging emulation.", lock));
        }
    }

    public interface TestInterface {
        /**
         * Check if a test ist running only an instance of EmulationActivity and will finish it.
         * @return if the test handles only a single EmulationActivity
         */
        boolean isIsolatedEmulationActivityTest();

        /**
         * Override Locale for test.
         * @return a test-specific locale if one was set, otherwise null.
         */
        Locale getTestLocale();
    }

    @Override
    public void finish() {
        disposeInstanceState(getViewModel().getConfiguration());
        EmulationConfiguration conf = getViewModel().getConfiguration();
        if (conf.getId().startsWith("_")) {
            Useropts.delete(this, conf.getEmulatorId(), conf.getId());
            new Deleter(getSnapshotPath(conf.getEmulatorId(), conf.getId()))
                    .run();
        }
        super.finish();
    }

    public static class EmulationWatchdogService extends Service {
        static boolean isRunning(final Context context) {
            final ActivityManager activityManager = (ActivityManager) context.getSystemService(
                    Context.ACTIVITY_SERVICE);
            final List<ActivityManager.RunningServiceInfo> services = activityManager
                    .getRunningServices(Integer.MAX_VALUE);
            String myService = EmulationWatchdogService.class.getName();
            for (ActivityManager.RunningServiceInfo runningServiceInfo : services) {
                Log.d(TAG, String.format("Service:%s", runningServiceInfo.service.getClassName()));
                if (runningServiceInfo.service.getClassName().equals(myService)) {
                    return true;
                }
            }
            return false;
        }
        private File mFile = null;

        @Nullable
        @Override
        public final IBinder onBind(final Intent intent) {
            return null;
        }

        @Override
        public final int onStartCommand(final Intent intent, final int flags, final int startId) {
            if (intent != null) {
                String path = intent.getStringExtra("path");
                if (path != null) {
                    mFile = new File(path);
                }
            }
            return START_STICKY;
        }

        @Override
        public final void onDestroy() {
            super.onDestroy();
        }

        @Override
        public final void onTaskRemoved(final Intent rootIntent) {
            if (mFile != null && mFile.exists()) {
                if (!mFile.delete()) {
                    mFile.deleteOnExit();
                }
            }
            super.onTaskRemoved(rootIntent);
        }
    }
}
