//  ---------------------------------------------------------------------------
//  This file is part of 8-Bit Wonders, a retro emulator for android.
//  Copyright (C) 2022  Rainer Hock <eight.bit.wonders@gmail.com>
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  ---------------------------------------------------------------------------


package de.rainerhock.eightbitwonders;

import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.net.Uri;
import android.view.KeyEvent;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import java.io.File;
import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * All Emulations have to implement this interface.
 * Methods like <pre>getXXXFunctions()</pre> are optional and can return null if the emulation
 * does not support the functions described in the Types of these functions.
 * All other functions have to be implemented.
 */
public interface Emulation {
    /**
     * Enum with all the input device types to be handled by an emulation:
     * {@link #DSTICK}: Atari style joystick with four directions and one fire button.
     */
    enum InputDeviceType { DSTICK, PADDLE, MOUSE }

    /**
     * Enum with all the type an emulation can understand.
     * * FIRE: Fire button on digital joystick
     * * ANALOG_PRIMARY: X-Button of paddle, left button of mouse
     * * ANALOG_SECONDARY: Y-Button of paddle, right button of mouse
     */
    enum InputButton { FIRE, ANALOG_PRIMARY, ANALOG_SECONDARY }
    /**
     * This is an additional META-State for KeyEvents passing an Umlaut.
     * In this cases the keycode is the code of the base character.
     */

    int META_UMLAUT = 0x10000000;

    /**
     * Start the emulation in a background thread.
     */
    void startThread();

    /**
     * Current state of the emulation, running or not.
     * A non-running emulation can be started by calling {@link #startThread()},
     * a running emulation can be stopped by calling {@link #terminate(Runnable)}
     * @return true for running emulation, false for not running.
     */
    boolean isRunning();
    /**
     * Request Termination the emulation's background thread.
     * After termination non null Runnable runAfterTermination will be called.
     * @param runAfterTermination task to be run after termination.
     */
    void terminate(@Nullable  Runnable runAfterTermination);

    /**
     * Get the ID of the emulator running the emulation.
     * @return unique ID specific for each implementation of {@link Emulation}
     */
    String getEmulatorId();

    /**
     * Get the configuration of the emulation.
     * @return unique ID specific for each configuration of {@link Emulation}
     */
    String getConfigurationId();

    /**
     * Set the object to handle user-interaction.
     * @param ui an implementation of {@link EmulationUi} handling user-interactions
     */
    void setEmulationUI(EmulationUi ui);

    /**
     * pause or unpause the emulation.
     * @param pause start/stop pause
     */
    void setPaused(boolean pause);

    void setPaused(boolean pause, Runnable runAfter);

    /**
     * Check if the emulation is paused.
     * @return if emulation is paused
     */
    boolean isPaused();
    /**
     * create a save state use for the next start of the emulation.
     * @param f File to store the data.
     * @return true if the current state was created, otherwise false.
     * Emulators have to use this state once and only once.
     */
    boolean createStateForNextRestart(File f);


    /**
     * Functions that are available or not depending on the state of the emulation will implement
     * this interface.
     */
    interface AvailabityChangingFunctions {
        /**
         * Indicate if the functions can be called at the moment of this call.
         *
         * @return availibilty of functions
         */
        boolean isAvailableNow();
    }
    /**
     * Return implementation of {@link JoystickFunctions} handling joystick actions.
     * @return instance of {@link JoystickFunctions} or null if the emulation does not support
     * joysticks
     */
    JoystickFunctions getJoystickFunctions();

    /**
     * Always show keyboard in compact mode, even if there is more space.
     */
    int SOFTKEYBOARD_ALWAYS_COMPACT = 1;
    /**
    * Show keyboard with a layout being as close as possible to original layout but
     * consider device size and ui guideline.
     */
    int SOFTKEYBOARD_EXACT_IF_POSSIBLE = 2;

    /**
     * Return implementation of {@link SoftwareKeyboardFunctions} handling virtual Keyboards.
     * @return instance of {@link SoftwareKeyboardFunctions} or null if the emulation does not
     * support software keyboards.
     */
    SoftwareKeyboardFunctions getSoftwareKeyboardFunctions();

    /**
     * Return implementation of {@link HardwareKeyboardFunctions} handling hardware Keyboards.
     * @return instance of {@link HardwareKeyboardFunctions} or null if the emulation does not
     * support hardware keyboards.
     */
    HardwareKeyboardFunctions getHardwareKeyboardFunctions();
    /**
     * Return implementation of {@link SpeedUpFunctions} handling switching from
     * the emulation's original speed to the maximum speed limited only by the hardware.
     * @return instance of {SpeedUpFunctions} or null if the emulation does not support speed-up.
     */
    SpeedUpFunctions getSpeedUpFunctions();

    /**
     * Return implementation of {@link FileFunctions} handling opening files and using them in the
     * emulation.
     * @return instance of {@link FileFunctions} or null if the emulation does not support
     * file-handling.
     */
    FileFunctions getFileFunctions();

    /**
     * Return implementation of {@link AnyRunnableFunctions} with reset functions.
     * @return implementation of {@link AnyRunnableFunctions} with functions for resetting the
     * emulation or null if the emulation does not support resetting.
     */
    AnyRunnableFunctions getResetFunctions();

    /**
     * Return implementation of {@link TimeMachineFunctions} for rewinding the emulation a few
     * seconds.
     * @return implementation of {@link TimeMachineFunctions} or null if
     * the emulation does not support rewinding the emulation.
     */
    TimeMachineFunctions getTimeMachineFunctions();

    /**
     * Return implementation of {@link MachineSettingsFunctions} to pass all user-settings.
     * except joystick-settings to the emulation.
     * @return  implementation of {@link MachineSettingsFunctions} or null if the emulation does
     * not have settings except joystick-settings
     */
    MachineSettingsFunctions getMachineSettingsFunction();

    /**
     * Return implementation of {@link TapeDeviceFunctions} for handling the virtual keys of
     * a tape device.
     * @return implementation of {@link TapeDeviceFunctions} for handling the keys of a
     * virtual tape-device or null if the emulation does not support tape-devices.
     */
    TapeDeviceFunctions getTapeDeviceFunctions();


    /**
     * Functions for user-settings (except joystick).
     */
    interface MachineSettingsFunctions {
        /**
         * Read the stored useropts.
         * @param useropts instance of {@link Useropts} with data for the current configuration.
         * @param runAfter this must be run after the useropts were applied.
         */
        void applyUseropts(Useropts useropts, Runnable runAfter);

        /**
         * List of all view-IDs visible in the layout of an {@link SettingsActivity}.
         * @param modelIsFix if set to true no views to change device model or memory map
         *                  (like PAL, NTSC, PET 30xx vs. 40xx vs. 80xx, VIC20 memory map)
         *                  shall be returned
         * @return List of View-IDs.
         */
        List<Integer> getLayouts(boolean modelIsFix);
    }

    /**
     * Functions for rewinding the emulation a few seconds.
     */
    interface TimeMachineFunctions {
        /**
         * Get the age of the newest recorded moment.
         * @return age of the newest recorded moment in miliseconds.
         */
        int getNewestMoment();
        /**
         * Get the age of the oldest recorded moment.
         * @return age of the oldest recorded moment in miliseconds.
         */
        int getOldestMoment();

        /**
         * Travel to a specific moment.
         * @param age the age of the moment in miliseconds.
         *            If there is none with the exact time, the closest matching older one
         *            will be used.
         */
        void travelTo(int age);

        /**
         * Get screenshot at a specific moment.
         * @param age the age of the moment in miliseconds.
         * @return a screenshot matching the interval.
         *         If there is none with the exact time, the closest matching older one
         *         will be used.
         */
        Bitmap getScreenPreview(int age);

    }
    /**
     * Function required for creating new emulator-specific files.
     */
    interface FileCreationFunction {
        /**
         * Create an Intent to start an activity.
         * The activity will have to call {@link android.app.Activity#setResult(int, Intent)}
         * before calling {@link android.app.Activity#finish()} with an Intent
         * for which
         * <ul>
         *     <li>
         * {@link Intent#putExtra(String, Serializable)} has been called with "createdata"
         * as the first parameter an any {@link Serializable} as the second parameter and
         *     </li>
         *     <li>
         * {@link Intent#putExtra(String, String)} has been called with "filename" as the first
         * parameter and default filename for the new file as the second parameter.
         *     </li>
         * </ul>
         * The value of the entry "createdata" will be passed
         * to {@link #createFile(ContentResolver, Uri, Serializable)}
         * to create and attach the file.
         *
         * @param context context to be used to create the intent.
         * @return an Intent that will be started and return values for
         * {@link #createFile(ContentResolver, Uri, Serializable)}
         */
        Intent getNewFilePropertiesIntent(Context context);
        /**
         * Create a new file using a given {@link ContentResolver} with the data given in a
         * {@link Serializable} created by the activity given in
         * {@link #getNewFilePropertiesIntent(Context)}}.
         * @param contentResolver  a contentresolver, use it to write data.
         * @param uri Uri to write the data to.
         * @param parameters emulator-specific values required for creating a file. Is a result
         *                   of the activity created by
         *                   {@link #getNewFilePropertiesIntent(Context)}}
         * @return true if the was successfully created
         */
        boolean createFile(ContentResolver contentResolver, Uri uri, Serializable parameters);
    }
    /**
     * Functions for using files within an emulation.
     */
    interface FileFunctions {
        /**
         * This method is called after the user picked a file to open and has to do the
         * appropriate action with the selected file.
         * @param uri URI of the file
         * @param title user-friendly title of the action to be done
         * @return an {@link Intent} for further use in {@link
         * EmulationActivity#startActivityForResult(Intent, int)}
         * if user action is required.
         */
        Intent getIntent(Uri uri, String title);
        /**
         * Return implementation of {@link AvailabityChangingRunnableFunctions}
         * with functions for detaching files.
         * that have been attached with the implementation of {@link FileFunctions} given in
         * {@link #getFileFunctions()}
         * @return implementation of {@link AnyRunnableFunctions} with functions for detaching
         * files from the emulation or null if the emulation does not support detaching.
         */
        AvailabityChangingRunnableFunctions getDetachFunctions();
    }
    interface MenuFeature {
        /**
         * Value for the {@link #getIconResource()} when there is no icon.
         */
        int NO_ICON = -1;
        /**
         * Get the text to be set as title.
         * @return resource id
         */
        String getName();

        /**
         * Get the id for the drawable ressource to be set as symbol.
         * @return resource id or {@link #NO_ICON} when there is no icon
         */
        int getIconResource();

        /**
         * Get the Runnable to be executed when the menuitem is selected.
         * @return Runnable to be executed.
         */
        Runnable getRunnable();
    }
    /**
     * Generic interface for a bunch of actions.
     */
    interface AnyRunnableFunctions {
        /**
         * Return an bunch of actions.
         * @return a {@link Map} with the labels of the functions as keys and the {@link Runnable}s
         * to be run after the user selected one of the action.
         */
        List<MenuFeature> getFunctions();
    }
    interface AvailabityChangingRunnableFunctions extends AnyRunnableFunctions,
            AvailabityChangingFunctions { }

    /**
     * Functions for Joystick interactions.
     */
    interface JoystickFunctions extends JoystickListener {

        /**
         * List of the emulation's available Joystickports.
         * @return {@link Map} with the available joystick ports as key and user friendly name
         * of the port as value.
         */

        Map<Integer, String> getJoystickports();

        /**
         * Make a difference if a joystick will always be visible in the settings activity
         * or visibility has to be turned on with the "show extended joysticks" option
         * if it was not connected.
         * @param port port number, must be in the returned keyset of {@link #getJoystickports()}
         * @return if a joystick port is common or less often used
         */
        boolean isExtentedJoystick(int port);
        /**
         * Make a difference if a joystick will always be visible in the settings activity
         * or visibility has to be turned on with the "show redirections to keyboard" option
         * if it was not connected.
         * @param port port number, must be in the returned keyset of {@link #getJoystickports()}
         * @return if a joystick port is common or less often used
         */
        boolean isSimulatingKeystrokes(int port);
        /**
         * Return all device types an emulation supports on the given port.
         * @param port port number, must be in the returned keyset of {@link #getJoystickports()}
         * @return all input device types support by the given port.
         */
        List<InputDeviceType> getAvailableInputDevicetypes(int port);

        /**
         * Get a {@link SoftkeyFunctions} that translates directions into keys.
         * @param port port to be handled
         * @return an implementation of {@link SoftkeyFunctions}
         * that translates directions into keys.
         */
        SoftkeyFunctions getJoystickToKeyboardMapper(Integer port);

        /**
         * Sets the current device type for the given port.
         * @param port port to set the device type for
         * @param newType new device type or null if no device is connected.
         */
        void setActiveInputDeviceType(int port, @Nullable Emulation.InputDeviceType newType);

        /**
         * Provide information how many "shots" can be autofired per minute.
         * @param port portnumber
         * @return maximal number of "shots" per minute.
         */
        int getAutofireFrequency(int port);


    }
    /**
     * Functions associated with the keys of a tape-device .
     */
    interface TapeDeviceFunctions extends AvailabityChangingFunctions {
        /**
         * {@link Runnable} to be run after the user has selected the
         * play key of the virtual tape-device.
         * @return a {@link Runnable} or null if the emulation does not support this key.
         */
        Runnable getPlayCommand();
        /**
         * {@link Runnable} to be run after the user has selected the
         * stop key of the virtual tape-device.
         * @return a {@link Runnable} or null if the emulation does not support this key.
         */
        Runnable getStopCommand();
        /**
         * {@link Runnable} to be run after the user has selected the
         * rewind key of the virtual tape-device.
         * @return a {@link Runnable} or null if the emulation does not support this key.
         */
        Runnable getRewindCommand();
        /**
         * {@link Runnable} to be run after the user has selected the
         * fast-forward key of the virtual tape-device.
         * @return a {@link Runnable} or null if the emulation does not support this key.
         */

        Runnable getForwardCommand();
        /**
         * {@link Runnable} to be run after the user has selected the
         * record key of the virtual tape-device.
         * @return a {@link Runnable} or null if the emulation does not support this key.
         */
        Runnable getRecordCommand();

    }
    /**
     * Functions for virtual keyboards.
     */
    interface SoftwareKeyboardFunctions {
        /**
         * Fragment to be added to the {@link EmulationActivity} the emulation ist running in.
         * All actions within the fragment are self-handled by the views within the fragment and
         * the emulation.
         *
         * @param type either {@link #SOFTKEYBOARD_ALWAYS_COMPACT} or
         * {@link #SOFTKEYBOARD_EXACT_IF_POSSIBLE}
         *
         * @return {@link KeyboardFragment} or null
         * if the emulation does not support a virtual keyboard.
         */
        KeyboardFragment createKeyboardFragment(int type);

        /**
         * Return true if the emulation supports differentiation between
         * always compact keyboard layouts and exact layouts if possible.
         * @return true if different layouts are supported
         */
        boolean supportsDifferentLayouts();
        /**
         * Called when a keyboard fragment is hidden.
         * @param f the {@link KeyboardFragment} that is being hidden.
         */
        void onKeyboardFragmentDismiss(KeyboardFragment f);


    }

    /**
     * Functions for switching from original speed to maximum speed.
     */
    interface SpeedUpFunctions {
        /**
         * Check if maximum speed mode is enabled.
         * @return true if maximum speed mode is enabled.
         */
        boolean getMaximumSpeedMode();

        /**
         * enable/disable maximum speed mode.
         * @param b true for maximum speed mode, false for original speed mode.
         */
        void setMaximumSpeedMode(boolean b);

    }
    interface GamepadFunctions {
        /**
         * Handle a button press.
         * @param keycode keycode according to constant values {@link KeyEvent}.KEYCODE_*
         *                or {@link EmulationUi}.PSEUDO_KEYCODE_*
         * @return true if the event was consumed otherwise false.
         */
        boolean onButtonPressed(int keycode);
        /**
         * Handle a button release.
         * @param keycode keycode according to constant values {@link KeyEvent}.KEYCODE_*
         *                or {@link EmulationUi}.PSEUDO_KEYCODE_*
         * @return true if the event was consumed otherwise false.
         */
        boolean onButtonReleased(int keycode);
    }
    /**
     * Functions for Hardware Keyboards.
     */
    interface HardwareKeyboardFunctions {
        /**
         * Instances of this interface populate a spinner in the user settings.
         */
        interface KeyboardMapping extends Comparable<KeyboardMapping> {
            /**
             * Unique internal ID for the Mapping.
             * @return an unique ID
             */
            String getId();

            /**
             * User-Friendly Name for the Mapping.
             * @return a user-friendly name.
             */
            String getName();
        }

        /**
         * Return available Mappings from Hardware to Emulation.
         * @param exactModel exact model of the emulated machine. Can be null if the
         *                   keyboard mapping does not depend on the exact machine.
         * @return List of Keyboardmappings
         */
        List<HardwareKeyboardFunctions.KeyboardMapping> getMappings();
        /**
         * Called when a key is pressed.
         * @param keycode {@link android.view.KeyEvent} normalized by
         * @return if the key was handled
         */
        boolean onKeyDown(KeyEvent keycode);
        /**
         * Called when a key is released.
         * @param keycode {@link android.view.KeyEvent} normalized by
         * @return if the key was handled
         */
        boolean onKeyUp(KeyEvent keycode);

        /**
         * Create fragment to be added to the settings for hardware keyboard configuration.
         * @return a fragment if hardware keyboard settings are supported or null if not.
         */
        SettingsFragment getSettingsFragment();
    }

    /**
     * Functions for storing the current state and configuration in a way it can be re-run and
     * transferred to other devices.
     */
    interface PackCurrentStateFunctions {
        /**
         * Get all properties to be parsed by the emulator on start.
         * They will be available by calling {@link EmulationConfiguration#getPresettings()}
         * @return Map with emulation specific-properties.
         */
        Map<String, String> getProperties();

        /**
         * Runnable to be called when a emulation "memory dump" has to be created.
         * @param milisecondsAgo Emulations not returning null in {@link #getTimeMachineFunctions()}
         *                       take the emulation state from there if greater than 0.
         *                       Otherwise (if value is 0 or {@link #getTimeMachineFunctions()}
         *                       returns null) the current state must be used.
         * @param encryptor #EmulationUi.Encryptor handling en- an decrypting.
         * @return Runnable to be called before the snapshot is stored to filesystem
         * or null if nothing has to be done
         */
        Runnable getDumpContentsRunnable(int milisecondsAgo,
                                         @NonNull EmulationUi.Encryptor encryptor);

        /**
         * Write a memory dump to a given file.
         * @param milisecondsAgo * All emulation take the latest dump if the value equals -1;
         *                       * Emulations not returning null in
         *                         {@link #getTimeMachineFunctions()}
         *                         take the emulation state from there
         *                         if the value is greater than 0.
         *                       * Otherwise (if value is 0 or {@link #getTimeMachineFunctions()}
         *                         returns null) false must be returned.
         * @param encryptor #EmulationUi.Encryptor handling en- an decrypting.
         * @param target File to write the dump to.
         * @return true if a snapshot existed and could be stored, otherwise false.
         */
        boolean writeDump(int milisecondsAgo, File target,
                          @NonNull EmulationUi.Encryptor encryptor);
        /**
         * load a snapshot.
         * @param filename path to read the snapshot from.
         * @param encryptor #EmulationUi.Encryptor handling en- an decrypting.
         * @return success
         */
        boolean loadSnapshot(String filename, @NonNull  EmulationUi.Encryptor encryptor);
        /**
         * Runnable to be called when a emulation "memory dump"/save state has to be imported.
         * @param source Path to existing and readable dump#
         * @param encryptor #EmulationUi.Encryptor handling en- an decrypting.
         * @return Runnable to be called before the snapshot is stored to filesystem
         * or null if nothing has to be done
         */
        Runnable getImportContentsRunnable(File source, @NonNull EmulationUi.Encryptor encryptor);

        /**
         * All local files currently in use that are not part of
         * the corresponding bare-metal-emulation.
         * @return List of local and readable files.
         */
        List<Uri> getAttachedFiles();

        /**
         * Choose wether an attachment shall be presented to the user in the share-activity.
         * @param uri uri to be shown
         * @return true if file shall be shown to user, otherwise false
         */
        boolean showFiletoUser(Uri uri);

    }

    /**
     * Return implementation of {@link PackCurrentStateFunctions} to be used for
     * re-running and transferring to other devices.
     *
     * @return instance of {@link PackCurrentStateFunctions} or null if
     * the emulation does not support packing.
     */
    PackCurrentStateFunctions getPackCurrentStateFunctions();

    /**
     * Functions for inserting/removing images from a fixed list without need to access the
     * file system.
     */
    interface FliplistFunctions {
        /**
         * Emulated devices to which an image can be attached.
         * @return Map of devices. Key ist the id of the device, value a user friendly name.
         */
        Map<Integer, String> getTargetDevices();

        /**
         * Currently attached images.
         * @return All the files in the list.
         */
        Set<Uri> getFilesInList();

        /**
         * Attach a file to device.
         * @param device Device to which the file is to be attached. Must be a key in the return
         *               value from {@link #getTargetDevices()}.
         * @param image image to attach.
         * @return if the image could be attached.
         */
        boolean attach(Integer device, Uri image);

        /**
         * Get the image attached to a device.
         * @param device Device to which an attached file is requested.
         * @return Attached content or null if no file or
         * at least no file from the list is attached.
         */
        Uri getAttachedImage(int device);

        /**
         * Check if content can be part of a fliplist.
         * @param image image to be checked
         * @return true if content can be part of a fliplist
         */
        boolean canBePart(Uri image);
    }

    /**
     * Return an implementation of #FliplistFunctions if the emulation supports fliplist functions
     * or null if not.
     * @param fliplistFromConfig Fliplist from Configuration. Emulation must use this, if it is not
     *                           null, otherwise an Emulation-owned can be created.
     * @return an implementation of #FliplistFunctions if the emulation supports fliplist functions
     *      * or null if not.
     */
    FliplistFunctions getFliplistFunctions(Set<Uri> fliplistFromConfig);

    /**
     * Audio-sepcific functions. Emulation can initiate sound output at any time, unless they have
     * been muted by {@link #mute()}, but not yet unmuted by {@link #unmute()}
     */
    interface SoundFunctions {
        /**
         * Mute the emulation. No Audio output is allowed.
         */
        void mute();

        /**
         * Unmute the emulation. Audio output is allowed.
         */
        void unmute();

        /**
         * Callback when audio (= no silence) is played.
         * @param r #Runnable to be called once.
         */
        void callOnAudioPlaying(Runnable r);
    }

    /**
     * Return an implementation of #SoundFunctions, if the emulation has audio output
     * or null if not.
     * @return an implemenation of #SoundFunctions, if the emulation has audio output
     * or null if not.
     */
    SoundFunctions getSoundFunctions();

    /**
     * Return an implementation of #FileCreationFunction, if the emulation supports creating files.
     * @return an implementation of #FileCreationFunction, if the emulation supports creating files.
     */
    FileCreationFunction getFileCreationFunction();
    interface SoftkeyFunctions {
        /**
         * Executed when a softkey created with #{@link EmulationUi#createSoftkey(String, String)}
         * was pressed.
         * @param id id, matches first parameter of
         *           #{@link EmulationUi#createSoftkey(String, String)}
         */
        void pressKey(String id);
        /**
         * Executed when a softkey created with #{@link EmulationUi#createSoftkey(String, String)}
         * was released.
         * @param id id, matches first parameter of
         *           #{@link EmulationUi#createSoftkey(String, String)}
         */
        void releaseKey(String id);

        /**
         * List all Keys that can be "pressed" by touching a softkey.
         * @return a list of keys.
         */
        @NonNull  List<AdditionalKey> getKeys();
        /**
         * List all Keys that can be "pressed" as modificator for another key by touching a softkey.
         * @return a list of keys.
         */
        @NonNull  List<AdditionalKey> getModificatorKeys();
    }
    /**
     * Return an Implementation of #SoftkeyFunctions, if the emulation adds Softkeys to the UI
     * by calling #{@link EmulationUi#createSoftkey(String, String)}.
     * @return an Implementation of #SoftkeyFunctions, if the emulation adds Softkeys to the UI
     *      * by calling #{@link EmulationUi#createSoftkey(String, String)} or null if not.
     */
    SoftkeyFunctions getSoftkeyFunctions();

    /**
     * Return an instance of {@link GamepadFunctions} if the emulator accepts
     * gamepad input events or null if not.
     * If an emulator does not support gamepad input events, gamepad actions
     * will be converted to joystick events if approbiate and sent
     * to the instance of {@link JoystickFunctions} return by getJoystickFunctions or
     * as keyboard events and sent to the instance of {@link HardwareKeyboardFunctions}
     * returned by getHardwareKeyboardFunctions
     * @return an instanceof {@link GamepadFunctions} or null if there is no specific
     * gamepad support.
     */
    GamepadFunctions getGamepadFunctions();
    /**
     * Common function for classes representing entries of an {@link android.widget.ArrayAdapter}.
     */
    interface DataEntry {
        /**
         * get Text displayed in the ui representation of the entry.
         * @return text of the ui representation.
         */
        String getText();
    }

    /**
     * Save custom savestate the emulation will be started with in the future.
     * If users can update the savestate the emulation is started with, this runnable
     * will create and save it.
     * @return Runnable that creates and saves the custom initial state or null if this
     * is not supported
     */
    Runnable getInitialSnapshotStorer();

    interface DualMonitorFunctions {
        /**
         * Return a map of displays.
         * @return a Map with the monitor ids as keys (theese ids must be passed as first parameter
         * in {@link EmulationUi#setBitmap(int, Bitmap)}
         * and {@link EmulationUi#onCanvasSizeChanged(int, int, int, float, float)}
         * values are user friendly names for each key.
         */
        @NonNull
        Map<Integer, String> getDisplays();

        /**
         * Get the default monitor id the emulation is calling
         * {@link EmulationUi#setBitmap(int, Bitmap)} and
         * {@link EmulationUi#onCanvasSizeChanged(int, int, int, float, float)} unless another
         * monitor or monitor setup is changed.
         * @return default monitor id.
         */
        int getDefaultMonitor();

        /**
         * Notify emulation about active monitor or monitor setup being changed by user.
         * @param monitor new monitor id
         */
        void setActiveMonitor(int monitor);
    }

    /**
     * Return an instance of {@link DualMonitorFunctions} to be called by ui if the
     * emulation supports more than monitor or monitor setup.
     * @return an instance of {@link DualMonitorFunctions} or null if there is only one
     * monitor or monitor setup.
     */
    @Nullable
    DualMonitorFunctions getDualMonitorFunctions();

    /**
     * Represantion of a key a joystick or button action can be mapped to.
     */
    interface AdditionalKey extends Serializable {
        /**
         * User friendly value of the key.
         * @return user friendly value of the key.
         */
        String getString();

        /**
         * Internal value of the key.
         * @return internal value of the key
         */
        String getId();
    }

}
