package de.rainerhock.eightbitwonders;

import static androidx.test.espresso.Espresso.onData;
import static androidx.test.espresso.Espresso.onView;
import static androidx.test.espresso.action.ViewActions.click;
import static androidx.test.espresso.action.ViewActions.scrollTo;
import static androidx.test.espresso.assertion.ViewAssertions.matches;
import static androidx.test.espresso.matcher.ViewMatchers.isDisplayed;
import static androidx.test.espresso.matcher.ViewMatchers.isEnabled;
import static androidx.test.espresso.matcher.ViewMatchers.isRoot;
import static androidx.test.espresso.matcher.ViewMatchers.withId;
import static androidx.test.espresso.matcher.ViewMatchers.withTagValue;
import static androidx.test.espresso.matcher.ViewMatchers.withText;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.anyOf;

import android.util.Log;
import android.view.KeyCharacterMap;
import android.view.KeyEvent;
import android.view.View;
import android.view.inputmethod.BaseInputConnection;

import androidx.test.espresso.UiController;
import androidx.test.espresso.ViewAction;
import androidx.test.platform.app.InstrumentationRegistry;

import junit.framework.AssertionFailedError;

import org.hamcrest.Matcher;
import org.junit.Test;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

public class HardwareKeyboardTest extends MainActivityTestBase {
    private void prepareSetting(Matcher<?> keyboard) {
        onView(withId(R.id.ib_hamburger_menu)).perform(tap());
        waitForIdle();
        onView(withText(R.string.IDS_MP_SETTINGS)).perform(click());
        waitForIdle();
        onView(withText(R.string.keyboard_input)).perform(click());
        onView(withId(R.id.sp_keyboard_mapping)).perform(click());
        onData(keyboard).perform(click());

        onView(withText(R.string.joysticks)).perform(scrollTo()).perform(click());
        waitForIdle();
        onView(withTagValue(equalTo("port#1"))).perform(scrollTo()).perform(click());
        onData(isOption(R.string.IDS_NONE)).perform(click());
        onView(withId(R.id.bn_apply)).perform(click());
        waitForIdle();
    }

    static List<ViewAction> createTypeActions(String keys, @SuppressWarnings("SameParameterValue") int minimalDownTime, @SuppressWarnings("SameParameterValue") int maximalDownTime) {
        KeyCharacterMap charMap = KeyCharacterMap.load(KeyCharacterMap.VIRTUAL_KEYBOARD);
        List<ViewAction> ret = new LinkedList<>();
        for (int i = 0; i < keys.length(); i++) {
            final int pos = i;
            int delay = ThreadLocalRandom.current().nextInt(minimalDownTime, maximalDownTime + 1);
            ret.add(new ViewAction() {
                @Override
                public String getDescription() {
                    return String.format("hold key representing %s for %d ms", keys.charAt(pos), delay);
                }

                @Override
                public Matcher<View> getConstraints() {
                    return isEnabled();
                }

                @Override
                public void perform(UiController uiController, View view) {
                    KeyEvent[] events = charMap.getEvents(keys.substring(pos, pos + 1).toCharArray());
                    BaseInputConnection conn = new BaseInputConnection(view, false);
                    if (events != null && events.length == 2 && events[0].getModifiers() == 0 && events[0].getKeyCode() != KeyEvent.KEYCODE_UNKNOWN) {
                        try {
                            KeyEvent e = new KeyEvent(KeyEvent.ACTION_DOWN, events[0].getKeyCode());
                            Log.v("simulate", "sending " + e + " and up after" + delay);
                            conn.sendKeyEvent(e);
                            Thread.sleep(delay);
                            conn.sendKeyEvent(new KeyEvent(KeyEvent.ACTION_UP, events[0].getKeyCode()));
                        }
                        catch (InterruptedException e) {
                            throw new RuntimeException(e);
                        }
                    } else {
                        KeyEvent ke = new KeyEvent(delay, keys.substring(pos, pos + 1), 0, 0);
                        conn.sendKeyEvent(ke);
                        try {
                            Thread.sleep(delay);
                        }
                        catch (InterruptedException e) {
                            throw new RuntimeException(e);
                        }
                    }
                }
            });
        }
        return ret;
    }

    @Test
    public void t_0270_us_c64() {
        t_0270_us(InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.name_c64),
                "screenshot-typein-en-pos.pixelbuffer",
                KeyEvent.KEYCODE_2);
    }

    @Test
    public void t_0270_us_vic20() {
        t_0270_us(InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.name_vic20),
                "screenshot-vic-typein-en-pos.pixelbuffer", KeyEvent.KEYCODE_1);
        typeRsRestoreShiftCbm("screenshot-vic-rs-restore-largecaps.pixelbuffer", "screenshot-vic-rs-restore-smallcaps.pixelbuffer");

    }

    private void t_0270_us(String machine, String assetTypeIn, int colorKeycode) {
        waitForIdle();
        waitForActivity(MainActivity.class,2,TimeUnit.SECONDS);
        onView(withText(machine))
                .perform(setJoysticksConnected(DpadJoystick.class, false))
                .perform(setJoysticksConnected(GameControllerJoystick.class, false))
                .perform(click());
        waitForIdle();
        waitForActivity(EmulationActivity.class, 2, TimeUnit.SECONDS);

        /*
        Action: enable keyboard german, by position and disable joystick, go back
        Expected result: -
         */
        prepareSetting(anyOf(isOption("English, by position"), isOption("Englisch, über Position")));
        /*
        Action: clear screen, type all characters
        Expected result: screen displays the correct characters.
         */
        waitForIdle(5, TimeUnit.SECONDS);
        typeUsKeys(colorKeycode, true);
        waitForIdle(1, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset(assetTypeIn, 1, TimeUnit.SECONDS)));
    }
    private void typeUsKeys(int colorKeycode, boolean functionKeys) {
        typeUsKeys(colorKeycode, functionKeys, true, true);
    }
    private void typeUsKeys(int colorKeycode, boolean functionKeys, boolean clearscreen, boolean shiftedKeys) {
        if (clearscreen) {
            onView(isC64Key("SHIFT")).perform(tap());
            onView(isC64Key("HOME")).perform(tap());
        }
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        // First row without shift
        for (ViewAction v : createTypeActions("`1234567890-=", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        for (ViewAction v : createTypeActions(Collections.singletonList(KeyEvent.KEYCODE_INSERT), 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }

        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);

        // First row with shift
        if (shiftedKeys) {
            for (ViewAction v : createTypeActions("!@#$%^&*(_+", 50, 500)) {
                onView(withId(R.id.screen)).perform(v);
            }
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_SHIFT_LEFT, 50, 500));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_INSERT, 50, 500));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_INSERT));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_SHIFT_LEFT));

            onView(isC64Key("SHIFT")).perform(tap());
            onView(isC64Key("RETURN")).perform(tap());
            waitForIdle(100, TimeUnit.MILLISECONDS);
        }
        // second row without shift
        for (ViewAction v : createTypeActions("qwertyuiop[]", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        for (ViewAction v : createTypeActions(Collections.singletonList(KeyEvent.KEYCODE_PAGE_DOWN), 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        // second row with shift
        if (shiftedKeys) {
            for (ViewAction v : createTypeActions("QWERTYUIOP{}", 50, 500)) {
                onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
                onView(withId(R.id.screen)).perform(v);
            }
            // second row with shift
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_SHIFT_LEFT, 50, 500));
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_PAGE_DOWN, 50, 500));
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_PAGE_DOWN));
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_SHIFT_LEFT));
            onView(isC64Key("SHIFT")).perform(tap());
            onView(isC64Key("RETURN")).perform(tap());
            waitForIdle(100, TimeUnit.MILLISECONDS);
        }
        // third row without shift
        for (ViewAction v : createTypeActions("asdfghjkl;'\\", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        // third row with shift
        if (shiftedKeys) {
            for (ViewAction v : createTypeActions("ASDFGHJKL:\"", 50, 500)) {
                onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
                onView(withId(R.id.screen)).perform(v);
            }
            onView(isC64Key("SHIFT")).perform(tap());
            onView(isC64Key("RETURN")).perform(tap());
            waitForIdle(100, TimeUnit.MILLISECONDS);
        }
        // fourth row without shift
        for (ViewAction v : createTypeActions("zxcvbnm,./", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        // fourth row with shift
        for (ViewAction v : createTypeActions("ZXCVBNM", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        waitForIdle(100, TimeUnit.MILLISECONDS);
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_SHIFT_LEFT, 50, 500));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_COMMA, 50, 500));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_COMMA));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_SHIFT_LEFT));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));

        for (ViewAction v : createTypeActions("?", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }

        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);

        // Cursor keys (press key and and a character to visualize where the cursor was)
        for (ViewAction v : createTypeActions(Arrays.asList(
                KeyEvent.KEYCODE_A, KeyEvent.KEYCODE_DPAD_RIGHT,
                KeyEvent.KEYCODE_B, KeyEvent.KEYCODE_DPAD_DOWN,
                KeyEvent.KEYCODE_C, KeyEvent.KEYCODE_DPAD_LEFT,
                KeyEvent.KEYCODE_D, KeyEvent.KEYCODE_DPAD_UP,
                KeyEvent.KEYCODE_E), 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);


        // LOAD and Run/Stop
        for (ViewAction v : createTypeActions("load", 50, 500)) {
            //onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_ESCAPE, 50, 500));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_ESCAPE));
        // Tab + 2 = white cursor
        if (colorKeycode != KeyEvent.KEYCODE_UNKNOWN) {
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_TAB, 50, 500));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, colorKeycode, 50, 500));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, colorKeycode));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_TAB));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_X, 50, 500));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_X));
        }
        // space
        for (ViewAction v : createTypeActions(" ", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        if (colorKeycode != KeyEvent.KEYCODE_UNKNOWN) {
            // Ctrl+7 = light blue cursor
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_CTRL_LEFT, 50, 500));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_7, 50, 500));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_7));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_CTRL_LEFT));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_X, 50, 500));
            onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_X));
            onView(isC64Key("SHIFT")).perform(tap());
            onView(isC64Key("RETURN")).perform(tap());
            waitForIdle(100, TimeUnit.MILLISECONDS);
        }


        for (ViewAction v : createTypeActions("@", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        if (functionKeys) {
            for (ViewAction v : createTypeActions(Arrays.asList(

                    KeyEvent.KEYCODE_F1, KeyEvent.KEYCODE_F2,
                    KeyEvent.KEYCODE_F3, KeyEvent.KEYCODE_F4,
                    KeyEvent.KEYCODE_F5, KeyEvent.KEYCODE_F6,
                    KeyEvent.KEYCODE_F7, KeyEvent.KEYCODE_F8), 50, 500)) {
                onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
                onView(withId(R.id.screen)).perform(v);
            }
        }
        for (ViewAction v : createTypeActions("@", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        // Some graphical symbols
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_CTRL_LEFT, 50, 500));
        waitForIdle(250, TimeUnit.MILLISECONDS);
        for (ViewAction v : createTypeActions("azy-=", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        for (ViewAction v : createTypeActions(Collections.singletonList(KeyEvent.KEYCODE_INSERT), 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_CTRL_LEFT));
        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
    }
    private void typeRsRestoreShiftCbm (String afterRsRestore, String afterShiftCbm) {
        // Run+Stop+Restore
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_ESCAPE, 50, 500));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating(3)));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_PAGE_UP, 50, 500));
        try {
            Thread.sleep(100);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }

        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_PAGE_UP));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating(3)));

        // Business as usual: restore sometimes has to pressed twice.
        try {
            Thread.sleep(100);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }

        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_PAGE_UP, 50, 500));
        try {
            Thread.sleep(100);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }

        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_PAGE_UP));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating(3)));

        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_ESCAPE));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset(afterRsRestore)));
        // Strg+Shift
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating(2)));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_CTRL_LEFT, 50, 500));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating(2)));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_SHIFT_LEFT, 50, 500));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating(2)));
        try {
            Thread.sleep(100);
        }
        catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_SHIFT_LEFT));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating(2)));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_CTRL_LEFT));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating(2)));
        waitForIdle();
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset(afterShiftCbm)));
    }

    private List<ViewAction> createDoubleKeyEvent(@SuppressWarnings("SameParameterValue") int code1, int code2, @SuppressWarnings("SameParameterValue") int minimalDownTime, @SuppressWarnings("SameParameterValue") int maximalDownTime) {
        List<ViewAction> ret = new LinkedList<>();
        int[] keycodes = new int[]{code1, code2, code2, code1};
        int[] actions = new int[]{KeyEvent.ACTION_DOWN, KeyEvent.ACTION_DOWN, KeyEvent.ACTION_UP, KeyEvent.ACTION_UP};
        for (int i = 0; i < 4; i++) {
            final int pos = i;
            int delay = ThreadLocalRandom.current().nextInt(minimalDownTime, maximalDownTime + 1);
            ret.add(new ViewAction() {
                @Override
                public String getDescription() {
                    return "triggering " + new KeyEvent(actions[pos], keycodes[pos]);
                }

                @Override
                public Matcher<View> getConstraints() {
                    return isEnabled();
                }

                @Override
                public void perform(UiController uiController, View view) {
                    BaseInputConnection conn = new BaseInputConnection(view, false);
                    conn.sendKeyEvent(new KeyEvent(actions[pos], keycodes[pos]));
                    try {
                        Thread.sleep(delay);
                    }
                    catch (InterruptedException e) {
                        throw new RuntimeException(e);
                    }

                }
            });
        }
        return ret;
    }

    @Test
    public void t_0010_de_c64() {
        t_0010_de(InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.name_c64),
                "screenshot-typein-de-pos.pixelbuffer");
    }

    @Test
    public void t_0010_de_vic20() {
        t_0010_de(InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.name_vic20),
                "screenshot-vic-typein-de-pos.pixelbuffer");


    }

    private void t_0010_de(String machine, String assetTypeIn) {
        waitForIdle();
        onView(withText(machine))
                .perform(setJoysticksConnected(DpadJoystick.class, false))
                .perform(setJoysticksConnected(GameControllerJoystick.class, false))
                .perform(click());
        waitForIdle();
        waitForActivity(EmulationActivity.class, 2, TimeUnit.SECONDS);
        prepareSetting(anyOf(isOption("German, by position"), isOption("Deutsch, über Position")));
        waitForIdle(5, TimeUnit.SECONDS);
        // first row has most characters different from us.
        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        for (ViewAction v : createTypeActions("^!\"", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        for (ViewAction v : createDoubleKeyEvent(KeyEvent.KEYCODE_SHIFT_LEFT, KeyEvent.KEYCODE_3, 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        for (ViewAction v : createTypeActions("$%&", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        for (ViewAction v : createDoubleKeyEvent(KeyEvent.KEYCODE_SHIFT_LEFT, KeyEvent.KEYCODE_7, 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_SHIFT_LEFT));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_SHIFT_LEFT));
        for (ViewAction v : createTypeActions("`", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_SHIFT_LEFT));
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(isC64Key("SHIFT")).perform(tap());
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        // Z and Y are swapped on german keyboards
        for (ViewAction v : createTypeActions("zy", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(isC64Key("SHIFT")).perform(tap());
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(isC64Key("RETURN")).perform(tap());// END ist the up arrow
        waitForIdle(100, TimeUnit.MILLISECONDS);
        for (ViewAction v : createTypeActions(Collections.singletonList(KeyEvent.KEYCODE_MOVE_END), 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        //onView(withId(R.id.monitorGLSurfaceView)).perform(captureEmulationScreen(assetTypeIn));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset(assetTypeIn)));
        waitForIdle();
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
    }

    private void test_restore_de(String machine, Matcher<?> matcher, int keycode, String colorkey, String assetScreenshotLoadingReady, String assetScreenshotAfterRestore) {
        waitForIdle();
        onView(withText(machine))
                .perform(setJoysticksConnected(DpadJoystick.class, false))
                .perform(setJoysticksConnected(GameControllerJoystick.class, false))
                .perform(click());
        waitForIdle();
        waitForActivity(EmulationActivity.class, 2, TimeUnit.SECONDS);
        prepareSetting(matcher);
        try {
            openDocument(extractTestAsset("testprograms.d64"));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        waitForIdle();
        onData(isOption("TESTRESTORE")).inAdapterView(withId(R.id.lv_imagecontents))
                .check(matches(isDisplayed()))
                .perform(click());
        waitForIdle();
        waitForActivity(EmulationActivity.class, 2, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset(assetScreenshotLoadingReady, 20, TimeUnit.SECONDS)));
        waitForIdle();
        onView(isC64Key("S")).perform(tap());
        onView(isC64Key("Y")).perform(tap());
        onView(isC64Key("S")).perform(tap());
        onView(isC64Key("4")).perform(tap());
        onView(isC64Key("3")).perform(tap());
        onView(isC64Key("5")).perform(tap());
        onView(isC64Key("2")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        onView(isC64Key("HOME")).perform(tap());
        onView(isC64Key("CTRL")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        onView(isC64Key(colorkey)).perform(tap());
        waitForIdle();
        for (ViewAction v : createTypeActions(Arrays.asList(keycode, keycode), 100, 500)) {
            waitForIdle();
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            waitForIdle();
            onView(withId(R.id.screen)).perform(v);
        }
        waitForIdle();
        try {
            Thread.sleep(500);
            onView(withId(R.id.gv_monitor)).perform(captureEmulationScreen("1.pixeldata"));
            Thread.sleep(100);
            onView(withId(R.id.gv_monitor)).perform(captureEmulationScreen("2.pixeldata"));
            Thread.sleep(100);
            onView(withId(R.id.gv_monitor)).perform(captureEmulationScreen("3.pixeldata"));
            Thread.sleep(100);
            onView(withId(R.id.gv_monitor)).perform(captureEmulationScreen("4.pixeldata"));
            Thread.sleep(100);
            onView(withId(R.id.gv_monitor)).perform(captureEmulationScreen(".pixeldata"));
        }
        catch (InterruptedException e) {
            // blerb
        }
        onView(withId(R.id.gv_monitor))
                .check(matches(isScreenUpdating()))
                .check(matches(showsBitmapEqualToAsset(assetScreenshotAfterRestore, 460, TimeUnit.MILLISECONDS)));
    }

    @Test
    public void test_restore_c64_de() {
        test_restore_de(InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.name_c64),
                anyOf(isOption("German, by position"), isOption("Deutsch, über Position")),
                KeyEvent.KEYCODE_F12,
                "7",
                "screenshot-c64-restore-prg-loaded.pixelbuffer",
                "screenshot-c64-restore.pixelbuffer");
    }

    @Test
    public void test_restore_c64_us() {
        test_restore_de(InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.name_c64),
                anyOf(isOption("English, by position"), isOption("Englisch, über Position")),
                KeyEvent.KEYCODE_PAGE_UP,
                "7",
                "screenshot-c64-restore-prg-loaded.pixelbuffer",
                "screenshot-c64-restore.pixelbuffer");
    }

    @Test
    public void test_restore_vic20_de() {
        test_restore_de(InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.name_vic20),
                anyOf(isOption("German, by position"), isOption("Deutsch, über Position")),
                KeyEvent.KEYCODE_F12,
                "2",
                "screenshot-vic20-restore-prg-loaded.pixelbuffer",
                "screenshot-vic20-restore.pixelbuffer");
    }

    @Test
    public void test_restore_vic20_us() {
        test_restore_de(InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.name_vic20),
                anyOf(isOption("English, by position"), isOption("Englisch, über Position")),
                KeyEvent.KEYCODE_PAGE_UP,
                "2",
                "screenshot-vic20-restore-prg-loaded.pixelbuffer",
                "screenshot-vic20-restore.pixelbuffer");
    }

    public void test_pet_keyboard(final Matcher<?> setting, boolean businessKeyboard) {
        waitForIdle();
        onView(isRoot())
                .perform(setJoysticksConnected(DpadJoystick.class, false))
                .perform(setJoysticksConnected(GameControllerJoystick.class, false));
        waitForIdle();
        waitForActivity(EmulationActivity.class, 2, TimeUnit.SECONDS);
        prepareSetting(setting);
        typeUsKeys(KeyEvent.KEYCODE_UNKNOWN, false, businessKeyboard, businessKeyboard);


    }

    @Test
    public void test_pet_business_keyboard() {
        waitForIdle();
        onView(withText(R.string.name_pet))
                .perform(setJoysticksConnected(DpadJoystick.class, false))
                .perform(setJoysticksConnected(GameControllerJoystick.class, false))
                .perform(click());
        waitForIdle();
        waitForActivity(EmulationActivity.class, 5, TimeUnit.SECONDS);
        switchToPetModel("8032");
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-machine-ready-pet8032.pixelbuffer", 2, TimeUnit.SECONDS)));
        test_pet_keyboard(anyOf(isOption("English, by position"), isOption("Englisch, über Position")), true);
        onView(allOf(isDisplayed(), isC64Key("SHIFT"))).perform(tap());
        onView(allOf(isDisplayed(), isC64Key("RETURN"))).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_DEL, 50, 500));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_DEL, 50, 500));
        try {
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-typein-pet-buen-pos.pixelbuffer")));
        } catch (AssertionFailedError e) {
            // Race condition...
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-typein-pet-buen-pos-variant.pixelbuffer")));
        }
    }
    @Test
    public void test_pet_graphical_keyboard() {
        waitForIdle();
        onView(withText(R.string.name_pet))
                .perform(setJoysticksConnected(DpadJoystick.class, false))
                .perform(setJoysticksConnected(GameControllerJoystick.class, false))
                .perform(click());
        waitForIdle();
        waitForActivity(EmulationActivity.class, 5, TimeUnit.SECONDS);
        switchToPetModel("4032");
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-machine-ready-pet4032.pixelbuffer")));
        test_pet_keyboard(anyOf(isOption("English, by position"), isOption("Englisch, über Position")), false);
        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_DEL, 50, 500));
        onView(withId(R.id.screen)).perform(createTypeAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_DEL, 50, 500));
        onView(isC64Key("SHIFT")).perform(tap());
        onView(isC64Key("RETURN")).perform(tap());
        onView(isC64Key("SHIFT")).perform(tap());
        waitForIdle(100, TimeUnit.MILLISECONDS);
        for (ViewAction v : createTypeActions("''", 50, 500)) {
            onView(withId(R.id.gv_monitor)).check(matches(isScreenUpdating()));
            onView(withId(R.id.screen)).perform(v);
        }

        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-typein-pet-gren-pos.pixelbuffer")));
    }
}
