package de.rainerhock.eightbitwonders;

import static androidx.test.espresso.Espresso.onData;
import static androidx.test.espresso.Espresso.onView;
import static androidx.test.espresso.action.MotionEvents.obtainMovement;
import static androidx.test.espresso.action.MotionEvents.obtainUpEvent;
import static androidx.test.espresso.action.ViewActions.click;
import static androidx.test.espresso.action.ViewActions.scrollTo;
import static androidx.test.espresso.assertion.ViewAssertions.matches;
import static androidx.test.espresso.matcher.ViewMatchers.isChecked;
import static androidx.test.espresso.matcher.ViewMatchers.isDescendantOfA;
import static androidx.test.espresso.matcher.ViewMatchers.isDisplayed;
import static androidx.test.espresso.matcher.ViewMatchers.isEnabled;
import static androidx.test.espresso.matcher.ViewMatchers.withChild;
import static androidx.test.espresso.matcher.ViewMatchers.withId;
import static androidx.test.espresso.matcher.ViewMatchers.withTagValue;
import static androidx.test.espresso.matcher.ViewMatchers.withText;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.not;

import android.content.Context;
import android.view.InputDevice;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.widget.SeekBar;

import androidx.annotation.NonNull;
import androidx.annotation.StringRes;
import androidx.test.espresso.NoMatchingViewException;
import androidx.test.espresso.UiController;
import androidx.test.espresso.ViewAction;
import androidx.test.espresso.action.GeneralClickAction;
import androidx.test.espresso.action.GeneralLocation;
import androidx.test.espresso.action.MotionEvents;
import androidx.test.espresso.action.Press;
import androidx.test.espresso.action.Tap;
import androidx.test.platform.app.InstrumentationRegistry;

import junit.framework.AssertionFailedError;

import org.hamcrest.Matcher;
import org.junit.Test;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

public class GameControllerJoystickTest extends C64JoystickTestBase {
    final static int DEVICE_ID = -73;

    static class UnitTestGameControllerJoystick extends GameControllerJoystick {
        private final int mFakeXAxis;
        private final int mFakeYAxis;
        private final String mFakeJoystickDescription;


        UnitTestGameControllerJoystick(final Context ctx, final InputDevice dev,
                                       final int xAxis, final int yAxis,
                                       final Integer[] triggerAxes,
                                       final Integer[] triggerKeys,
                                       final ValueType valuetype) {

            super(ctx, dev, xAxis, yAxis, triggerAxes, triggerKeys,valuetype);
            mFakeXAxis = xAxis;
            mFakeYAxis = yAxis;
            switch (xAxis) {
                case MotionEvent.AXIS_X:
                    mFakeJoystickDescription =
                            String.format(" (%s)", ctx.getString(R.string.left_stick));
                    break;
                case MotionEvent.AXIS_Z:
                    mFakeJoystickDescription =
                            String.format(" (%s)", ctx.getString(R.string.right_stick));
                    break;
                case MotionEvent.AXIS_HAT_X:
                    mFakeJoystickDescription =
                            String.format(" (%s)", ctx.getString(R.string.dpad));
                    break;
                default:
                    mFakeJoystickDescription = "";

            }
            
        }
        boolean deviceHasButtonWithKeycode(int keycode) {
            return Arrays.asList(KeyEvent.KEYCODE_BUTTON_X, KeyEvent.KEYCODE_BUTTON_Y)
                    .contains(keycode);
        }

        @Override
        protected float getFlatRange(final InputDevice device, final int axis, final int source) {
            return 0.25f;
        }
        @Override
        String getHardwareButtons() {
            return "KEYBUTTON_A";
        }
        @NonNull
        @Override
        public String toString() {
            return FAKE_GAMECONTROLLER_DEVICEID + mFakeJoystickDescription;
        }
        @Override
        String getId() {
            return FAKE_GAMECONTROLLER_DEVICEID + "-" + mFakeXAxis;
        }
        @Override
        String getHardwareDescriptor() {
            return FAKE_GAMECONTROLLER_DEVICEID + "-" + mFakeXAxis;
        }

        @Override
        int getHardwareId() {
            return DEVICE_ID;
        }
        @Override
        protected boolean isDeviceMatch(InputDevice dev) {
            return true;
        }
        @Override
        protected float getCurrentValue(MotionEvent event, final int axis, int historyPos) {
            if (axis == mFakeXAxis) {
                return event.getX();
            }
            if (axis == mFakeYAxis) {
                return event.getY();
            }

            return 0f;
        }
        @Override
        protected boolean isMatchingHardwareDescriptor(KeyEvent event) {
            return false;
        }
    }
    static ViewAction release(JoystickState state) {
        if (state.downEvent == null) {
            throw new IllegalStateException("Joystick not in use");
        }
        return new ViewAction() {
            @Override
            public String getDescription() {
                return null;
            }

            @Override
            public Matcher<View> getConstraints() {
                return isEnabled();
            }

            @Override
            public void perform(UiController uiController, View view) {
                MotionEvent moveEvent = obtainUpEvent(state.downEvent, new float[] {0f,0f});
                Objects.requireNonNull(getActivity()).dispatchGenericMotionEvent(moveEvent);
                state.downEvent = null;

            }
        };
    }
    static ViewAction setStickPosition(JoystickState state, float x, float y) {
        return new ViewAction() {
            @Override
            public String getDescription() {
                String s = state != null && state.downEvent != null
                        ? state.downEvent.toString() : "<null>";
                return String.format("move Gamecontroller stick to position [%f/%f], downEvent = %s",
                        x, y, s);
            }

            @Override
            public Matcher<View> getConstraints() {
                return isEnabled();
            }
            @Override
            public void perform(UiController uiController, View view) {
                if (state.downEvent == null) {
                    state.downEvent = MotionEvents.obtainDownEvent(new float[]{0f, 0f}, new float[]{1f, 1f});
                }
                Objects.requireNonNull(getActivity()).onGenericMotionEvent(state.downEvent);
                MotionEvent moveEvent = obtainMovement(state.downEvent, new float[] {x, y});
                Objects.requireNonNull(getActivity()).onGenericMotionEvent(moveEvent);
            }
        };
    }

    static class JoystickState {

        public MotionEvent downEvent = null;
    }

    @Test
    public void t_0010_joystick_1() {
        /*
        Action: open settings.
        Expected result: Joystick named "Joystick _UNITTEST_GAMECONTROLLER_JOYSTICK_ (Joystick 1)" exists

         */
        onView(withId(R.id.ib_hamburger_menu)).perform(tap());
        waitForIdle();
        onView(withText(R.string.IDS_MP_SETTINGS)).perform(click());
        waitForIdle();
        onView(withText(R.string.joysticks)).perform(click());
        waitForIdle();
        onView(withTagValue(equalTo("port#1"))).perform(click());
        /* Action
        Choose Joystick "Joystick _UNITTEST_GAMECONTROLLER_JOYSTICK_ (Joystick 1)" and load JOYSTICKTEST from testprograms.
        Expected result: 255 is displayed
         */
        onData(isOption(String.format("%s (%s)", FAKE_GAMECONTROLLER_DEVICEID, InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.left_stick)))).perform(click());
        waitForIdle();
        onView(withId(R.id.bn_apply)).perform(click());
        waitForIdle();

        try {
            openDocument(extractTestAsset("testprograms.d64"));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        onData(isOption("JOYSTICKTEST")).inAdapterView(withId(R.id.lv_imagecontents))
                .check(matches(isDisplayed()))
                .perform(click());
        waitForIdle();
        waitForActivity(EmulationActivity.class, 5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-centered.pixelbuffer", 20, TimeUnit.SECONDS)));
        JoystickState state = new JoystickState();
        /*
        Action: move joystick left
        Expected result: 251 is shown
         */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, -1f, 0f))
                .check(matches(showsBitmapEqualToAsset("screenshot-joystick-left.pixelbuffer", 20, TimeUnit.SECONDS)));
        /*
        Action: recenter joystick
        Expected result: 255 is displayed.
         */
        onView(withId(R.id.gv_monitor))
                .perform(release(state))
                .check(matches(showsBitmapEqualToAsset("screenshot-joystick-centered.pixelbuffer", 20, TimeUnit.SECONDS)));
        /*
        Action: move joystick right, only slightly up
        Expected result: 247 is shown
         */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 1f, 0.2f))
                .check(matches(showsBitmapEqualToAsset("screenshot-joystick-right.pixelbuffer", 20, TimeUnit.SECONDS)));
        /*
        Action: move joystick back to center, but not fully.
        Expected result: 255 is displayed.
         */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0.2f, 0.2f))
                .check(matches(showsBitmapEqualToAsset("screenshot-joystick-centered.pixelbuffer", 20, TimeUnit.SECONDS)));


        /*
        Action: move joystick  up
        Expected result: 254 is shown
        */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0f, -1f))
                .check(matches(showsBitmapEqualToAsset("screenshot-joystick-up.pixelbuffer", 20, TimeUnit.SECONDS)));
        /*
        Action: move joystick  up
        Expected result: 254 is shown
        */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0f, -1f))
                .check(matches(showsBitmapEqualToAsset("screenshot-joystick-up.pixelbuffer", 20, TimeUnit.SECONDS)));
        /*
        Action: move joystick  down
        Expected result: 253 is shown
        */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0f, 1f))
                .check(matches(showsBitmapEqualToAsset("screenshot-joystick-down.pixelbuffer", 20, TimeUnit.SECONDS)));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_THUMBL, -73));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-down-fire.pixelbuffer", 20, TimeUnit.SECONDS)));

    }
    @Test
    public void t_0310_firebuttons() {
        try {
            openDocument(extractTestAsset("testprograms.d64"));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        onData(isOption("JOYSTICKTEST")).inAdapterView(withId(R.id.lv_imagecontents))
                .check(matches(isDisplayed()))
                .perform(click());
        waitForIdle(1, TimeUnit.SECONDS);
        //onView(isRoot()).perform(waitForMatch(withId(R.id.hamburger_menu),1, TimeUnit.SECONDS));
        onView(withId(R.id.ib_hamburger_menu)).perform(tap());
        waitForIdle();
        onView(withText(R.string.IDS_MP_SETTINGS)).perform(click());
        waitForIdle();
        onView(withText(R.string.joysticks)).perform(click());
        waitForIdle();
        onView(withTagValue(equalTo("port#1"))).perform(click());
        onData(isOption(String.format("%s (%s)", FAKE_GAMECONTROLLER_DEVICEID, InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.right_stick)))).perform(click());
        onView(withTagValue(equalTo("port#2"))).perform(scrollTo()).perform(click());
        onData(isOption(InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.IDS_NONE))).perform(click());
        onView(withId(R.id.bn_apply)).perform(click());
        waitForIdle();
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-centered.pixelbuffer", 20, TimeUnit.SECONDS)));
        for (int keycode: Arrays.asList(KeyEvent.KEYCODE_BUTTON_L1, KeyEvent.KEYCODE_BUTTON_L2,
                KeyEvent.KEYCODE_BUTTON_R1, KeyEvent.KEYCODE_BUTTON_R2,
                KeyEvent.KEYCODE_BUTTON_THUMBL, KeyEvent.KEYCODE_BUTTON_THUMBR)) {
            onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, keycode, DEVICE_ID));
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-fire.pixelbuffer", 20, TimeUnit.SECONDS)));
            onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, keycode, DEVICE_ID));
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-centered.pixelbuffer", 20, TimeUnit.SECONDS)));
        }
        onView(withId(R.id.ib_hamburger_menu)).perform(tap());
        waitForIdle();
        onView(withText(R.string.IDS_MP_SETTINGS)).perform(click());
        waitForIdle();
        onView(withTagValue(equalTo("port#1"))).perform(click());
        onData(isOption(String.format("%s (%s)", FAKE_GAMECONTROLLER_DEVICEID, InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.dpad)))).perform(click());
        onView(withTagValue(equalTo("port#2"))).perform(click());
        onData(isOption(String.format("%s (%s)", FAKE_GAMECONTROLLER_DEVICEID, InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.left_stick)))).perform(click());
        onView(withId(R.id.bn_apply)).perform(click());
        waitForIdle();
        for (int keycode: Arrays.asList(KeyEvent.KEYCODE_BUTTON_L1,KeyEvent.KEYCODE_BUTTON_R1, KeyEvent.KEYCODE_BUTTON_R2)) {
            onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, keycode, DEVICE_ID));
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-fire.pixelbuffer", 20, TimeUnit.SECONDS)));
            onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, keycode, DEVICE_ID));
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-centered.pixelbuffer", 20, TimeUnit.SECONDS)));
        }
        for (int keycode: Collections.singletonList(KeyEvent.KEYCODE_BUTTON_L2)) {
            onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, keycode, DEVICE_ID));
            onView(withId(R.id.gv_monitor)).check(matches(not(showsBitmapEqualToAsset("screenshot-joystick-fire.pixelbuffer", 20, TimeUnit.SECONDS))));
            onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, keycode, DEVICE_ID));
        }
    }
    @Test
    public void test_diagonals_settings() {
        JoystickState state = new JoystickState();
        /*
        Action: open settings
        Expected result: Checkbox "Lock diagonals" is not checked.
         */
        check_diagonals_not_checked(isOption(String.format("%s (%s)", FAKE_GAMECONTROLLER_DEVICEID, InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.left_stick))));
        run_joystick_test_program();
        /*
        Action: press direction at about 11:00 clock position
        Expected result: 250 is shown (up left)
         */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, -0.7f, -0.9f));

        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-left-up.pixelbuffer",10, TimeUnit.SECONDS)));
        /*
        Action: move finger on direction to about 10:00 clock position
        Expected result: 250 is shown (up left)
         */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, -0.9f, -0.7f));

        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-left-up.pixelbuffer",10, TimeUnit.SECONDS)));

        /* Action: Remove finger from direction, open settings, check "Log diagonals" and go back.
        Expected result: 255 is displayed
         */
        onView(withId(R.id.gv_monitor))
                .perform(release(state));
        set_check_in_lock_diagonals();

        /*
        Action: press direction at about 11:00 clock position
        Expected result: 254 is shown (up)
         */
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-centered.pixelbuffer", 10, TimeUnit.SECONDS)));
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, -0.7f, -0.9f));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-up.pixelbuffer",10, TimeUnit.SECONDS)));
        /* Action: Remove finger from direction, open settings, check "Log diagonals" and go back.
        Expected result: 255 is displayed
         */
        onView(withId(R.id.gv_monitor))
                .perform(release(state));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-centered.pixelbuffer",10, TimeUnit.SECONDS)));
        /*
        Action: press direction at about 10:00 clock position
        Expected result: 251 is shown (left)
         */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, -0.9f, -0.7f));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-left.pixelbuffer",10, TimeUnit.SECONDS)));
        /*
        Action: open settings and uncheck "Lock settings"
        Expected result: -
         */
        onView(withId(R.id.gv_monitor))
                .perform(release(state));
        unset_check_in_lock_diagonals();

    }
    private void setSensitivy(GeneralLocation location) {
        onView(withId(R.id.ib_hamburger_menu)).perform(tap());
        waitForIdle();
        onView(withText(R.string.IDS_MP_SETTINGS)).perform(scrollTo()).perform(click());
        waitForIdle();

        onView(withTagValue(equalTo("port#1"))).perform(scrollTo()).perform(click());
        onData(isOption(String.format("%s (%s)", FAKE_GAMECONTROLLER_DEVICEID, InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.left_stick)))).perform(click());
        onView(withTagValue(equalTo("port#2"))).perform(scrollTo()).perform(click());
        onData(isOption(R.string.IDS_NONE)).perform(click());
        onView(withTagValue(equalTo("port#1"))).perform(scrollTo());
        waitForView(allOf(isDisplayed(), withId(R.id.sb_sensitivy)), 5, TimeUnit.SECONDS);

        //noinspection deprecation
        onView(allOf(isDisplayed(), withId(R.id.sb_sensitivy)))
                .perform(scrollTo())
                .perform(new GeneralClickAction(Tap.SINGLE, location, Press.FINGER));
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class, 1, TimeUnit.SECONDS);
    }
    @Test
    @FlakyMouseTest
    public void testSensitivy() {
        JoystickState state = new JoystickState();
        run_joystick_test_program();
        onView(withId(R.id.ib_hamburger_menu)).perform(tap());
        waitForIdle();
        onView(withText(R.string.IDS_MP_SETTINGS)).perform(click());
        onView(withText(R.string.joysticks)).perform(click());
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class, 1, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-centered.pixelbuffer", 20, TimeUnit.SECONDS)));
        /*
        Action: set sensitivity low
        Expected result: no reaction to medium thumbstick amplitudes
         */
        setSensitivy(GeneralLocation.CENTER_LEFT);
        /*
        Action: move joystick 70% up
        Expected result: 255 is shown
         */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0f, -0.65f));
        waitForIdle(1, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-joystick-centered.pixelbuffer", 1, TimeUnit.SECONDS)));
        /*
        Action: set sensitivity high
        Expected result: reaction to same thumbstick amplitude as above
         */
        setSensitivy(GeneralLocation.CENTER_RIGHT);
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0f, -0.65f))
                .check(matches(showsBitmapEqualToAsset("screenshot-joystick-up.pixelbuffer", 20, TimeUnit.SECONDS)));
    }

    private void opensettings(@StringRes int devicetypeoption) {
        setControllerPort1(
                String.format("%s (%s)", FAKE_GAMECONTROLLER_DEVICEID, InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.left_stick)),
                devicetypeoption, true);
    }

    @Test
    public void testPaddleMovement() {
        JoystickState state = new JoystickState();
        run_potentiometer_test_program();
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, -1f, 0f));
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor))
                .check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 10, TimeUnit.SECONDS)))
                .perform(release(state));
        opensettings(R.string.paddle);
        Matcher<View> checkbox = allOf(isDescendantOfA(withChild(withTagValue(equalTo("port#1")))),withId(R.id.cb_lock_diagonals));
        Matcher<View> seekbar = allOf(isDescendantOfA(withChild(withTagValue(equalTo("port#1")))),withId(R.id.sb_sensitivy));

        onView(checkbox).check(matches(not(isDisplayed())));
        //noinspection deprecation
        onView(seekbar).perform(new GeneralClickAction(Tap.SINGLE, GeneralLocation.CENTER, Press.FINGER));
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.gv_monitor))
            .check(matches(showsBitmapEqualToAsset("screenshot-paddle-top-left.pixelbuffer", 10, TimeUnit.SECONDS)));

        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0, 0.5f));
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor))
                .check(matches(showsBitmapEqualToAsset("screenshot-paddle-bottom-left.pixelbuffer", 10, TimeUnit.SECONDS)))
                .perform(release(state));

        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-button-x.pixelbuffer", 20, TimeUnit.SECONDS)));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));

        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L2, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-button-y.pixelbuffer", 20, TimeUnit.SECONDS)));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L2, DEVICE_ID));
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 5, TimeUnit.SECONDS)));

        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, -0.5f, 0f));
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor))
                .check(matches(showsBitmapEqualToAsset("screenshot-paddle-bottom-left.pixelbuffer", 1, TimeUnit.SECONDS)))
                .perform(release(state));
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0.5f, 0f));
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor))
                .check(matches(showsBitmapEqualToAsset("screenshot-paddle-bottom-right.pixelbuffer", 1, TimeUnit.SECONDS)))
                .perform(release(state));
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, -0.5f, -0.5f));
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor))
                .check(matches(showsBitmapEqualToAsset("screenshot-paddle-top-left.pixelbuffer", 1, TimeUnit.SECONDS)))
                .perform(release(state));
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0.5f, 0));
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor))
                .check(matches(showsBitmapEqualToAsset("screenshot-paddle-top-right.pixelbuffer", 1, TimeUnit.SECONDS)))
                .perform(release(state));
    }
    @Test
    public void testPaddleSensitivy() {
        JoystickState state = new JoystickState();
        run_potentiometer_test_program();
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 10, TimeUnit.SECONDS)));
        opensettings(R.string.paddle);
        Matcher<View> spinner = allOf(isDescendantOfA(withChild(withTagValue(equalTo("port#1")))),withId(R.id.sp_typeselector));
        Matcher<View> seekbar = allOf(isDescendantOfA(withChild(withTagValue(equalTo("port#1")))),withId(R.id.sb_sensitivy));

        onView(spinner).perform(click());
        onView(withText(R.string.paddle)).perform(click());
        onData(isOption(R.string.paddle)).inAdapterView(spinner).check(matches(isDisplayed()));
        //noinspection deprecation
        onView(seekbar).perform(new GeneralClickAction(Tap.SINGLE, GeneralLocation.CENTER_RIGHT, Press.FINGER));
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-top-left.pixelbuffer", 10, TimeUnit.SECONDS)));
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0.5f, 0));
        waitForIdle(1, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).perform(release(state));
        waitForIdle(5, TimeUnit.SECONDS);
        //onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-slow-right.pixelbuffer", 2, TimeUnit.SECONDS)));
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, -0.6f, 0));
        waitForIdle(2, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor))
                .check(matches(not(showsBitmapEqualToAsset("screenshot-paddle-bottom-right.pixelbuffer", 10, TimeUnit.SECONDS))))
                .perform(release(state));
        opensettings(R.string.paddle);
        //noinspection deprecation
        onView(seekbar).perform(new GeneralClickAction(Tap.SINGLE, GeneralLocation.CENTER_LEFT, Press.FINGER));
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0.5f, 0));
        waitForIdle(1, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).perform(release(state));
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(not(showsBitmapEqualToAsset("screenshot-paddle-bottom-right.pixelbuffer", 2, TimeUnit.SECONDS))));

    }
    @Test
    public void testPaddleButtons() {
        run_potentiometer_test_program();
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 10, TimeUnit.SECONDS)));
        opensettings(R.string.paddle);
        Matcher<View> spinner = allOf(isDescendantOfA(withChild(withTagValue(equalTo("port#1")))),withId(R.id.sp_typeselector));
        JoystickState state = new JoystickState();
        onView(spinner).perform(click());
        onView(withText(R.string.paddle)).perform(click());
        onData(isOption(R.string.paddle)).inAdapterView(spinner).check(matches(isDisplayed()));
        onView(allOf(isDisplayed(), withId(R.id.cb_swapbuttons))).check(matches(not(isChecked())));
        onView(withId(R.id.mouse_usage_hint)).check(matches(not(isDisplayed())));
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0, 0.5f));
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor))
                .check(matches(showsBitmapEqualToAsset("screenshot-paddle-bottom-left.pixelbuffer", 10, TimeUnit.SECONDS)))
                .perform(release(state));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_THUMBL, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-button-x.pixelbuffer", 25, TimeUnit.SECONDS)));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_THUMBL, DEVICE_ID));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L2, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-button-y.pixelbuffer", 25, TimeUnit.SECONDS)));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L2, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 25, TimeUnit.SECONDS)));
        opensettings(R.string.paddle);
        onView(allOf(isDisplayed(), withId(R.id.cb_swapbuttons))).perform(click()).check(matches(isChecked()));
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0, 0.5f));
        waitForIdle(5, TimeUnit.SECONDS);
        onView(withId(R.id.gv_monitor))
                .check(matches(showsBitmapEqualToAsset("screenshot-paddle-bottom-left.pixelbuffer", 10, TimeUnit.SECONDS)))
                .perform(release(state));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_THUMBL, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-button-y.pixelbuffer", 25, TimeUnit.SECONDS)));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_THUMBL, DEVICE_ID));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L2, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-button-x.pixelbuffer", 25, TimeUnit.SECONDS)));
        //onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L2, DEVICE_ID));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_THUMBL, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-button-xy.pixelbuffer", 25, TimeUnit.SECONDS)));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_THUMBL, DEVICE_ID));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L2, DEVICE_ID));

    }
    @Test
    @FlakyMouseTest
    public void testMouseMovement() throws InterruptedException {
        final int length = 775 * 8;
        /*
        Action: run Test Program
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        run_potentiometer_test_program();
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 15, TimeUnit.SECONDS)));
        /*
        Actions:
        * open settings and connect device as paddles to port 1
        Expected result:
        * show potentiometer values 25/0 and 255 (defaults for just plugged in device)
         */
        opensettings(R.string.mouse);
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        JoystickState state = new JoystickState();
        /*
        Action: Move stick to the right
        Expected result: show potentiometer values 18/6 and 255
         */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0.225f, 0));
        Thread.sleep(length);
        onView(withId(R.id.gv_monitor))
                .perform(release(state));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-+120x.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: Move stick to the right with half speed
        Expected result: show potentiometer values 11/6 and 255
         */

        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0.2125f, 0));
        Thread.sleep(length);
        onView(withId(R.id.gv_monitor))
                .perform(release(state));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-+180x.pixeldata", 10, TimeUnit.SECONDS)));


        /*
        Action: Move stick to the left with full speed, 1 1/2 times of other steps length
        Expected result: show potentiometer values 6/6 and 255 (returned to original position)
         */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, -0.225f, 0));
        Thread.sleep(length * 3 / 2);
        onView(withId(R.id.gv_monitor))
                .perform(release(state));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: Move stick down with full speed
        Expected result: show potentiometer values 6/7 and 255
         */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0, 0.20625f));
        Thread.sleep(length * 4);
        onView(withId(R.id.gv_monitor))
                .perform(release(state));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-+120y.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: Move stick up with full speed
        Expected result: show potentiometer values 6/7 and 255
         */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0, -0.20625f));
        Thread.sleep(length * 4);
        onView(withId(R.id.gv_monitor))
                .perform(release(state));
        try {
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        } catch (AssertionFailedError e) {
            onView(withId(R.id.gv_monitor))
                    .perform(setStickPosition(state, 0, -0.20625f));
            Thread.sleep(length / 6);
            onView(withId(R.id.gv_monitor))
                    .perform(release(state));
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        }
    }
    @Test
    public void testMouseButtons() {
        /*
        Action: run Test Program
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        run_potentiometer_test_program();
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 15, TimeUnit.SECONDS)));
        /*
        Actions:
        * open settings and connect device as paddles to port 1
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        opensettings(R.string.mouse);
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));

        /*
        Action: press primary button
        Expected result: digital value 239 is displayed
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-primary-button.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: press secondary button
        Expected result: digital value 238 is displayed
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L2, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-both-buttons.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: release primary button
        Expected result: digital value 254 is displayed
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-secondary-button.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: release secondary button
        Expected result: digital value 255 is displayed
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L2, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: open settings and click on checkbox "swap buttons"
        Expected result: checkbox is checked
         */
        opensettings(R.string.mouse);
        onView(allOf(isDisplayed(), withId(R.id.cb_swapbuttons))).perform(click()).check(matches(isChecked()));
        /*
        Action: press button appy
        Expected result: Emulation is running, digital value 255 is displayed
         */
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: press primary button
        Expected result: digital value 254 is displayed
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-secondary-button.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: release primary button
        Expected result: digital value 255 is displayed
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
        /*
        Action: press secondary button
        Expected result: digital value 239 is displayed
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L2, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-primary-button.pixelbuffer", 10, TimeUnit.SECONDS)));
        /*
        Action: release secondary button
        Expected result: digital value 255 is displayed
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L2, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 10, TimeUnit.SECONDS)));
    }
    @Test
    @FlakyMouseTest
    public void testMouseSpeed() throws InterruptedException {
        final int length = 775 * 8;
        /*
        Action: run Test Program
        Expected result: showing potentiometer values 25/25 (no paddle connected)
        and digital value 255 (no button pressed)
         */
        run_potentiometer_test_program();
        JoystickState state = new JoystickState();
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-paddle-default.pixelbuffer", 15, TimeUnit.SECONDS)));
        /*
        Actions:
        * open settings and connect device as paddles to port 1, set speed to minimum
        Expected result:
        * show potentiometer values 6/6 and 255 (defaults for just plugged in device)
         */
        opensettings(R.string.mouse);
        onView(allOf(isDisplayed(), withId(R.id.sb_sensitivy)))
                .perform(scrollTo())
                .perform(new GeneralClickAction(Tap.SINGLE, GeneralLocation.CENTER_LEFT, Press.FINGER));
        onView(withId(R.id.bn_apply)).perform(click());
        waitForActivity(EmulationActivity.class);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-default.pixeldata", 15, TimeUnit.SECONDS)));
        /*
        Action: Move stick to the right for three times the original time
        Expected result: show potentiometer values 18/6 and 255
        */
        onView(withId(R.id.gv_monitor))
                .perform(setStickPosition(state, 0.225f, 0));
        Thread.sleep(length * 3 - 500);
        onView(withId(R.id.gv_monitor))
                .perform(release(state));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-mouse-+120x.pixeldata", 10, TimeUnit.SECONDS)));

    }

    @Test
    public void test_secondary_button_alt_functions() {
        JoystickState state = new JoystickState();


        test_specific_button(
                () -> onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID)),
                () -> onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID)),
                () -> onView(withId(R.id.gv_monitor)).perform(setStickPosition(state, 1f, 0)),
                () -> onView(withId(R.id.gv_monitor)).perform(setStickPosition(state, 0, 0)),
                String.format("%s (%s)", FAKE_GAMECONTROLLER_DEVICEID, InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.left_stick))
        );
    }
    private ViewAction slideToFrequency(int frequency) {

        return new ViewAction() {
            @Override
            public Matcher<View> getConstraints() {
                return instanceOf(SeekBar.class);
            }

            @Override
            public String getDescription() {
                return "Move frequency slider to frequency" + frequency;
            }

            @Override
            public void perform(UiController uiController, View view) {
                ((SeekBar) view).setProgress(frequency-1);
            }
        };
    }
    private void setPort2SecondaryFunction(String device, int optionId, int frequency) {
        onView(withId(R.id.ib_hamburger_menu)).perform(tap());
        waitForIdle();
        onView(withText(R.string.IDS_MP_SETTINGS)).perform(click());
        waitForIdle();
        try {
            onView(withTagValue(equalTo("port#2"))).check(matches(isDisplayed()));
        } catch (AssertionFailedError e) {
            onView(withText(R.string.joysticks)).perform(click());
            waitForIdle();
        }
        onView(withTagValue(equalTo("port#1"))).perform(scrollTo()).perform(click());
        onData(isOption(R.string.IDS_NONE)).perform(click());
        onView(withTagValue(equalTo("port#2"))).perform(scrollTo()).perform(click());
        /* Action
        Choose Joystick "Joystick _UNITTEST_GAMECONTROLLER_JOYSTICK_ (Joystick 1)" and load JOYSTICKTEST from testprograms.
        Expected result: 255 is displayed
         */
        onData(isOption(device)).perform(click());
        waitForIdle();
        onView(allOf(isDisplayed(), withId(R.id.sp_secondary_button)))
                .perform(scrollTo())
                .perform(click());
        onView(withText(optionId)).perform(click());
        try {
            onView(withText(R.string.paddle)).check(matches(isDisplayed()));
            pressBack();
        } catch (NoMatchingViewException e) {
            // that's ok
        }
        waitForIdle();

        onView(withId(R.id.gh_c64_settings)).perform(scrollTo());
        onView(allOf(isDisplayed(), withId(R.id.sb_frequency)))
                .perform(scrollTo())
                .perform(slideToFrequency(frequency));
        String expectedText = InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.shots_per_second, frequency);
        onView(allOf(isDisplayed(),withId(R.id.tv_frequency))).check(matches(withText(expectedText)));

        onView(withText(R.string.apply)).perform(click());
        waitForActivity(EmulationActivity.class);
    }
    @Test
    public void test_autofire() throws IOException {
        /*
        Action intitialize Emulation
        Expected result: start screen shown
         */
        String device = String.format("%s (%s)", FAKE_GAMECONTROLLER_DEVICEID, InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.left_stick));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("snapshot-machine-ready.pixelbuffer", 5, TimeUnit.SECONDS)));
        /*
        Action: run test program
        Expected result: black border (i.e. no button pressed)
         */
        openDocument(extractTestAsset("testautofire.prg"));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-released-fire.pixelbuffer", 1, TimeUnit.SECONDS)));
        /*
        Actions:
        - Open settings
        - Autofire shall be turned on while 2nd button is pressed
        - Frequency is 1/sec
        - Close settings
        Expected result: Emulation is shown
         */
        setPort2SecondaryFunction(device, R.string.autofire_while_pressed, 1);
        /*
        Action: press button
        Expected result: Border flashing, ~ twice a second
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        waitForIdle(100, TimeUnit.MILLISECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-pressed-fire.pixelbuffer", 100, TimeUnit.MILLISECONDS)));
        waitForIdle(450, TimeUnit.MILLISECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-released-fire.pixelbuffer", 100, TimeUnit.MILLISECONDS)));
        waitForIdle(450, TimeUnit.MILLISECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-pressed-fire.pixelbuffer", 100, TimeUnit.MILLISECONDS)));
        /*
        Action: release button
        Expected result: border stops flashing
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-released-fire.pixelbuffer", 100, TimeUnit.MILLISECONDS)));
        for (int i = 0; i < 10; i++) {
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-released-fire.pixelbuffer", 100, TimeUnit.MILLISECONDS)));
            waitForIdle(100, TimeUnit.MILLISECONDS);
        }
        /*
        Actions:
        - Open settings
        - Autofire shall be toggled when 2nd button is pressed
        - Frequency 1 / sec
        - Close settings
        Expected result: Emulation is shown
         */
        setPort2SecondaryFunction(device, R.string.autofire_toggle, 1);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-released-fire.pixelbuffer", 100, TimeUnit.MILLISECONDS)));
        /*
        Action: short button press
        Expected result: Border flashing, ~ twice a second
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        waitForIdle();
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-pressed-fire.pixelbuffer", 500, TimeUnit.MILLISECONDS)));
        for (int i = 0; i < 3; i++) {
            waitForIdle(400, TimeUnit.MILLISECONDS);
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-released-fire.pixelbuffer", 200, TimeUnit.MILLISECONDS)));
            waitForIdle(400, TimeUnit.MILLISECONDS);
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-pressed-fire.pixelbuffer", 200, TimeUnit.MILLISECONDS)));
        }
        /*
        Action: short button press
        Expected result: Border stops flashing
         */
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        waitForIdle();
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        for (int i = 0; i < 10; i++) {
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-released-fire.pixelbuffer", 100, TimeUnit.MILLISECONDS)));
            waitForIdle(100, TimeUnit.MILLISECONDS);
        }
        /*
        Actions:
        - Open settings
        - Autofire shall be toggled when 2nd button is pressed
        - Frequency 5 / sec
        - Close settings
        Expected result: Emulation is shown
         */
        setPort2SecondaryFunction(device, R.string.autofire_toggle, 5);
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        waitForIdle();
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
    }
    @Test
    @SpecialTest
    public void test_autofire_speed() throws IOException {
        String device = String.format("%s (%s)", FAKE_GAMECONTROLLER_DEVICEID, InstrumentationRegistry.getInstrumentation().getTargetContext().getString(R.string.left_stick));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("snapshot-machine-ready.pixelbuffer", 5, TimeUnit.SECONDS)));
        /*
        Action: run test program
        Expected result: black border (i.e. no button pressed)
         */
        openDocument(extractTestAsset("testautofire.prg"));
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-released-fire.pixelbuffer", 1, TimeUnit.SECONDS)));
        /*
        Actions:
        - Open settings
        - Autofire shall be turned on while 2nd button is pressed
        - Frequency is 1/sec
        - Close settings
        Expected result: Emulation is shown
         */
        setPort2SecondaryFunction(device, R.string.autofire_toggle, 5);
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        waitForIdle();
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        for (int i = 0; i < 3; i++) {
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-pressed-fire.pixelbuffer", 120, TimeUnit.MILLISECONDS)));
            waitForIdle();
            onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-released-fire.pixelbuffer", 100, TimeUnit.MILLISECONDS)));
            waitForIdle();

        }
        /*
        Actions:
        - Open settings
        - Autofire shall be toggled when 2nd button is pressed
        - Frequency 1 / sec
        - Close settings
        Expected result: Emulation is shown, border still flashing, ~ twice a second
         */
        setPort2SecondaryFunction(device, R.string.autofire_toggle, 1);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-released-fire.pixelbuffer", 600, TimeUnit.MILLISECONDS)));
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));
        waitForIdle();
        onView(withId(R.id.screen)).perform(keyAction(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_BUTTON_L1, DEVICE_ID));

        waitForIdle(50, TimeUnit.MILLISECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-pressed-fire.pixelbuffer", 600, TimeUnit.MILLISECONDS)));
        waitForIdle(200, TimeUnit.MILLISECONDS);
        onView(withId(R.id.gv_monitor)).check(matches(showsBitmapEqualToAsset("screenshot-pressed-fire.pixelbuffer", 100, TimeUnit.MILLISECONDS)));



    }
}