package de.questmaster.wettkampf_funk_trainer.ui;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.DiffUtil;
import androidx.recyclerview.widget.RecyclerView;

import android.content.SharedPreferences;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import java.util.List;

import de.questmaster.wettkampf_funk_trainer.R;
import de.questmaster.wettkampf_funk_trainer.data.FunkSpruch;
import de.questmaster.wettkampf_funk_trainer.databinding.FragmentWtBinding;

/**
 * {@link RecyclerView.Adapter} that can display a {@link FunkSpruch}.
 */
public class MyFunkRecyclerViewAdapter extends RecyclerView.Adapter<MyFunkRecyclerViewAdapter.ViewHolder> {

    // ViewType-Konstanten für unterschiedliche Item-Typen
    private static final int VIEW_TYPE_NORMAL = 0;
    private static final int VIEW_TYPE_PLACEHOLDER = 1;

    public interface OnItemClickListener {
        void onItemClick(int position); // Or pass the FunkSpruch itself, or an ID
    }

    /**
     * DiffUtil.Callback für intelligente RecyclerView-Updates.
     * Vergleicht alte und neue Liste und berechnet minimale Änderungen.
     *
     * Performance-Optimierung: RecyclerView animiert nur tatsächlich geänderte Items.
     */
    private static class FunkspruchDiffCallback extends DiffUtil.Callback {
        private final List<FunkSpruch> oldList;
        private final List<FunkSpruch> newList;

        FunkspruchDiffCallback(List<FunkSpruch> oldList, List<FunkSpruch> newList) {
            this.oldList = oldList;
            this.newList = newList;
        }

        @Override
        public int getOldListSize() {
            return oldList == null ? 0 : oldList.size();
        }

        @Override
        public int getNewListSize() {
            return newList == null ? 0 : newList.size();
        }

        @Override
        public boolean areItemsTheSame(int oldItemPosition, int newItemPosition) {
            // Items sind gleich wenn Sprecher und Position identisch sind
            FunkSpruch oldItem = oldList.get(oldItemPosition);
            FunkSpruch newItem = newList.get(newItemPosition);

            return oldItem.getSprecher() == newItem.getSprecher()
                && oldItem.getSpruch().equals(newItem.getSpruch());
        }

        @Override
        public boolean areContentsTheSame(int oldItemPosition, int newItemPosition) {
            // Content ist gleich wenn der Spruch-Text identisch ist
            FunkSpruch oldItem = oldList.get(oldItemPosition);
            FunkSpruch newItem = newList.get(newItemPosition);

            return oldItem.getSpruch().equals(newItem.getSpruch());
        }
    }

    private final OnItemClickListener mListener;
    private List<FunkSpruch> mLfunkSpruch;
    private final SharedPreferences mSharedPreference;

    public MyFunkRecyclerViewAdapter(OnItemClickListener listener, SharedPreferences sp) {
        this.mSharedPreference = sp;
        this.mListener = listener;

        // Performance-Optimierung: Stable IDs ermöglichen besseres Item-Tracking
        setHasStableIds(true);
    }

    @NonNull
    @Override
    public ViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        return new ViewHolder(FragmentWtBinding.inflate(LayoutInflater.from(parent.getContext()), parent, false));
    }

    @Override
    public void onBindViewHolder(@NonNull final ViewHolder holder, int position) {
        if (mLfunkSpruch == null) {
            return;
        }

        // Performance-Optimierung: Nur 1x List-Access statt 4x
        FunkSpruch item = mLfunkSpruch.get(position);
        String spruch = item.getSpruch(); // String zwischenspeichern
        FunkSpruch.Sprecher itemSprecher = item.getSprecher();

        switch(itemSprecher) {
            case A_TRUPP:
                holder.mEinheitView.setImageResource(R.drawable.ic_list_at);
                break;
            case W_TRUPP:
                holder.mEinheitView.setImageResource(R.drawable.ic_list_wt);
                break;
            case S_TRUPP:
                holder.mEinheitView.setImageResource(R.drawable.ic_list_st);
                break;
            case E_LEITER:
                if (mSharedPreference.getBoolean("einheitGruppe", true)) {
                    holder.mEinheitView.setImageResource(R.drawable.ic_list_gf);
                } else {
                    holder.mEinheitView.setImageResource(R.drawable.ic_list_sf);
                }
                break;
            case MASCHI:
                holder.mEinheitView.setImageResource(R.drawable.ic_list_ma);
                break;
            case ZLST:
                holder.mEinheitView.setImageResource(R.drawable.ic_list_zlst);
                break;
            default:
                holder.mEinheitView.setImageResource(R.drawable.ic_list_leer);
        }
        holder.mIdView.setText(itemSprecher.toNameString());
        holder.mContentView.setText(spruch);
        holder.mContentView.setContentDescription(position + ". " + spruch);
    }

    @Override
    public int getItemCount() {
        return (mLfunkSpruch == null) ? 0 : mLfunkSpruch.size();
    }

    @Override
    public long getItemId(int position) {
        if (mLfunkSpruch == null || position < 0 || position >= mLfunkSpruch.size()) {
            return RecyclerView.NO_ID;
        }

        // Stabile ID basierend auf Position und Spruch-Content
        // Kombiniert Position mit Hash des Inhalts für eindeutige ID
        FunkSpruch item = mLfunkSpruch.get(position);
        return (long) position * 31 + item.getSpruch().hashCode();
    }

    @Override
    public int getItemViewType(int position) {
        if (mLfunkSpruch == null || position < 0 || position >= mLfunkSpruch.size()) {
            return VIEW_TYPE_NORMAL;
        }

        // Placeholder-Items ("???") bekommen eigenen ViewType für bessere Wiederverwendung
        FunkSpruch item = mLfunkSpruch.get(position);
        if (item.getSpruch().equals("???")) {
            return VIEW_TYPE_PLACEHOLDER;
        }

        return VIEW_TYPE_NORMAL;
    }

    public void updateFunkspruchListe(@NonNull List<FunkSpruch> lFunkSprüche, @NonNull RecyclerView rv) {
        List<FunkSpruch> oldList = this.mLfunkSpruch;
        List<FunkSpruch> newList = lFunkSprüche;

        // DiffUtil berechnet die minimalen Änderungen
        DiffUtil.DiffResult diffResult = DiffUtil.calculateDiff(
            new FunkspruchDiffCallback(oldList, newList)
        );

        // Liste aktualisieren
        this.mLfunkSpruch = newList;

        // DiffUtil dispatcht die optimalen notify-Calls
        diffResult.dispatchUpdatesTo(this);

        // Intelligentes Scrolling: Nur beim aktiven Abspielen, nicht bei Tab-Wechseln
        if (oldList != null && !newList.isEmpty()) {
            int oldSize = oldList.size();
            int newSize = newList.size();

            // Fall 1: Liste ist gewachsen (neue Items hinzugefügt)
            if (newSize > oldSize) {
                if (newSize - oldSize < 4) {
                    // Wenige Items hinzugefügt: Scroll zum Ende (typisch beim Abspielen)
                    rv.smoothScrollToPosition(newSize - 1);
                } else if (oldSize == 0) {
                    // Initial-Load: Scroll zum Anfang
                    rv.smoothScrollToPosition(0);
                }
                // Sonst: User-Position respektieren (kein Auto-Scroll bei vielen neuen Items)
            }
            // Fall 2: Gleiche Größe, aber letztes Element geändert ("???" → echter Text)
            // Dies passiert NUR beim Abspielen, nicht bei Tab-Wechseln
            else if (newSize == oldSize && newSize > 0) {
                FunkSpruch oldLastItem = oldList.get(oldSize - 1);
                FunkSpruch newLastItem = newList.get(newSize - 1);

                // Nur scrollen wenn sich das letzte Element tatsächlich geändert hat
                // (verhindert Scroll bei Tab-Wechseln wo die Liste identisch bleibt)
                if (!oldLastItem.getSpruch().equals(newLastItem.getSpruch())) {
                    rv.smoothScrollToPosition(newSize - 1);
                }
            }
        }
    }

    public class ViewHolder extends RecyclerView.ViewHolder {
        public final TextView mIdView;
        public final ImageView mEinheitView;
        public final TextView mContentView;

        public ViewHolder(FragmentWtBinding binding) {
            super(binding.getRoot());

            mIdView = binding.itemNumber;
            mEinheitView = binding.einheitBild;
            mContentView = binding.content;

            // Set the click listener using data binding
            View rootView = binding.getRoot();
            rootView.setOnClickListener(v -> {
                if (mListener != null) {
                    mListener.onItemClick(getBindingAdapterPosition());
                }
            });
        }

        @NonNull
        @Override
        public String toString() {
            return super.toString() + " '" + mContentView.getText() + "'";
        }
    }
}
