package de.questmaster.wettkampf_funk_trainer.ui;

import android.graphics.Canvas;
import android.graphics.Rect;
import android.util.Pair;
import android.view.MotionEvent;
import android.view.View;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.recyclerview.widget.GridLayoutManager;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import me.zhanghai.android.fastscroll.FastScroller;
import me.zhanghai.android.fastscroll.PopupTextProvider;
import me.zhanghai.android.fastscroll.Predicate;

public class FastScrollViewHelper implements FastScroller.ViewHelper {
    @NonNull
    private final RecyclerView mView;
    @Nullable
    private final PopupTextProvider mPopupTextProvider;

    @NonNull
    private final Rect mTempRect = new Rect();

    public FastScrollViewHelper(@NonNull RecyclerView view,
                              @Nullable PopupTextProvider popupTextProvider) {
        mView = view;
        mPopupTextProvider = popupTextProvider;
    }

    @Override
    public void addOnPreDrawListener(@NonNull Runnable onPreDraw) {
        mView.addItemDecoration(new RecyclerView.ItemDecoration() {
            @Override
            public void onDraw(@NonNull Canvas canvas, @NonNull RecyclerView parent,
                               @NonNull RecyclerView.State state) {
                onPreDraw.run();
            }
        });
    }

    @Override
    public void addOnScrollChangedListener(@NonNull Runnable onScrollChanged) {
        mView.addOnScrollListener(new RecyclerView.OnScrollListener() {
            @Override
            public void onScrolled(@NonNull RecyclerView recyclerView, int dx, int dy) {
                onScrollChanged.run();
            }
        });
    }

    @Override
    public void addOnTouchEventListener(@NonNull Predicate<MotionEvent> onTouchEvent) {
        mView.addOnItemTouchListener(new RecyclerView.SimpleOnItemTouchListener() {
            @Override
            public boolean onInterceptTouchEvent(@NonNull RecyclerView recyclerView,
                                                 @NonNull MotionEvent event) {
                return onTouchEvent.test(event);
            }
            @Override
            public void onTouchEvent(@NonNull RecyclerView recyclerView,
                                     @NonNull MotionEvent event) {
                onTouchEvent.test(event);
            }
        });
    }
/*
    @Override
    public int getScrollRange() {
        int itemCount = getItemCount();
        if (itemCount == 0) {
            return 0;
        }

        return mView.getPaddingTop() + getRecyclerViewContentHeight(mView, -1) + mView.getPaddingBottom();
    }

    @Override
    public int getScrollOffset() {
        int firstItemPosition = getFirstItemPosition();
        if (firstItemPosition == RecyclerView.NO_POSITION) {
            return 0;
        }
        int firstItemTop = getFirstItemOffset();
        return mView.getPaddingTop() + getRecyclerViewContentHeight(mView, firstItemPosition) - firstItemTop;
    }
*/
    @Override
    public int getScrollRange() {
        return mView.computeVerticalScrollRange();
    }

    @Override
    public int getScrollOffset() {
        return mView.computeVerticalScrollOffset();
    }

    @Override
    public void scrollTo(int offset) {
        // Stop any scroll in progress for RecyclerView.
        mView.stopScroll();
        offset -= mView.getPaddingTop();
        // firstItemPosition should be non-negative even if paddingTop is greater than item height.
        Pair<Integer, Integer> pair = getRecyclerViewContentByOffset(offset);
        scrollToPositionWithOffset(pair.first, pair.second);
    }

    @Nullable
    @Override
    public CharSequence getPopupText() {
        PopupTextProvider popupTextProvider = mPopupTextProvider;
        if (popupTextProvider == null) {
            RecyclerView.Adapter<?> adapter = mView.getAdapter();
            if (adapter instanceof PopupTextProvider) {
                popupTextProvider = (PopupTextProvider) adapter;
            }
        }
        if (popupTextProvider == null) {
            return null;
        }
        int position = getFirstItemAdapterPosition();
        if (position == RecyclerView.NO_POSITION) {
            return null;
        }
        return popupTextProvider.getPopupText(mView, position);
    }

    private int getItemCount() {
        LinearLayoutManager linearLayoutManager = getVerticalLinearLayoutManager();
        if (linearLayoutManager == null) {
            return 0;
        }
        int itemCount = linearLayoutManager.getItemCount();
        if (itemCount == 0) {
            return 0;
        }
        if (linearLayoutManager instanceof GridLayoutManager) {
            GridLayoutManager gridLayoutManager = (GridLayoutManager) linearLayoutManager;
            itemCount = (itemCount - 1) / gridLayoutManager.getSpanCount() + 1;
        }
        return itemCount;
    }

    private int getFirstItemPosition() {
        int position = getFirstItemAdapterPosition();
        LinearLayoutManager linearLayoutManager = getVerticalLinearLayoutManager();
        if (linearLayoutManager == null) {
            return RecyclerView.NO_POSITION;
        }
        if (linearLayoutManager instanceof GridLayoutManager) {
            GridLayoutManager gridLayoutManager = (GridLayoutManager) linearLayoutManager;
            position /= gridLayoutManager.getSpanCount();
        }
        return position;
    }

    /**
     * Retrieves the adapter position of the first visible item in the RecyclerView.
     *
     * @return The adapter position of the first visible item, or {@link RecyclerView#NO_POSITION}
     *         if no items are visible or the layout manager is not properly set.
     */
    private int getFirstItemAdapterPosition() {
        if (mView.getChildCount() == 0) {
            return RecyclerView.NO_POSITION;
        }
        View itemView = mView.getChildAt(0);
        LinearLayoutManager linearLayoutManager = getVerticalLinearLayoutManager();
        if (linearLayoutManager == null) {
            return RecyclerView.NO_POSITION;
        }
        return linearLayoutManager.getPosition(itemView);
    }

    /**
     * Retrieves the height of the first visible item in the RecyclerView.
     *
     * @return The height of the first visible item, or 0 if no items are visible.
     */
    private int getFirstItemOffset() {
        if (mView.getChildCount() == 0) {
            return RecyclerView.NO_POSITION;
        }
        View itemView = mView.getChildAt(0);
        mView.getDecoratedBoundsWithMargins(itemView, mTempRect);
        return mTempRect.top;
    }

    private void scrollToPositionWithOffset(int position, int offset) {
        LinearLayoutManager linearLayoutManager = getVerticalLinearLayoutManager();
        if (linearLayoutManager == null) {
            return;
        }
        if (linearLayoutManager instanceof GridLayoutManager) {
            GridLayoutManager gridLayoutManager = (GridLayoutManager) linearLayoutManager;
            position *= gridLayoutManager.getSpanCount();
        }
        // LinearLayoutManager actually takes offset from paddingTop instead of top of RecyclerView.
        offset -= mView.getPaddingTop();
        linearLayoutManager.scrollToPositionWithOffset(position, offset);
    }

    @Nullable
    private LinearLayoutManager getVerticalLinearLayoutManager() {
        RecyclerView.LayoutManager layoutManager = mView.getLayoutManager();
        if (!(layoutManager instanceof LinearLayoutManager)) {
            return null;
        }
        LinearLayoutManager linearLayoutManager = (LinearLayoutManager) layoutManager;
        if (linearLayoutManager.getOrientation() != RecyclerView.VERTICAL) {
            return null;
        }
        return linearLayoutManager;
    }

    private int getRecyclerViewContentHeight(RecyclerView recyclerView, int position) {
        RecyclerView.Adapter adapter = recyclerView.getAdapter();
        if (adapter == null) {
            return 0;
        }
        int countPosition = position >= 0 ? position : adapter.getItemCount();;
        int totalHeight = 0;
        for (int i = 0; i < countPosition; i++) {
            totalHeight += getElementHeight(recyclerView, i);
        }
        return totalHeight;
    }
/*
    private Pair<Integer, Integer> getRecyclerViewContentByOffset(int offset) {
        //RecyclerView.Adapter adapter = recyclerView.getAdapter();
        //if (adapter == null) {
        //    return 0;
        //}
        //adapter.getItemCount();
        int totalHeight = 0;
        for (int i = 0; i < mView.getChildCount(); i++) {
            int height = getElementHeight(mView, i);

            // Prüfe, ob offset erreicht ist
            if (totalHeight + height > offset) {
                return new Pair<>(i - 1, totalHeight - offset);
            }

            // Addiere die Höhe des aktuellen Elements
            totalHeight += height;
        }
        return new Pair<>(0, -offset);
    }*/

    private Pair<Integer, Integer> getRecyclerViewContentByOffset(int offset) {
        int currentOffset = 0;
        LinearLayoutManager layoutManager = getVerticalLinearLayoutManager();
        if (layoutManager == null) {
            return new Pair<>(0, -offset); // Or some other default
        }

        for (int i = 0; i < mView.getChildCount(); i++) {
            View child = mView.getChildAt(i);
            if (child == null) continue;

            // Get the actual measured height of the visible child
            int childHeight = layoutManager.getDecoratedMeasuredHeight(child);
            // Or if you need to include margins:
            // RecyclerView.LayoutParams lp = (RecyclerView.LayoutParams) child.getLayoutParams();
            // int childHeightWithMargins = child.getHeight() + lp.topMargin + lp.bottomMargin;


            int itemPosition = layoutManager.getPosition(child);
            // Adjust itemPosition for GridLayoutManager if your FastScroller logic needs "row" position
            if (layoutManager instanceof GridLayoutManager) {
                // This logic might need to align with how getFirstItemPosition handles GridLayoutManager
                itemPosition = itemPosition / ((GridLayoutManager) layoutManager).getSpanCount();
            }


            if (currentOffset + childHeight > offset) {
                // The target offset falls within this item
                // The pair should likely be (itemAdapterPosition, offsetWithinThatItem)
                // The second value in the pair for scrollToPositionWithOffset is the offset from the top edge of the item.
                int offsetWithinItem = offset - currentOffset;
                return new Pair<>(layoutManager.getPosition(child), -offsetWithinItem); // scrollToPositionWithOffset typically expects negative offset from item top
            }
            currentOffset += childHeight;
        }

        // Offset is beyond the currently laid out children
        // This might indicate a need to scroll to the last item or handle it differently
        RecyclerView.Adapter<?> adapter = mView.getAdapter();
        if (adapter != null && adapter.getItemCount() > 0) {
            return new Pair<>(adapter.getItemCount() -1, 0);
        }
        return new Pair<>(0, -offset); // Fallback
    }

    private static int getElementHeight(RecyclerView recyclerView, int i) {
        RecyclerView.Adapter adapter = recyclerView.getAdapter();
        if (adapter == null) {
            return 0;
        }
        // Erzeuge ein exakt bemessenes Width-Spec basierend auf der RecyclerView-Breite
        int widthSpec = View.MeasureSpec.makeMeasureSpec(recyclerView.getWidth(), View.MeasureSpec.EXACTLY);
        // Bei Höhe wird UNSPECIFIED genutzt, damit alle Höhen ermittelt werden können
        int heightSpec = View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED);

        int viewType = adapter.getItemViewType(i);
        RecyclerView.ViewHolder holder = adapter.createViewHolder(recyclerView, viewType);
        adapter.bindViewHolder(holder, i);
        // Messe das View-Element
        holder.itemView.measure(widthSpec, heightSpec);
        return holder.itemView.getMeasuredHeight();
    }
}
