package de.questmaster.wettkampf_funk_trainer.ui;

import android.content.Context;
import androidx.arch.core.executor.testing.InstantTaskExecutorRule;
import androidx.lifecycle.MutableLiveData;
import androidx.test.core.app.ApplicationProvider;
import androidx.test.ext.junit.runners.AndroidJUnit4;
import org.hamcrest.CoreMatchers;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;
import de.questmaster.wettkampf_funk_trainer.data.FunkSpruch;
import de.questmaster.wettkampf_funk_trainer.data.FunkspruchPlayer;
import de.questmaster.wettkampf_funk_trainer.data.IFunkSprüche;
import static androidx.test.platform.app.InstrumentationRegistry.getInstrumentation;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.when;

@RunWith(AndroidJUnit4.class)
public class EinheitViewModelInstrumentedTest {

    @Rule
    public InstantTaskExecutorRule instantTaskExecutorRule = new InstantTaskExecutorRule();

    private Context context;
    private EinheitViewModel viewModel;
    private FunkspruchPlayer mockedPlayer;

    @Before
    public void setUp() throws Exception {
        context = ApplicationProvider.getApplicationContext();
        viewModel = new EinheitViewModel();
        viewModel.init(FunkSpruch.Sprecher.A_TRUPP, context);

        // Erstelle einen Mockito-Mock für FunkspruchPlayer
        mockedPlayer = Mockito.mock(FunkspruchPlayer.class);
        when(mockedPlayer.isStarted()).thenReturn(true);
        when(mockedPlayer.getSprecher()).thenReturn(FunkSpruch.Sprecher.A_TRUPP);

        // Dummy-Liste für getAktuelleFunkSprüche()
        List<FunkSpruch> dummyList = Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Initial Message")
        );
        when(mockedPlayer.getAktuelleFunkSprüche()).thenReturn(dummyList);

        // Stub für next() und hasNext() in speakNext()
        when(mockedPlayer.hasNext()).thenReturn(true);
        when(mockedPlayer.next()).thenReturn(new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Next Test"));

        // Stub für get(0) hinzufügen, um NullPointerException zu vermeiden
        when(mockedPlayer.get(0)).thenReturn(new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Test Message"));

        // Injiziere den Mock in das private LiveData-Feld funkspruchPlayer via Reflection
        Field field = EinheitViewModel.class.getDeclaredField("funkspruchPlayer");
        field.setAccessible(true);
        @SuppressWarnings("unchecked")
        MutableLiveData<FunkspruchPlayer> liveData = (MutableLiveData<FunkspruchPlayer>) field.get(viewModel);
        liveData.setValue(mockedPlayer);
    }

    private <T> T getLiveDataValue(final androidx.lifecycle.LiveData<T> liveData) throws InterruptedException {
        final Object[] data = new Object[1];
        getInstrumentation().waitForIdleSync();
        liveData.observeForever(value -> data[0] = value);
        TimeUnit.SECONDS.sleep(1);
        @SuppressWarnings("unchecked")
        T result = (T) data[0];
        return result;
    }

    @Test
    public void testOnFabClicked_StartTraining() throws InterruptedException {
        // Simuliere, dass das Training nicht gestartet ist
        when(mockedPlayer.isStarted()).thenReturn(false);
        getInstrumentation().runOnMainSync(() -> viewModel.onFabClicked());
        Mockito.verify(mockedPlayer).start();
        boolean isStarted = getLiveDataValue(viewModel.getIsTrainingStarted());
        assertThat(isStarted, CoreMatchers.is(true));
        List<FunkSpruch> spruchList = getLiveDataValue(viewModel.getFunkSpruecheList());
        assertEquals(1, spruchList.size());
    }

    @Test
    public void testOnFabClicked_StopTraining() throws InterruptedException {
        when(mockedPlayer.isStarted()).thenReturn(true);
        getInstrumentation().runOnMainSync(() -> viewModel.onFabClicked());
        Mockito.verify(mockedPlayer).stop();
        boolean isStarted = getLiveDataValue(viewModel.getIsTrainingStarted());
        assertThat(isStarted, CoreMatchers.is(false));
    }

    @Test
    public void testSpeakNext_NotFinished() throws InterruptedException {
        // Starte das Training zuerst, falls noch nicht gestartet.
        when(mockedPlayer.isStarted()).thenReturn(false);
        getInstrumentation().runOnMainSync(() -> viewModel.onFabClicked());
        boolean isStartedInitial = getLiveDataValue(viewModel.getIsTrainingStarted());
        assertThat(isStartedInitial, CoreMatchers.is(true));

        // Simuliere, dass der Player nach next() nicht finished ist
        when(mockedPlayer.isFinished()).thenReturn(false);

        // Simuliere eine aktualisierte Liste nach next() mit 2 Einträgen
        List<FunkSpruch> dummyListAfterNext = Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Initial Message"),
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Next Test")
        );
        when(mockedPlayer.getAktuelleFunkSprüche()).thenReturn(dummyListAfterNext);

        viewModel.speakNext();

        // Überprüfe, dass die Liste der Funksprüche aktualisiert wurde
        List<FunkSpruch> spruchList = getLiveDataValue(viewModel.getFunkSpruecheList());
        assertEquals(2, spruchList.size());

        // Da der Player nicht als finished gilt, bleibt isTrainingStarted true
        boolean isStarted = getLiveDataValue(viewModel.getIsTrainingStarted());
        assertThat(isStarted, CoreMatchers.is(true));
    }

    @Test
    public void testSpeakNext_Finished() throws InterruptedException {
        // Simuliere, dass der Player nach next() finished ist
        when(mockedPlayer.isFinished()).thenReturn(true);
        viewModel.speakNext();
        boolean isStarted = getLiveDataValue(viewModel.getIsTrainingStarted());
        assertThat(isStarted, CoreMatchers.is(false));
    }

    @Test
    public void testSpeakNext_NoNext_NotFinished() throws InterruptedException {
        // Starte das Training, falls noch nicht gestartet.
        when(mockedPlayer.isStarted()).thenReturn(false);
        getInstrumentation().runOnMainSync(() -> viewModel.onFabClicked());
        boolean isStartedInitial = getLiveDataValue(viewModel.getIsTrainingStarted());
        assertThat(isStartedInitial, CoreMatchers.is(true));

        // Simuliere, dass der Player keinen weiteren Funkspruch hat und auch nicht finished ist.
        when(mockedPlayer.hasNext()).thenReturn(false);
        when(mockedPlayer.isFinished()).thenReturn(false);

        // Sicherstellen, dass die Liste unverändert bleibt.
        List<FunkSpruch> initialList = Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Initial Message")
        );
        when(mockedPlayer.getAktuelleFunkSprüche()).thenReturn(initialList);

        viewModel.speakNext();

        List<FunkSpruch> spruchList = getLiveDataValue(viewModel.getFunkSpruecheList());
        assertEquals(1, spruchList.size());
        boolean isStartedAfter = getLiveDataValue(viewModel.getIsTrainingStarted());
        assertThat(isStartedAfter, CoreMatchers.is(true));
    }

}