// Datei: app/src/androidTest/java/de/questmaster/wettkampf_funk_trainer/ui/TestEinheitFragmentInstrumentedTest.java
package de.questmaster.wettkampf_funk_trainer.ui;

import android.content.Context;
import android.view.View;

import androidx.fragment.app.FragmentFactory;
import androidx.fragment.app.testing.FragmentScenario;
import androidx.lifecycle.Lifecycle;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.test.core.app.ApplicationProvider;
import androidx.test.espresso.NoMatchingViewException;
import androidx.test.espresso.ViewAssertion;
import androidx.test.espresso.assertion.ViewAssertions;
import androidx.test.espresso.matcher.ViewMatchers;
import androidx.test.espresso.Espresso;
import androidx.test.espresso.action.ViewActions;
import androidx.test.ext.junit.runners.AndroidJUnit4;
import androidx.test.filters.LargeTest;
import org.hamcrest.Matcher;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import de.questmaster.wettkampf_funk_trainer.R;
import de.questmaster.wettkampf_funk_trainer.data.FunkSpruch;

import static androidx.test.espresso.Espresso.onView;
import static androidx.test.espresso.Espresso.onData;
import static androidx.test.espresso.Espresso.onView;
import static androidx.test.espresso.matcher.ViewMatchers.withId;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.Matchers.anything;
import static org.hamcrest.Matchers.notNullValue;

@RunWith(AndroidJUnit4.class)
@LargeTest
public class EinheitFragmentInstrumentedTest {

    private Context appContext;

    // Test‑Fragment, das EinheitFragment erweitert
    public static class TestEinheitFragment extends EinheitFragment {
        @Override
        protected FunkSpruch.Sprecher getLernSprecher() {
            return FunkSpruch.Sprecher.A_TRUPP;
        }
    }

    @Before
    public void setUp() {
        appContext = ApplicationProvider.getApplicationContext();
    }

    // Test: Fragmentstart, RecyclerView sollte sichtbar sein und initial Elemente besitzen.
    @Test
    public void testFragmentLaunchAndRecyclerViewPopulated() {
        FragmentScenario<TestEinheitFragment> scenario =
                FragmentScenario.launchInContainer(TestEinheitFragment.class, null, R.style.Theme_Wettkampf_Funk_Trainer, (FragmentFactory) null);
        // Prüfe, ob RecyclerView im Layout existiert und sichtbar ist.
        onView(withId(R.id.list))
                .check(ViewAssertions.matches(ViewMatchers.isDisplayed()));
        // Da das ViewModel in init() den initialen Listeneintrag lädt, wird mindestens ein Element erwartet.
        // Hier kann z. B. mit einem Custom RecyclerViewItemCountAssertion die Anzahl gebaut werden:
        onView(withId(R.id.list))
                .check(new RecyclerViewItemCountAssertion(not(0)));
    }

    // Test: Klick auf FAB startet Training (und ändert den Icon-Zustand) und lädt neue Listeneinträge.
    @Test
    public void testFabClickStartStopTraining() throws InterruptedException {
        FragmentScenario<TestEinheitFragment> scenario =
                FragmentScenario.launchInContainer(TestEinheitFragment.class, null, R.style.Theme_Wettkampf_Funk_Trainer, (FragmentFactory) null);
        // Klick auf FAB, um Training zu starten
        onView(withId(R.id.fab)).perform(ViewActions.click());
        // Nach dem Start sollte isTrainingStarted true sein – der FAB Icon-Wechsel (hier implizit über den Ressourcenzugriff) kann geprüft werden.
        onView(withId(R.id.fab))
                .check(ViewAssertions.matches(ViewMatchers.withTagValue(org.hamcrest.Matchers.is((Object) "started"))));
        // Simulation: Ein erneuter Klick stoppt das Training
        onView(withId(R.id.fab)).perform(ViewActions.click());
        onView(withId(R.id.fab))
                .check(ViewAssertions.matches(ViewMatchers.withTagValue(org.hamcrest.Matchers.is((Object) "stopped"))));
    }

    @Test
    public void testRecyclerViewInteractions() {
        FragmentScenario<TestEinheitFragment> scenario =
                FragmentScenario.launchInContainer(TestEinheitFragment.class, null, R.style.Theme_Wettkampf_Funk_Trainer, (FragmentFactory) null);

        // Verify RecyclerView is visible and populated
        onView(withId(R.id.list))
                .check(ViewAssertions.matches(ViewMatchers.isDisplayed()))
                .check(new RecyclerViewItemCountAssertion(not(0)));

        // Simulate a click on the first item in the RecyclerView
        onView(withId(R.id.list))
                .perform(androidx.test.espresso.contrib.RecyclerViewActions.actionOnItemAtPosition(0, ViewActions.click()));

        // Verify the RecyclerView updates after the click
        onView(withId(R.id.list))
                .check(new RecyclerViewItemCountAssertion(not(0)));
    }

    @Ignore
    @Test
    public void testScrollPositionRestoration() {
        // Starte das Test-Fragment
        FragmentScenario<TestEinheitFragment> scenario =
                FragmentScenario.launchInContainer(TestEinheitFragment.class, null, R.style.Theme_Wettkampf_Funk_Trainer, (FragmentFactory) null);

        // Warte, bis das Fragment fertig aufgebaut ist
        scenario.onFragment(fragment -> {
            RecyclerView recyclerView = fragment.getView().findViewById(R.id.list);
            LinearLayoutManager layoutManager = (LinearLayoutManager) recyclerView.getLayoutManager();
            // Simuliere, dass die Liste weit gescrollt wurde, z. B. zu Position 3
            recyclerView.scrollToPosition(3);

            // Ermitteln der ersten sichtbaren Position vor Rotation
            int firstVisibleBefore = layoutManager.findFirstVisibleItemPosition();
            org.junit.Assert.assertEquals(3, firstVisibleBefore);
        });

        // Simuliere einen Konfigurationswechsel, z. B. Rotation
        scenario.recreate();

        // Nach Neuanlage: Überprüfe, ob die Scrollposition wiederhergestellt wurde
        scenario.onFragment(fragment -> {
            RecyclerView recyclerView = fragment.getView().findViewById(R.id.list);
            LinearLayoutManager layoutManager = (LinearLayoutManager) recyclerView.getLayoutManager();
            onView(withId(R.id.list)).check(new RecyclerViewFirstVisiblePositionAssertion(3));
        });
    }

    // Hilfsklasse zur Überprüfung der ersten sichtbaren Position in einer RecyclerView
    public static class RecyclerViewFirstVisiblePositionAssertion implements ViewAssertion {
        private final int expectedPosition;

        public RecyclerViewFirstVisiblePositionAssertion(int expectedPosition) {
            this.expectedPosition = expectedPosition;
        }

        @Override
        public void check(View view, NoMatchingViewException noViewFoundException) {
            RecyclerView recyclerView = (RecyclerView) view;
            LinearLayoutManager layoutManager = (LinearLayoutManager) recyclerView.getLayoutManager();
            int firstVisible = layoutManager.findFirstVisibleItemPosition();
            org.junit.Assert.assertEquals("Erste sichtbare Position stimmt nicht überein", expectedPosition, firstVisible);
        }
    }

    // Hilfsklasse zur Überprüfung der Item-Anzahl in einer RecyclerView.
    public static class RecyclerViewItemCountAssertion implements ViewAssertion {
        private final Matcher<Integer> matcher;

        public RecyclerViewItemCountAssertion(Matcher<Integer> matcher) {
            this.matcher = matcher;
        }

        @Override
        public void check(android.view.View view, androidx.test.espresso.NoMatchingViewException noViewFoundException) {
            androidx.recyclerview.widget.RecyclerView recyclerView = (androidx.recyclerview.widget.RecyclerView) view;
            androidx.recyclerview.widget.RecyclerView.Adapter adapter = recyclerView.getAdapter();
            org.junit.Assert.assertThat(adapter.getItemCount(), matcher);
        }
    }
}
