package de.questmaster.wettkampf_funk_trainer.data;

import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mockito;
import org.mockito.junit.MockitoJUnitRunner;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.NoSuchElementException;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThrows;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.when;

import android.content.SharedPreferences;

@RunWith(MockitoJUnitRunner.class)
public class FunkspruchPlayerUnitTest {

    private IFunkSprüche mockSprüche;
    private List<FunkSpruch> funkSpruchList;
    private FunkSpruch.Sprecher sprecher;
    private FunkspruchPlayer player;

    @Before
    public void setUp() {
        // Initialisierung des Mocks und der Liste
        mockSprüche = Mockito.mock(IFunkSprüche.class);
        funkSpruchList = new ArrayList<>();
        sprecher = FunkSpruch.Sprecher.KONTEXT;
        when(mockSprüche.funksprueche()).thenReturn(funkSpruchList);
        when(mockSprüche.getSprecher()).thenReturn(sprecher);
    }

    private void createPlayer(FunkSpruch.Sprecher sprecher) {
        this.sprecher = sprecher;
        when(mockSprüche.getSprecher()).thenReturn(this.sprecher);
        player = new FunkspruchPlayer(mockSprüche);
    }

    @Test
    public void testConstructor() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2"),
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 3")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);

        // Act & Assert
        assertEquals(funkSpruchList.size() - 1, player.verfügbarePosition);
    }

    @Test
    public void testStart() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2"),
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 3")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.verfügbarePosition = 2;

        // Act
        player.start();

        // Assert
        Assert.assertEquals(0, player.verfügbarePosition);
    }

    @Test
    public void testStop() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2"),
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 3")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.verfügbarePosition = 0;

        // Act
        player.stop();

        // Assert
        Assert.assertEquals(funkSpruchList.size() - 1, player.verfügbarePosition);
    }

    @Test
    public void testHasNextEmptyList() {
        // Arrange
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);

        // Act
        boolean hasNext = player.hasNext();

        // Assert
        assertFalse(hasNext);
    }

    @Test
    public void testHasNextAtBeginning() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();

        // Act
        boolean hasNext = player.hasNext();

        // Assert
        Assert.assertTrue(hasNext);
    }

    @Test
    public void testHasNextInMiddle() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2"),
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 3")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.verfügbarePosition = 1;

        // Act
        boolean hasNext = player.hasNext();

        // Assert
        Assert.assertTrue(hasNext);
    }

    @Test
    public void testHasNextAtEnd() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.verfügbarePosition = 1;

        // Act
        boolean hasNext = player.hasNext();

        // Assert
        assertFalse(hasNext);
    }

    @Test
    public void testNextEmptyList() {
        // Arrange
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);

        // Act & Assert
        assertThrows(NoSuchElementException.class, () -> player.next());
    }

    @Test
    public void testNextSingleMessage() {
        // Arrange
        funkSpruchList.add(new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 1"));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();

        // Act & Assert
        assertThrows(NoSuchElementException.class, () -> player.next());
    }

    @Test
    public void testNextMultipleMessagesDifferentSpeakers() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.W_TRUPP, "Message 2"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 3")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();

        // Act
        FunkSpruch next = player.next();

        // Assert
        assertNotNull(next);
        assertEquals(FunkSpruch.Sprecher.KONTEXT, next.getSprecher());
        assertEquals("Message 2\n\nMessage 3\n\n", next.getSpruch());
    }

    @Test
    public void testNextMultipleMessagesSameSpeaker() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2"),
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 3")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();

        // Act
        FunkSpruch next = player.next();

        // Assert
        assertNotNull(next);
        assertEquals(FunkSpruch.Sprecher.KONTEXT, next.getSprecher());
        assertEquals("Message 2\n\n", next.getSpruch());
    }

    @Test
    public void testNextReachingEnd() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.W_TRUPP, "Message 2")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();

        // Act
        player.next();

        // Assert
        assertThrows(NoSuchElementException.class, () -> player.next());
    }

    @Test
    public void testGetValidPosition() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2"),
                new FunkSpruch(FunkSpruch.Sprecher.W_TRUPP, "Message 3")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();

        // Act
        FunkSpruch spruch = player.get(1);

        // Assert
        assertNotNull(spruch);
        assertEquals("Message 2", spruch.getSpruch());
        assertEquals(FunkSpruch.Sprecher.E_LEITER, spruch.getSprecher());
    }

    @Test
    public void testGetInvalidPositionNegative() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);

        // Act & Assert
        assertThrows(IndexOutOfBoundsException.class, () -> player.get(-1));
    }

    @Test
    public void testGetInvalidPositionOutOfBounds() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);

        // Act & Assert
        assertThrows(IndexOutOfBoundsException.class, () -> player.get(2));
    }

    @Test
    public void testGetNextPositionWrongSpeaker() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2"),
                new FunkSpruch(FunkSpruch.Sprecher.W_TRUPP, "Message 3")
        ));
        createPlayer(FunkSpruch.Sprecher.E_LEITER);
        player.verfügbarePosition = 0;

        // Act
        FunkSpruch spruch = player.get(1);

        // Assert
        assertNotNull(spruch);
        assertEquals("???", spruch.getSpruch());
        assertEquals(FunkSpruch.Sprecher.E_LEITER, spruch.getSprecher());
    }

    @Test
    public void testSizeEmptyList() {
        // Arrange
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);

        // Act
        int size = player.size();

        // Assert
        assertEquals(0, size);
    }

    @Test
    public void testSizeSingleElement() {
        // Arrange
        funkSpruchList.add(new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);

        // Act
        int size = player.size();

        // Assert
        assertEquals(1, size);
    }

    @Test
    public void testSizeMultipleElementsWithAdditionalElement() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 2"),
                new FunkSpruch(FunkSpruch.Sprecher.W_TRUPP, "Message 3")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();

        // Act
        int size = player.size();

        // Assert
        // Erwartet wird eine Größe von 2, da der dritte Eintrag nicht zum aktiven Sprecher passt
        assertEquals(2, size);
    }

    @Test
    public void testUpdateSprücheResetsStateAndCounter() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();

        SharedPreferences mockSharedPreferences = Mockito.mock(SharedPreferences.class);
        // Mock isChangedInLastUpdate() to return true to trigger reset behavior
        when(mockSprüche.isChangedInLastUpdate()).thenReturn(true);

        // Act
        player.updateSprüche(mockSharedPreferences);

        // Assert
        // Verify that both the counter is reset and the state is changed
        assertEquals(funkSpruchList.size() - 1, player.verfügbarePosition);
        assertFalse(player.isStarted());
        
        // Verify updateItems was called with the shared preferences
        Mockito.verify(mockSprüche).updateItems(mockSharedPreferences);
    }

    @Test
    public void testGetAktuelleFunkSprüche() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();

        // Act
        List<FunkSpruch> aktuelleListe = player.getAktuelleFunkSprüche();

        // Assert
        // Da aktuellePosition 0 ist, erwarte ich genau 1 Element oder 2 falls der zusätzliche Platzhalter hinzugerechnet wird
        if (aktuelleListe.size() > 1) {
            assertEquals("???", aktuelleListe.get(aktuelleListe.size() - 1).getSpruch());
        } else {
            assertEquals(1, aktuelleListe.size());
        }
    }

    @Test
    public void testNextAggregatesMessagesFromDifferentSpeakers() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.W_TRUPP, "Message 2"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 3")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();

        // Act
        FunkSpruch aggregated = player.next();

        // Assert
        assertNotNull(aggregated);
        assertEquals(FunkSpruch.Sprecher.KONTEXT, aggregated.getSprecher());
        // The first message (E_LEITER) should be included in the aggregated message
        assertEquals("Message 2\n\nMessage 3\n\n", aggregated.getSpruch());
    }

    @Test
    public void testGetAktuelleFunkSprücheIncludesPlaceholder() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();
        
        // Set up the conditions for size() to return a value greater than verfügbarePosition + 1
        // This is needed because getAktuelleFunkSprüche checks if (verfügbarePosition + 1) < size()
        // The key condition is that the verfügbarePosition (0) element should have a different Sprecher
        // than the player's Sprecher to make size() return a value larger than just position+1
        
        // Override the funkSpruchList with messages that will trigger the placeholder condition
        funkSpruchList.clear();
        funkSpruchList.add(new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 1"));
        funkSpruchList.add(new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 2"));
        
        // Act
        List<FunkSpruch> aktuelleListe = player.getAktuelleFunkSprüche();

        // Assert
        assertEquals(2, aktuelleListe.size());
        assertEquals("???", aktuelleListe.get(1).getSpruch());
    }

    @Test
    public void testGetSprecher() {
        // Arrange
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);

        // Act
        FunkSpruch.Sprecher result = player.getSprecher();

        // Assert
        assertEquals(FunkSpruch.Sprecher.A_TRUPP, result);
    }

    @Test
    public void testIsStarted_initiallyFalse() {
        // Arrange
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);

        // Act & Assert
        assertFalse(player.isStarted());
    }

    @Test
    public void testIsStarted_afterStart() {
        // Arrange
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();

        // Act & Assert
        assertTrue(player.isStarted());
    }

    @Test
    public void testIsStarted_afterStop() {
        // Arrange
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();
        player.stop();

        // Act & Assert
        assertFalse(player.isStarted());
    }

    @Test
    public void testIsFinished_initiallyTrue() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        
        // The player is initially not started, and has no messages available (hasNext() would be false)
        // According to isFinished() = !hasNext() && !mStarted
        // Since hasNext() is false, isFinished() should be true

        // Act & Assert
        assertFalse(player.hasNext()); // Verify hasNext() returns true initially
        assertFalse(player.isStarted()); // Player is not started
        assertTrue(player.isFinished()); // Should be true because hasNext() is false
    }

    @Test
    public void testIsFinished_whenNoMoreMessages() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.W_TRUPP, "Message 2")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        player.start();
        player.next(); // Navigate through all messages
        
        // Verify hasNext() returns false after navigating
        assertFalse(player.hasNext());
        
        // Act & Assert
        assertFalse(player.isStarted());
        assertTrue(player.isFinished());
    }

    @Test
    public void testUpdateSprücheWithNewContent() {
        // Arrange
        funkSpruchList.addAll(Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "Message 2")
        ));
        createPlayer(FunkSpruch.Sprecher.A_TRUPP);
        
        // Setup mock for content change notification
        when(mockSprüche.isChangedInLastUpdate()).thenReturn(true);
        
        // Create new content to be returned after update
        List<FunkSpruch> updatedList = Arrays.asList(
                new FunkSpruch(FunkSpruch.Sprecher.A_TRUPP, "New Message 1"),
                new FunkSpruch(FunkSpruch.Sprecher.E_LEITER, "New Message 2"),
                new FunkSpruch(FunkSpruch.Sprecher.W_TRUPP, "New Message 3")
        );
        
        // Setup mock to return the new content
        SharedPreferences mockSharedPreferences = Mockito.mock(SharedPreferences.class);
        
        // Prepare the mock to return new data after update
        when(mockSprüche.funksprueche()).thenReturn(updatedList);
        
        // Act
        player.updateSprüche(mockSharedPreferences);
        
        // Assert
        Mockito.verify(mockSprüche).updateItems(mockSharedPreferences);
        assertEquals(updatedList.size() - 1, player.verfügbarePosition);
        assertFalse(player.isStarted());
        
        // Verify we can access the updated content
        FunkSpruch updatedMessage = player.get(0);
        assertEquals("New Message 1", updatedMessage.getSpruch());
    }
}