package de.questmaster.wettkampf_funk_trainer.ui;

import android.content.Context;
import android.content.SharedPreferences;
import android.speech.tts.TextToSpeech;

import androidx.annotation.NonNull;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;
import androidx.lifecycle.ViewModel;
import androidx.preference.PreferenceManager;

import java.util.List;
import java.util.Locale;

import de.questmaster.wettkampf_funk_trainer.data.AtFunkSprüche;
import de.questmaster.wettkampf_funk_trainer.data.ElFunkSprüche;
import de.questmaster.wettkampf_funk_trainer.data.FunkSpruch;
import de.questmaster.wettkampf_funk_trainer.data.FunkSprüche;
import de.questmaster.wettkampf_funk_trainer.data.FunkspruchPlayer;
import de.questmaster.wettkampf_funk_trainer.data.MaFunkSprüche;
import de.questmaster.wettkampf_funk_trainer.data.StFunkSprüche;
import de.questmaster.wettkampf_funk_trainer.data.WtFunkSprüche;
import de.questmaster.wettkampf_funk_trainer.utilities.PrefsHelper;
import timber.log.Timber;

/**
 * ViewModel für Einheit-Fragments (AT, WT, EL, MA, ST).
 * Verwaltet den FunkspruchPlayer und TextToSpeech-Engine.
 *
 * <p>Performance-Optimierungen:</p>
 * <ul>
 *   <li>Guard-Clause in init() verhindert mehrfache TTS-Initialisierung bei Rotation</li>
 *   <li>handleOnResume() updatet nur bei tatsächlichen Content-Änderungen</li>
 *   <li>Application Context für PrefsHelper verhindert Memory-Leaks</li>
 * </ul>
 *
 * <p>Lifecycle:</p>
 * <ul>
 *   <li>Fragment-scoped: Jedes Fragment hat eigene ViewModel-Instanz</li>
 *   <li>Überlebt Config-Changes (Rotation)</li>
 *   <li>Cleanup in onCleared() - TTS wird ordnungsgemäß released</li>
 * </ul>
 *
 * @see EinheitFragment
 * @see FunkspruchPlayer
 */
public class EinheitViewModel extends ViewModel implements TextToSpeech.OnInitListener {

    private final MutableLiveData<List<FunkSpruch>> funkSpruecheList = new MutableLiveData<>();
    private final MutableLiveData<Boolean> isTrainingStarted = new MutableLiveData<>(false);
    private final MutableLiveData<FunkSpruch.Sprecher> currentSprecher = new MutableLiveData<>();
    private final MutableLiveData<FunkspruchPlayer> funkspruchPlayer = new MutableLiveData<>();
    private TextToSpeech textToSpeech;
    private boolean ttsIsInitialized = false;
    private SharedPreferences mSharedPreference;
    private PrefsHelper mPrefsHelper;

    // Lifecycle-Management: Verhindert mehrfache Initialisierung
    private boolean isInitialized = false;

    public EinheitViewModel() {
        Timber.d("EinheitViewModel created - instance: %s", this.hashCode());
    }

    /**
     * Initialisiert das ViewModel mit Sprecher und Context.
     *
     * <p>Performance-Optimierung: Guard-Clause verhindert mehrfache Initialisierung
     * bei Fragment-Recreation (z.B. Screen-Rotation).</p>
     *
     * @param sprecher Der Sprecher für den dieses ViewModel zuständig ist
     * @param context Context für Ressourcen-Zugriff (wird zu Application Context konvertiert)
     */
    // Initialize the ViewModel (called from the Fragment)
    public void init(FunkSpruch.Sprecher sprecher, Context context) {
        Timber.d("EinheitViewModel init() called - instance: %s", this.hashCode());

        // Performance-Optimierung: Guard-Clause verhindert mehrfache Initialisierung
        // Bei Fragment-Recreation (Rotation) wird init() nicht erneut durchlaufen
        if (isInitialized && currentSprecher.getValue() == sprecher) {
            Timber.d("ViewModel already initialized for %s, skipping", sprecher.toNameString());
            return;
        }

        if (mSharedPreference == null) {
            mSharedPreference = PreferenceManager.getDefaultSharedPreferences(context);
        }

        // init TTS: Nur beim ersten Mal oder bei Sprecher-Wechsel
        if (textToSpeech == null) {
            textToSpeech = new TextToSpeech(context.getApplicationContext(), this); // Use application context
            Timber.d("TextToSpeech initialized");
        }

        // PrefsHelper mit Application Context erstellen (verhindert Context-Leaks)
        if (mPrefsHelper == null) {
            mPrefsHelper = new PrefsHelper(context.getApplicationContext());
            Timber.d("PrefsHelper initialized with Application Context");
        }

        isInitialized = true;
        Timber.d("ViewModel initialization completed for %s", sprecher.toNameString());
    }

    public LiveData<Boolean> getIsTrainingStarted() {
        return isTrainingStarted;
    }
    // Add a get method to expose the current Sprecher
    public LiveData<FunkSpruch.Sprecher> getCurrentSprecher() {
        return currentSprecher;
    }
    // Expose the list of Funksprüche via LiveData
    public LiveData<List<FunkSpruch>> getFunkSpruecheList() {
        return funkSpruecheList;
    }

    public void onFabClicked() {
        FunkspruchPlayer player = funkspruchPlayer.getValue();
        if (player != null) {
            if (!player.isStarted()) {
                player.start();
                speak(player.get(0).getSpruch());
                isTrainingStarted.setValue(true);
                countTrainingsStart(player.getSprecher());
            } else {
                player.stop(); // Or speakNext if you are not planning to stop the player
                isTrainingStarted.setValue(false);
            }
            // Update List after state change
            funkSpruecheList.setValue(player.getAktuelleFunkSprüche());
        }
    }

    /**
     * Wird bei onResume() des Fragments aufgerufen.
     * Aktualisiert die Funksprüche aus Präferenzen.
     *
     * <p>Performance-Optimierung: Erstellt Player nur beim ersten Mal,
     * danach werden nur Präferenzen aktualisiert. LiveData wird nur bei
     * tatsächlichen Inhalts-Änderungen aktualisiert.</p>
     *
     * @param sprecher Der aktuelle Sprecher (sollte mit init() übereinstimmen)
     */
    // Updates the FunkspruchPlayer with the provided IFunkSprüche
    public void handleOnResume(@NonNull FunkSpruch.Sprecher sprecher) {
        FunkspruchPlayer player = funkspruchPlayer.getValue();

        if (player == null) {
            // Erstes Mal: Player erstellen
            Timber.d("handleOnResume: Creating new FunkspruchPlayer for %s", sprecher.toNameString());
            setupFunkspruchPlayer(sprecher);
        } else {
            // Bereits vorhanden: Nur Präferenzen updaten
            Timber.d("handleOnResume: Updating existing FunkspruchPlayer");
            player.updateSprüche(mSharedPreference);

            // Nur wenn sich Inhalt geändert hat, Liste neu setzen
            if (player.isChangedInLastUpdate()) {
                Timber.d("handleOnResume: Content changed, updating LiveData");
                funkSpruecheList.setValue(player.getAktuelleFunkSprüche());
                isTrainingStarted.setValue(player.isStarted());
            } else {
                Timber.d("handleOnResume: No content change, keeping current state");
            }
        }
    }

    // Creates the FunkspruchPlayer instance
    private void setupFunkspruchPlayer(FunkSpruch.Sprecher sprecher) {
        FunkSprüche funkSprüche;
        FunkspruchPlayer player;
        if (funkspruchPlayer.getValue() == null){
            switch (sprecher) {
                case E_LEITER:
                    funkSprüche = new ElFunkSprüche(mSharedPreference);
                    break;
                case MASCHI:
                    funkSprüche = new MaFunkSprüche(mSharedPreference);
                    break;
                case W_TRUPP:
                    funkSprüche = new WtFunkSprüche(mSharedPreference);
                    break;
                case A_TRUPP:
                    funkSprüche = new AtFunkSprüche(mSharedPreference);
                    break;
                case S_TRUPP:
                    funkSprüche = new StFunkSprüche(mSharedPreference);
                    break;
                default:
                    throw new IllegalArgumentException("Invalid Sprecher: " + sprecher);
            }
            player = new FunkspruchPlayer(funkSprüche);
            funkspruchPlayer.setValue(player);
        } else {
            player = funkspruchPlayer.getValue();
            if (player != null) {
                player.updateSprüche(mSharedPreference);
            }
        }

        // Initialize liveData after creating the player
        funkSpruecheList.setValue(player.getAktuelleFunkSprüche());
        isTrainingStarted.setValue(player.isStarted()); // Set initial state
        currentSprecher.setValue(player.getSprecher());
    }

    @Override
    public void onInit(int status) {
        if (status == TextToSpeech.SUCCESS) {
            textToSpeech.setSpeechRate(1.5f);
            int result = textToSpeech.setLanguage(Locale.GERMAN);
            if (result == TextToSpeech.LANG_MISSING_DATA || result == TextToSpeech.LANG_NOT_SUPPORTED) {
                Timber.e( "Language is not available.");
            } else {
                ttsIsInitialized = true;
                Timber.d("TTS Engine initialized successfully");
            }
        } else {
            Timber.e("Initialization failed.");
            textToSpeech = null;
        }
    }

    private void speak(String text) {
        if (ttsIsInitialized && textToSpeech != null) {
            if (mSharedPreference.getBoolean("vorlesen", true)) {
                textToSpeech.setSpeechRate(Float.parseFloat(mSharedPreference.getString("lesegeschwindigkeit", "1.5")));
                textToSpeech.speak(text, TextToSpeech.QUEUE_FLUSH, null, null); // Consider adding request identifier
            }
        } else {
            Timber.w("speak failed: not bound to TTS engine");
        }
    }

    public void speakNext() {
        FunkspruchPlayer player = funkspruchPlayer.getValue();
        if (player != null) {
            if (player.hasNext()) {
                FunkSpruch nextSpruch = player.next();
                speak(nextSpruch.getSpruch());
                funkSpruecheList.setValue(player.getAktuelleFunkSprüche()); // Update
            }
            if (player.isFinished()) { // Trigger End training event if training is finished
                isTrainingStarted.setValue(false);
                countTrainingsEnd(player.getSprecher());
            }
        }
    }

    @Override
    protected void onCleared() {
        if (textToSpeech != null) {
            textToSpeech.stop();
            textToSpeech.shutdown();
            textToSpeech = null;
            ttsIsInitialized = false;
            Timber.d("TextToSpeech released.");
        }
        super.onCleared();
    }

    private void countTrainingsStart(FunkSpruch.Sprecher sprecher) {
        PrefsHelper ph = mPrefsHelper;
        // count trainig start TODO: add MA + ST
        switch (sprecher) {
            case E_LEITER:
                ph.addStartTrainingElCount();
                break;
            case A_TRUPP:
                ph.addStartTrainingAtCount();
                break;
            case W_TRUPP:
                ph.addStartTrainingWtCount();
        }
    }

    private void countTrainingsEnd(FunkSpruch.Sprecher sprecher) {
        PrefsHelper ph = mPrefsHelper;
        // count trainig finished TODO: add MA + ST
        switch (sprecher) {
            case E_LEITER:
                ph.addFinishTrainingElCount();
                break;
            case A_TRUPP:
                ph.addFinishTrainingAtCount();
                break;
            case W_TRUPP:
                ph.addFinishTrainingWtCount();
        }
    }
}
