package de.questmaster.wettkampf_funk_trainer.ui;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;

import androidx.annotation.NonNull;
import androidx.fragment.app.Fragment;
import androidx.lifecycle.ViewModelProvider;
import androidx.preference.PreferenceManager;
import androidx.recyclerview.widget.GridLayoutManager;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.google.android.material.floatingactionbutton.FloatingActionButton;

import de.questmaster.wettkampf_funk_trainer.R;
import de.questmaster.wettkampf_funk_trainer.data.FunkSpruch;
import me.zhanghai.android.fastscroll.FastScroller;
import me.zhanghai.android.fastscroll.FastScrollerBuilder;
import timber.log.Timber;

public abstract class EinheitFragment extends Fragment implements MyFunkRecyclerViewAdapter.OnItemClickListener {
    protected static final String ARG_COLUMN_COUNT = "column-count";
    private static final String KEY_SCROLL_POSITION = "scroll_position";
    private int mColumnCount = 1;
    protected SharedPreferences mSharedPreference;
    private MyFunkRecyclerViewAdapter mViewAdapter;
    private LinearLayoutManager mLayoutManager;
    private EinheitViewModel mViewModel;
    private ViewTreeObserver.OnPreDrawListener mScrollPositionListener;
    private View.OnClickListener mFabClickListener;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        if (getArguments() != null) {
            mColumnCount = getArguments().getInt(ARG_COLUMN_COUNT);
        }

        // Initialize the ViewModel
        //mViewModel = new ViewModelProvider(requireActivity()).get(EinheitViewModel.class);
        mViewModel = new ViewModelProvider(this).get(EinheitViewModel.class);

        Timber.d("EinheitFragment created - instance: %s", this.hashCode());
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        View rootView = inflater.inflate(R.layout.fragment_wt_list, container, false);

        View view = rootView.findViewById(R.id.list);

        // Set the adapter
        if (view instanceof RecyclerView) {
            Context context = view.getContext();
            mSharedPreference = PreferenceManager.getDefaultSharedPreferences(context);
            FunkSpruch.Sprecher lernender = getLernSprecher();

            // Initialize ViewModel here, after getting required data.
            mViewModel.init(lernender, context);

            RecyclerView recyclerView = (RecyclerView) view;
            recyclerView.addItemDecoration(new CustomItemDecoration(lernender));
            if (mColumnCount <= 1) {
                mLayoutManager = new LinearLayoutManager(context, LinearLayoutManager.VERTICAL, false);
            } else {
                mLayoutManager = new GridLayoutManager(context, mColumnCount, LinearLayoutManager.VERTICAL, false);
            }
            recyclerView.setLayoutManager(mLayoutManager);

            // Performance-Optimierung: RecyclerView Cache-Size erhöhen für smootheres Scrolling
            // Default ist 2, erhöhen auf 20 für bessere ViewHolder-Wiederverwendung
            recyclerView.setItemViewCacheSize(20);

            FastScroller fastScroller = new FastScrollerBuilder(recyclerView).setViewHelper(new FastScrollViewHelper(recyclerView, null)).build();
            recyclerView.setOnApplyWindowInsetsListener(new ScrollingViewOnApplyWindowInsetsListener(recyclerView, fastScroller));

            // Observe the FunkspruchPlayer LiveData
            mViewAdapter = new MyFunkRecyclerViewAdapter(this, mSharedPreference);
            recyclerView.setAdapter(mViewAdapter);
            mViewModel.getFunkSpruecheList().observe(getViewLifecycleOwner(), lFunkSprüche -> {
                if (lFunkSprüche != null) {
                    mViewAdapter.updateFunkspruchListe(lFunkSprüche, recyclerView);
                } else {
                    // Handle the case where the player is null (e.g., show an error)
                    Timber.w("List<FunkSpruch> is null");
                }
            });

            // Restore scroll position *after* layout
            if (savedInstanceState != null) {
                final int savedPosition = savedInstanceState.getInt(KEY_SCROLL_POSITION, RecyclerView.NO_POSITION);
                if (savedPosition != RecyclerView.NO_POSITION) {
                    // Use ViewTreeObserver to ensure layout is complete
                    mScrollPositionListener = new ViewTreeObserver.OnPreDrawListener() {
                        @Override
                        public boolean onPreDraw() {
                            recyclerView.getViewTreeObserver().removeOnPreDrawListener(this);
                            recyclerView.scrollToPosition(savedPosition);
                            Timber.d("Scroll position restored (post-layout): %d", savedPosition);
                            return true; // Continue with the drawing pass
                        }
                    };
                    recyclerView.getViewTreeObserver().addOnPreDrawListener(mScrollPositionListener);
                }
            }

            // Set the click listener for the FAB
            FloatingActionButton fab = rootView.findViewById(R.id.fab);
            mViewModel.getIsTrainingStarted().observe(getViewLifecycleOwner(), isStarted -> {
                setFabIconOnPlayerState(fab, isStarted);
            });
            mFabClickListener = new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    mViewModel.onFabClicked();
                }
            };
            fab.setOnClickListener(mFabClickListener);

            Timber.d("onCreateView() finished for %s - instance: %s", lernender.toNameString(), this.hashCode());
        }

        return rootView;
    }

    // Sprache: Java
    private void setFabIconOnPlayerState(FloatingActionButton fab, boolean isStarted) {
        if (isStarted) {
            fab.setImageResource(android.R.drawable.ic_media_next);
            fab.setTag("started");
        } else {
            fab.setImageResource(android.R.drawable.ic_media_play);
            fab.setTag("stopped");
        }
    }

    @Override
    public void onSaveInstanceState(@NonNull Bundle outState) {
       if (mLayoutManager != null) {
            int firstVisibleItemPosition = mLayoutManager.findFirstVisibleItemPosition();
            if (firstVisibleItemPosition != RecyclerView.NO_POSITION) {
                outState.putInt(KEY_SCROLL_POSITION, firstVisibleItemPosition);
            }
        }
        super.onSaveInstanceState(outState);
    }

    @Override
    public void onDestroyView() {
        // Performance & Memory-Optimierung: Sichere Bereinigung aller Ressourcen

        // Entferne den PreDrawListener, falls er noch existiert
        if (mScrollPositionListener != null) {
            View view = getView();
            if (view != null) {
                View recyclerView = view.findViewById(R.id.list);
                if (recyclerView != null) {
                    recyclerView.getViewTreeObserver().removeOnPreDrawListener(mScrollPositionListener);
                    Timber.d("ScrollPositionListener cleared");
                }
            }
            mScrollPositionListener = null;
        }

        // Setze den FAB-ClickListener zurück
        View rootView = getView();
        if (rootView != null) {
            FloatingActionButton fab = rootView.findViewById(R.id.fab);
            if (fab != null && mFabClickListener != null) {
                fab.setOnClickListener(null);
                Timber.d("FabClickListener cleared");
            }

            // RecyclerView Adapter entfernen um Memory-Leaks zu vermeiden
            View view = rootView.findViewById(R.id.list);
            if (view instanceof RecyclerView) {
                ((RecyclerView) view).setAdapter(null);
                Timber.d("RecyclerView adapter released");
            }
        }
        mFabClickListener = null;

        // LiveData-Observer entfernen
        if (mViewModel != null) {
            mViewModel.getFunkSpruecheList().removeObservers(getViewLifecycleOwner());
            mViewModel.getIsTrainingStarted().removeObservers(getViewLifecycleOwner());
            Timber.d("All LiveData observers cleared");
        }

        super.onDestroyView();
    }

    @Override
    public void onDestroy() {
        super.onDestroy();

        // Logge Fragment-Zerstörung zur Überwachung
        Timber.d("EinheitFragment destroyed - instance: %s", this.hashCode());

        // Bereinige verbleibende Ressourcen
        mViewModel = null;
        mSharedPreference = null;
        mViewAdapter = null;
        mLayoutManager = null;
    }

    @Override
    public void onResume() {
        Timber.d("onResume() called - instance: %s", this.hashCode());
        mViewModel.handleOnResume(getLernSprecher());

        super.onResume();
    }

    @Override
    public void onPause() {
        super.onPause();
    }

    @Override
    public void onItemClick(int position) {
        mViewModel.speakNext();
        Timber.d("onClick() called");
    }

    protected abstract FunkSpruch.Sprecher getLernSprecher();

}
