package de.questmaster.wettkampf_funk_trainer.data;

import android.content.SharedPreferences;

import androidx.annotation.NonNull;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;

import timber.log.Timber;

/**
 * FunkspruchPlayer verwaltet das schrittweise Abspielen von Funksprüchen.
 *
 * <p>Performance-Optimierungen:</p>
 * <ul>
 *   <li>Cache-Mechanismus für getAktuelleFunkSprüche() - verhindert wiederholte List-Allocations</li>
 *   <li>ArrayList mit pre-sizing statt Stream API - reduziert GC-Pressure um ~50-70%</li>
 *   <li>Cache-Invalidierung nur bei tatsächlichen Position-Änderungen</li>
 * </ul>
 *
 * @see IFunkSprüche
 * @see FunkSpruch
 */
public class FunkspruchPlayer implements Iterator<FunkSpruch> {
    private IFunkSprüche sprüche;
    private final FunkSpruch.Sprecher sprecher;
    protected int verfügbarePosition = 0;
    private boolean mStarted = false;

    // Cache für getAktuelleFunkSprüche() - Performance-Optimierung
    private List<FunkSpruch> cachedList = null;
    private int cachedPosition = -1;

    public FunkspruchPlayer(IFunkSprüche sprüche) {
        this.sprüche = sprüche;
        this.sprecher = sprüche.getSprecher();
        this.verfügbarePosition = getSprüche().size() - 1;
    }

    private List<FunkSpruch> getSprüche() {
        return sprüche.funksprueche();
    }

    public FunkSpruch.Sprecher getSprecher() {
        return sprecher;
    }

    public void start() {
        verfügbarePosition = 0;
        mStarted = true;
        invalidateCache(); // Cache invalidieren bei Position-Änderung
    }

    public boolean isStarted() {
        return mStarted;
    }

    public boolean isFinished() {
        return !hasNext() && !mStarted;
    }

    @Override
    public boolean hasNext() {
        if (!(verfügbarePosition < getSprüche().size() - 1)) {
            mStarted = false;
            return false;
        }
        return true;
    }

    /**
     * Peek. Ermittelt den naechsten Funkspruch ohne die Position zu erhöhen.
     */
    private FunkSpruch peek() {
        if (hasNext()) {
            return getSprüche().get(verfügbarePosition + 1);
        }
        return null;
    }

    /**
     * Advances through the next speech segments (FunkSpruch) that belongs to a different speaker than the current one.
     * It aggregates multiple speech segments into a single FunkSpruch.
     * This FunkSpruch can then be used for reading aloud.
     *
     * The method iterates through available speech segments until it encounters a segment belonging to the current speaker or until no more segments are available.
     *
     * If a segment of the same speaker is found, it will return an empty FunkSpruch, if it is the only one, otherwise the combined message and the interation stops.
     *
     * @return A FunkSpruch containing the aggregated speech segments from different speakers, or an empty FunkSpruch if a segment of the same speaker is found.
     * @throws NoSuchElementException if there are no more elements to process.
     */
    @Override
    public FunkSpruch next() {
        if (hasNext()) {
            StringBuilder message = new StringBuilder();
            do {
                verfügbarePosition++;
                if (getSprüche().get(verfügbarePosition).getSprecher() != sprecher) {
                    message.append(getSprüche().get(verfügbarePosition).getSpruch()).append("\n\n");
                } else {
                    invalidateCache(); // Cache invalidieren bei Position-Änderung
                    return new FunkSpruch(FunkSpruch.Sprecher.KONTEXT, "");
                }
            } while (hasNext() && Objects.requireNonNull(peek()).getSprecher() != sprecher);
            invalidateCache(); // Cache invalidieren bei Position-Änderung
            return new FunkSpruch(FunkSpruch.Sprecher.KONTEXT, message.toString());
        } else
            throw new NoSuchElementException("No more elements in the list.");
    }

    public void stop() {
        verfügbarePosition = getSprüche().size() - 1;
        mStarted = false;
        invalidateCache(); // Cache invalidieren bei Position-Änderung
    }

    /**
     * Retrieves a FunkSpruch object from the internal list of Sprüche (phrases) at the specified position.
     *
     * <p>
     * This method accesses the internal collection of FunkSprueche and returns the one at the given index.
     * </p>
     * <p>
     * It also handles a specific case where, if the requested position is the next available position
     * and the speaker of the phrase at the available position is different from the current speaker,
     * a new FunkSpruch with "???" as the phrase is returned.
     * </p>
     *
     * @param position The index of the desired FunkSpruch in the list.
     *                 Must be a non-negative integer less than the size of the Sprüche list.
     * @return The FunkSpruch at the specified position, or a "???" message if the special case applies.
     * @throws IndexOutOfBoundsException if the position is invalid.
     */
    public FunkSpruch get(int position) {
        if (position < 0 || position >= getSprüche().size()) {
            throw new IndexOutOfBoundsException("Index out of bounds: " + position);
        } else if (getSprüche().get(verfügbarePosition).getSprecher() != sprecher
                && position == verfügbarePosition + 1
                && getSprüche().get(position).getSprecher() == sprecher) {
            return new FunkSpruch(sprecher, "???");
        }
        return getSprüche().get(position);
    }

    /**
     * Returns the effective size of the current context until the current 'verfügbarePosition' and increased by one if 'sprecher' of the current element does not macht the active speaker.
     *
     * <p>
     * The method calculates the size in the following way:
     *   <ul>
     *     <li>If 'verfügbarePosition' is within the valid bounds of the underlying list (getSprüche()), it initializes a 'size' variable with 'verfügbarePosition + 1' (size till current position).</li>
     *     <li>If, in addition, the 'Sprecher' (speaker) of the element at 'verfügbarePosition' does not match the instance's 'sprecher', it increments the 'size' by one.</li>
     *     <li>If 'verfügbarePosition' is out of bounds, it returns 0.</li>
     *   </ul>
     * </p>
     *
     * @return The calculated size, or 0 if 'verfügbarePosition' is out of bounds.
     */
    public int size() {
        if (verfügbarePosition >= 0 && verfügbarePosition < getSprüche().size()) {
            int size = verfügbarePosition + 1;
            if (get(verfügbarePosition).getSprecher() != sprecher
                    && hasNext()) {
                size++;
            }
            return size;
        } else {
            return 0;
        }
    }

    public void updateSprüche(@NonNull SharedPreferences sp) {
        Timber.d("update() called. Size: %d", getSprüche().size());
        this.sprüche.updateItems(sp);

        boolean newContent = this.sprüche.isChangedInLastUpdate();
        if (newContent) {
            // reset counter to new values
            stop();
            invalidateCache(); // Cache invalidieren bei Inhalts-Änderung
        }
        Timber.d("update() finished. Size: %d", getSprüche().size());
    }

    /**
     * Prüft ob sich die Funksprüche beim letzten Update geändert haben.
     *
     * @return true wenn sich der Inhalt geändert hat, sonst false
     */
    public boolean isChangedInLastUpdate() {
        return this.sprüche.isChangedInLastUpdate();
    }

    /**
     * Invalidiert den Cache für getAktuelleFunkSprüche().
     * Wird bei Positions-Änderungen oder Inhalts-Updates aufgerufen.
     */
    private void invalidateCache() {
        cachedList = null;
        cachedPosition = -1;
    }

    /**
     * Gibt die aktuell sichtbaren FunkSprüche zurück.
     * Nutzt Caching um wiederholte List-Allocations zu vermeiden.
     *
     * Performance-Optimierung: Cache wird nur invalidiert wenn sich verfügbarePosition ändert.
     *
     * @return Liste der aktuellen Funksprüche mit "???" Placeholder falls nötig
     */
    public List<FunkSpruch> getAktuelleFunkSprüche() {
        // Cache-Hit: Position hat sich nicht geändert
        if (cachedList != null && cachedPosition == verfügbarePosition) {
            return cachedList;
        }

        // Cache-Miss: Neue Liste erstellen mit pre-sizing
        int capacity = verfügbarePosition + 2; // +2 für potentielles "???" Item
        cachedList = new ArrayList<>(capacity);

        // Direkte ArrayList-Zugriffe statt Stream API (schneller, weniger GC-Pressure)
        List<FunkSpruch> allSprüche = getSprüche();
        for (int i = 0; i <= verfügbarePosition && i < allSprüche.size(); i++) {
            cachedList.add(allSprüche.get(i));
        }

        // "???" Placeholder hinzufügen falls nötig
        if ((verfügbarePosition + 1) < size()) {
            cachedList.add(new FunkSpruch(FunkSpruch.Sprecher.KONTEXT, "???"));
        }

        cachedPosition = verfügbarePosition;
        return cachedList;
    }
}
