package de.questmaster.wettkampf_funk_trainer;

import static android.os.Build.VERSION_CODES.UPSIDE_DOWN_CAKE;

import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;

import androidx.activity.OnBackPressedCallback;
import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.view.MenuProvider;
import androidx.core.view.OnApplyWindowInsetsListener;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.core.view.WindowInsetsControllerCompat;
import androidx.drawerlayout.widget.DrawerLayout;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.lifecycle.Lifecycle;
import androidx.navigation.NavController;
import androidx.navigation.Navigation;
import androidx.navigation.fragment.NavHostFragment;
import androidx.navigation.ui.AppBarConfiguration;
import androidx.navigation.ui.NavigationUI;

import com.google.android.material.navigation.NavigationView;

import de.questmaster.wettkampf_funk_trainer.databinding.ActivityMainBinding;
import de.questmaster.wettkampf_funk_trainer.utilities.PrefsHelper;
import de.questmaster.wettkampf_funk_trainer.utilities.ReleaseTree;
import de.questmaster.wettkampf_funk_trainer.BuildConfig;
import timber.log.Timber;

public class MainActivity extends AppCompatActivity {

    private AppBarConfiguration mAppBarConfiguration;
    private ActivityMainBinding binding;
    private Fragment mCurrentFragment = null;
    private NavController navController;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        // Prüfe, ob bereits ein Tree gepflanzt wurde
        if (Timber.forest().isEmpty()) {
            if (BuildConfig.DEBUG) {
                Timber.plant(new Timber.DebugTree());
            } else {
                Timber.plant(new ReleaseTree());
            }
        }

        binding = ActivityMainBinding.inflate(getLayoutInflater());
        setContentView(binding.getRoot());

        setSupportActionBar(binding.appBarMain.toolbar);

        DrawerLayout drawer = binding.drawerLayout;
        NavigationView navigationView = binding.navView;
        // Passing each menu ID as a set of Ids because each
        // menu should be considered as top level destinations.
        mAppBarConfiguration = new AppBarConfiguration.Builder(
                R.id.nav_home, R.id.nav_el, R.id.nav_ma, R.id.nav_at, R.id.nav_wt, R.id.nav_st)
                .setOpenableLayout(drawer)
                .build();

        // NavHostFragment korrekt finden und einrichten
        NavHostFragment navHostFragment =
                (NavHostFragment) getSupportFragmentManager().findFragmentById(R.id.nav_host_fragment_content_main);

        if (navHostFragment != null) {
            navController = navHostFragment.getNavController();

            // Navigation-UI einrichten - Navigation Component übernimmt die Fragment-Verwaltung
            NavigationUI.setupActionBarWithNavController(this, navController, mAppBarConfiguration);
            NavigationUI.setupWithNavController(navigationView, navController);

            // Logger für Debug-Zwecke: Trackt Navigation-Ereignisse
            navController.addOnDestinationChangedListener((controller, destination, arguments) -> {
                Timber.d("Navigation zu Destination: %s (ID: %d)",
                        destination.getLabel(), destination.getId());
            });
        } else {
            Timber.e("NavHostFragment nicht gefunden!");
        }

        // Back-Callback
        getOnBackPressedDispatcher().addCallback(this, new OnBackPressedCallback(true) {
            @Override
            public void handleOnBackPressed() {
                if (!navController.popBackStack()) {
                    finish(); // Beendet die Activity, wenn kein BackStack mehr vorhanden ist
                }
            }
        });

        // Handle insets for Android API 35+
        if (Build.VERSION.SDK_INT >= UPSIDE_DOWN_CAKE) {
            ViewCompat.setOnApplyWindowInsetsListener(navigationView, new OnApplyWindowInsetsListener() {
                @NonNull
                @Override
                public WindowInsetsCompat onApplyWindowInsets(@NonNull View view, @NonNull WindowInsetsCompat insets) {
                    WindowInsetsControllerCompat controller = ViewCompat.getWindowInsetsController(view);
                    if (controller != null) {
//                        controller.setAppearanceLightStatusBars(true); // Optional: Make status bar icons dark if desired
                        controller.setSystemBarsBehavior(WindowInsetsControllerCompat.BEHAVIOR_SHOW_TRANSIENT_BARS_BY_SWIPE);
                    }
                    int systemBarsTop = insets.getInsets(WindowInsetsCompat.Type.systemBars()).top;
                    view.setPadding(view.getPaddingLeft(), systemBarsTop, view.getPaddingRight(), view.getPaddingBottom());
                    return insets;
                }
            });
        }

        // Registriere Back-Callback - Ersatz für onBackPressed()
        getOnBackPressedDispatcher().addCallback(this, new OnBackPressedCallback(true) {
            @Override
            public void handleOnBackPressed() {
                // Hilft beim Debuggen von Navigation-Problemen
                if (mCurrentFragment != null) {
                    Timber.d("onBackPressed() mit aktivem Fragment: %s (hash: %s)",
                            mCurrentFragment.getClass().getSimpleName(),
                            mCurrentFragment.hashCode());
                }

                if (navController == null) {
                    NavController navController = Navigation.findNavController(
                            MainActivity.this, R.id.nav_host_fragment_content_main);
                }

                // Wenn NavController nicht zurückgehen kann, beende den Callback
                if (!navController.popBackStack()) {
                    this.setEnabled(false); // Deaktiviert den Callback, um Default-Verhalten zuzulassen
                    getOnBackPressedDispatcher().onBackPressed(); // Trigger default back behavior
                    this.setEnabled(true); // Reaktiviere für zukünftige Back-Ereignisse
                }
            }
        });

        // Das hier nicht vergessen!
        monitorFragmentChanges();

        addMenuProvider(new MenuProvider() {
            @Override
            public void onCreateMenu(@NonNull Menu menu, @NonNull MenuInflater menuInflater) {
                menuInflater.inflate(R.menu.main, menu);
            }

            @Override
            public boolean onMenuItemSelected(@NonNull MenuItem menuItem) {
                int id = menuItem.getItemId();
                if (id == R.id.action_settings) {
                    startActivity(new Intent(MainActivity.this, SettingsActivity.class));
                    return true;
                } else if (id == R.id.action_rate_app) {
                    openStoreVotingLink();
                    return true;
                } else if (id == R.id.action_donate_app) {
                    openDonationLink();
                    return true;
                } else if (id == R.id.action_issue_app) {
                    openIssueLink();
                    return true;
                }
                return false; // Let the system handle other items or super.onOptionsItemSelected
            }
        }, this, Lifecycle.State.RESUMED); // Add provider with lifecycle owner and state

        // count app start
        PrefsHelper ph = new PrefsHelper(this);
        ph.addAppLaunch();
        // check if first start
        if (ph.isFirstLaunch()) {
            // Show onboarding activity
            startActivity(new Intent(this, OnboardingActivity.class));

            // Set first launch flag to false
            ph.setFirstLaunch(false);
        }
    }

    // Method to track current visible fragment
    private void monitorFragmentChanges() {
        NavHostFragment navHostFragment =
                (NavHostFragment) getSupportFragmentManager().findFragmentById(R.id.nav_host_fragment_content_main);

        if (navHostFragment != null) {
            // Register FragmentManager callback to monitor Fragment lifecycles
            navHostFragment.getChildFragmentManager().registerFragmentLifecycleCallbacks(
                    new FragmentManager.FragmentLifecycleCallbacks() {
                        @Override
                        public void onFragmentStarted(@NonNull FragmentManager fm, @NonNull Fragment f) {
                            super.onFragmentStarted(fm, f);
                            mCurrentFragment = f;
                            Timber.d("Active fragment: %s (hash: %s)",
                                    f.getClass().getSimpleName(), f.hashCode());
                        }
                    },
                    false // Recursive = false, nur direkte Kinder überwachen
            );
        }
    }

    // Don't forget to release the TextToSpeech instance when you're done
    @Override
    public void onDestroy() {
        super.onDestroy();
    }

    private void openIssueLink() {
        startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("https://gitlab.com/questmaster97/wettkampf_funk_trainer/-/issues")));
    }

    private void openStoreVotingLink() {
        String packageName = getPackageName();
        try {
            startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("market://details?id=" + packageName)));
        } catch (android.content.ActivityNotFoundException anfe) {
            startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("https://play.google.com/store/apps/details?id=" + packageName)));
        }
    }

    private void openDonationLink() {
        startActivity(new Intent(Intent.ACTION_VIEW, Uri.parse("https://github.com/sponsors/questmaster?frequency=one-time")));
    }

    @Override
    public boolean onSupportNavigateUp() {
        if (navController == null) {
            // Fallback to find NavController if not set
            navController = Navigation.findNavController(this, R.id.nav_host_fragment_content_main);
        }
        return NavigationUI.navigateUp(navController, mAppBarConfiguration)
                || super.onSupportNavigateUp();
    }

}