package de.ntdote.medicalcalendarlog

import de.ntdote.medicalcalendarlog.data.DecayType
import de.ntdote.medicalcalendarlog.data.Drug
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.data.TemplateType
import org.junit.Test
import org.junit.Assert.*

class SourceTemplateDetectionTest {

    @Test
    fun testSourceTemplateDetection() {
        // Create a source template for Aspirin
        val sourceTemplate = Template(
            id = "source-1",
            name = "Aspirin 500mg",
            templateType = TemplateType.DECAYING,
            drugs = listOf(
                Drug.createSource(
                    drugType = "Aspirin",
                    factor = 500.0,
                    unit = "mg",
                    decayType = DecayType.HALF_LIFE,
                    halfLifeHours = 4.0
                )
            )
        )

        // Create a derived template for the same drug
        val derivedTemplate = Template(
            id = "derived-1",
            name = "Aspirin 250mg",
            templateType = TemplateType.DECAYING,
            drugs = listOf(
                Drug.createDerived(
                    drugType = "Aspirin",
                    factor = 250.0,
                    unit = "mg"
                )
            )
        )

        val templates = listOf(sourceTemplate, derivedTemplate)

        // Test: Find source drugs in the collection
        val sourceDrugs = templates.flatMap { it.getSourceDrugs() }
        assertEquals("Should find one source drug", 1, sourceDrugs.size)
        assertEquals("Source drug should be Aspirin", "Aspirin", sourceDrugs.first().drugType)
        assertEquals("Source should have decay parameters", DecayType.HALF_LIFE, sourceDrugs.first().decayType)

        // Test: Check template types
        assertTrue("First template should contain source drugs", sourceTemplate.hasSourceDrugs())
        assertFalse("Second template should not contain source drugs", derivedTemplate.hasSourceDrugs())

        // Test: Get all drug types (both source and derived)
        val allDrugTypes = templates.flatMap { it.drugs }.map { it.drugType }.distinct()
        assertEquals("Should have drugs for one type (Aspirin)", 1, allDrugTypes.size)
        assertTrue("Should contain Aspirin", allDrugTypes.contains("Aspirin"))
    }

    @Test
    fun testNoSourceTemplateExists() {
        // Create only derived templates
        val derived1 = Template(
            id = "derived-1",
            name = "Aspirin 250mg",
            templateType = TemplateType.DECAYING,
            drugs = listOf(
                Drug.createDerived(
                    drugType = "Aspirin",
                    factor = 250.0,
                    unit = "mg"
                )
            )
        )

        val derived2 = Template(
            id = "derived-2",
            name = "Aspirin 100mg",
            templateType = TemplateType.DECAYING,
            drugs = listOf(
                Drug.createDerived(
                    drugType = "Aspirin",
                    factor = 100.0,
                    unit = "mg"
                )
            )
        )

        val templates = listOf(derived1, derived2)

        // Test: Should not find any source drugs
        val sourceDrugs = templates.flatMap { it.getSourceDrugs() }
        assertEquals("Should not find any source drugs", 0, sourceDrugs.size)

        // Test: Both templates should not have source drugs
        assertFalse("First template should not have source drugs", derived1.hasSourceDrugs())
        assertFalse("Second template should not have source drugs", derived2.hasSourceDrugs())
    }

    @Test
    fun testMultipleDrugTypes() {
        // Create templates for different drugs
        val aspirinSource = Template(
            id = "aspirin-source",
            name = "Aspirin 500mg",
            templateType = TemplateType.DECAYING,
            drugs = listOf(
                Drug.createSource(
                    drugType = "Aspirin",
                    factor = 500.0,
                    unit = "mg",
                    decayType = DecayType.HALF_LIFE,
                    halfLifeHours = 4.0
                )
            )
        )

        val ibuprofenSource = Template(
            id = "ibuprofen-source",
            name = "Ibuprofen 400mg",
            templateType = TemplateType.DECAYING,
            drugs = listOf(
                Drug.createSource(
                    drugType = "Ibuprofen",
                    factor = 400.0,
                    unit = "mg",
                    decayType = DecayType.CONSTANT,
                    hourlyDecayRate = 50.0
                )
            )
        )

        val aspirinDerived = Template(
            id = "aspirin-derived",
            name = "Aspirin 250mg",
            templateType = TemplateType.DECAYING,
            drugs = listOf(
                Drug.createDerived(
                    drugType = "Aspirin",
                    factor = 250.0,
                    unit = "mg"
                )
            )
        )

        val templates = listOf(aspirinSource, ibuprofenSource, aspirinDerived)

        // Test: Find source drugs
        val sourceDrugs = templates.flatMap { it.getSourceDrugs() }
        assertEquals("Should find two source drugs", 2, sourceDrugs.size)

        val aspirinSourceDrug = sourceDrugs.find { it.drugType == "Aspirin" }
        val ibuprofenSourceDrug = sourceDrugs.find { it.drugType == "Ibuprofen" }

        assertNotNull("Should find Aspirin source drug", aspirinSourceDrug)
        assertNotNull("Should find Ibuprofen source drug", ibuprofenSourceDrug)

        assertEquals("Aspirin should use half-life decay", DecayType.HALF_LIFE, aspirinSourceDrug?.decayType)
        assertEquals("Ibuprofen should use constant decay", DecayType.CONSTANT, ibuprofenSourceDrug?.decayType)

        // Test: Check which templates have sources
        assertTrue("Aspirin source template should have source drugs", aspirinSource.hasSourceDrugs())
        assertTrue("Ibuprofen source template should have source drugs", ibuprofenSource.hasSourceDrugs())
        assertFalse("Aspirin derived template should not have source drugs", aspirinDerived.hasSourceDrugs())
    }

    @Test
    fun testMigrationScenario() {
        // Simulate a migration scenario where we need to avoid duplicate sources
        val existingTemplates = listOf(
            Template(
                id = "existing-source",
                name = "Aspirin Source",
                templateType = TemplateType.DECAYING,
                drugs = listOf(
                    Drug.createSource(
                        drugType = "Aspirin",
                        factor = 500.0,
                        unit = "mg",
                        decayType = DecayType.HALF_LIFE,
                        halfLifeHours = 4.0
                    )
                )
            )
        )

        // Get existing source drug types
        val existingSourceDrugTypes = existingTemplates
            .flatMap { it.getSourceDrugs() }
            .map { it.drugType }
            .toSet()

        assertTrue("Should have Aspirin as existing source", existingSourceDrugTypes.contains("Aspirin"))

        // Now create a new template - it should create derived drugs to avoid conflict
        val newTemplate = Template(
            id = "new-template",
            name = "Aspirin 250mg",
            templateType = TemplateType.DECAYING,
            drugs = listOf(
                // This should be created as derived since Aspirin source already exists
                if (existingSourceDrugTypes.contains("Aspirin")) {
                    Drug.createDerived(
                        drugType = "Aspirin",
                        factor = 250.0,
                        unit = "mg"
                    )
                } else {
                    Drug.createSource(
                        drugType = "Aspirin",
                        factor = 250.0,
                        unit = "mg",
                        decayType = DecayType.HALF_LIFE,
                        halfLifeHours = 4.0
                    )
                }
            )
        )

        // Verify the new template has derived drugs, not source drugs
        assertFalse("New template should not have source drugs", newTemplate.hasSourceDrugs())
        assertEquals("New template should have one drug", 1, newTemplate.drugs.size)
        assertEquals("New template drug should be Aspirin", "Aspirin", newTemplate.drugs.first().drugType)
        assertNull("New template drug should not have decay type (derived)", newTemplate.drugs.first().decayType)
    }
}
