package de.ntdote.medicalcalendarlog

import de.ntdote.medicalcalendarlog.data.*
import de.ntdote.medicalcalendarlog.utils.AdvancedCalculationEngine
import org.junit.Test
import org.junit.Assert.*
import java.util.*

class AlcoholConcentrationTest {

    @Test
    fun testAlcoholConcentrationWithBeerAndWine() {
        // Create beer template (source) - 0.32 factor, 0.1 decay per hour
        val beerTemplate = Template(
            id = "beer_template",
            name = "Beer",
            templateType = TemplateType.DECAYING,
            extendedData = ExtendedTemplateData(
                factor = 0.32,
                unit = "BAC",
                typeOfDrug = "alcohol",
                decayType = DecayType.CONSTANT,
                hourlyDecayRate = 0.1,
                halfLifeHours = null
            )
        )

        // Create wine template (secondary) - 0.1 factor, 0.1 decay per hour
        val wineTemplate = Template(
            id = "wine_template", 
            name = "Wine",
            templateType = TemplateType.DECAYING,
            extendedData = ExtendedTemplateData(
                factor = 0.1,
                unit = "BAC",
                typeOfDrug = "alcohol",
                decayType = DecayType.CONSTANT,
                hourlyDecayRate = 0.1,
                halfLifeHours = null
            )
        )

        val templates = listOf(beerTemplate, wineTemplate)

        // Create test events
        val baseTime = Date()
        val events = listOf(
            // First beer at time 0
            CalendarEvent(
                id = 1,
                title = "Beer",
                startTime = baseTime,
                endTime = Date(baseTime.time + 60000), // 1 minute later
                calendarId = 1,
                dosage = 1.0
            ),
            // Wine at time +1 hour
            CalendarEvent(
                id = 2,
                title = "Wine", 
                startTime = Date(baseTime.time + 3600000), // 1 hour later
                endTime = Date(baseTime.time + 3660000), // 1 hour 1 minute later
                calendarId = 1,
                dosage = 1.0
            ),
            // Second beer at time +2 hours
            CalendarEvent(
                id = 3,
                title = "Beer",
                startTime = Date(baseTime.time + 7200000), // 2 hours later
                endTime = Date(baseTime.time + 7260000), // 2 hours 1 minute later
                calendarId = 1,
                dosage = 1.0
            )
        )

        // Test concentration calculations at key time points
        println("=== Alcohol Concentration Test ===")
        
        // Time 0: Right after first beer
        val conc0 = calculateConcentrationAtTime(templates, events, baseTime)
        println("Time 0 (after 1st beer): $conc0 BAC")
        println("Expected: 0.32 (1 beer × 0.32 factor)")
        
        // Time +30 minutes: Beer decaying
        val conc30min = calculateConcentrationAtTime(templates, events, Date(baseTime.time + 1800000))
        println("Time +30min: $conc30min BAC")
        println("Expected: ~0.27 (0.32 - 0.05 decay)")
        
        // Time +1 hour: Right after wine added
        val conc1h = calculateConcentrationAtTime(templates, events, Date(baseTime.time + 3600000))
        println("Time +1h (after wine): $conc1h BAC")
        println("Expected: ~0.32 (0.22 from beer + 0.1 from wine)")
        
        // Time +2 hours: Right after second beer
        val conc2h = calculateConcentrationAtTime(templates, events, Date(baseTime.time + 7200000))
        println("Time +2h (after 2nd beer): $conc2h BAC")
        println("Expected: ~0.54 (0.12 from 1st beer + 0.0 from wine + 0.32 from 2nd beer)")

        // Verify that wine ADDS to concentration, doesn't subtract
        // At +30min: beer = 0.27, at +1h: beer = 0.22, wine = 0.1, total = 0.32
        // So wine correctly adds 0.1, making total higher than just beer decay alone
        val expectedBeerAt1h = 0.32 - 0.1 // Beer after 1 hour decay = 0.22
        val expectedTotalAt1h = expectedBeerAt1h + 0.1 // Add wine contribution = 0.32
        assertEquals("Wine should add 0.1 to concentration", expectedTotalAt1h, conc1h, 0.01)
        
        // Verify that each beer adds 0.32
        assertEquals("First beer should add 0.32", 0.32, conc0, 0.01)
        
        // Verify wine is actually contributing positively
        assertTrue("Wine should increase total concentration", conc1h > expectedBeerAt1h)
        
        // Check individual contributions
        println("\n=== Individual Event Analysis ===")
        events.forEachIndexed { index, event ->
            val template = templates.find { it.name == event.title }
            println("Event ${index + 1}: ${event.title}")
            println("  Template factor: ${template?.extendedData?.factor}")
            println("  Expected contribution: ${event.dosage * (template?.extendedData?.factor ?: 0.0)}")
        }
    }

    private fun calculateConcentrationAtTime(
        templates: List<Template>,
        events: List<CalendarEvent>,
        targetTime: Date
    ): Double {
        var totalConcentration = 0.0
        
        // Get all events for alcohol that occurred at or before the target time
        val relevantEvents = events.filter { event ->
            event.startTime.time <= targetTime.time &&
            templates.any { template -> 
                template.extendedData?.typeOfDrug == "alcohol" && matchesTemplate(event, template)
            }
        }
        
        println("\n--- Calculating concentration at ${targetTime} ---")
        
        relevantEvents.forEach { event ->
            // Find the template for this event
            val template = templates.find { template ->
                template.extendedData?.typeOfDrug == "alcohol" && matchesTemplate(event, template)
            }
            
            template?.let { tmpl ->
                val extendedData = tmpl.extendedData
                if (extendedData != null) {
                    val timeDiffHours = (targetTime.time - event.startTime.time) / (1000.0 * 60.0 * 60.0)
                    val dosage = event.dosage
                    val factor = extendedData.factor ?: 0.0
                    
                    val concentration = when {
                        extendedData.hourlyDecayRate != null && extendedData.hourlyDecayRate!! > 0 -> {
                            // Linear decay using constant elimination
                            val remaining = dosage * factor - (extendedData.hourlyDecayRate!! * timeDiffHours)
                            maxOf(0.0, remaining)
                        }
                        else -> 0.0
                    }
                    
                    println("Event: ${event.title} at ${event.startTime}")
                    println("  Time diff: ${timeDiffHours} hours")
                    println("  Dosage: $dosage, Factor: $factor")
                    println("  Decay rate: ${extendedData.hourlyDecayRate}/hour")
                    println("  Contribution: $concentration")
                    
                    totalConcentration += concentration
                }
            }
        }
        
        println("Total concentration: $totalConcentration")
        return totalConcentration
    }
    
    private fun matchesTemplate(event: CalendarEvent, template: Template): Boolean {
        val templateName = template.name
        return event.title.contains(templateName) || 
               event.title.startsWith(templateName) ||
               templateName.contains(event.title)
    }
}
