package de.ntdote.medicalcalendarlog.viewmodel

import android.app.Application
import android.app.usage.UsageStatsManager
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.os.Build
import android.os.PowerManager
import android.provider.Settings
import android.util.Log
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import androidx.work.WorkInfo
import androidx.work.WorkManager
import de.ntdote.medicalcalendarlog.data.CalendarEvent
import de.ntdote.medicalcalendarlog.data.CalendarInfo
import de.ntdote.medicalcalendarlog.data.ConcentrationData
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.data.TemplateType
import de.ntdote.medicalcalendarlog.repository.CalendarRepository
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.service.HourlyReminderWorker
import de.ntdote.medicalcalendarlog.service.ReminderService
import de.ntdote.medicalcalendarlog.service.TemplateWidgetProvider
import de.ntdote.medicalcalendarlog.utils.ConcentrationCalculator
import de.ntdote.medicalcalendarlog.utils.ConfigExportImport
import de.ntdote.medicalcalendarlog.utils.DrugResolution
import de.ntdote.medicalcalendarlog.utils.TimeDuration
import de.ntdote.medicalcalendarlog.utils.TimeFormatter
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.launch
import java.util.Date
import java.util.UUID

class MainViewModel(
    private val application: Application,
    private val calendarRepository: CalendarRepository,
    private val preferencesRepository: PreferencesRepository
) : ViewModel() {

    private val _uiState = MutableStateFlow(UiState())
    val uiState: StateFlow<UiState> = _uiState.asStateFlow()

    private val _availableCalendars = MutableStateFlow<List<CalendarInfo>>(emptyList())
    val availableCalendars: StateFlow<List<CalendarInfo>> = _availableCalendars.asStateFlow()

    private val _calendarStatus = MutableStateFlow<CalendarRepository.CalendarStatus?>(
        CalendarRepository.CalendarStatus(
            hasPermissions = false,
            calendarsFound = 0,
            errorMessage = "Calendar status not loaded yet"
        )
    )
    val calendarStatus: StateFlow<CalendarRepository.CalendarStatus?> = _calendarStatus.asStateFlow()

    private val _events = MutableStateFlow<List<CalendarEvent>>(emptyList())
    val events: StateFlow<List<CalendarEvent>> = _events.asStateFlow()

    private val _concentrationData = MutableStateFlow<List<ConcentrationData>>(emptyList())
    val concentrationData: StateFlow<List<ConcentrationData>> = _concentrationData.asStateFlow()

    private val _visibleDrugs = MutableStateFlow<Set<String>>(emptySet())
    val visibleDrugs: StateFlow<Set<String>> = _visibleDrugs.asStateFlow()

    private val _showEventText = MutableStateFlow(true)
    val showEventText: StateFlow<Boolean> = _showEventText.asStateFlow()
    
    // Current time used for calculations
    private val _currentTime = MutableStateFlow(Date())

    private val configExportImport by lazy { ConfigExportImport(application.contentResolver) }

    data class UiState(
        val isLoading: Boolean = false,
        val selectedCalendarId: Long? = null,
        val daysBack: Int = 90,
        val templates: List<Template> = emptyList(),
        val error: String? = null,
        val hasCalendarPermission: Boolean = false,
        val exportInProgress: Boolean = false,
        val importInProgress: Boolean = false,
        val debugModeEnabled: Boolean = false,
        val nextReminderTimestamp: Long = 0L,
        val nextReminderTemplateName: String = "",
        val hourlyWorkerDeltas: List<Int> = emptyList(),
        val nextWorkerRunTimestamp: Long = 0L,
        val appStandbyBucket: Int = 0
    )

    init {
        loadInitialData()
        observePreferences()
    }

    private fun loadInitialData() {
        viewModelScope.launch {
            _uiState.value = _uiState.value.copy(isLoading = true)
            
            try {
                loadAvailableCalendars()
                
                // Wait for preferences to be loaded first
                // We'll handle auto-selection in observePreferences() after we know
                // if a calendar was previously selected
                
            } catch (e: Exception) {
                _uiState.value = _uiState.value.copy(error = e.message)
            } finally {
                _uiState.value = _uiState.value.copy(isLoading = false)
            }
        }
    }

    private fun observePreferences() {
        // Observer 1: Calendar ID and DaysBack changes → reload events
        viewModelScope.launch {
            combine(
                preferencesRepository.selectedCalendarId,
                preferencesRepository.daysBack
            ) { calendarId, daysBack ->
                Pair(calendarId, daysBack)
            }.collect { (calendarId, daysBack) ->
                val previousCalendarId = _uiState.value.selectedCalendarId
                
                _uiState.value = _uiState.value.copy(
                    selectedCalendarId = calendarId,
                    daysBack = daysBack
                )
                
                // Auto-select preferred calendar only if no calendar was previously selected
                if (calendarId == null && previousCalendarId == null) {
                    val preferredCalendar = calendarRepository.findPreferredCalendar()
                    preferredCalendar?.let {
                        preferencesRepository.setSelectedCalendarId(it.id)
                    }
                } else if (calendarId != null) {
                    // Load events when calendar or daysBack changes
                    loadEvents(calendarId, daysBack)
                }
            }
        }
        
        // Observer 2: Templates changes → recalculate concentration (use cached events)
        viewModelScope.launch {
            preferencesRepository.templates.collect { templates ->
                _uiState.value = _uiState.value.copy(templates = templates)
                
                // Only recalculate if we have events cached
                if (_events.value.isNotEmpty()) {
                    updateConcentrationData(templates, _events.value)
                }
            }
        }
        
        // Observer 3: Debug mode → update UI state only (no computation)
        viewModelScope.launch {
            preferencesRepository.debugModeEnabled.collect { debugModeEnabled ->
                _uiState.value = _uiState.value.copy(debugModeEnabled = debugModeEnabled)
            }
        }
        
        // Observer 4: Next reminder info → update UI state only
        viewModelScope.launch {
            combine(
                preferencesRepository.nextReminderTimestamp,
                preferencesRepository.nextReminderTemplateName
            ) { nextReminder, nextReminderName ->
                Pair(nextReminder, nextReminderName)
            }.collect { (nextReminder, nextReminderName) ->
                _uiState.value = _uiState.value.copy(
                    nextReminderTimestamp = nextReminder,
                    nextReminderTemplateName = nextReminderName
                )
            }
        }
        
        // Observer 5: Hourly worker deltas → update UI state only
        viewModelScope.launch {
            preferencesRepository.hourlyWorkerRuns.collect { runs ->
                // Calculate deltas from the runs
                val deltas = if (runs.isEmpty()) {
                    emptyList()
                } else {
                    val deltaList = mutableListOf<Int>()
                    
                    // First, add time since last run (most recent run to now)
                    val timeSinceLastRun = (System.currentTimeMillis() - runs[0]) / (1000 * 60)
                    deltaList.add(timeSinceLastRun.toInt())
                    
                    // Then add deltas between consecutive runs
                    if (runs.size >= 2) {
                        for (i in 0 until runs.size - 1) {
                            val deltaMs = runs[i] - runs[i + 1]  // More recent - older
                            val deltaMinutes = (deltaMs / (1000 * 60)).toInt()
                            deltaList.add(deltaMinutes)
                        }
                    }
                    
                    deltaList
                }
                
                _uiState.value = _uiState.value.copy(
                    hourlyWorkerDeltas = deltas
                )
            }
        }
    }

    fun loadAvailableCalendars() {
        viewModelScope.launch {
            try {
                val calendars = calendarRepository.getAvailableCalendars()
                _availableCalendars.value = calendars
                _uiState.value = _uiState.value.copy(hasCalendarPermission = true)
                
                // Update calendar status
                val status = calendarRepository.getCalendarStatus()
                _calendarStatus.value = status
                
                Log.d("MCL", "MainViewModel: Loaded ${calendars.size} calendars, status=$status")
            } catch (e: SecurityException) {
                _uiState.value = _uiState.value.copy(
                    hasCalendarPermission = false,
                    error = "Calendar permission required"
                )
                _calendarStatus.value = CalendarRepository.CalendarStatus(
                    hasPermissions = false,
                    calendarsFound = 0,
                    errorMessage = "Security exception: ${e.message}",
                    exception = e
                )
            } catch (e: Exception) {
                _calendarStatus.value = CalendarRepository.CalendarStatus(
                    hasPermissions = calendarRepository.hasCalendarPermissions(),
                    calendarsFound = 0,
                    errorMessage = "Error loading calendars: ${e.message}",
                    exception = e
                )
            }
        }
    }
    
    /**
     * Refresh calendar status and available calendars
     */
    fun refreshCalendarStatus() {
        viewModelScope.launch {
            try {
                val status = calendarRepository.getCalendarStatus()
                _calendarStatus.value = status
                
                // Also refresh the calendars list
                if (status.hasPermissions && status.calendarsFound > 0) {
                    loadAvailableCalendars()
                }
                
                Log.d("MCL", "MainViewModel: Refreshed calendar status: $status")
            } catch (e: Exception) {
                _calendarStatus.value = CalendarRepository.CalendarStatus(
                    hasPermissions = calendarRepository.hasCalendarPermissions(),
                    calendarsFound = 0,
                    errorMessage = "Error refreshing calendar status: ${e.message}",
                    exception = e
                )
            }
        }
    }

    private fun loadEvents(calendarId: Long, daysBack: Int) {
        viewModelScope.launch {
            try {
                val events = calendarRepository.getEventsForPeriod(calendarId, daysBack)
                _events.value = events
                updateConcentrationData(_uiState.value.templates, events)
            } catch (e: Exception) {
                _uiState.value = _uiState.value.copy(error = e.message)
            }
        }
    }

    private fun updateConcentrationData(templates: List<Template>, events: List<CalendarEvent>) {
        // Only calculate concentration data for enabled templates, but use all templates for cross-template calculations
        val enabledTemplates = templates.filter { it.enabled }
        Log.d("MCL", "MainViewModel: updateConcentrationData - Processing ${enabledTemplates.size} enabled templates out of ${templates.size} total")
        
        val concentrationData = enabledTemplates.map { template ->
            val isDerived = template.drugs.any { it.decayType == null }
            Log.d("MCL", "MainViewModel: Processing template '${template.name}' (${template.templateType}, derived=$isDerived, drugs=${template.drugs.size})")
            
            val concentrations = if (template.templateType == TemplateType.DECAYING) {
                // For decaying templates, calculate cross-template drug concentrations using ALL templates
                val drugConcentrations = template.drugs.associate { drug ->
                    val crossTemplateConcentration = ConcentrationCalculator.calculateCurrentDrugConcentration(
                        drug.drugType, templates, events  // Use all templates, not just enabled ones
                    )
                    Log.d("MCL", "MainViewModel: Drug '${drug.drugType}' concentration = $crossTemplateConcentration")
                    drug.drugType to crossTemplateConcentration
                }
                Log.d("MCL", "MainViewModel: Template '${template.name}' concentrations map: $drugConcentrations")
                drugConcentrations
            } else {
                // For non-decaying templates, use the original template-only calculation
                val drugConcentrations = ConcentrationCalculator.calculateConcentrationsPerDrug(template, events, Date(), templates)
                Log.d("MCL", "MainViewModel: Non-decaying template '${template.name}' concentrations: $drugConcentrations")
                drugConcentrations
            }
            
            val peakConcentrations = if (template.templateType == TemplateType.DECAYING) {
                // For decaying templates, calculate cross-template peak concentrations using ALL templates
                template.drugs.associate { drug ->
                    // Generate peak calculation across all templates for this drug (not just enabled ones)
                    val history = ConcentrationCalculator.generateDrugConcentrationHistory(
                        drug.drugType, templates, events, _uiState.value.daysBack  // Use all templates
                    )
                    drug.drugType to (history.maxOfOrNull { it.second } ?: 0.0)
                }
            } else {
                ConcentrationCalculator.calculatePeakConcentrationsPerDrug(
                    template, events, _uiState.value.daysBack
                )
            }
            
            val lastEventTime = if (template.templateType == TemplateType.DECAYING) {
                // For decaying templates, get the last event time across all templates containing any of the drugs
                template.drugs.mapNotNull { drug ->
                    ConcentrationCalculator.getLastDrugEventTime(drug.drugType, templates, events)  // Use all templates
                }.maxOrNull()
            } else {
                ConcentrationCalculator.getLastEventTime(template, events)
            }
            
            ConcentrationData(
                templateId = template.id,
                concentrations = concentrations,
                peakConcentrations = peakConcentrations,
                lastEventTime = lastEventTime
            )
        }
        _concentrationData.value = concentrationData
    }

    fun createEventFromTemplate(template: Template, dosage: Double? = null) {
        viewModelScope.launch {
            val calendarId = _uiState.value.selectedCalendarId
            if (calendarId == null) {
                _uiState.value = _uiState.value.copy(error = "No calendar selected")
                return@launch
            }

            try {
                // Use centralized event creation (handles notification cancellation and widget updates)
                calendarRepository.createEventFromTemplate(calendarId, template, dosage)
                
                // Refresh all data immediately to update UI and calculations
                refreshData()
                
                // Trigger UI refresh to update time displays immediately
                triggerUiRefresh()
                
            } catch (e: Exception) {
                _uiState.value = _uiState.value.copy(error = e.message)
            }
        }
    }

    fun setSelectedCalendar(calendarId: Long) {
        viewModelScope.launch {
            preferencesRepository.setSelectedCalendarId(calendarId)
        }
    }

    fun setDaysBack(days: Int) {
        viewModelScope.launch {
            preferencesRepository.setDaysBack(days)
        }
    }

    fun setDebugModeEnabled(enabled: Boolean) {
        viewModelScope.launch {
            preferencesRepository.setDebugModeEnabled(enabled)
        }
    }

    fun addTemplate(template: Template) {
        viewModelScope.launch {
            val templateWithId = template.copy(id = UUID.randomUUID().toString())
            preferencesRepository.addTemplate(templateWithId)
            
            // Update widgets to reflect the new template
            TemplateWidgetProvider.updateAllWidgets(application)
            
            // Recalculate reminders when a new template is added
            ReminderService.recalculateAllReminders(application, "template_added")
        }
    }

    fun updateTemplate(template: Template, recalculateReminders: Boolean = true) {
        viewModelScope.launch {
            preferencesRepository.updateTemplate(template)
            
            // Handle derivance logic for drug templates
            if (template.templateType == TemplateType.DECAYING && template.drugs.isNotEmpty()) {
                val allTemplates = _uiState.value.templates
                
                // Check if this template has any source drugs and update derived templates accordingly
                val sourceDrugs = template.getSourceDrugs()
                
                sourceDrugs.forEach { sourceDrug ->
                    // Find all derived templates that reference this source drug
                    val derivedTemplates = DrugResolution.getDerivedTemplates(template, allTemplates)
                        .filter { derivedTemplate ->
                            derivedTemplate.drugs.any { drug -> 
                                drug.drugType == sourceDrug.drugType && drug.decayType == null
                            }
                        }
                    
                    // Update enabled/visible state for all derived templates
                    derivedTemplates.forEach { derivedTemplate ->
                        val updatedDerived = derivedTemplate.copy(
                            enabled = template.enabled,
                            visible = if (template.enabled) derivedTemplate.visible else false
                        )
                        preferencesRepository.updateTemplate(updatedDerived)
                    }
                }
            }
            
            // Update widgets as threshold changes may affect badge colors
            TemplateWidgetProvider.updateAllWidgets(application)
            
            // Recalculate reminders when template settings change (skip for visibility-only updates)
            if (recalculateReminders) {
                ReminderService.recalculateAllReminders(application, "template_updated")
            }
        }
    }

    fun removeTemplate(templateId: String) {
        viewModelScope.launch {
            preferencesRepository.removeTemplate(templateId)
            
            // Recalculate reminders when a template is removed
            ReminderService.recalculateAllReminders(application, "template_removed")
        }
    }

    // Signal for UI refresh
    private val _refreshSignal = MutableStateFlow(0)
    val refreshSignal: StateFlow<Int> = _refreshSignal.asStateFlow()
    
    /**
     * Trigger UI refresh by incrementing the refresh signal
     */
    fun triggerUiRefresh() {
        _refreshSignal.value += 1
        Log.d("MCL", "MainViewModel: Triggered UI refresh, signal=${_refreshSignal.value}")
    }
    
    /**
     * Refresh data to ensure UI displays current information
     * This is called periodically when the user is idle on the main screen
     */
    fun refreshData() {
        viewModelScope.launch {
            val calendarId = _uiState.value.selectedCalendarId ?: return@launch
            val daysBack = _uiState.value.daysBack
            
            try {
                Log.d("MCL", "MainViewModel: Starting data refresh at ${Date()}")
                
                // Reload events from calendar
                val events = calendarRepository.getEventsForPeriod(calendarId, daysBack)
                Log.d("MCL", "MainViewModel: Loaded ${events.size} events from calendar")
                _events.value = events
                
                // Update concentration data with fresh events
                updateConcentrationData(_uiState.value.templates, events)
                Log.d("MCL", "MainViewModel: Updated concentration data for ${_concentrationData.value.size} templates")
                
                // Update current time for calculations
                _currentTime.value = Date()
                Log.d("MCL", "MainViewModel: Data refresh completed successfully")
            } catch (e: Exception) {
                Log.e("MCL", "MainViewModel: Error refreshing data", e)
                _uiState.value = _uiState.value.copy(error = e.message)
            }
        }
    }

    fun clearError() {
        _uiState.value = _uiState.value.copy(error = null)
    }

    fun getTimeSinceLastEvent(templateId: String): String {
        val concentrationData = _concentrationData.value.find { it.templateId == templateId }
        // Use the current time to ensure time calculations are up-to-date
        val duration = TimeDuration.between(concentrationData?.lastEventTime ?: return "Never", Date())
        return TimeFormatter.format(duration)
    }

    fun getPeakConcentration(templateId: String): Double? {
        val template = _uiState.value.templates.find { it.id == templateId }
        if (template?.templateType != TemplateType.DECAYING) return null
        
        return ConcentrationCalculator.calculatePeakConcentration(
            template, _events.value, _uiState.value.daysBack
        )
    }

    fun getLatestMetricValue(templateId: String): Double? {
        val template = _uiState.value.templates.find { it.id == templateId }
        if (template?.templateType != TemplateType.METRIC) return null
        
        val relevantEvents = _events.value.filter { event ->
            event.title.startsWith(template.name)
        }.sortedByDescending { it.startTime }
        
        return relevantEvents.firstOrNull()?.let { event ->
            extractMetricValue(event.title, template.name)
        }
    }

    private fun extractMetricValue(eventTitle: String, templateName: String): Double? {
        // Extract value from standardized "TemplateNameVALUE" format (no spaces or colons)
        val pattern = "${templateName}([0-9]+\\.?[0-9]*)".toRegex()
        return pattern.find(eventTitle)?.groupValues?.get(1)?.toDoubleOrNull()
    }

    fun getConcentrationHistory(template: Template): List<Pair<Date, Double>> {
        return if (template.templateType == TemplateType.METRIC) {
            generateMetricHistory(template, _events.value, _uiState.value.daysBack)
        } else {
            ConcentrationCalculator.generateConcentrationHistory(
                template, _events.value, _uiState.value.daysBack
            )
        }
    }

    fun toggleDrugVisibility(drugName: String) {
        val currentVisible = _visibleDrugs.value
        val newVisible = if (currentVisible.contains(drugName)) {
            currentVisible - drugName
        } else {
            currentVisible + drugName
        }
        _visibleDrugs.value = newVisible
        
        // Update the persistent template visibility
        updateTemplateVisibility(drugName, newVisible.contains(drugName))
    }

    fun initializeVisibleDrugs(drugNames: List<String>) {
        // Initialize visibility based on template's persistent visible attribute
        val templates = _uiState.value.templates
        val visibleDrugs = drugNames.filter { drugName ->
            // For decaying templates, use drug resolution to find the correct template
            val template = templates.find { template ->
                when (template.templateType) {
                    TemplateType.DECAYING -> {
                        // Use DrugResolution to get all drugs with resolved decay info for this template
                        val resolvedDrugs = DrugResolution.getDrugsWithDecayInfo(template, templates)
                        resolvedDrugs.any { drugWithDecay -> drugWithDecay.drug.drugType == drugName }
                    }
                    else -> template.name == drugName
                }
            }
            template?.visible ?: true // Default to visible if template not found
        }.toSet()
        
        _visibleDrugs.value = visibleDrugs
    }

    private fun updateTemplateVisibility(drugName: String, isVisible: Boolean) {
        viewModelScope.launch {
            val templates = _uiState.value.templates
            
            // Find all templates that contain this drug name
            val templatesToUpdate = templates.filter { template ->
                when (template.templateType) {
                    TemplateType.DECAYING -> {
                        // Use DrugResolution to get all drugs with resolved decay info for this template
                        val resolvedDrugs = DrugResolution.getDrugsWithDecayInfo(template, templates)
                        resolvedDrugs.any { drugWithDecay -> drugWithDecay.drug.drugType == drugName }
                    }
                    else -> template.name == drugName
                }
            }
            
            // Update all matching templates
            templatesToUpdate.forEach { template ->
                val updatedTemplate = template.copy(visible = isVisible)
                // Skip reminder recalculation for visibility-only changes
                updateTemplate(updatedTemplate, recalculateReminders = false)
            }
        }
    }

    fun toggleEventTextVisibility() {
        _showEventText.value = !_showEventText.value
    }

    fun getConcentrationHistoryForTimeRange(
        template: Template, 
        startTime: Date, 
        endTime: Date
    ): List<Pair<Date, Double>> {
        val calendarId = _uiState.value.selectedCalendarId ?: return emptyList()
        
        // Always use the full daysBack period to ensure all contributing events are included
        // This prevents events from "dropping out" when scrolling/zooming
        val fullDaysBack = _uiState.value.daysBack
        
        return try {
            // Get events for the full period to ensure accurate concentration calculation
            val events = calendarRepository.getEventsForPeriod(calendarId, fullDaysBack)
            
            if (template.templateType == TemplateType.METRIC) {
                generateMetricHistoryForTimeRange(template, events, startTime, endTime)
            } else {
                ConcentrationCalculator.generateConcentrationHistoryForTimeRange(
                    template, events, startTime, endTime
                )
            }
        } catch (e: Exception) {
            emptyList()
        }
    }

    fun loadEventsForTimeRange(startTime: Date, endTime: Date): List<CalendarEvent> {
        val calendarId = _uiState.value.selectedCalendarId ?: return emptyList()
        
        // Calculate days back from the start time to now
        val daysFromStart = ((Date().time - startTime.time) / (24 * 60 * 60 * 1000)).toInt() + 1
        
        return try {
            calendarRepository.getEventsForPeriod(calendarId, daysFromStart).filter { event ->
                event.startTime.after(startTime) && event.startTime.before(endTime)
            }
        } catch (e: Exception) {
            emptyList()
        }
    }

    private fun generateMetricHistory(template: Template, events: List<CalendarEvent>, daysBack: Int): List<Pair<Date, Double>> {
        if (template.templateType != TemplateType.METRIC) return emptyList()
        
        val relevantEvents = events.filter { event ->
            event.title.startsWith(template.name)
        }.sortedBy { it.startTime }
        
        if (relevantEvents.isEmpty()) return emptyList()
        
        val history = mutableListOf<Pair<Date, Double>>()
        val currentTime = Date()
        val startTime = Date(currentTime.time - (daysBack * 24 * 60 * 60 * 1000L))
        
        // Extract actual data points
        val dataPoints = relevantEvents.mapNotNull { event ->
            val value = extractMetricValue(event.title, template.name)
            if (value != null) Pair(event.startTime, value) else null
        }.filter { it.first.after(startTime) }
        
        if (dataPoints.isEmpty()) return emptyList()
        
        // Generate smooth continuous line with interpolation
        val totalTimeMillis = currentTime.time - startTime.time
        val timeStepMillis = totalTimeMillis / 500L // 500 data points for smooth curves
        
        var time = startTime.time
        var currentValueIndex = 0
        
        for (i in 0..500) {
            val pointTime = Date(time)
            
            // Find the appropriate value for this time point
            val value = when {
                // Before first data point - use first value
                pointTime.before(dataPoints.first().first) -> dataPoints.first().second
                
                // After last data point - use last value
                pointTime.after(dataPoints.last().first) -> dataPoints.last().second
                
                // Between data points - use step function (hold previous value until next measurement)
                else -> {
                    // Find the most recent measurement before or at this time
                    var mostRecentValue = dataPoints.first().second
                    for (dataPoint in dataPoints) {
                        if (dataPoint.first.time <= time) {
                            mostRecentValue = dataPoint.second
                        } else {
                            break
                        }
                    }
                    mostRecentValue
                }
            }
            
            history.add(Pair(pointTime, value))
            time += timeStepMillis
            
            if (time > currentTime.time) break
        }
        
        return history
    }

    private fun generateMetricHistoryForTimeRange(
        template: Template, 
        events: List<CalendarEvent>, 
        startTime: Date, 
        endTime: Date
    ): List<Pair<Date, Double>> {
        if (template.templateType != TemplateType.METRIC) return emptyList()
        
        val relevantEvents = events.filter { event ->
            event.title.startsWith(template.name)
        }.sortedBy { it.startTime }
        
        if (relevantEvents.isEmpty()) return emptyList()
        
        // Extract actual data points within the time range
        val dataPoints = relevantEvents.mapNotNull { event ->
            val value = extractMetricValue(event.title, template.name)
            if (value != null) Pair(event.startTime, value) else null
        }.filter { it.first.after(startTime) && it.first.before(endTime) }
        
        if (dataPoints.isEmpty()) {
            // If no data points in range, check for the most recent value before startTime
            val lastValueBeforeRange = relevantEvents.mapNotNull { event ->
                val value = extractMetricValue(event.title, template.name)
                if (value != null && event.startTime.before(startTime)) Pair(event.startTime, value) else null
            }.maxByOrNull { it.first }
            
            if (lastValueBeforeRange != null) {
                // Create a flat line with the last known value
                val history = mutableListOf<Pair<Date, Double>>()
                val totalTimeMillis = endTime.time - startTime.time
                val timeStepMillis = totalTimeMillis / 100L // 100 data points for smooth curves
                
                var time = startTime.time
                for (i in 0..100) {
                    history.add(Pair(Date(time), lastValueBeforeRange.second))
                    time += timeStepMillis
                    if (time > endTime.time) break
                }
                return history
            }
            return emptyList()
        }
        
        val history = mutableListOf<Pair<Date, Double>>()
        
        // Generate smooth continuous line with step function
        val totalTimeMillis = endTime.time - startTime.time
        val timeStepMillis = totalTimeMillis / 200L // 200 data points for smooth curves
        
        var time = startTime.time
        
        for (i in 0..200) {
            val pointTime = Date(time)
            
            // Find the appropriate value for this time point using step function
            val value = when {
                // Before first data point - check if there's a value before the range
                pointTime.before(dataPoints.first().first) -> {
                    val lastValueBeforeRange = relevantEvents.mapNotNull { event ->
                        val value = extractMetricValue(event.title, template.name)
                        if (value != null && event.startTime.before(startTime)) Pair(event.startTime, value) else null
                    }.maxByOrNull { it.first }
                    lastValueBeforeRange?.second ?: dataPoints.first().second
                }
                
                // After last data point - use last value
                pointTime.after(dataPoints.last().first) -> dataPoints.last().second
                
                // Between data points - use step function (hold previous value until next measurement)
                else -> {
                    // Find the most recent measurement before or at this time
                    var mostRecentValue = dataPoints.first().second
                    for (dataPoint in dataPoints) {
                        if (dataPoint.first.time <= time) {
                            mostRecentValue = dataPoint.second
                        } else {
                            break
                        }
                    }
                    mostRecentValue
                }
            }
            
            history.add(Pair(pointTime, value))
            time += timeStepMillis
            
            if (time > endTime.time) break
        }
        
        return history
    }

    /**
     * Export current templates to calendar
     */
    fun exportTemplatesToCalendar(): Result<Unit> {
        val calendarId = _uiState.value.selectedCalendarId
        if (calendarId == null) {
            return Result.failure(Exception("No calendar selected"))
        }

        _uiState.value = _uiState.value.copy(exportInProgress = true)
        
        return try {
            val result = configExportImport.exportTemplatesToCalendar(calendarId, _uiState.value.templates)
            _uiState.value = _uiState.value.copy(exportInProgress = false)
            result
        } catch (e: Exception) {
            _uiState.value = _uiState.value.copy(exportInProgress = false)
            Result.failure(e)
        }
    }

    /**
     * Import templates from calendar, replacing current templates
     */
    fun importTemplatesFromCalendar(): Result<List<Template>> {
        val calendarId = _uiState.value.selectedCalendarId
        if (calendarId == null) {
            return Result.failure(Exception("No calendar selected"))
        }

        _uiState.value = _uiState.value.copy(importInProgress = true)
        
        return try {
            val result = configExportImport.importTemplatesFromCalendar(calendarId)
            _uiState.value = _uiState.value.copy(importInProgress = false)
            
            if (result.isSuccess) {
                val importedTemplates = result.getOrNull() ?: emptyList()
                // Replace all current templates with imported ones
                viewModelScope.launch {
                    preferencesRepository.replaceAllTemplates(importedTemplates)
                    
                    // Recalculate reminders after importing templates
                    ReminderService.recalculateAllReminders(application, "templates_imported")
                }
            }
            
            result
        } catch (e: Exception) {
            _uiState.value = _uiState.value.copy(importInProgress = false)
            Result.failure(e)
        }
    }

    /**
     * Check if config exists in calendar
     */
    fun hasConfigInCalendar(): Boolean {
        val calendarId = _uiState.value.selectedCalendarId ?: return false
        return configExportImport.hasConfigInCalendar(calendarId)
    }

    /**
     * Get config metadata from calendar
     */
    fun getConfigMetadata(): Result<de.ntdote.medicalcalendarlog.utils.ConfigMetadata?> {
        val calendarId = _uiState.value.selectedCalendarId
        if (calendarId == null) {
            return Result.failure(Exception("No calendar selected"))
        }
        return configExportImport.getConfigMetadata(calendarId)
    }

    /**
     * Delete config from calendar
     */
    fun deleteConfigFromCalendar(): Result<Unit> {
        val calendarId = _uiState.value.selectedCalendarId
        if (calendarId == null) {
            return Result.failure(Exception("No calendar selected"))
        }
        return configExportImport.deleteConfigFromCalendar(calendarId)
    }

    /**
     * Manually trigger the hourly worker to evaluate notifications
     * (equivalent to the ADB broadcast command)
     */
    fun triggerManualWorkerRun() {
        Log.d("MCL", "MainViewModel: Manually triggering hourly worker")
        HourlyReminderWorker.evaluateNotifications(application)
    }

    /**
     * Check if the app is ignoring battery optimizations (unrestricted background jobs)
     */
    fun isBackgroundUnrestricted(): Boolean {
        val powerManager = application.getSystemService(Context.POWER_SERVICE) as PowerManager
        return powerManager.isIgnoringBatteryOptimizations(application.packageName)
    }

    /**
     * Create an intent to request battery optimization exemption
     */
    fun createBatteryOptimizationIntent(): Intent {
        return Intent(Settings.ACTION_REQUEST_IGNORE_BATTERY_OPTIMIZATIONS).apply {
            data = Uri.parse("package:${application.packageName}")
        }
    }

    /**
     * Get the next scheduled worker run timestamp from WorkManager
     */
    fun getNextWorkerRunTimestamp(): Long {
        return try {
            Log.d("MCL", "MainViewModel: Querying WorkManager for next hourly worker run timestamp")
            val workInfos = WorkManager.getInstance(application)
                .getWorkInfosForUniqueWork("hourly_reminder_worker")
                .get()
            
            Log.d("MCL", "MainViewModel: Found ${workInfos.size} work items for hourly_reminder_worker")
            workInfos.forEach { workInfo ->
                Log.d("MCL", "MainViewModel: WorkInfo - id=${workInfo.id}, state=${workInfo.state}, nextScheduleTime=${workInfo.nextScheduleTimeMillis}")
            }
            
            // Find the earliest next run time among all scheduled work
            val enqueuedWork = workInfos.filter { it.state == WorkInfo.State.ENQUEUED }
            Log.d("MCL", "MainViewModel: Found ${enqueuedWork.size} enqueued hourly worker items")
            
            val scheduledTimes = enqueuedWork.mapNotNull { it.nextScheduleTimeMillis }
            Log.d("MCL", "MainViewModel: Hourly worker scheduled times: $scheduledTimes")
            
            val nextRunTime = scheduledTimes.minOrNull() ?: 0L
            Log.d("MCL", "MainViewModel: Next hourly worker run timestamp: $nextRunTime")
            nextRunTime
        } catch (e: Exception) {
            Log.e("MCL", "MainViewModel: Error getting next hourly worker run timestamp: ${e.message}", e)
            -1L // Use -1 to distinguish from "not scheduled" (0L)
        }
    }

    /**
     * Get the app standby bucket
     */
    fun getAppStandbyBucket(): Int {
        return try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                val usageStatsManager = application.getSystemService(Context.USAGE_STATS_SERVICE) as UsageStatsManager
                usageStatsManager.appStandbyBucket
            } else {
                0 // Unknown/not available on older Android versions
            }
        } catch (e: Exception) {
            Log.w("MCL", "MainViewModel: Could not get app standby bucket", e)
            0
        }
    }

    /**
     * Update debug info (next worker run and app standby bucket)
     */
    fun updateDebugInfo() {
        viewModelScope.launch {
            Log.d("MCL", "MainViewModel: updateDebugInfo() called - querying worker run and standby bucket")
            val nextWorkerRun = getNextWorkerRunTimestamp()
            val standbyBucket = getAppStandbyBucket()
            
            Log.d("MCL", "MainViewModel: updateDebugInfo() - nextWorkerRun=$nextWorkerRun, standbyBucket=$standbyBucket")
            _uiState.value = _uiState.value.copy(
                nextWorkerRunTimestamp = nextWorkerRun,
                appStandbyBucket = standbyBucket
            )
            Log.d("MCL", "MainViewModel: updateDebugInfo() completed - UI state updated")
        }
    }
}
