package de.ntdote.medicalcalendarlog.utils

/**
 * Formats TimeDuration objects into user-friendly strings.
 * This is the single source of truth for all time formatting in the app.
 */
object TimeFormatter {
    
    /**
     * Format a time duration for display.
     * 
     * @param duration The duration to format
     * @param compact If true, uses compact format (e.g., "3d12" instead of "3d 12h")
     * @param widgetMode If true, never shows minutes - minimum is 0h (for widget display)
     * @return Formatted string representation
     * 
     * Format rules:
     * - Minutes: "45m" (when < 60 minutes, unless widgetMode is true)
     * - Hours: "12h" (when < 24 hours)
     * - Days with hours: "3d 12h" or "3d12" (compact)
     * - Days only: "7d" (when days > 10)
     * - Never: "Never" (when duration is NEVER)
     * - Widget mode: Never shows minutes, minimum is "0h"
     */
    fun format(duration: TimeDuration, compact: Boolean = false, widgetMode: Boolean = false): String {
        // Handle special cases
        if (duration.isNever()) return "Never"
        
        val days = duration.days
        val hours = duration.hours
        val minutes = duration.minutes
        
        return when {
            // Widget mode: never show minutes, minimum is 0h
            widgetMode && duration.totalHours < 24 -> {
                if (hours == 0) "0h" else "${hours}h"
            }
            
            // Less than 60 minutes: show minutes (unless widget mode)
            duration.totalMinutes < 60 -> {
                if (widgetMode) "0h" else "${minutes}m"
            }
            
            // Less than 24 hours: show hours
            duration.totalHours < 24 -> "${hours}h"
            
            // 2-9 days: show days and hours
            days < 10 -> {
                if (compact) {
                    "${days}d${hours}"
                } else {
                    "${days}d ${hours}h"
                }
            }
            
            // 10+ days in compact mode: show only days
            // 10+ days in normal mode: show days and hours
            else -> {
                if (compact) {
                    "${days}d"
                } else {
                    "${days}d ${hours}h"
                }
            }
        }
    }
    
    /**
     * Format a time duration with precise detail (for debugging).
     * Always shows days, hours, minutes, and seconds.
     * 
     * @param duration The duration to format
     * @return Formatted string with full precision (e.g., "3d 12h 45m 30s")
     */
    fun formatPrecise(duration: TimeDuration): String {
        if (duration.isNever()) return "Never"
        
        val days = duration.days
        val hours = duration.hours
        val minutes = duration.minutes
        val seconds = duration.seconds
        
        return when {
            days > 0 -> "${days}d ${hours}h ${minutes}m ${seconds}s"
            hours > 0 -> "${hours}h ${minutes}m ${seconds}s"
            minutes > 0 -> "${minutes}m ${seconds}s"
            else -> "${seconds}s"
        }
    }
    
    /**
     * Check if a duration exceeds a threshold (for reminder/overdue checks).
     * 
     * @param duration The duration to check
     * @param thresholdHours The threshold in hours
     * @return true if the duration is >= threshold, or if duration is NEVER
     */
    fun isOverdue(duration: TimeDuration, thresholdHours: Int): Boolean {
        if (duration.isNever()) return true
        return duration.totalHours >= thresholdHours
    }
}
