package de.ntdote.medicalcalendarlog.utils

import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.os.Build
import androidx.core.app.NotificationCompat
import androidx.core.app.RemoteInput
import de.ntdote.medicalcalendarlog.MainActivity
import de.ntdote.medicalcalendarlog.R
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.service.NotificationActionReceiver

/**
 * Unified notification builder for all reminder notifications
 * Ensures consistent formatting and behavior across initial and update notifications
 */
object NotificationBuilder {
    
    private const val CHANNEL_ID_INITIAL = "medication_reminders_initial"
    private const val CHANNEL_ID_UPDATES = "medication_reminders_updates"
    
    /**
     * Create notification channels (required for Android O and above)
     */
    fun createNotificationChannels(notificationManager: NotificationManager) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            // Create "Initial Reminders" channel
            val initialChannel = NotificationChannel(
                CHANNEL_ID_INITIAL,
                "Initial Reminders",
                NotificationManager.IMPORTANCE_DEFAULT
            ).apply {
                description = "First notification when it's time to log medication or health metrics"
            }
            notificationManager.createNotificationChannel(initialChannel)
            
            // Create "Reminder Updates" channel
            val updatesChannel = NotificationChannel(
                CHANNEL_ID_UPDATES,
                "Reminder Updates",
                NotificationManager.IMPORTANCE_DEFAULT
            ).apply {
                description = "Hourly updates for existing reminders"
            }
            notificationManager.createNotificationChannel(updatesChannel)
        }
    }
    
    /**
     * Build a time-based notification
     * 
     * @param context Application context
     * @param template The template this notification is for
     * @param drugName Display name for the drug/template
     * @param hoursOverdue How many hours overdue this is (0 if just due)
     * @param isOverlooked Whether this is an overlooked notification
     * @param hoursSinceOriginalNotification Hours since first notification (0 for initial)
     * @return Notification and notification ID
     */
    fun buildTimeBasedNotification(
        context: Context,
        template: Template,
        drugName: String,
        hoursOverdue: Long = 0,
        isOverlooked: Boolean = false,
        hoursSinceOriginalNotification: Long = 0
    ): Pair<android.app.Notification, Int> {
        
        val notificationId = template.id.hashCode()
        
        val intent = Intent(context, MainActivity::class.java).apply {
            flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
        }
        
        val pendingIntent = PendingIntent.getActivity(
            context, 
            notificationId,
            intent, 
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
        
        // Create streamlined notification content
        val title = when {
            isOverlooked -> "Overlooked: $drugName"
            hoursSinceOriginalNotification >= 2 -> "${hoursSinceOriginalNotification}h Since First Alert: $drugName"
            else -> "Reminder: $drugName"
        }
        
        // Simplified content text
        val contentText = "Time to log your $drugName" + if (hoursOverdue > 0) " (${hoursOverdue}h overdue)" else ""
        
        // Create a more concise message for the big text style
        val thresholdMillis = template.getEffectiveReminderThresholdMillis()!!
        val thresholdHours = thresholdMillis / (60 * 60 * 1000.0)
        val totalHours = thresholdHours + hoursOverdue
        
        val bigText = buildString {
            if (thresholdHours >= 1.0) {
                append("It's been ${totalHours.toInt()} hours since your last entry. ")
            } else {
                val totalMinutes = (totalHours * 60).toInt()
                append("It's been $totalMinutes minutes since your last entry. ")
            }
            
            if (hoursOverdue > 0 && hoursSinceOriginalNotification >= 2) {
                append("First alert was ${hoursSinceOriginalNotification}h ago.")
            } else if (hoursOverdue > 0) {
                append("This reminder is ${hoursOverdue}h overdue.")
            }
        }
        
        // Set priority based on how overdue it is or how long since original notification
        val priority = if (hoursOverdue > 2 || hoursSinceOriginalNotification >= 2) {
            NotificationCompat.PRIORITY_HIGH
        } else {
            NotificationCompat.PRIORITY_DEFAULT
        }
        
        // Choose channel based on whether this is the first notification or an update
        val channelId = if (hoursSinceOriginalNotification == 0L) {
            CHANNEL_ID_INITIAL // First notification - use initial channel
        } else {
            CHANNEL_ID_UPDATES // Subsequent update - use updates channel
        }
        
        // Create mute action buttons
        val mute24hIntent = Intent(context, NotificationActionReceiver::class.java).apply {
            action = NotificationActionReceiver.ACTION_MUTE_24H
            putExtra(NotificationActionReceiver.EXTRA_TEMPLATE_ID, template.id)
            putExtra(NotificationActionReceiver.EXTRA_NOTIFICATION_ID, notificationId)
            putExtra(NotificationActionReceiver.EXTRA_IS_CONCENTRATION, false)
        }
        val mute24hPendingIntent = PendingIntent.getBroadcast(
            context,
            (template.id + "_mute24h").hashCode(),
            mute24hIntent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
        
        val muteRelogIntent = Intent(context, NotificationActionReceiver::class.java).apply {
            action = NotificationActionReceiver.ACTION_MUTE_UNTIL_RELOG
            putExtra(NotificationActionReceiver.EXTRA_TEMPLATE_ID, template.id)
            putExtra(NotificationActionReceiver.EXTRA_NOTIFICATION_ID, notificationId)
            putExtra(NotificationActionReceiver.EXTRA_IS_CONCENTRATION, false)
        }
        val muteRelogPendingIntent = PendingIntent.getBroadcast(
            context,
            (template.id + "_muterelog").hashCode(),
            muteRelogIntent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
        
        // Create Log action with RemoteInput for templates requiring input
        val logIntent = Intent(context, NotificationActionReceiver::class.java).apply {
            action = NotificationActionReceiver.ACTION_LOG
            putExtra(NotificationActionReceiver.EXTRA_TEMPLATE_ID, template.id)
            putExtra(NotificationActionReceiver.EXTRA_TEMPLATE_NAME, drugName)
            putExtra(NotificationActionReceiver.EXTRA_NOTIFICATION_ID, notificationId)
            putExtra(NotificationActionReceiver.EXTRA_IS_CONCENTRATION, false)
            putExtra(NotificationActionReceiver.EXTRA_REQUIRES_INPUT, template.requiresInput)
        }
        val logPendingIntent = PendingIntent.getBroadcast(
            context,
            (template.id + "_log").hashCode(),
            logIntent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_MUTABLE // Must be mutable for RemoteInput
        )
        
        val logActionBuilder = NotificationCompat.Action.Builder(
            R.drawable.ic_launcher_foreground,
            "Log",
            logPendingIntent
        )
        
        // Add RemoteInput for templates requiring input
        if (template.requiresInput) {
            val remoteInput = RemoteInput.Builder(NotificationActionReceiver.KEY_TEXT_REPLY)
                .setLabel("Enter value")
                .build()
            logActionBuilder.addRemoteInput(remoteInput)
        }
        
        val logAction = logActionBuilder.build()
        
        val notification = NotificationCompat.Builder(context, channelId)
            .setSmallIcon(R.drawable.ic_launcher_foreground)
            .setContentTitle(title)
            .setContentText(contentText)
            .setStyle(NotificationCompat.BigTextStyle().bigText(bigText))
            .setPriority(priority)
            .setContentIntent(pendingIntent)
            .setAutoCancel(true) // Auto-dismiss when user taps the notification
            .setOnlyAlertOnce(true) // Only alert on the first notification, not on updates
            .addAction(logAction)
            .addAction(R.drawable.ic_launcher_foreground, "Mute 24h", mute24hPendingIntent)
            .addAction(R.drawable.ic_launcher_foreground, "Mute this time", muteRelogPendingIntent)
            .build()
        
        return Pair(notification, notificationId)
    }
    
    /**
     * Build a concentration-based notification
     * 
     * @param context Application context
     * @param template The template this notification is for
     * @param drugName Display name for the drug
     * @param currentConcentration Current concentration value
     * @param hoursSinceOriginalNotification Hours since first notification (0 for initial)
     * @return Notification and notification ID
     */
    fun buildConcentrationNotification(
        context: Context,
        template: Template,
        drugName: String,
        currentConcentration: Double,
        hoursSinceOriginalNotification: Long = 0
    ): Pair<android.app.Notification, Int> {
        
        val notificationId = (template.id + "_concentration").hashCode()
        
        val intent = Intent(context, MainActivity::class.java).apply {
            flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
        }
        
        val pendingIntent = PendingIntent.getActivity(
            context, 
            notificationId,
            intent, 
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
        
        val threshold = template.concentrationReminderThreshold ?: 0.0
        // Get unit from the first drug, or fallback to metric unit for non-drug templates
        val unit = when {
            template.drugs.isNotEmpty() -> template.drugs[0].unit
            else -> template.metricUnit
        }
        
        // Streamlined notification content
        val title = if (hoursSinceOriginalNotification >= 2) {
            "${hoursSinceOriginalNotification}h Since First Alert: Low $drugName"
        } else {
            "Low Concentration: $drugName"
        }
        
        // Create a more concise message for the big text style
        val bigText = buildString {
            append("${String.format("%.2f", currentConcentration)} $unit (below threshold of $threshold $unit)")
            
            if (hoursSinceOriginalNotification >= 2) {
                append("\nFirst alert was ${hoursSinceOriginalNotification}h ago.")
            }
        }
        
        // Set priority based on how long since original notification
        val priority = if (hoursSinceOriginalNotification >= 2) {
            NotificationCompat.PRIORITY_HIGH
        } else {
            NotificationCompat.PRIORITY_DEFAULT
        }
        
        // Choose channel based on whether this is the first notification or an update
        val channelId = if (hoursSinceOriginalNotification == 0L) {
            CHANNEL_ID_INITIAL // First notification - use initial channel
        } else {
            CHANNEL_ID_UPDATES // Subsequent update - use updates channel
        }
        
        // Create mute action buttons
        val mute24hIntent = Intent(context, NotificationActionReceiver::class.java).apply {
            action = NotificationActionReceiver.ACTION_MUTE_24H
            putExtra(NotificationActionReceiver.EXTRA_TEMPLATE_ID, template.id)
            putExtra(NotificationActionReceiver.EXTRA_NOTIFICATION_ID, notificationId)
            putExtra(NotificationActionReceiver.EXTRA_IS_CONCENTRATION, true)
        }
        val mute24hPendingIntent = PendingIntent.getBroadcast(
            context,
            (template.id + "_concentration_mute24h").hashCode(),
            mute24hIntent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
        
        val muteRelogIntent = Intent(context, NotificationActionReceiver::class.java).apply {
            action = NotificationActionReceiver.ACTION_MUTE_UNTIL_RELOG
            putExtra(NotificationActionReceiver.EXTRA_TEMPLATE_ID, template.id)
            putExtra(NotificationActionReceiver.EXTRA_NOTIFICATION_ID, notificationId)
            putExtra(NotificationActionReceiver.EXTRA_IS_CONCENTRATION, true)
        }
        val muteRelogPendingIntent = PendingIntent.getBroadcast(
            context,
            (template.id + "_concentration_muterelog").hashCode(),
            muteRelogIntent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
        )
        
        // Create Log action with RemoteInput for templates requiring input
        val logIntent = Intent(context, NotificationActionReceiver::class.java).apply {
            action = NotificationActionReceiver.ACTION_LOG
            putExtra(NotificationActionReceiver.EXTRA_TEMPLATE_ID, template.id)
            putExtra(NotificationActionReceiver.EXTRA_TEMPLATE_NAME, drugName)
            putExtra(NotificationActionReceiver.EXTRA_NOTIFICATION_ID, notificationId)
            putExtra(NotificationActionReceiver.EXTRA_IS_CONCENTRATION, true)
            putExtra(NotificationActionReceiver.EXTRA_REQUIRES_INPUT, template.requiresInput)
        }
        val logPendingIntent = PendingIntent.getBroadcast(
            context,
            (template.id + "_concentration_log").hashCode(),
            logIntent,
            PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_MUTABLE // Must be mutable for RemoteInput
        )
        
        val logActionBuilder = NotificationCompat.Action.Builder(
            R.drawable.ic_launcher_foreground,
            "Log",
            logPendingIntent
        )
        
        // Add RemoteInput for templates requiring input
        if (template.requiresInput) {
            val remoteInput = RemoteInput.Builder(NotificationActionReceiver.KEY_TEXT_REPLY)
                .setLabel("Enter value")
                .build()
            logActionBuilder.addRemoteInput(remoteInput)
        }
        
        val logAction = logActionBuilder.build()
        
        val notification = NotificationCompat.Builder(context, channelId)
            .setSmallIcon(R.drawable.ic_launcher_foreground)
            .setContentTitle(title)
            .setContentText("Current: ${String.format("%.2f", currentConcentration)} $unit (Threshold: $threshold $unit)")
            .setStyle(NotificationCompat.BigTextStyle().bigText(bigText))
            .setPriority(priority)
            .setContentIntent(pendingIntent)
            .setAutoCancel(true) // Auto-dismiss when user taps the notification
            .setOnlyAlertOnce(true) // Only alert on the first notification, not on updates
            .addAction(logAction)
            .addAction(R.drawable.ic_launcher_foreground, "Mute 24h", mute24hPendingIntent)
            .addAction(R.drawable.ic_launcher_foreground, "Mute this time", muteRelogPendingIntent)
            .build()
        
        return Pair(notification, notificationId)
    }
    
    /**
     * Store notification metadata in SharedPreferences
     */
    fun storeNotificationMetadata(
        context: Context,
        notificationId: Int,
        templateId: String,
        isConcentration: Boolean,
        originalTime: Long = System.currentTimeMillis()
    ) {
        val prefs = context.getSharedPreferences("reminder_prefs", Context.MODE_PRIVATE)
        val key = if (isConcentration) {
            "notification_id_${templateId}_concentration"
        } else {
            "notification_id_${templateId}"
        }
        
        val originalTimeKey = if (isConcentration) {
            "original_notification_time_${templateId}_concentration"
        } else {
            "original_notification_time_${templateId}"
        }
        
        prefs.edit()
            .putInt(key, notificationId)
            .putLong(originalTimeKey, originalTime)
            .apply()
    }
}
