package de.ntdote.medicalcalendarlog.utils

import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Paint
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.graphics.drawscope.Stroke
import androidx.compose.ui.graphics.nativeCanvas
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.data.TemplateType
import java.text.SimpleDateFormat
import java.util.*
import kotlin.math.*

object GraphUtils {
    
    data class GraphData(
        val template: Template,
        val history: List<Pair<Date, Double>>,
        val color: Color,
        val unit: String
    )
    
    data class AxisInfo(
        val minValue: Double,
        val maxValue: Double,
        val unit: String,
        val color: Color
    )
    
    fun getTimeAxisLabels(startTime: Date, endTime: Date, daysBack: Int): List<Pair<Date, String>> {
        val labels = mutableListOf<Pair<Date, String>>()
        val calendar = Calendar.getInstance()
        
        when {
            daysBack > 30 -> {
                // Long ranges: Monday 00:00 markers
                calendar.time = startTime
                calendar.set(Calendar.HOUR_OF_DAY, 0)
                calendar.set(Calendar.MINUTE, 0)
                calendar.set(Calendar.SECOND, 0)
                calendar.set(Calendar.MILLISECOND, 0)
                
                // Find next Monday
                while (calendar.get(Calendar.DAY_OF_WEEK) != Calendar.MONDAY) {
                    calendar.add(Calendar.DAY_OF_YEAR, 1)
                }
                
                val format = SimpleDateFormat("MMM dd", Locale.getDefault())
                while (calendar.time.before(endTime)) {
                    labels.add(Pair(calendar.time, format.format(calendar.time)))
                    calendar.add(Calendar.WEEK_OF_YEAR, 1)
                }
            }
            daysBack > 7 -> {
                // Medium ranges: Daily 00:00 markers
                calendar.time = startTime
                calendar.set(Calendar.HOUR_OF_DAY, 0)
                calendar.set(Calendar.MINUTE, 0)
                calendar.set(Calendar.SECOND, 0)
                calendar.set(Calendar.MILLISECOND, 0)
                calendar.add(Calendar.DAY_OF_YEAR, 1)
                
                val format = SimpleDateFormat("MMM dd", Locale.getDefault())
                while (calendar.time.before(endTime)) {
                    labels.add(Pair(calendar.time, format.format(calendar.time)))
                    calendar.add(Calendar.DAY_OF_YEAR, 1)
                }
            }
            else -> {
                // Short ranges: Hourly markers
                calendar.time = startTime
                calendar.set(Calendar.MINUTE, 0)
                calendar.set(Calendar.SECOND, 0)
                calendar.set(Calendar.MILLISECOND, 0)
                calendar.add(Calendar.HOUR_OF_DAY, 1)
                
                val format = SimpleDateFormat("HH:mm", Locale.getDefault())
                while (calendar.time.before(endTime)) {
                    labels.add(Pair(calendar.time, format.format(calendar.time)))
                    calendar.add(Calendar.HOUR_OF_DAY, if (daysBack <= 1) 2 else 6)
                }
            }
        }
        
        return labels
    }
    
    fun getYAxisLabels(minValue: Double, maxValue: Double): List<Double> {
        val range = maxValue - minValue
        val stepCount = 5
        val rawStep = range / stepCount
        
        // Round step to nice number
        val magnitude = 10.0.pow(floor(log10(rawStep)))
        val normalizedStep = rawStep / magnitude
        
        val niceStep = when {
            normalizedStep <= 1.0 -> 1.0
            normalizedStep <= 2.0 -> 2.0
            normalizedStep <= 5.0 -> 5.0
            else -> 10.0
        } * magnitude
        
        val labels = mutableListOf<Double>()
        var value = ceil(minValue / niceStep) * niceStep
        
        while (value <= maxValue) {
            labels.add(value)
            value += niceStep
        }
        
        return labels
    }
    
    data class ZoomPanState(
        val scale: Float = 1f,
        val offsetX: Float = 0f,
        val offsetY: Float = 0f
    )
    
    fun calculateVisibleTimeRange(
        originalStartTime: Date,
        originalEndTime: Date,
        zoomPanState: ZoomPanState,
        canvasWidth: Float
    ): Pair<Date, Date> {
        val originalTimeRange = originalEndTime.time - originalStartTime.time
        
        // Calculate visible time range based on zoom and pan
        val visibleTimeRange = (originalTimeRange / zoomPanState.scale).toLong()
        
        // Calculate time offset based on pan
        val timeOffsetRatio = -zoomPanState.offsetX / (canvasWidth * zoomPanState.scale)
        val timeOffset = (originalTimeRange * timeOffsetRatio).toLong()
        
        val visibleStartTime = Date(originalStartTime.time + timeOffset)
        val visibleEndTime = Date(visibleStartTime.time + visibleTimeRange)
        
        return Pair(visibleStartTime, visibleEndTime)
    }
    
    fun DrawScope.drawCombinedGraph(
        graphDataList: List<GraphData>,
        padding: Float = 60f
    ) {
        drawCombinedGraphWithZoom(graphDataList, ZoomPanState(), padding)
    }
    
    fun DrawScope.drawCombinedGraphWithEvents(
        graphDataList: List<GraphData>,
        events: List<de.ntdote.medicalcalendarlog.data.CalendarEvent>,
        templates: List<de.ntdote.medicalcalendarlog.data.Template>,
        visibleItems: Set<String> = emptySet(),
        showEventText: Boolean = true,
        padding: Float = 60f
    ) {
        drawCombinedGraphWithZoomAndEvents(graphDataList, events, templates, visibleItems, showEventText, ZoomPanState(), padding)
    }
    
    fun DrawScope.drawCombinedGraphWithZoomAndEvents(
        graphDataList: List<GraphData>,
        events: List<de.ntdote.medicalcalendarlog.data.CalendarEvent>,
        templates: List<de.ntdote.medicalcalendarlog.data.Template>,
        visibleItems: Set<String> = emptySet(),
        showEventText: Boolean = true,
        zoomPanState: ZoomPanState,
        padding: Float = 60f
    ) {
        if (graphDataList.isEmpty()) return
        
        val graphWidth = size.width - 2 * padding
        val graphHeight = size.height - 2 * padding
        
        // Use the provided graphDataList directly (it's already filtered by the caller)
        val visibleGraphData = graphDataList
        
        // Find overall time range from visible data only
        val allHistory = visibleGraphData.flatMap { it.history }
        if (allHistory.isEmpty()) return
        
        val minTime = allHistory.minOf { it.first.time }
        val maxTime = allHistory.maxOf { it.first.time }
        val timeRange = maxTime - minTime
        
        // Create individual axis info for each drug with distinct scaling
        val axisInfos = visibleGraphData.map { graphData ->
            val drugValues = graphData.history.map { it.second }
            val minValue = 0.0  // Always start Y-axis at 0
            val maxValue = if (drugValues.isNotEmpty()) drugValues.maxOf { it } * 1.05 else 1.0
            
            AxisInfo(
                minValue = minValue,
                maxValue = maxValue,
                unit = graphData.unit,
                color = graphData.color
            )
        }
        
        // Draw background grid using the first axis for reference (time grid only)
        if (axisInfos.isNotEmpty()) {
            drawTimeGrid(padding, graphWidth, graphHeight, minTime, maxTime, timeRange)
        }
        
        // Draw each visible drug line using its individual scaling
        visibleGraphData.forEachIndexed { index, graphData ->
            val axisInfo = axisInfos[index]
            drawConcentrationLine(
                graphData.history,
                graphData.color,
                padding,
                graphWidth,
                graphHeight,
                minTime,
                timeRange,
                axisInfo.minValue,
                axisInfo.maxValue
            )
        }
        
        // Draw event markers using individual scaling for positioning
        drawEventMarkersWithIndividualScaling(
            events,
            templates,
            visibleItems,
            showEventText,
            padding,
            graphWidth,
            graphHeight,
            minTime,
            timeRange,
            visibleGraphData,
            axisInfos
        )
        
        // Draw axes
        drawAxes(padding, graphWidth, graphHeight)
        
        // Draw axis labels with individual axes
        drawAxisLabels(
            padding,
            graphWidth,
            graphHeight,
            Date(minTime),
            Date(maxTime),
            axisInfos
        )
    }
    
    fun DrawScope.drawCombinedGraphWithZoom(
        graphDataList: List<GraphData>,
        zoomPanState: ZoomPanState,
        padding: Float = 60f
    ) {
        if (graphDataList.isEmpty()) return
        
        val graphWidth = size.width - 2 * padding
        val graphHeight = size.height - 2 * padding
        
        // Find overall time range
        val allHistory = graphDataList.flatMap { it.history }
        if (allHistory.isEmpty()) return
        
        val minTime = allHistory.minOf { it.first.time }
        val maxTime = allHistory.maxOf { it.first.time }
        val timeRange = maxTime - minTime
        
        // Calculate unified Y-axis range for all drugs (truly unified scaling)
        val allValues = graphDataList.flatMap { it.history.map { point -> point.second } }
        val globalMinValue = 0.0  // Always start Y-axis at 0
        val globalMaxValue = if (allValues.isNotEmpty()) allValues.maxOf { it } * 1.05 else 1.0
        
        // Create single unified axis info for all drugs
        val unifiedAxisInfo = AxisInfo(globalMinValue, globalMaxValue, "concentration", Color.White)
        
        // Draw background grid using unified axis
        drawGrid(padding, graphWidth, graphHeight, minTime, maxTime, timeRange, unifiedAxisInfo)
        
        // Draw each drug line using the same unified scaling
        graphDataList.forEach { graphData ->
            drawConcentrationLine(
                graphData.history,
                graphData.color,
                padding,
                graphWidth,
                graphHeight,
                minTime,
                timeRange,
                globalMinValue,
                globalMaxValue
            )
        }
        
        // Draw axes
        drawAxes(padding, graphWidth, graphHeight)
        
        // Draw axis labels with unified axis
        drawAxisLabels(
            padding,
            graphWidth,
            graphHeight,
            Date(minTime),
            Date(maxTime),
            listOf(unifiedAxisInfo)
        )
    }
    
    private fun DrawScope.drawGrid(
        padding: Float,
        graphWidth: Float,
        graphHeight: Float,
        minTime: Long,
        maxTime: Long,
        timeRange: Long,
        axisInfo: AxisInfo
    ) {
        val gridColor = Color.Gray.copy(alpha = 0.3f)
        
        // Vertical grid lines (time)
        val timeLabels = getTimeAxisLabels(Date(minTime), Date(maxTime), ((maxTime - minTime) / (24 * 60 * 60 * 1000)).toInt())
        timeLabels.forEach { (time, _) ->
            val x = padding + ((time.time - minTime).toFloat() / timeRange.toFloat()) * graphWidth
            drawLine(
                color = gridColor,
                start = Offset(x, padding),
                end = Offset(x, size.height - padding),
                strokeWidth = 1f
            )
        }
        
        // Horizontal grid lines (concentration)
        val yLabels = getYAxisLabels(axisInfo.minValue, axisInfo.maxValue)
        yLabels.forEach { value ->
            val y = size.height - padding - ((value - axisInfo.minValue).toFloat() / (axisInfo.maxValue - axisInfo.minValue).toFloat()) * graphHeight
            drawLine(
                color = gridColor,
                start = Offset(padding, y),
                end = Offset(size.width - padding, y),
                strokeWidth = 1f
            )
        }
    }
    
    private fun DrawScope.drawTimeGrid(
        padding: Float,
        graphWidth: Float,
        graphHeight: Float,
        minTime: Long,
        maxTime: Long,
        timeRange: Long
    ) {
        val gridColor = Color.Gray.copy(alpha = 0.3f)
        
        // Only draw vertical grid lines (time) - no horizontal lines for individual scaling
        val timeLabels = getTimeAxisLabels(Date(minTime), Date(maxTime), ((maxTime - minTime) / (24 * 60 * 60 * 1000)).toInt())
        timeLabels.forEach { (time, _) ->
            val x = padding + ((time.time - minTime).toFloat() / timeRange.toFloat()) * graphWidth
            drawLine(
                color = gridColor,
                start = Offset(x, padding),
                end = Offset(x, size.height - padding),
                strokeWidth = 1f
            )
        }
        
        // Draw midnight lines
        drawMidnightLines(padding, graphWidth, graphHeight, minTime, maxTime, timeRange)
        
        // Draw current time line
        drawCurrentTimeLine(padding, graphWidth, graphHeight, minTime, maxTime, timeRange)
    }
    
    private fun DrawScope.drawMidnightLines(
        padding: Float,
        graphWidth: Float,
        graphHeight: Float,
        minTime: Long,
        maxTime: Long,
        timeRange: Long
    ) {
        val midnightColor = Color.Blue.copy(alpha = 0.4f)
        val calendar = Calendar.getInstance()
        
        // Find all midnight times within the visible range
        calendar.timeInMillis = minTime
        calendar.set(Calendar.HOUR_OF_DAY, 0)
        calendar.set(Calendar.MINUTE, 0)
        calendar.set(Calendar.SECOND, 0)
        calendar.set(Calendar.MILLISECOND, 0)
        
        // Start from the next midnight after minTime
        calendar.add(Calendar.DAY_OF_YEAR, 1)
        
        while (calendar.timeInMillis <= maxTime) {
            val x = padding + ((calendar.timeInMillis - minTime).toFloat() / timeRange.toFloat()) * graphWidth
            
            // Draw midnight line
            drawLine(
                color = midnightColor,
                start = Offset(x, padding),
                end = Offset(x, size.height - padding),
                strokeWidth = 2f
            )
            
            calendar.add(Calendar.DAY_OF_YEAR, 1)
        }
    }
    
    private fun DrawScope.drawCurrentTimeLine(
        padding: Float,
        graphWidth: Float,
        graphHeight: Float,
        minTime: Long,
        maxTime: Long,
        timeRange: Long
    ) {
        val currentTime = System.currentTimeMillis()
        
        // Only draw if current time is within the visible range
        if (currentTime >= minTime && currentTime <= maxTime) {
            val x = padding + ((currentTime - minTime).toFloat() / timeRange.toFloat()) * graphWidth
            val currentTimeColor = Color.Red.copy(alpha = 0.8f)
            
            // Draw current time line
            drawLine(
                color = currentTimeColor,
                start = Offset(x, padding),
                end = Offset(x, size.height - padding),
                strokeWidth = 3f
            )
            
            // Draw "NOW" label
            val paint = Paint().asFrameworkPaint().apply {
                isAntiAlias = true
                textSize = 32f
                color = android.graphics.Color.RED
                isFakeBoldText = true
            }
            
            drawContext.canvas.nativeCanvas.drawText(
                "NOW",
                x - paint.measureText("NOW") / 2,
                padding - 10f,
                paint
            )
        }
    }
    
    private fun DrawScope.drawConcentrationLine(
        history: List<Pair<Date, Double>>,
        color: Color,
        padding: Float,
        graphWidth: Float,
        graphHeight: Float,
        minTime: Long,
        timeRange: Long,
        minConcentration: Double,
        maxConcentration: Double
    ) {
        if (history.size < 2) return
        
        val concentrationRange = maxConcentration - minConcentration
        
        for (i in 0 until history.size - 1) {
            val (time1, concentration1) = history[i]
            val (time2, concentration2) = history[i + 1]
            
            val x1 = padding + ((time1.time - minTime).toFloat() / timeRange.toFloat()) * graphWidth
            val y1 = size.height - padding - ((concentration1 - minConcentration).toFloat() / concentrationRange.toFloat()) * graphHeight
            
            val x2 = padding + ((time2.time - minTime).toFloat() / timeRange.toFloat()) * graphWidth
            val y2 = size.height - padding - ((concentration2 - minConcentration).toFloat() / concentrationRange.toFloat()) * graphHeight
            
            drawLine(
                color = color,
                start = Offset(x1, y1),
                end = Offset(x2, y2),
                strokeWidth = 2f
            )
        }
        
        // Draw data points
        history.forEach { (time, concentration) ->
            val x = padding + ((time.time - minTime).toFloat() / timeRange.toFloat()) * graphWidth
            val y = size.height - padding - ((concentration - minConcentration).toFloat() / concentrationRange.toFloat()) * graphHeight
            
            drawCircle(
                color = color,
                radius = 3f,
                center = Offset(x, y)
            )
        }
    }
    
    private fun DrawScope.drawEventMarkers(
        events: List<de.ntdote.medicalcalendarlog.data.CalendarEvent>,
        templates: List<de.ntdote.medicalcalendarlog.data.Template>,
        visibleItems: Set<String>,
        showEventText: Boolean,
        padding: Float,
        graphWidth: Float,
        graphHeight: Float,
        minTime: Long,
        timeRange: Long,
        globalMinValue: Double = 0.0,
        globalMaxValue: Double = 1.0
    ) {
        val paint = Paint().asFrameworkPaint().apply {
            isAntiAlias = true
            textSize = 24f
            color = android.graphics.Color.WHITE
        }
        
        // Filter events that are within the visible time range
        val visibleEvents = events.filter { event ->
            event.startTime.time >= minTime && event.startTime.time <= minTime + timeRange
        }
        
        visibleEvents.forEach { event ->
            // Get matching template for this event using unified matching logic
            val matchingTemplate = templates.find { template ->
                EventMatching.matchesTemplate(event, template)
            }
            
            // Only show events that match configured templates
            val shouldShowEvent = if (matchingTemplate != null) {
                val itemName = if (matchingTemplate.templateType == TemplateType.DECAYING) {
                    matchingTemplate.extendedData?.typeOfDrug ?: matchingTemplate.name
                } else {
                    matchingTemplate.name
                }
                visibleItems.isEmpty() || visibleItems.contains(itemName)
            } else {
                // Skip events that don't match any configured template
                false
            }
            
            if (!shouldShowEvent) return@forEach
            
            val x = padding + ((event.startTime.time - minTime).toFloat() / timeRange.toFloat()) * graphWidth
            
            // Show the actual event title (which already contains the dosage appended)
            val eventText = event.title
            
            // Calculate Y position based on concentration at event time using unified scaling
            val y = if (matchingTemplate?.templateType == TemplateType.DECAYING || matchingTemplate?.templateType == TemplateType.METRIC) {
                // Calculate peak concentration immediately after this event for stable positioning
                val concentration = if (matchingTemplate.templateType == TemplateType.DECAYING) {
                    de.ntdote.medicalcalendarlog.utils.ConcentrationCalculator
                        .calculatePeakAfterEvent(matchingTemplate, events, event)
                } else {
                    // For metric templates, use the dosage value directly
                    event.dosage
                }
                
                // Use the unified global scaling for consistent positioning
                val valueRange = globalMaxValue - globalMinValue
                if (valueRange > 0) {
                    size.height - padding - ((concentration - globalMinValue).toFloat() / valueRange.toFloat()) * graphHeight
                } else {
                    size.height - padding - 30f
                }
            } else {
                // For general templates, position at bottom (timeline view)
                size.height - padding - 30f
            }
            
            // Truncate long event titles for display
            val displayText = if (eventText.length > 15) {
                eventText.take(12) + "..."
            } else {
                eventText
            }
            
            // Set larger text size (3x original)
            paint.textSize = 48f
            
            // Calculate text width for background sizing with larger text
            val textWidth = paint.measureText(displayText)
            val backgroundWidth = maxOf(textWidth + 16f, 30f)
            val backgroundHeight = 60f
            
            // Draw event marker - either with text or as a simple dot
            if (showEventText) {
                // Draw only the black border (no white background for transparency)
                drawRect(
                    color = Color.Black,
                    topLeft = Offset(x - backgroundWidth / 2, y - backgroundHeight / 2),
                    size = androidx.compose.ui.geometry.Size(backgroundWidth, backgroundHeight),
                    style = Stroke(width = 1.5f)
                )
                
                // Draw event text with larger size
                drawContext.canvas.nativeCanvas.drawText(
                    displayText,
                    x - textWidth / 2,
                    y + paint.textSize / 3,
                    paint
                )
            } else {
                // Draw a simple marker dot when text is hidden
                drawCircle(
                    color = Color.White,
                    radius = 4f,
                    center = Offset(x, y)
                )
                drawCircle(
                    color = Color.Black,
                    radius = 4f,
                    center = Offset(x, y),
                    style = Stroke(width = 1f)
                )
            }
            
            // Dosage numbers removed - now shown in popup dialog instead
        }
    }
    
    private fun DrawScope.drawEventMarkersWithIndividualScaling(
        events: List<de.ntdote.medicalcalendarlog.data.CalendarEvent>,
        templates: List<de.ntdote.medicalcalendarlog.data.Template>,
        visibleItems: Set<String>,
        showEventText: Boolean,
        padding: Float,
        graphWidth: Float,
        graphHeight: Float,
        minTime: Long,
        timeRange: Long,
        visibleGraphData: List<GraphData>,
        axisInfos: List<AxisInfo>
    ) {
        val paint = Paint().asFrameworkPaint().apply {
            isAntiAlias = true
            textSize = 24f
            color = android.graphics.Color.WHITE
        }
        
        // Filter events that are within the visible time range
        val visibleEvents = events.filter { event ->
            event.startTime.time >= minTime && event.startTime.time <= minTime + timeRange
        }
        
        visibleEvents.forEach { event ->
            // Get matching template for this event using unified matching logic
            val matchingTemplate = templates.find { template ->
                EventMatching.matchesTemplate(event, template)
            }
            
            // Only show events that match configured templates
            val shouldShowEvent = if (matchingTemplate != null) {
                val itemName = if (matchingTemplate.templateType == TemplateType.DECAYING) {
                    matchingTemplate.extendedData?.typeOfDrug ?: matchingTemplate.name
                } else {
                    matchingTemplate.name
                }
                visibleItems.isEmpty() || visibleItems.contains(itemName)
            } else {
                // Skip events that don't match any configured template
                false
            }
            
            if (!shouldShowEvent) return@forEach
            
            val x = padding + ((event.startTime.time - minTime).toFloat() / timeRange.toFloat()) * graphWidth
            
            // Show the actual event title (which already contains the dosage appended)
            val eventText = event.title
            
            // Calculate Y position using individual scaling for the matching drug
            val y = if (matchingTemplate?.templateType == TemplateType.DECAYING || matchingTemplate?.templateType == TemplateType.METRIC) {
                // Find the corresponding graph data and axis info for this template
                val graphDataIndex = visibleGraphData.indexOfFirst { graphData ->
                    val drugName = graphData.template.extendedData?.typeOfDrug ?: graphData.template.name
                    val templateDrugName = matchingTemplate.extendedData?.typeOfDrug ?: matchingTemplate.name
                    drugName == templateDrugName
                }
                
                if (graphDataIndex >= 0 && graphDataIndex < axisInfos.size) {
                    val axisInfo = axisInfos[graphDataIndex]
                    
                    // Calculate peak concentration immediately after this event for stable positioning
                    val concentration = if (matchingTemplate.templateType == TemplateType.DECAYING) {
                        de.ntdote.medicalcalendarlog.utils.ConcentrationCalculator
                            .calculatePeakAfterEvent(matchingTemplate, events, event)
                    } else {
                        // For metric templates, use the dosage value directly
                        event.dosage
                    }
                    
                    // Use the individual drug's scaling for positioning
                    val valueRange = axisInfo.maxValue - axisInfo.minValue
                    if (valueRange > 0) {
                        size.height - padding - ((concentration - axisInfo.minValue).toFloat() / valueRange.toFloat()) * graphHeight
                    } else {
                        size.height - padding - 30f
                    }
                } else {
                    // Fallback to bottom if no matching graph data found
                    size.height - padding - 30f
                }
            } else {
                // For general templates, position at bottom (timeline view)
                size.height - padding - 30f
            }
            
            // Truncate long event titles for display
            val displayText = if (eventText.length > 15) {
                eventText.take(12) + "..."
            } else {
                eventText
            }
            
            // Set larger text size (3x original)
            paint.textSize = 48f
            
            // Calculate text width for background sizing with larger text
            val textWidth = paint.measureText(displayText)
            val backgroundWidth = maxOf(textWidth + 16f, 30f)
            val backgroundHeight = 60f
            
            // Draw event marker - either with text or as a simple dot
            if (showEventText) {
                // Draw only the black border (no white background for transparency)
                drawRect(
                    color = Color.Black,
                    topLeft = Offset(x - backgroundWidth / 2, y - backgroundHeight / 2),
                    size = androidx.compose.ui.geometry.Size(backgroundWidth, backgroundHeight),
                    style = Stroke(width = 1.5f)
                )
                
                // Draw event text with larger size
                drawContext.canvas.nativeCanvas.drawText(
                    displayText,
                    x - textWidth / 2,
                    y + paint.textSize / 3,
                    paint
                )
            } else {
                // Draw a simple marker dot when text is hidden
                drawCircle(
                    color = Color.White,
                    radius = 4f,
                    center = Offset(x, y)
                )
                drawCircle(
                    color = Color.Black,
                    radius = 4f,
                    center = Offset(x, y),
                    style = Stroke(width = 1f)
                )
            }
            
            // Dosage numbers removed - now shown in popup dialog instead
        }
    }
    
    
    private fun DrawScope.drawAxes(
        padding: Float,
        graphWidth: Float,
        graphHeight: Float
    ) {
        // X-axis
        drawLine(
            color = Color.Black,
            start = Offset(padding, size.height - padding),
            end = Offset(size.width - padding, size.height - padding),
            strokeWidth = 2f
        )
        
        // Y-axis
        drawLine(
            color = Color.Black,
            start = Offset(padding, padding),
            end = Offset(padding, size.height - padding),
            strokeWidth = 2f
        )
    }
    
    private fun DrawScope.drawAxisLabels(
        padding: Float,
        graphWidth: Float,
        graphHeight: Float,
        startTime: Date,
        endTime: Date,
        axisInfos: List<AxisInfo>
    ) {
        val paint = Paint().asFrameworkPaint().apply {
            isAntiAlias = true
            textSize = 28f
            color = android.graphics.Color.WHITE
        }
        
        // Time labels (X-axis) - 4-7 labels evenly spaced (reduced by half)
        val daysBack = ((endTime.time - startTime.time) / (24 * 60 * 60 * 1000)).toInt()
        val allTimeLabels = getTimeAxisLabels(startTime, endTime, daysBack)
        val timeLabels = when {
            allTimeLabels.size <= 4 -> allTimeLabels
            allTimeLabels.size <= 7 -> allTimeLabels
            else -> {
                val targetCount = when {
                    daysBack <= 30 -> 7
                    daysBack <= 60 -> 6
                    else -> 5
                }
                val step = maxOf(1, allTimeLabels.size / targetCount)
                allTimeLabels.filterIndexed { index, _ -> index % step == 0 }.take(targetCount)
            }
        }
        val timeRange = endTime.time - startTime.time
        
        timeLabels.forEach { (time, label) ->
            val x = padding + ((time.time - startTime.time).toFloat() / timeRange.toFloat()) * graphWidth
            drawContext.canvas.nativeCanvas.drawText(
                label,
                x - paint.measureText(label) / 2,
                size.height - padding + 35f,
                paint
            )
        }
        
        // Multiple Y-axes with color-coded labels
        axisInfos.forEachIndexed { index, axisInfo ->
            val yLabels = getYAxisLabels(axisInfo.minValue, axisInfo.maxValue)
            val concentrationRange = axisInfo.maxValue - axisInfo.minValue
            
            // Set paint color to match the drug line color
            paint.color = android.graphics.Color.argb(
                255,
                (axisInfo.color.red * 255).toInt(),
                (axisInfo.color.green * 255).toInt(),
                (axisInfo.color.blue * 255).toInt()
            )
            
            // Calculate X position for this axis with better spacing
            val axisX = when (index) {
                0 -> padding // Left axis
                1 -> size.width - padding // Right axis
                2 -> padding + 50f // Right side of left axis
                3 -> size.width - padding - 50f // Left side of right axis
                else -> padding + 50f + (index - 2) * 40f // Additional axes with better spacing
            }
            
            yLabels.forEach { value ->
                val y = size.height - padding - ((value - axisInfo.minValue).toFloat() / concentrationRange.toFloat()) * graphHeight
                val label = if (value >= 1) "%.1f".format(value) else "%.2f".format(value)
                
                val labelX = when (index) {
                    0 -> axisX - paint.measureText(label) - 10f // Left of left axis
                    1 -> axisX + 10f // Right of right axis
                    2 -> axisX - paint.measureText(label) - 5f // Left of secondary left
                    3 -> axisX + 5f // Right of secondary right
                    else -> axisX - paint.measureText(label) / 2 // Centered for others
                }
                
                drawContext.canvas.nativeCanvas.drawText(
                    label,
                    labelX,
                    y + paint.textSize / 3,
                    paint
                )
            }
            
            // Y-axis unit label
            drawContext.canvas.nativeCanvas.save()
            val unitX = when (index) {
                0 -> 20f
                1 -> size.width - 20f
                2 -> 60f
                3 -> size.width - 60f
                else -> 20f + (index - 4) * 30f
            }
            drawContext.canvas.nativeCanvas.rotate(-90f, unitX, size.height / 2)
            drawContext.canvas.nativeCanvas.drawText(
                axisInfo.unit,
                unitX - paint.measureText(axisInfo.unit) / 2,
                size.height / 2,
                paint
            )
            drawContext.canvas.nativeCanvas.restore()
        }
        
        // Time axis label (white)
        paint.color = android.graphics.Color.WHITE
        drawContext.canvas.nativeCanvas.drawText(
            "Time",
            size.width / 2 - paint.measureText("Time") / 2,
            size.height - 50f,
            paint
        )
    }
    
    fun getDrugColors(): List<Color> {
        return listOf(
            Color(0xFF2196F3), // Blue
            Color(0xFF4CAF50), // Green
            Color(0xFFFF9800), // Orange
            Color(0xFF00E5FF), // Bright Cyan (replaced purple)
            Color(0xFFF44336), // Red
            Color(0xFF00BCD4), // Cyan
            Color(0xFFFFEB3B), // Yellow
            Color(0xFF795548), // Brown
            Color(0xFF607D8B), // Blue Grey
            Color(0xFFE91E63)  // Pink
        )
    }
}
