package de.ntdote.medicalcalendarlog.utils

import de.ntdote.medicalcalendarlog.data.CalendarEvent
import de.ntdote.medicalcalendarlog.data.Template

/**
 * Unified event matching logic for the entire application.
 * 
 * Events are created with template.name as the base:
 * - If template.requiresInput: event.title = template.name + number (e.g., "Tylrnol 500" + "1.5" = "Tylrnol 5001.5")
 * - If !template.requiresInput: event.title = template.name (e.g., "🤧" = "🤧")
 * 
 * This function ensures consistent matching across all features (reminders, graphs, widgets, etc.)
 */
object EventMatching {
    
    /**
     * Check if a calendar event matches a template based on its title.
     * 
     * @param event The calendar event to check
     * @param template The template to match against
     * @return true if the event matches the template, false otherwise
     */
    fun matchesTemplate(event: CalendarEvent, template: Template): Boolean {
        return if (template.requiresInput) {
            // Event title must be template.name followed by a number (with optional decimal point)
            // Example: "Tylrnol 500" matches events like "Tylrnol 5001.5" or "Tylrnol 500500"
            if (event.title.startsWith(template.name)) {
                val suffix = event.title.substring(template.name.length)
                // Check if suffix is a valid number (digits with optional decimal point)
                suffix.matches(Regex("""^\d+(\.\d+)?$"""))
            } else {
                false
            }
        } else {
            // Event title must exactly match template name
            // Example: "🤧" matches only "🤧"
            event.title == template.name
        }
    }
    
    /**
     * Filter a list of events to only those matching a specific template.
     * 
     * @param events The list of events to filter
     * @param template The template to match against
     * @return A list of events that match the template
     */
    fun filterEventsForTemplate(events: List<CalendarEvent>, template: Template): List<CalendarEvent> {
        return events.filter { event -> matchesTemplate(event, template) }
    }
}
