package de.ntdote.medicalcalendarlog.utils

import de.ntdote.medicalcalendarlog.data.Drug
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.data.TemplateType

/**
 * Utility for resolving drug relationships between templates.
 * Handles source/derived drug relationships where multiple templates can reference the same drug type.
 */
object DrugResolution {

    /**
     * Find the source drug (with decay parameters) for a given drug type across all templates.
     * @param drugType The type of drug to find the source for
     * @param allTemplates All available templates to search
     * @return The source Drug instance, or null if no source found
     */
    fun findSourceDrug(drugType: String, allTemplates: List<Template>): Drug? {
        return allTemplates
            .filter { it.templateType == TemplateType.DECAYING }
            .flatMap { it.drugs }
            .find { it.drugType == drugType && it.decayType != null }
    }

    /**
     * Get all drugs in a template with their decay information resolved.
     * For derived drugs (no decay info), finds the source drug that provides the decay parameters.
     * @param template The template to process
     * @param allTemplates All available templates for source lookup
     * @return List of drugs with resolved decay information
     */
    fun getDrugsWithDecayInfo(template: Template, allTemplates: List<Template>): List<DrugWithDecayInfo> {
        if (template.templateType != TemplateType.DECAYING) {
            return emptyList()
        }

        return template.drugs.map { drug ->
            val isSource = drug.decayType != null
            val sourceDrug = if (isSource) {
                null // This drug IS the source
            } else {
                findSourceDrug(drug.drugType, allTemplates)
            }

            DrugWithDecayInfo(
                drug = drug,
                sourceDrug = sourceDrug,
                isSource = isSource
            )
        }
    }

    /**
     * Check if a template contains any source drugs (drugs with decay parameters).
     * @param template The template to check
     * @return True if template has at least one drug with decay parameters
     */
    fun isSourceTemplate(template: Template): Boolean {
        return template.templateType == TemplateType.DECAYING && 
               template.drugs.any { it.decayType != null }
    }

    /**
     * Find all templates that are derived from the given source template.
     * A derived template has drugs with the same drugType but no decay parameters.
     * @param sourceTemplate The source template to find derivatives for
     * @param allTemplates All available templates to search
     * @return List of templates that derive from the source
     */
    fun getDerivedTemplates(sourceTemplate: Template, allTemplates: List<Template>): List<Template> {
        val sourceDrugTypes = sourceTemplate.drugs
            .filter { it.decayType != null } // Only consider source drugs
            .map { it.drugType }
            .toSet()

        if (sourceDrugTypes.isEmpty()) {
            return emptyList()
        }

        return allTemplates.filter { template ->
            template.id != sourceTemplate.id && // Exclude the source template itself
            template.templateType == TemplateType.DECAYING &&
            template.drugs.any { drug ->
                drug.drugType in sourceDrugTypes && drug.decayType == null
            }
        }
    }

    /**
     * Find all source templates that provide decay info for the given derived template.
     * @param derivedTemplate The template with derived drugs (no decay parameters)
     * @param allTemplates All available templates to search
     * @return List of templates that provide source drug information
     */
    fun getSourceTemplates(derivedTemplate: Template, allTemplates: List<Template>): List<Template> {
        val derivedDrugTypes = derivedTemplate.drugs
            .filter { it.decayType == null } // Only consider derived drugs
            .map { it.drugType }
            .toSet()

        if (derivedDrugTypes.isEmpty()) {
            return emptyList()
        }

        return allTemplates.filter { template ->
            template.id != derivedTemplate.id && // Exclude the derived template itself
            template.templateType == TemplateType.DECAYING &&
            template.drugs.any { drug ->
                drug.drugType in derivedDrugTypes && drug.decayType != null
            }
        }
    }

    /**
     * Get the effective decay information for a drug, resolving to source if needed.
     * @param drug The drug to get decay info for
     * @param allTemplates All templates for source lookup
     * @return The drug with decay parameters, or null if no decay info available
     */
    fun getEffectiveDecayInfo(drug: Drug, allTemplates: List<Template>): Drug? {
        return if (drug.decayType != null) {
            drug // This drug has its own decay info
        } else {
            findSourceDrug(drug.drugType, allTemplates) // Find source drug
        }
    }

    /**
     * Check if two templates share any drug types.
     * @param template1 First template
     * @param template2 Second template
     * @return True if they share at least one drug type
     */
    fun sharesDrugTypes(template1: Template, template2: Template): Boolean {
        val drugTypes1 = template1.drugs.map { it.drugType }.toSet()
        val drugTypes2 = template2.drugs.map { it.drugType }.toSet()
        return drugTypes1.intersect(drugTypes2).isNotEmpty()
    }

    /**
     * Get all unique drug types from a list of templates.
     * @param templates The templates to extract drug types from
     * @return Set of unique drug type names
     */
    fun getAllDrugTypes(templates: List<Template>): Set<String> {
        return templates
            .filter { it.templateType == TemplateType.DECAYING }
            .flatMap { it.drugs.map { drug -> drug.drugType } }
            .toSet()
    }
}

/**
 * Data class representing a drug with its resolved decay information.
 * @param drug The original drug from the template
 * @param sourceDrug The source drug that provides decay info (null if this drug IS the source)
 * @param isSource True if this drug contains its own decay parameters
 */
data class DrugWithDecayInfo(
    val drug: Drug,
    val sourceDrug: Drug?, // The source drug with decay info, or null if this drug IS the source
    val isSource: Boolean
) {
    /**
     * Get the effective decay parameters for this drug.
     * @return The drug instance that contains the decay parameters, or null if none available
     */
    fun getEffectiveDecaySource(): Drug? {
        return if (isSource) drug else sourceDrug
    }

    /**
     * Check if this drug has effective decay information available.
     */
    fun hasDecayInfo(): Boolean {
        return getEffectiveDecaySource() != null
    }
}
