package de.ntdote.medicalcalendarlog.utils

import android.content.ContentResolver
import android.content.ContentUris
import android.content.ContentValues
import android.database.Cursor
import android.provider.CalendarContract
import android.util.Log
import de.ntdote.medicalcalendarlog.BuildConfig
import de.ntdote.medicalcalendarlog.data.Template
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json
import kotlinx.serialization.encodeToString
import kotlinx.serialization.decodeFromString
import java.text.SimpleDateFormat
import java.util.*

@Serializable
data class ConfigExportData(
    val metadata: ConfigMetadata,
    val templates: List<Template>
)

@Serializable
data class ConfigMetadata(
    val appVersion: String,
    val versionCode: Int? = null, // Added in version 36 - null for old exports (assume 34)
    val exportTimestamp: String,
    val templateCount: Int
)

class ConfigExportImport(private val contentResolver: ContentResolver) {
    
    companion object {
        private const val TAG = "MCL"
        const val BACKUP_EVENT_PREFIX = "[MCL] Configuration Backup"
        private const val CONFIG_EVENT_TITLE_PREFIX = BACKUP_EVENT_PREFIX // For backward compatibility
        
        private val json = Json {
            prettyPrint = true
            ignoreUnknownKeys = true
        }
        
        // Search for config events within the last 30 days
        private const val CONFIG_SEARCH_DAYS_BACK = 30
        
        /**
         * Sanitize HTML-enriched JSON data from calendar providers
         * 
         * Google Calendar and other providers may convert plain JSON into HTML
         * by adding <br> tags and encoding special characters.
         * This function strips HTML tags and decodes entities.
         */
        fun sanitizeJsonData(jsonData: String): String {
            var sanitized = jsonData
            
            // Check if the data contains HTML tags
            val containsHtml = sanitized.contains("<br>", ignoreCase = true) ||
                              sanitized.contains("<br/>", ignoreCase = true) ||
                              sanitized.contains("&nbsp;") ||
                              sanitized.contains("&quot;") ||
                              sanitized.contains("&lt;") ||
                              sanitized.contains("&gt;")
            
            if (containsHtml) {
                Log.w(TAG, "ConfigExportImport: HTML content detected in JSON data - sanitizing...")
                
                // Remove <br> and <br/> tags (replace with nothing, not newlines)
                sanitized = sanitized.replace(Regex("<br\\s*/?>", RegexOption.IGNORE_CASE), "")
                
                // Remove other common HTML tags (but keep their content)
                sanitized = sanitized.replace(Regex("<[^>]+>"), "")
                
                // Decode common HTML entities
                sanitized = sanitized
                    .replace("&nbsp;", " ")
                    .replace("&quot;", "\"")
                    .replace("&#34;", "\"")
                    .replace("&apos;", "'")
                    .replace("&#39;", "'")
                    .replace("&lt;", "<")
                    .replace("&gt;", ">")
                    .replace("&amp;", "&") // Must be last to avoid double-decoding
                
                Log.d(TAG, "ConfigExportImport: Sanitization complete - removed HTML formatting")
            }
            
            return sanitized
        }
        
        /**
         * Parse backup metadata from JSON string
         */
        fun parseBackupMetadata(jsonData: String): ConfigMetadata {
            val sanitizedData = sanitizeJsonData(jsonData)
            val exportData = json.decodeFromString<ConfigExportData>(sanitizedData)
            return exportData.metadata
        }
        
        /**
         * Parse template count from JSON string
         */
        fun parseTemplateCount(jsonData: String): Int {
            val sanitizedData = sanitizeJsonData(jsonData)
            val exportData = json.decodeFromString<ConfigExportData>(sanitizedData)
            return exportData.templates.size
        }
    }
    
    /**
     * Export templates to calendar as a special config event
     */
    fun exportTemplatesToCalendar(calendarId: Long, templates: List<Template>): Result<Unit> {
        return try {
            // NOTE: Don't delete old backups - let users manage them via backup selection screen
            
            val currentTime = System.currentTimeMillis()
            val timestamp = SimpleDateFormat("yyyy-MM-dd HH:mm", Locale.US).format(Date(currentTime))
            
            // Create export data with metadata (include both version name and code)
            val exportData = ConfigExportData(
                metadata = ConfigMetadata(
                    appVersion = BuildConfig.VERSION_NAME,
                    versionCode = BuildConfig.VERSION_CODE,
                    exportTimestamp = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'", Locale.US).apply {
                        timeZone = TimeZone.getTimeZone("UTC")
                    }.format(Date(currentTime)),
                    templateCount = templates.size
                ),
                templates = templates
            )
            
            Log.i(TAG, "ConfigExportImport: Creating backup with ${templates.size} templates, version ${BuildConfig.VERSION_NAME} (${BuildConfig.VERSION_CODE})")
            
            // Serialize to JSON
            val jsonString = json.encodeToString(exportData)
            
            // Create calendar event with current time and descriptive title
            val eventTitle = "$CONFIG_EVENT_TITLE_PREFIX - $timestamp"
            val values = ContentValues().apply {
                put(CalendarContract.Events.DTSTART, currentTime)
                put(CalendarContract.Events.DTEND, currentTime + (60 * 60 * 1000)) // 1 hour duration
                put(CalendarContract.Events.TITLE, eventTitle)
                put(CalendarContract.Events.DESCRIPTION, jsonString)
                put(CalendarContract.Events.CALENDAR_ID, calendarId)
                put(CalendarContract.Events.EVENT_TIMEZONE, TimeZone.getDefault().id)
                put(CalendarContract.Events.ALL_DAY, 0)
            }
            
            val uri = contentResolver.insert(CalendarContract.Events.CONTENT_URI, values)
            if (uri != null) {
                Result.success(Unit)
            } else {
                Result.failure(Exception("Failed to create config event in calendar"))
            }
        } catch (e: Exception) {
            Result.failure(e)
        }
    }
    
    /**
     * Import templates from calendar config event
     */
    fun importTemplatesFromCalendar(calendarId: Long): Result<List<Template>> {
        return try {
            val configEvent = findConfigEvent(calendarId)
            if (configEvent == null) {
                return Result.failure(Exception("No configuration found in calendar"))
            }
            
            // Parse JSON from event description (sanitize HTML first)
            val sanitizedDescription = sanitizeJsonData(configEvent.description)
            val exportData = json.decodeFromString<ConfigExportData>(sanitizedDescription)
            
            // Validate metadata
            if (exportData.metadata.templateCount != exportData.templates.size) {
                return Result.failure(Exception("Configuration data integrity check failed"))
            }
            
            Log.i(TAG, "ConfigExportImport: Importing config from version ${exportData.metadata.appVersion} (${exportData.metadata.versionCode ?: 34})")
            
            // Apply unified migration framework (includes ### migration)
            val migratedTemplates = DataMigrations.migrateTemplates(
                templates = exportData.templates,
                sourceVersion = exportData.metadata.versionCode
            )
            
            // Post-migration cleanup: Auto-set requiresInput for METRIC templates
            val cleanedTemplates = migratedTemplates.map { template ->
                var updatedTemplate = template
                
                // Auto-set requiresInput for METRIC templates if not already set
                if (template.templateType == de.ntdote.medicalcalendarlog.data.TemplateType.METRIC && !updatedTemplate.requiresInput) {
                    updatedTemplate = updatedTemplate.copy(requiresInput = true)
                }
                
                // Clean up extended data to ensure proper nulling
                if (updatedTemplate.extendedData != null) {
                    updatedTemplate.copy(
                        extendedData = updatedTemplate.extendedData.let { extData ->
                            when (extData.decayType) {
                                de.ntdote.medicalcalendarlog.data.DecayType.CONSTANT -> 
                                    extData.copy(halfLifeHours = null)
                                de.ntdote.medicalcalendarlog.data.DecayType.HALF_LIFE -> 
                                    extData.copy(hourlyDecayRate = null)
                                null -> extData.copy(hourlyDecayRate = null, halfLifeHours = null)
                            }
                        }
                    )
                } else {
                    updatedTemplate
                }
            }
            
            Log.i(TAG, "ConfigExportImport: Import complete: ${cleanedTemplates.size} templates ready")
            Result.success(cleanedTemplates)
        } catch (e: Exception) {
            Result.failure(e)
        }
    }
    
    /**
     * Check if config exists in calendar
     */
    fun hasConfigInCalendar(calendarId: Long): Boolean {
        return findConfigEvent(calendarId) != null
    }
    
    /**
     * Get config metadata without importing full templates
     */
    fun getConfigMetadata(calendarId: Long): Result<ConfigMetadata?> {
        return try {
            val configEvent = findConfigEvent(calendarId)
            if (configEvent == null) {
                Result.success(null)
            } else {
                val exportData = json.decodeFromString<ConfigExportData>(configEvent.description)
                Result.success(exportData.metadata)
            }
        } catch (e: Exception) {
            Result.failure(e)
        }
    }
    
    /**
     * Delete config event from calendar
     */
    fun deleteConfigFromCalendar(calendarId: Long): Result<Unit> {
        return try {
            // Find and delete all config events (there should only be one, but clean up any extras)
            val configEvents = findAllConfigEvents(calendarId)
            var deletedCount = 0
            
            for (configEvent in configEvents) {
                val uri = ContentUris.withAppendedId(CalendarContract.Events.CONTENT_URI, configEvent.id)
                val deletedRows = contentResolver.delete(uri, null, null)
                if (deletedRows > 0) {
                    deletedCount++
                }
            }
            
            Result.success(Unit)
        } catch (e: Exception) {
            Result.failure(e)
        }
    }
    
    private data class ConfigEvent(
        val id: Long,
        val title: String,
        val description: String,
        val startTime: Long
    )
    
    private fun findConfigEvent(calendarId: Long): ConfigEvent? {
        val configEvents = findAllConfigEvents(calendarId)
        // Return the most recent config event
        return configEvents.maxByOrNull { it.startTime }
    }
    
    private fun findAllConfigEvents(calendarId: Long): List<ConfigEvent> {
        val configEvents = mutableListOf<ConfigEvent>()
        
        val projection = arrayOf(
            CalendarContract.Events._ID,
            CalendarContract.Events.TITLE,
            CalendarContract.Events.DESCRIPTION,
            CalendarContract.Events.DTSTART
        )
        
        // Search for events with our title prefix within the last 30 days
        val searchStartTime = System.currentTimeMillis() - (CONFIG_SEARCH_DAYS_BACK * 24 * 60 * 60 * 1000L)
        
        val selection = "${CalendarContract.Events.CALENDAR_ID} = ? AND ${CalendarContract.Events.TITLE} LIKE ? AND ${CalendarContract.Events.DTSTART} >= ?"
        val selectionArgs = arrayOf(
            calendarId.toString(), 
            "$CONFIG_EVENT_TITLE_PREFIX%",
            searchStartTime.toString()
        )
        
        val cursor: Cursor? = contentResolver.query(
            CalendarContract.Events.CONTENT_URI,
            projection,
            selection,
            selectionArgs,
            "${CalendarContract.Events.DTSTART} DESC" // Most recent first
        )
        
        cursor?.use {
            while (it.moveToNext()) {
                val id = it.getLong(0)
                val title = it.getString(1) ?: ""
                val description = it.getString(2) ?: ""
                val startTime = it.getLong(3)
                
                // Verify this is our config event by checking the title prefix
                if (title.startsWith(CONFIG_EVENT_TITLE_PREFIX)) {
                    configEvents.add(ConfigEvent(id, title, description, startTime))
                }
            }
        }
        
        return configEvents
    }
    
    /**
     * Import templates from a specific backup
     * @param backupJsonData The JSON data from the backup
     * @return Result containing imported templates or error
     */
    fun importFromBackup(backupJsonData: String): Result<List<Template>> {
        return try {
            // Sanitize HTML-enriched JSON (e.g., from Google Calendar)
            val sanitizedData = sanitizeJsonData(backupJsonData)
            
            // Parse JSON
            val exportData = json.decodeFromString<ConfigExportData>(sanitizedData)
            
            // Validate metadata
            if (exportData.metadata.templateCount != exportData.templates.size) {
                return Result.failure(Exception("Configuration data integrity check failed"))
            }
            
            val versionName = exportData.metadata.appVersion
            val versionCode = exportData.metadata.versionCode ?: 34
            
            Log.i(TAG, "ConfigExportImport: Importing from backup version $versionName ($versionCode)")
            
            // Apply unified migration framework
            val migratedTemplates = DataMigrations.migrateTemplates(
                templates = exportData.templates,
                sourceVersion = versionCode
            )
            
            // Post-migration cleanup
            val cleanedTemplates = migratedTemplates.map { template ->
                var updatedTemplate = template
                
                // Auto-set requiresInput for METRIC templates if not already set
                if (template.templateType == de.ntdote.medicalcalendarlog.data.TemplateType.METRIC && !updatedTemplate.requiresInput) {
                    updatedTemplate = updatedTemplate.copy(requiresInput = true)
                }
                
                // Clean up extended data to ensure proper nulling
                if (updatedTemplate.extendedData != null) {
                    updatedTemplate.copy(
                        extendedData = updatedTemplate.extendedData.let { extData ->
                            when (extData.decayType) {
                                de.ntdote.medicalcalendarlog.data.DecayType.CONSTANT -> 
                                    extData.copy(halfLifeHours = null)
                                de.ntdote.medicalcalendarlog.data.DecayType.HALF_LIFE -> 
                                    extData.copy(hourlyDecayRate = null)
                                null -> extData.copy(hourlyDecayRate = null, halfLifeHours = null)
                            }
                        }
                    )
                } else {
                    updatedTemplate
                }
            }
            
            Log.i(TAG, "ConfigExportImport: Backup import complete: ${cleanedTemplates.size} templates ready")
            Result.success(cleanedTemplates)
        } catch (e: Exception) {
            Log.e(TAG, "ConfigExportImport: Failed to import from backup", e)
            Result.failure(e)
        }
    }
}
