package de.ntdote.medicalcalendarlog.utils

import de.ntdote.medicalcalendarlog.data.CalendarEvent
import de.ntdote.medicalcalendarlog.data.DecayType
import de.ntdote.medicalcalendarlog.data.Drug
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.data.TemplateType
import java.util.Date
import kotlin.math.pow

/**
 * Core concentration calculation engine for multi-drug support.
 * 
 * Architecture:
 * - Each Template can contain multiple Drug objects
 * - Each Drug has its own decay parameters (half-life or constant rate)
 * - Calculations work per-drug, aggregating contributions from all sources
 */
object ConcentrationCalculator {

    /**
     * Calculate total concentration for a template at a specific time.
     * For multi-drug templates, this returns the sum of all drug concentrations.
     */
    fun calculateConcentrationAtTime(
        template: Template,
        events: List<CalendarEvent>,
        targetTime: Date
    ): Double {
        return calculateCurrentConcentration(template, events, targetTime)
    }

    /**
     * Calculate current total concentration for a template.
     * Sums contributions from all drugs in the template.
     */
    fun calculateCurrentConcentration(
        template: Template,
        events: List<CalendarEvent>,
        currentTime: Date = Date(),
        allTemplates: List<Template> = emptyList()
    ): Double {
        if (template.templateType != TemplateType.DECAYING || template.drugs.isEmpty()) {
            return 0.0
        }

        // Get relevant events for this template
        val relevantEvents = EventMatching.filterEventsForTemplate(events, template).sortedBy { it.startTime }
        
        // Sum concentration contributions from all drugs
        return template.drugs.sumOf { drug ->
            calculateDrugConcentration(drug, relevantEvents, currentTime, allTemplates)
        }
    }

    /**
     * Calculate concentration for a single drug within a template.
     * For derived drugs (no decay parameters), resolves to source drug across all templates.
     */
    private fun calculateDrugConcentration(
        drug: Drug,
        events: List<CalendarEvent>,
        currentTime: Date,
        allTemplates: List<Template> = emptyList()
    ): Double {
        return when (drug.decayType) {
            DecayType.CONSTANT -> calculateLinearDecayConcentration(events, drug, currentTime)
            DecayType.HALF_LIFE -> calculateHalfLifeConcentration(events, drug, currentTime)
            null -> {
                // This is a derived drug - find its source across all templates
                if (allTemplates.isNotEmpty()) {
                    val sourceDrug = DrugResolution.findSourceDrug(drug.drugType, allTemplates)
                    if (sourceDrug != null) {
                        // Use source drug's decay parameters with current drug's factor
                        val adjustedDrug = sourceDrug.copy(factor = drug.factor)
                        calculateDrugConcentration(adjustedDrug, events, currentTime, allTemplates)
                    } else {
                        0.0 // No source found
                    }
                } else {
                    0.0 // No templates provided, can't resolve derived drug
                }
            }
        }
    }

    private fun calculateLinearDecayConcentration(
        events: List<CalendarEvent>,
        drug: Drug,
        targetTime: Date
    ): Double {
        val hourlyDecayRate = drug.hourlyDecayRate ?: 0.0
        if (hourlyDecayRate <= 0.0) return 0.0

        // Create timeline of dose events
        val doseEvents = events.filter { it.startTime.time <= targetTime.time }
            .map { event ->
                Pair(event.startTime.time, event.dosage * drug.factor)
            }
            .sortedBy { it.first }

        if (doseEvents.isEmpty()) return 0.0

        // Simulate concentration over time
        var totalConcentration = 0.0
        var lastTime = doseEvents.first().first

        for ((eventTime, dosage) in doseEvents) {
            // Apply decay from last time to this event
            val timeDiffHours = (eventTime - lastTime) / (1000.0 * 60.0 * 60.0)
            totalConcentration = maxOf(0.0, totalConcentration - (hourlyDecayRate * timeDiffHours))
            
            // Add new dose
            totalConcentration += dosage
            lastTime = eventTime
        }

        // Apply final decay from last event to target time
        val finalTimeDiffHours = (targetTime.time - lastTime) / (1000.0 * 60.0 * 60.0)
        totalConcentration = maxOf(0.0, totalConcentration - (hourlyDecayRate * finalTimeDiffHours))

        return totalConcentration
    }

    private fun calculateHalfLifeConcentration(
        events: List<CalendarEvent>,
        drug: Drug,
        currentTime: Date
    ): Double {
        var totalConcentration = 0.0

        for (event in events) {
            // Skip events that are in the future
            if (event.startTime.time > currentTime.time) continue
            
            val dosage = event.dosage * drug.factor
            val hoursElapsed = (currentTime.time - event.startTime.time) / (1000.0 * 60.0 * 60.0)

            val halfLife = drug.halfLifeHours ?: 24.0
            if (halfLife > 0 && hoursElapsed >= 0) {
                totalConcentration += dosage * (0.5).pow(hoursElapsed / halfLife)
            }
        }

        return totalConcentration
    }

    /**
     * Calculate concentration for a specific drug type across all templates.
     * This is used for drug-based graphing where multiple templates can contribute to the same drug.
     */
    fun calculateDrugConcentrationAtTime(
        drugType: String,
        templates: List<Template>,
        events: List<CalendarEvent>,
        targetTime: Date
    ): Double {
        return calculateCurrentDrugConcentration(drugType, templates, events, targetTime)
    }

    /**
     * Calculate current concentration for a specific drug type across all templates.
     */
    fun calculateCurrentDrugConcentration(
        drugType: String,
        templates: List<Template>,
        events: List<CalendarEvent>,
        currentTime: Date = Date()
    ): Double {
        // Find the source drug definition (with proper decay parameters) using DrugResolution
        val canonicalDrug = DrugResolution.findSourceDrug(drugType, templates) ?: return 0.0

        // Find all templates that contain this drug type
        val drugTemplates = templates.filter { template ->
            template.templateType == TemplateType.DECAYING &&
            template.drugs.any { it.drugType.equals(drugType, ignoreCase = true) }
        }

        // Collect all events from templates containing this drug
        val allRelevantEvents = drugTemplates.flatMap { template ->
            val templateEvents = EventMatching.filterEventsForTemplate(events, template)
            // Find the drug in this template
            val drugInTemplate = template.drugs.find { it.drugType.equals(drugType, ignoreCase = true) }
            // Map events to (time, dosage) pairs using this template's drug factor
            templateEvents.map { event ->
                val effectiveDosage = event.dosage * (drugInTemplate?.factor ?: 1.0)
                Pair(event.startTime.time, effectiveDosage)
            }
        }.filter { it.second > 0.0 }
            .sortedBy { it.first }

        return when (canonicalDrug.decayType) {
            DecayType.CONSTANT -> {
                calculateLinearDecayConcentrationWithDoses(allRelevantEvents, canonicalDrug, currentTime)
            }
            DecayType.HALF_LIFE -> {
                calculateHalfLifeConcentrationWithDoses(allRelevantEvents, canonicalDrug, currentTime)
            }
            null -> 0.0 // No decay type means no concentration tracking
        }
    }

    private fun calculateLinearDecayConcentrationWithDoses(
        doseEvents: List<Pair<Long, Double>>,
        drug: Drug,
        targetTime: Date
    ): Double {
        val hourlyDecayRate = drug.hourlyDecayRate ?: 0.0
        if (hourlyDecayRate <= 0.0) return 0.0

        // Filter events that happened before target time
        val filteredEvents = doseEvents.filter { it.first <= targetTime.time }
            .sortedBy { it.first }

        if (filteredEvents.isEmpty()) return 0.0

        // Simulate concentration over time
        var totalConcentration = 0.0
        var lastTime = filteredEvents.first().first

        for ((eventTime, dosage) in filteredEvents) {
            // Apply decay from last time to this event
            val timeDiffHours = (eventTime - lastTime) / (1000.0 * 60.0 * 60.0)
            totalConcentration = maxOf(0.0, totalConcentration - (hourlyDecayRate * timeDiffHours))
            
            // Add new dose
            totalConcentration += dosage
            lastTime = eventTime
        }

        // Apply final decay from last event to target time
        val finalTimeDiffHours = (targetTime.time - lastTime) / (1000.0 * 60.0 * 60.0)
        totalConcentration = maxOf(0.0, totalConcentration - (hourlyDecayRate * finalTimeDiffHours))

        return totalConcentration
    }

    private fun calculateHalfLifeConcentrationWithDoses(
        doseEvents: List<Pair<Long, Double>>,
        drug: Drug,
        currentTime: Date
    ): Double {
        var totalConcentration = 0.0

        for ((eventTime, dosage) in doseEvents) {
            // Skip events that are in the future
            if (eventTime > currentTime.time) continue
            
            val hoursElapsed = (currentTime.time - eventTime) / (1000.0 * 60.0 * 60.0)

            val halfLife = drug.halfLifeHours ?: 24.0
            if (halfLife > 0 && hoursElapsed >= 0) {
                totalConcentration += dosage * (0.5).pow(hoursElapsed / halfLife)
            }
        }

        return totalConcentration
    }

    /**
     * Calculate current concentration for each drug in a template.
     * Returns a map of drugType -> current concentration.
     */
    fun calculateConcentrationsPerDrug(
        template: Template,
        events: List<CalendarEvent>,
        currentTime: Date = Date(),
        allTemplates: List<Template> = emptyList()
    ): Map<String, Double> {
        if (template.templateType != TemplateType.DECAYING || template.drugs.isEmpty()) {
            return emptyMap()
        }

        val relevantEvents = EventMatching.filterEventsForTemplate(events, template).sortedBy { it.startTime }
        
        return template.drugs.associate { drug ->
            drug.drugType to calculateDrugConcentration(drug, relevantEvents, currentTime, allTemplates)
        }
    }

    /**
     * Calculate peak concentration for each drug in a template.
     * Returns a map of drugType -> peak concentration.
     */
    fun calculatePeakConcentrationsPerDrug(
        template: Template,
        events: List<CalendarEvent>,
        daysBack: Int = 90
    ): Map<String, Double> {
        if (template.templateType != TemplateType.DECAYING || template.drugs.isEmpty()) {
            return emptyMap()
        }

        val currentTime = Date()
        val startTime = Date(currentTime.time - (daysBack * 24 * 60 * 60 * 1000L))
        val relevantEvents = EventMatching.filterEventsForTemplate(events, template).sortedBy { it.startTime }
        
        return template.drugs.associate { drug ->
            val history = generateDrugHistoryForTimeRange(drug, relevantEvents, startTime, currentTime)
            drug.drugType to (history.maxOfOrNull { it.second } ?: 0.0)
        }
    }

    /**
     * Generate concentration history for a single drug.
     */
    private fun generateDrugHistoryForTimeRange(
        drug: Drug,
        events: List<CalendarEvent>,
        startTime: Date,
        endTime: Date
    ): List<Pair<Date, Double>> {
        val history = mutableListOf<Pair<Date, Double>>()
        val totalTimeMillis = endTime.time - startTime.time
        val timeStepMillis = totalTimeMillis / 500L
        
        var time = startTime.time
        for (i in 0..500) {
            val pointTime = Date(time)
            val concentration = calculateDrugConcentration(drug, events, pointTime)
            history.add(Pair(pointTime, concentration))
            time += timeStepMillis
            if (time > endTime.time) break
        }
        
        return history
    }

    fun calculatePeakConcentration(
        template: Template,
        events: List<CalendarEvent>,
        daysBack: Int = 90
    ): Double {
        if (template.templateType != TemplateType.DECAYING || template.drugs.isEmpty()) {
            return 0.0
        }

        val currentTime = Date()
        val startTime = Date(currentTime.time - (daysBack * 24 * 60 * 60 * 1000L))
        
        // Generate concentration history and find the peak
        val history = generateConcentrationHistoryForTimeRange(template, events, startTime, currentTime)
        return history.maxOfOrNull { it.second } ?: 0.0
    }

    fun getLastEventTime(template: Template, events: List<CalendarEvent>): Date? {
        val relevantEvents = EventMatching.filterEventsForTemplate(events, template)
        return relevantEvents.maxByOrNull { it.startTime }?.startTime
    }

    /**
     * Get the time duration since the last event.
     * Returns TimeDuration.NEVER if no event exists.
     * Use TimeFormatter.format() to convert to a display string.
     */
    fun getTimeSinceLastEvent(lastEventTime: Date?, currentTime: Date = Date()): TimeDuration {
        if (lastEventTime == null) return TimeDuration.NEVER
        return TimeDuration.between(lastEventTime, currentTime)
    }

    /**
     * Calculate the peak concentration immediately after a specific event.
     * This is used for stable event marker positioning in graphs.
     */
    fun calculatePeakAfterEvent(
        template: Template,
        events: List<CalendarEvent>,
        targetEvent: CalendarEvent
    ): Double {
        if (template.templateType != TemplateType.DECAYING || template.drugs.isEmpty()) {
            return 0.0
        }

        // Calculate concentration at the moment right after this event (1 second later)
        val momentAfterEvent = Date(targetEvent.startTime.time + 1000)
        return calculateCurrentConcentration(template, events, momentAfterEvent)
    }

    fun generateConcentrationHistory(
        template: Template,
        events: List<CalendarEvent>,
        daysBack: Int
    ): List<Pair<Date, Double>> {
        if (template.templateType != TemplateType.DECAYING || template.drugs.isEmpty()) {
            return emptyList()
        }

        val relevantEvents = EventMatching.filterEventsForTemplate(events, template).sortedBy { it.startTime }
        val history = mutableListOf<Pair<Date, Double>>()
        val currentTime = Date()
        val startTime = Date(currentTime.time - (daysBack * 24 * 60 * 60 * 1000L))

        // Generate exactly 1000 data points
        val totalTimeMillis = currentTime.time - startTime.time
        val timeStepMillis = totalTimeMillis / 1000L
        
        var time = startTime.time
        val endTime = currentTime.time

        for (i in 0..1000) {
            if (time > endTime) break
            
            val pointTime = Date(time)
            // Sum concentration from all drugs
            val concentration = template.drugs.sumOf { drug ->
                when (drug.decayType) {
                    DecayType.CONSTANT -> calculateLinearDecayConcentration(relevantEvents, drug, pointTime)
                    DecayType.HALF_LIFE -> calculateHalfLifeConcentration(relevantEvents, drug, pointTime)
                    null -> 0.0 // No decay type means no concentration tracking
                }
            }

            history.add(Pair(pointTime, concentration))
            time += timeStepMillis
        }

        return history
    }

    fun generateConcentrationHistoryForTimeRange(
        template: Template,
        events: List<CalendarEvent>,
        startTime: Date,
        endTime: Date
    ): List<Pair<Date, Double>> {
        if (template.templateType != TemplateType.DECAYING || template.drugs.isEmpty()) {
            return emptyList()
        }

        val relevantEvents = EventMatching.filterEventsForTemplate(events, template).sortedBy { it.startTime }
        val history = mutableListOf<Pair<Date, Double>>()
        
        // Generate data points for the specified time range
        val totalTimeMillis = endTime.time - startTime.time
        val timeStepMillis = totalTimeMillis / 500L // 500 data points for smooth curves
        
        var time = startTime.time

        for (i in 0..500) {
            val pointTime = Date(time)
            // Sum concentration from all drugs
            val concentration = template.drugs.sumOf { drug ->
                when (drug.decayType) {
                    DecayType.CONSTANT -> calculateLinearDecayConcentration(relevantEvents, drug, pointTime)
                    DecayType.HALF_LIFE -> calculateHalfLifeConcentration(relevantEvents, drug, pointTime)
                    null -> 0.0 // No decay type means no concentration tracking
                }
            }

            history.add(Pair(pointTime, concentration))
            time += timeStepMillis
            
            if (time > endTime.time) break
        }

        return history
    }

    fun generateDrugConcentrationHistory(
        drugType: String,
        templates: List<Template>,
        events: List<CalendarEvent>,
        daysBack: Int
    ): List<Pair<Date, Double>> {
        // Find the source drug definition (with proper decay parameters) using DrugResolution
        val canonicalDrug = DrugResolution.findSourceDrug(drugType, templates) ?: return emptyList()

        // Find all templates that contain this drug type
        val drugTemplates = templates.filter { template ->
            template.templateType == TemplateType.DECAYING &&
            template.drugs.any { it.drugType.equals(drugType, ignoreCase = true) }
        }

        // Collect all events from templates containing this drug
        val allRelevantEvents = drugTemplates.flatMap { template ->
            val templateEvents = EventMatching.filterEventsForTemplate(events, template)
            val drugInTemplate = template.drugs.find { it.drugType.equals(drugType, ignoreCase = true) }
            templateEvents.map { event ->
                val effectiveDosage = event.dosage * (drugInTemplate?.factor ?: 1.0)
                Pair(event.startTime.time, effectiveDosage)
            }
        }.filter { it.second > 0.0 }
            .sortedBy { it.first }

        val history = mutableListOf<Pair<Date, Double>>()
        val currentTime = Date()
        val startTime = Date(currentTime.time - (daysBack * 24 * 60 * 60 * 1000L))

        // Generate exactly 1000 data points
        val totalTimeMillis = currentTime.time - startTime.time
        val timeStepMillis = totalTimeMillis / 1000L
        
        var time = startTime.time
        val endTime = currentTime.time

        for (i in 0..1000) {
            if (time > endTime) break
            
            val pointTime = Date(time)
            val concentration = when (canonicalDrug.decayType) {
                DecayType.CONSTANT -> {
                    calculateLinearDecayConcentrationWithDoses(allRelevantEvents, canonicalDrug, pointTime)
                }
                DecayType.HALF_LIFE -> {
                    calculateHalfLifeConcentrationWithDoses(allRelevantEvents, canonicalDrug, pointTime)
                }
                null -> 0.0 // No decay type means no concentration tracking
            }

            history.add(Pair(pointTime, concentration))
            time += timeStepMillis
        }

        return history
    }

    fun getLastDrugEventTime(drugType: String, templates: List<Template>, events: List<CalendarEvent>): Date? {
        // Get all templates for this drug type
        val drugTemplates = templates.filter { template ->
            template.templateType == TemplateType.DECAYING &&
            template.drugs.any { it.drugType.equals(drugType, ignoreCase = true) }
        }

        // Get all events from all templates for this drug type
        val relevantEvents = drugTemplates.flatMap { template ->
            EventMatching.filterEventsForTemplate(events, template)
        }.distinct()

        return relevantEvents.maxByOrNull { it.startTime }?.startTime
    }
}
