package de.ntdote.medicalcalendarlog.ui.screens

import android.text.util.Linkify
import android.widget.TextView
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.compose.ui.viewinterop.AndroidView
import de.ntdote.medicalcalendarlog.R
import io.noties.markwon.Markwon
import io.noties.markwon.linkify.LinkifyPlugin

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun HelpScreen(
    onNavigateBack: () -> Unit
) {
    val context = LocalContext.current
    val readmeContent = remember {
        try {
            context.resources.openRawResource(R.raw.readme)
                .bufferedReader()
                .use { it.readText() }
        } catch (e: Exception) {
            "# Error Loading Help\n\nError loading help content: ${e.message}\n\nPlease visit:\n[Medical Calendar Log on Codeberg](https://codeberg.org/svewa/MedicalCalendarLog)"
        }
    }

    // Get Material3 colors for theming
    val textColor = MaterialTheme.colorScheme.onBackground.hashCode()
    val linkColor = MaterialTheme.colorScheme.primary.hashCode()
    val backgroundColor = MaterialTheme.colorScheme.background.hashCode()

    Scaffold(
        topBar = {
            TopAppBar(
                title = { Text("Help") },
                navigationIcon = {
                    IconButton(onClick = onNavigateBack) {
                        Icon(
                            Icons.AutoMirrored.Filled.ArrowBack,
                            contentDescription = "Back"
                        )
                    }
                }
            )
        }
    ) { paddingValues ->
        Column(
            modifier = Modifier
                .fillMaxSize()
                .padding(paddingValues)
        ) {
            AndroidView(
                factory = { ctx ->
                    TextView(ctx).apply {
                        // Configure Markwon with basic markdown support
                        val markwon = Markwon.builder(ctx)
                            .usePlugin(LinkifyPlugin.create(
                                // Only enable web URLs and email, disable phone numbers
                                Linkify.WEB_URLS or Linkify.EMAIL_ADDRESSES
                            ))
                            .build()

                        // Set text color and link color from Material3 theme
                        setTextColor(textColor)
                        setLinkTextColor(linkColor)
                        setBackgroundColor(backgroundColor)
                        
                        // Set padding
                        setPadding(
                            (16 * resources.displayMetrics.density).toInt(),
                            (16 * resources.displayMetrics.density).toInt(),
                            (16 * resources.displayMetrics.density).toInt(),
                            (16 * resources.displayMetrics.density).toInt()
                        )

                        // Render markdown
                        markwon.setMarkdown(this, readmeContent)
                    }
                },
                modifier = Modifier
                    .fillMaxSize()
                    .verticalScroll(rememberScrollState())
            )
        }
    }
}
