package de.ntdote.medicalcalendarlog.ui.screens

import android.content.Intent
import android.net.Uri
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.ArrowDropDown
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material.icons.filled.Edit
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.graphicsLayer
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.SpanStyle
import androidx.compose.ui.text.buildAnnotatedString
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextDecoration
import androidx.compose.ui.text.withStyle
import androidx.compose.ui.unit.dp
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.compose.LifecycleResumeEffect
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import androidx.lifecycle.viewmodel.compose.viewModel
import de.ntdote.medicalcalendarlog.data.DecayType
import de.ntdote.medicalcalendarlog.data.ExtendedTemplateData
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.data.TemplateType
import de.ntdote.medicalcalendarlog.repository.CalendarRepository
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.ui.BackupSelectionActivity
import de.ntdote.medicalcalendarlog.ui.components.TemplateDialog
import de.ntdote.medicalcalendarlog.viewmodel.MainViewModel

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun ConfigurationScreen(
    onNavigateBack: () -> Unit,
    viewModel: MainViewModel? = null
) {
    val context = LocalContext.current
    val actualViewModel = viewModel ?: viewModel {
        MainViewModel(
            application = context.applicationContext as android.app.Application,
            calendarRepository = CalendarRepository(context.contentResolver, context),
            preferencesRepository = PreferencesRepository(context)
        )
    }
    val uiState by actualViewModel.uiState.collectAsStateWithLifecycle()
    val availableCalendars by actualViewModel.availableCalendars.collectAsStateWithLifecycle()
    val calendarStatus by actualViewModel.calendarStatus.collectAsStateWithLifecycle()
    
    var showTemplateDialog by remember { mutableStateOf(false) }
    var editingTemplate by remember { mutableStateOf<Template?>(null) }
    var showCalendarPicker by remember { mutableStateOf(false) }
    var showCalendarTroubleshooting by remember { mutableStateOf(false) }

    Scaffold(
        topBar = {
            TopAppBar(
                title = { Text("Configuration") },
                navigationIcon = {
                    IconButton(onClick = onNavigateBack) {
                        Icon(Icons.AutoMirrored.Filled.ArrowBack, contentDescription = "Back")
                    }
                }
            )
        },
        floatingActionButton = {
            FloatingActionButton(
                onClick = { 
                    editingTemplate = null
                    showTemplateDialog = true 
                }
            ) {
                Icon(Icons.Default.Add, contentDescription = "Add Template")
            }
        }
    ) { paddingValues ->
        LazyColumn(
            modifier = Modifier
                .fillMaxSize()
                .padding(paddingValues)
                .padding(16.dp),
            verticalArrangement = Arrangement.spacedBy(16.dp)
        ) {
            // Calendar Selection
            item {
                Card {
                    Column(
                        modifier = Modifier.padding(16.dp)
                    ) {
                        Text(
                            text = "Calendar Settings",
                            style = MaterialTheme.typography.titleMedium,
                            fontWeight = FontWeight.Bold
                        )
                        Spacer(modifier = Modifier.height(8.dp))
                        
                        OutlinedButton(
                            onClick = { showCalendarPicker = true },
                            modifier = Modifier.fillMaxWidth()
                        ) {
                            val selectedCalendar = availableCalendars.find { it.id == uiState.selectedCalendarId }
                            Text(
                                text = selectedCalendar?.displayName ?: "Select Calendar"
                            )
                        }
                        
                        Spacer(modifier = Modifier.height(8.dp))
                        
                        var daysBackText by remember { mutableStateOf(uiState.daysBack.toString()) }
                        
                        // Update text when uiState changes (e.g., from other sources)
                        LaunchedEffect(uiState.daysBack) {
                            daysBackText = uiState.daysBack.toString()
                        }
                        
                        // Handle focus loss - validate and set default if needed
                                DisposableEffect(Unit) {
                                    onDispose {
                                        if (daysBackText.isBlank() || daysBackText.toIntOrNull() == null || daysBackText.toIntOrNull() == 0) {
                                            actualViewModel.setDaysBack(1)
                                            daysBackText = "1"
                                        }
                                    }
                                }
                        
                        // Days Back and Debug in one row
                        Row(
                            modifier = Modifier.fillMaxWidth(),
                            horizontalArrangement = Arrangement.spacedBy(8.dp),
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            OutlinedTextField(
                                value = daysBackText,
                                onValueChange = { value ->
                                    daysBackText = value
                                    
                                    // Allow empty field temporarily
                                    if (value.isEmpty()) {
                                        return@OutlinedTextField
                                    }
                                    
                                    // Validate and update
                                    val days = value.toIntOrNull()
                                    if (days != null && days > 0) {
                                        actualViewModel.setDaysBack(days)
                                    }
                                },
                                label = { Text("Days Back") },
                                keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                                modifier = Modifier.weight(1f),
                                isError = daysBackText.isNotEmpty() && (daysBackText.toIntOrNull() == null || daysBackText.toIntOrNull() == 0),
                                supportingText = {
                                    if (daysBackText.isNotEmpty() && (daysBackText.toIntOrNull() == null || daysBackText.toIntOrNull() == 0)) {
                                        Text(
                                            text = "Must be positive",
                                            color = MaterialTheme.colorScheme.error
                                        )
                                    }
                                }
                            )
                            
                            // Debug Mode Setting
                            Row(
                                verticalAlignment = Alignment.CenterVertically,
                                horizontalArrangement = Arrangement.spacedBy(8.dp)
                            ) {
                                Text(
                                    text = "Debug",
                                    style = MaterialTheme.typography.bodyMedium
                                )
                                Switch(
                                    checked = uiState.debugModeEnabled,
                                    onCheckedChange = { enabled ->
                                        actualViewModel.setDebugModeEnabled(enabled)
                                    }
                                )
                            }
                        }
                        
                        Spacer(modifier = Modifier.height(12.dp))
                        
                        // Calendar Status Display (show only when there are issues OR debug mode is on)
                        calendarStatus?.let { status ->
                            // Show status card only if:
                            // 1. There are issues (no permissions, no calendars, or error)
                            // 2. OR debug mode is enabled (to show "X calendars available")
                            if (!status.hasPermissions || status.calendarsFound == 0 || status.errorMessage != null || uiState.debugModeEnabled) {
                                CalendarStatusCard(
                                    status = status,
                                    debugModeEnabled = uiState.debugModeEnabled,
                                    onRefresh = { actualViewModel.refreshCalendarStatus() },
                                    onShowTroubleshooting = { showCalendarTroubleshooting = true }
                                )
                            }
                        }
                    }
                }
            }
            
            // Calendar Troubleshooting Card (only shown when there are issues)
            calendarStatus?.let { status ->
                if (!status.hasPermissions || status.calendarsFound == 0 || status.errorMessage != null) {
                    item {
                        CalendarTroubleshootingCard(
                            status = status,
                            onRefresh = { actualViewModel.refreshCalendarStatus() }
                        )
                    }
                }
            }

            // Export/Import Section
            item {
                Card {
                    Column(
                        modifier = Modifier.padding(16.dp)
                    ) {
                        Text(
                            text = "Template Backup",
                            style = MaterialTheme.typography.titleMedium,
                            fontWeight = FontWeight.Bold
                        )
                        Spacer(modifier = Modifier.height(12.dp))
                        
                        // Single row: Backup and Restore buttons
                        Row(
                            modifier = Modifier.fillMaxWidth(),
                            horizontalArrangement = Arrangement.spacedBy(8.dp)
                        ) {
                            OutlinedButton(
                                onClick = {
                                    val result = actualViewModel.exportTemplatesToCalendar()
                                    // TODO: Show success/error message
                                },
                                enabled = !uiState.exportInProgress && uiState.templates.isNotEmpty(),
                                modifier = Modifier.weight(1f)
                            ) {
                                if (uiState.exportInProgress) {
                                    CircularProgressIndicator(
                                        modifier = Modifier.size(16.dp),
                                        strokeWidth = 2.dp
                                    )
                                } else {
                                    Text("Backup")
                                }
                            }
                            
                            OutlinedButton(
                                onClick = {
                                    val intent = Intent(context, BackupSelectionActivity::class.java)
                                    context.startActivity(intent)
                                },
                                modifier = Modifier.weight(1f)
                            ) {
                                Text("Restore")
                            }
                        }
                    }
                }
            }

            // Background Jobs Card
            item {
                BackgroundJobsCard(viewModel = actualViewModel)
            }

            // Templates Section
            item {
                Text(
                    text = "Templates",
                    style = MaterialTheme.typography.titleMedium,
                    fontWeight = FontWeight.Bold
                )
            }

            if (uiState.templates.isEmpty()) {
                item {
                    Card {
                        Column(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(16.dp),
                            horizontalAlignment = Alignment.CenterHorizontally
                        ) {
                            Text(
                                text = "No templates configured",
                                style = MaterialTheme.typography.bodyMedium
                            )
                            Spacer(modifier = Modifier.height(8.dp))
                            Text(
                                text = "Tap the + button to add your first template",
                                style = MaterialTheme.typography.bodySmall,
                                color = MaterialTheme.colorScheme.onSurfaceVariant
                            )
                        }
                    }
                }
            } else {
                items(uiState.templates) { template ->
                    TemplateCard(
                        template = template,
                        allTemplates = uiState.templates,
                        onEdit = {
                            editingTemplate = template
                            showTemplateDialog = true
                        },
                        onDelete = {
                            actualViewModel.removeTemplate(template.id)
                        }
                    )
                }
            }
            
            // Build Information Section
            item {
                Spacer(modifier = Modifier.height(16.dp))
                Card {
                    Column(
                        modifier = Modifier.padding(16.dp)
                    ) {
                        Text(
                            text = "Build Information",
                            style = MaterialTheme.typography.titleMedium,
                            fontWeight = FontWeight.Bold
                        )
                        Spacer(modifier = Modifier.height(8.dp))
                        
                        Text(
                            text = "Version: ${de.ntdote.medicalcalendarlog.BuildConfig.VERSION_NAME} (${de.ntdote.medicalcalendarlog.BuildConfig.VERSION_CODE})",
                            style = MaterialTheme.typography.bodySmall
                        )
                        
                        // Clickable repository link
                        Text(
                            text = buildAnnotatedString {
                                append("Repository: ")
                                withStyle(
                                    style = SpanStyle(
                                        color = MaterialTheme.colorScheme.primary,
                                        textDecoration = TextDecoration.Underline
                                    )
                                ) {
                                    append("codeberg.org/svewa/...")
                                }
                            },
                            style = MaterialTheme.typography.bodySmall,
                            modifier = Modifier.clickable {
                                val intent = Intent(Intent.ACTION_VIEW, Uri.parse(de.ntdote.medicalcalendarlog.BuildConfig.REPOSITORY_URL))
                                context.startActivity(intent)
                            }
                        )
                        
                        Text(
                            text = "Build type: ${if (de.ntdote.medicalcalendarlog.BuildConfig.IS_DEBUG) "Debug" else "Release"}",
                            style = MaterialTheme.typography.bodySmall
                        )
                    }
                }
            }
        }
    }

    // Calendar Picker Dialog
    if (showCalendarPicker) {
        AlertDialog(
            onDismissRequest = { showCalendarPicker = false },
            title = { Text("Select Calendar") },
            text = {
                LazyColumn {
                    items(availableCalendars) { calendar ->
                        TextButton(
                            onClick = {
                                actualViewModel.setSelectedCalendar(calendar.id)
                                showCalendarPicker = false
                            },
                            modifier = Modifier.fillMaxWidth()
                        ) {
                            Column(
                                modifier = Modifier.fillMaxWidth()
                            ) {
                                Text(
                                    text = calendar.displayName,
                                    style = MaterialTheme.typography.bodyMedium
                                )
                                Text(
                                    text = calendar.accountName,
                                    style = MaterialTheme.typography.bodySmall,
                                    color = MaterialTheme.colorScheme.onSurfaceVariant
                                )
                            }
                        }
                    }
                }
            },
            confirmButton = {
                TextButton(onClick = { showCalendarPicker = false }) {
                    Text("Cancel")
                }
            }
        )
    }

    // Template Dialog
    if (showTemplateDialog) {
        TemplateDialog(
            template = editingTemplate,
            existingTemplates = uiState.templates,
            onDismiss = { 
                showTemplateDialog = false
                editingTemplate = null
            },
            onSave = { template ->
                if (editingTemplate != null) {
                    actualViewModel.updateTemplate(template)
                } else {
                    actualViewModel.addTemplate(template)
                }
                showTemplateDialog = false
                editingTemplate = null
            }
        )
    }
}

@Composable
private fun BackgroundJobsCard(viewModel: MainViewModel) {
    val context = LocalContext.current
    var isExpanded by remember { mutableStateOf(false) }
    var isUnrestricted by remember { mutableStateOf(viewModel.isBackgroundUnrestricted()) }
    
    // Update status when card is expanded
    LaunchedEffect(isExpanded) {
        if (isExpanded) {
            isUnrestricted = viewModel.isBackgroundUnrestricted()
        }
    }
    
    // Update status when screen resumes (user returns from system settings)
    LifecycleResumeEffect(Unit) {
        isUnrestricted = viewModel.isBackgroundUnrestricted()
        onPauseOrDispose {
            // No cleanup needed
        }
    }

    Card(
        modifier = Modifier.fillMaxWidth()
    ) {
        Column(
            modifier = Modifier
                .fillMaxWidth()
                .clickable { isExpanded = !isExpanded }
                .padding(16.dp)
        ) {
            // Collapsed header - always visible
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.CenterVertically
            ) {
                Text(
                    text = "Background jobs: ",
                    style = MaterialTheme.typography.bodyMedium
                )
                Row(
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.spacedBy(4.dp)
                ) {
                    Text(
                        text = if (isUnrestricted) "unrestricted" else "restricted",
                        style = MaterialTheme.typography.bodyMedium,
                        color = if (isUnrestricted) 
                            MaterialTheme.colorScheme.primary 
                        else 
                            MaterialTheme.colorScheme.error,
                        fontWeight = FontWeight.Bold
                    )
                    Icon(
                        imageVector = Icons.Default.ArrowDropDown,
                        contentDescription = if (isExpanded) "Collapse" else "Expand",
                        modifier = Modifier.graphicsLayer {
                            rotationZ = if (isExpanded) 180f else 0f
                        }
                    )
                }
            }
            
            // Expanded content
            AnimatedVisibility(visible = isExpanded) {
                Column(
                    modifier = Modifier.fillMaxWidth()
                ) {
                    Spacer(modifier = Modifier.height(12.dp))
                    
                    Text(
                        text = "Background job restrictions can prevent scheduled notifications from appearing on time and may cause widgets to display outdated data.",
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                    
                    Spacer(modifier = Modifier.height(8.dp))
                    
                    Text(
                        text = "This setting allows the app to run background tasks without restrictions, ensuring timely notifications and accurate widget updates.",
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                    
                    Spacer(modifier = Modifier.height(8.dp))
                    
                    Text(
                        text = "Note: Some device manufacturers implement additional battery optimizations beyond stock Android. If you continue to experience issues with notifications or widgets, check your device's battery settings for manufacturer-specific options.",
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant,
                        fontWeight = FontWeight.Light
                    )
                    
                    // Show button only if restricted
                    if (!isUnrestricted) {
                        Spacer(modifier = Modifier.height(12.dp))
                        
                        Button(
                            onClick = {
                                val intent = viewModel.createBatteryOptimizationIntent()
                                context.startActivity(intent)
                                // Update status after a delay to allow user to complete the action
                                android.os.Handler(android.os.Looper.getMainLooper()).postDelayed({
                                    isUnrestricted = viewModel.isBackgroundUnrestricted()
                                }, 1000)
                            },
                            modifier = Modifier.fillMaxWidth()
                        ) {
                            Text("Remove Restrictions")
                        }
                    }
                }
            }
        }
    }
}

@Composable
private fun CalendarStatusCard(
    status: CalendarRepository.CalendarStatus,
    debugModeEnabled: Boolean,
    onRefresh: () -> Unit,
    onShowTroubleshooting: () -> Unit
) {
    Card(
        colors = CardDefaults.cardColors(
            containerColor = when {
                !status.hasPermissions -> MaterialTheme.colorScheme.errorContainer
                status.calendarsFound == 0 -> MaterialTheme.colorScheme.tertiaryContainer
                else -> MaterialTheme.colorScheme.surfaceVariant
            }
        )
    ) {
        Column(
            modifier = Modifier.padding(12.dp)
        ) {
            // Status header
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.CenterVertically
            ) {
                Column {
                    Text(
                        text = "Calendar Status",
                        style = MaterialTheme.typography.labelMedium,
                        fontWeight = FontWeight.Bold,
                        color = when {
                            !status.hasPermissions -> MaterialTheme.colorScheme.error
                            status.calendarsFound == 0 -> MaterialTheme.colorScheme.onTertiaryContainer
                            else -> MaterialTheme.colorScheme.onSurfaceVariant
                        }
                    )
                    
                    Spacer(modifier = Modifier.height(4.dp))
                    
                    Text(
                        text = when {
                            !status.hasPermissions -> "Permissions missing"
                            status.calendarsFound == 0 -> "No calendars found"
                            else -> "${status.calendarsFound} calendar${if (status.calendarsFound != 1) "s" else ""} available"
                        },
                        style = MaterialTheme.typography.bodySmall,
                        color = when {
                            !status.hasPermissions -> MaterialTheme.colorScheme.error
                            status.calendarsFound == 0 -> MaterialTheme.colorScheme.onTertiaryContainer
                            else -> MaterialTheme.colorScheme.onSurface
                        }
                    )
                }
                
                Row(
                    verticalAlignment = Alignment.CenterVertically,
                    horizontalArrangement = Arrangement.spacedBy(8.dp)
                ) {
                    IconButton(
                        onClick = onRefresh,
                        modifier = Modifier.size(32.dp)
                    ) {
                        Icon(
                            imageVector = Icons.Default.ArrowDropDown, // Using available icon
                            contentDescription = "Refresh",
                            modifier = Modifier.graphicsLayer {
                                rotationZ = 90f
                            }
                        )
                    }
                }
            }
            
            // Error message if present
            status.errorMessage?.let { errorMessage ->
                Spacer(modifier = Modifier.height(8.dp))
                Text(
                    text = errorMessage,
                    style = MaterialTheme.typography.bodySmall,
                    color = MaterialTheme.colorScheme.error
                )
            }
            
            // Troubleshooting button for issues
            if (!status.hasPermissions || status.calendarsFound == 0) {
                Spacer(modifier = Modifier.height(8.dp))
                Button(
                    onClick = onShowTroubleshooting,
                    modifier = Modifier.fillMaxWidth(),
                    colors = ButtonDefaults.buttonColors(
                        containerColor = when {
                            !status.hasPermissions -> MaterialTheme.colorScheme.error
                            else -> MaterialTheme.colorScheme.primary
                        }
                    )
                ) {
                    Text("Troubleshoot")
                }
            }
        }
    }
}

@Composable
private fun CalendarTroubleshootingCard(
    status: CalendarRepository.CalendarStatus,
    onRefresh: () -> Unit
) {
    Card(
        colors = CardDefaults.cardColors(
            containerColor = MaterialTheme.colorScheme.surfaceVariant
        )
    ) {
        Column(
            modifier = Modifier.padding(16.dp)
        ) {
            Text(
                text = "Calendar Troubleshooting",
                style = MaterialTheme.typography.titleMedium,
                fontWeight = FontWeight.Bold
            )
            
            Spacer(modifier = Modifier.height(12.dp))
            
            when {
                !status.hasPermissions -> {
                    Text(
                        text = "Calendar permissions are required for this app to function.",
                        style = MaterialTheme.typography.bodyMedium
                    )
                    
                    Spacer(modifier = Modifier.height(8.dp))
                    
                    Text(
                        text = """1. Go back to the main screen to grant permissions
2. If permissions were denied, open app settings to grant them
3. Make sure you grant both READ and WRITE calendar permissions""",
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                }
                
                status.calendarsFound == 0 -> {
                    Text(
                        text = "No calendars found on your device.",
                        style = MaterialTheme.typography.bodyMedium
                    )
                    
                    Spacer(modifier = Modifier.height(8.dp))
                    
                    Text(
                        text = """Common issues on virtual tablets:

1. Calendar sync may be disabled
   • Open your calendar app
   • Check if calendars are syncing
   • Enable sync in system Accounts settings

2. No calendars exist
   • Open your calendar app
   • Create at least one calendar
   • Recommended: create a 'medilog' calendar

3. Calendar content provider unavailable
   • Make sure to have some calendar sync adapter or local
   • Or use a physical device for testing""",
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                    
                    Spacer(modifier = Modifier.height(12.dp))
                    
                    Button(
                        onClick = onRefresh,
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Text("Check Again")
                    }
                }
                
                else -> {
                    Text(
                        text = "Unknown calendar issue.",
                        style = MaterialTheme.typography.bodyMedium
                    )
                    
                    Spacer(modifier = Modifier.height(8.dp))
                    
                    Text(
                        text = "Error: ${status.errorMessage}",
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.error
                    )
                }
            }
        }
    }
}

@Composable
private fun TemplateCard(
    template: Template,
    allTemplates: List<Template> = emptyList(),
    onEdit: () -> Unit,
    onDelete: () -> Unit
) {
    Card {
        Column(
            modifier = Modifier.padding(16.dp)
        ) {
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.Top
            ) {
                Column(
                    modifier = Modifier.weight(1f)
                ) {
                    Text(
                        text = template.name,
                        style = MaterialTheme.typography.titleSmall,
                        fontWeight = FontWeight.Bold
                    )
                    
                    if (template.templateType == TemplateType.DECAYING && template.drugs.isNotEmpty()) {
                        Spacer(modifier = Modifier.height(4.dp))
                        
                        // Display all drugs in template
                        Text(
                            text = if (template.isSingleDrug()) {
                                "Drug: ${template.getDrugTypesDisplay()}"
                            } else {
                                "Drugs: ${template.getDrugTypesDisplay()}"
                            },
                            style = MaterialTheme.typography.bodySmall
                        )
                        
                        // For single drug templates, show additional details
                        if (template.isSingleDrug()) {
                            val drug = template.drugs.first()
                            Text(
                                text = "Factor: ${drug.factor} ${drug.unit}",
                                style = MaterialTheme.typography.bodySmall
                            )
                            
                            // Handle decay display
                            val decayText = if (drug.decayType != null) {
                                "Decay: ${drug.decayType}"
                            } else {
                                "Decay: derived"
                            }
                            
                            Text(
                                text = decayText,
                                style = MaterialTheme.typography.bodySmall
                            )
                        } else {
                            // For multi-drug templates, show count and source/derived info
                            val sourceCount = template.getSourceDrugs().size
                            val derivedCount = template.getDerivedDrugs().size
                            
                            Text(
                                text = "Source drugs: $sourceCount, Derived: $derivedCount",
                                style = MaterialTheme.typography.bodySmall
                            )
                        }
                    }
                }
                
                Row {
                    IconButton(onClick = onEdit) {
                        Icon(Icons.Default.Edit, contentDescription = "Edit")
                    }
                    IconButton(onClick = onDelete) {
                        Icon(Icons.Default.Delete, contentDescription = "Delete")
                    }
                }
            }
        }
    }
}
