package de.ntdote.medicalcalendarlog.ui.components

import androidx.compose.foundation.layout.*
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.selection.selectable
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material.icons.filled.Close
import androidx.compose.material.icons.filled.ExpandLess
import androidx.compose.material.icons.filled.ExpandMore
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.semantics.Role
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import androidx.compose.ui.window.Dialog
import de.ntdote.medicalcalendarlog.data.DecayType
import de.ntdote.medicalcalendarlog.data.Drug
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.data.TemplateType
import java.util.UUID

// State holder for a drug being edited
data class DrugEditState(
    val drugType: String = "",
    val factor: String = "",
    val unit: String = "",
    val decayType: DecayType? = DecayType.HALF_LIFE,
    val hourlyDecayRate: String = "",
    val halfLifeHours: String = "",
    val concentrationReminderEnabled: Boolean = false,
    val concentrationReminderThreshold: String = "",
    val isExpanded: Boolean = true,
    val isSourceConflict: Boolean = false  // Flag to indicate source drug conflict
) {
    fun toDrug(allTemplates: List<Template> = emptyList()): Drug? {
        val drugTypeTrimmed = drugType.trim()
        val factorValue = factor.toDoubleOrNull()
        
        if (drugTypeTrimmed.isBlank() || factorValue == null || factorValue <= 0) {
            return null
        }
        
        val threshold = if (concentrationReminderEnabled) {
            concentrationReminderThreshold.toDoubleOrNull()
        } else null
        
        return try {
            if (decayType == null) {
                // This is a derived drug - deriv unit from source
                val sourceDrug = de.ntdote.medicalcalendarlog.utils.DrugResolution.findSourceDrug(drugTypeTrimmed, allTemplates)
                val effectiveUnit = sourceDrug?.unit ?: "mg" // Default to "mg" if no source found
                
                Drug.createDerived(
                    drugType = drugTypeTrimmed,
                    factor = factorValue,
                    unit = effectiveUnit,
                    concentrationReminderEnabled = concentrationReminderEnabled,
                    concentrationReminderThreshold = threshold
                )
            } else {
                // This is a source drug - use the unit specified in the form
                val unitTrimmed = unit.trim()
                if (unitTrimmed.isBlank()) {
                    return null
                }
                
                Drug.createSource(
                    drugType = drugTypeTrimmed,
                    factor = factorValue,
                    unit = unitTrimmed,
                    decayType = decayType,
                    hourlyDecayRate = if (decayType == DecayType.CONSTANT) hourlyDecayRate.toDoubleOrNull() else null,
                    halfLifeHours = if (decayType == DecayType.HALF_LIFE) halfLifeHours.toDoubleOrNull() else null,
                    concentrationReminderEnabled = concentrationReminderEnabled,
                    concentrationReminderThreshold = threshold
                )
            }
        } catch (e: Exception) {
            null
        }
    }
    
    companion object {
        fun fromDrug(drug: Drug, expanded: Boolean = true): DrugEditState {
            return DrugEditState(
                drugType = drug.drugType,
                factor = drug.factor.toString(),
                unit = drug.unit,
                decayType = drug.decayType,
                hourlyDecayRate = drug.hourlyDecayRate?.toString() ?: "",
                halfLifeHours = drug.halfLifeHours?.toString() ?: "",
                concentrationReminderEnabled = drug.concentrationReminderEnabled,
                concentrationReminderThreshold = drug.concentrationReminderThreshold?.toString() ?: "",
                isExpanded = expanded
            )
        }
    }
}

@Composable
fun TemplateDialog(
    template: Template?,
    existingTemplates: List<Template>,
    onDismiss: () -> Unit,
    onSave: (Template) -> Unit
) {
    var name by remember { mutableStateOf(template?.name ?: "") }
    var templateType by remember { mutableStateOf(template?.templateType ?: TemplateType.GENERAL) }
    var metricUnit by remember { mutableStateOf(template?.metricUnit ?: "") }
    
    // Drugs list for DECAYING templates
    val initialDrugs = template?.drugs?.mapIndexed { index, drug -> 
        DrugEditState.fromDrug(drug, expanded = index == 0) 
    } ?: emptyList()
    var drugs by remember { mutableStateOf(initialDrugs.toMutableList()) }
    
    // Reminder fields - convert from millis to hours for display
    var reminderEnabled by remember { mutableStateOf(template?.reminderEnabled ?: false) }
    var reminderThresholdHours by remember { 
        mutableStateOf(
            template?.getEffectiveReminderThresholdMillis()?.let { millis ->
                val hours = millis / (60.0 * 60.0 * 1000.0)
                if (hours == hours.toLong().toDouble()) {
                    hours.toLong().toString()
                } else {
                    String.format("%.2f", hours)
                }
            } ?: ""
        )
    }
    
    // Template enabled/disabled state
    var templateEnabled by remember { mutableStateOf(template?.enabled ?: true) }
    
    // Requires input state
    var requiresInput by remember { mutableStateOf(template?.requiresInput ?: false) }
    
    val focusRequester = remember { FocusRequester() }

    LaunchedEffect(Unit) {
        focusRequester.requestFocus()
    }

    Dialog(onDismissRequest = onDismiss) {
        Card(
            modifier = Modifier
                .fillMaxWidth()
                .padding(16.dp)
        ) {
            Column(
                modifier = Modifier
                    .padding(16.dp)
                    .verticalScroll(rememberScrollState())
            ) {
                Text(
                    text = if (template != null) "Edit Template" else "Add Template",
                    style = MaterialTheme.typography.titleLarge,
                    fontWeight = FontWeight.Bold
                )
                
                Spacer(modifier = Modifier.height(16.dp))
                
                OutlinedTextField(
                    value = name,
                    onValueChange = { name = it },
                    label = { Text("Template Name") },
                    placeholder = { Text(
                        when (templateType) {
                            TemplateType.METRIC -> "e.g., Weight: ⚖️"
                            TemplateType.DECAYING -> "e.g., Painkillers: 💊"
                            TemplateType.GENERAL -> "e.g., Water: 💧"
                        }
                    ) },
                    modifier = Modifier
                        .fillMaxWidth()
                        .focusRequester(focusRequester),
                    supportingText = { 
                        Text(
                            when (templateType) {
                                TemplateType.METRIC -> "Metric templates always prompt for value"
                                TemplateType.DECAYING -> "Track drug concentration over time"
                                TemplateType.GENERAL -> "Simple logging without calculations"
                            }
                        )
                    }
                )
                
                Spacer(modifier = Modifier.height(16.dp))
                
                // Template enabled/disabled switch
                Card(
                    modifier = Modifier.fillMaxWidth(),
                    colors = CardDefaults.cardColors(
                        containerColor = if (templateEnabled) 
                            MaterialTheme.colorScheme.primaryContainer 
                        else 
                            MaterialTheme.colorScheme.errorContainer
                    )
                ) {
                    Row(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(16.dp),
                        verticalAlignment = Alignment.CenterVertically,
                        horizontalArrangement = Arrangement.SpaceBetween
                    ) {
                        Column(
                            modifier = Modifier.weight(1f)
                        ) {
                            Text(
                                text = if (templateEnabled) "Template Enabled" else "Template Disabled",
                                style = MaterialTheme.typography.titleMedium,
                                fontWeight = FontWeight.Bold,
                                color = if (templateEnabled) 
                                    MaterialTheme.colorScheme.onPrimaryContainer 
                                else 
                                    MaterialTheme.colorScheme.onErrorContainer
                            )
                            Text(
                                text = if (templateEnabled) 
                                    "Template appears in main and graph screen" 
                                else 
                                    "Template is hidden from main and graph screen",
                                style = MaterialTheme.typography.bodySmall,
                                color = if (templateEnabled) 
                                    MaterialTheme.colorScheme.onPrimaryContainer.copy(alpha = 0.8f) 
                                else 
                                    MaterialTheme.colorScheme.onErrorContainer.copy(alpha = 0.8f)
                            )
                        }
                        
                        Switch(
                            checked = templateEnabled,
                            onCheckedChange = { templateEnabled = it }
                        )
                    }
                }
                
                if (!templateEnabled) {
                    Spacer(modifier = Modifier.height(8.dp))
                    Card(
                        modifier = Modifier.fillMaxWidth(),
                        colors = CardDefaults.cardColors(
                            containerColor = MaterialTheme.colorScheme.surfaceVariant
                        )
                    ) {
                        Text(
                            text = "Disabled templates are temporarily hidden but their configuration is preserved.",
                            style = MaterialTheme.typography.bodySmall,
                            color = MaterialTheme.colorScheme.onSurfaceVariant,
                            modifier = Modifier.padding(12.dp)
                        )
                    }
                }
                
                Spacer(modifier = Modifier.height(16.dp))
                
                // Requires Input checkbox (for GENERAL and DECAYING templates, not METRIC)
                if (templateType == TemplateType.GENERAL || templateType == TemplateType.DECAYING) {
                    Card(
                        modifier = Modifier.fillMaxWidth(),
                        colors = CardDefaults.cardColors(
                            containerColor = MaterialTheme.colorScheme.secondaryContainer
                        )
                    ) {
                        Row(
                            modifier = Modifier
                                .fillMaxWidth()
                                .padding(16.dp),
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            Checkbox(
                                checked = requiresInput,
                                onCheckedChange = { requiresInput = it }
                            )
                            Spacer(modifier = Modifier.width(12.dp))
                            Column(
                                modifier = Modifier.weight(1f)
                            ) {
                                Text(
                                    text = "Requires Input",
                                    style = MaterialTheme.typography.titleSmall,
                                    fontWeight = FontWeight.Bold,
                                    color = MaterialTheme.colorScheme.onSecondaryContainer
                                )
                                Text(
                                    text = if (templateType == TemplateType.DECAYING) {
                                        "If enabled, prompts for dosage. If disabled, uses dosage = 1."
                                    } else {
                                        "If enabled, always prompts for a value when clicked."
                                    },
                                    style = MaterialTheme.typography.bodySmall,
                                    color = MaterialTheme.colorScheme.onSecondaryContainer.copy(alpha = 0.8f)
                                )
                            }
                        }
                    }
                    
                    Spacer(modifier = Modifier.height(16.dp))
                }
                
                Text(
                    text = "Template Type",
                    style = MaterialTheme.typography.bodyMedium,
                    fontWeight = FontWeight.Medium
                )
                
                Column {
                    Row(
                        modifier = Modifier
                            .fillMaxWidth()
                            .selectable(
                                selected = templateType == TemplateType.METRIC,
                                onClick = { templateType = TemplateType.METRIC },
                                role = Role.RadioButton
                            ),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        RadioButton(
                            selected = templateType == TemplateType.METRIC,
                            onClick = null
                        )
                        Spacer(modifier = Modifier.width(8.dp))
                        Column {
                            Text("Metric")
                            Text(
                                "weight, mood, etc.",
                                style = MaterialTheme.typography.bodySmall,
                                color = MaterialTheme.colorScheme.onSurfaceVariant
                            )
                        }
                    }
                    
                    Row(
                        modifier = Modifier
                            .fillMaxWidth()
                            .selectable(
                                selected = templateType == TemplateType.DECAYING,
                                onClick = { templateType = TemplateType.DECAYING },
                                role = Role.RadioButton
                            ),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        RadioButton(
                            selected = templateType == TemplateType.DECAYING,
                            onClick = null
                        )
                        Spacer(modifier = Modifier.width(8.dp))
                        Column {
                            Text("Decaying")
                            Text(
                                "concentration tracking",
                                style = MaterialTheme.typography.bodySmall,
                                color = MaterialTheme.colorScheme.onSurfaceVariant
                            )
                        }
                    }
                    
                    Row(
                        modifier = Modifier
                            .fillMaxWidth()
                            .selectable(
                                selected = templateType == TemplateType.GENERAL,
                                onClick = { templateType = TemplateType.GENERAL },
                                role = Role.RadioButton
                            ),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        RadioButton(
                            selected = templateType == TemplateType.GENERAL,
                            onClick = null
                        )
                        Spacer(modifier = Modifier.width(8.dp))
                        Column {
                            Text("General")
                            Text(
                                "basic event logging",
                                style = MaterialTheme.typography.bodySmall,
                                color = MaterialTheme.colorScheme.onSurfaceVariant
                            )
                        }
                    }
                }
                
                Spacer(modifier = Modifier.height(16.dp))
                
                // Metric-specific fields
                if (templateType == TemplateType.METRIC) {
                    OutlinedTextField(
                        value = metricUnit,
                        onValueChange = { metricUnit = it },
                        label = { Text("Unit") },
                        placeholder = { Text("e.g., kg, /10, %") },
                        modifier = Modifier.fillMaxWidth(),
                        supportingText = { Text("Unit for this metric") }
                    )
                    
                    Spacer(modifier = Modifier.height(16.dp))
                }
                
                // Decaying template fields - Drugs list
                if (templateType == TemplateType.DECAYING) {
                    Card(
                        modifier = Modifier.fillMaxWidth(),
                        colors = CardDefaults.cardColors(
                            containerColor = MaterialTheme.colorScheme.tertiaryContainer
                        )
                    ) {
                        Column(
                            modifier = Modifier.padding(16.dp)
                        ) {
                            Row(
                                modifier = Modifier.fillMaxWidth(),
                                horizontalArrangement = Arrangement.SpaceBetween,
                                verticalAlignment = Alignment.CenterVertically
                            ) {
                                Text(
                                    text = "Drugs in This Template",
                                    style = MaterialTheme.typography.titleMedium,
                                    fontWeight = FontWeight.Bold,
                                    color = MaterialTheme.colorScheme.onTertiaryContainer
                                )
                                IconButton(
                                    onClick = {
                                        drugs = (drugs + DrugEditState()).toMutableList()
                                    }
                                ) {
                                    Icon(
                                        Icons.Default.Add,
                                        contentDescription = "Add drug",
                                        tint = MaterialTheme.colorScheme.onTertiaryContainer
                                    )
                                }
                            }
                            
                            if (drugs.isEmpty()) {
                                Text(
                                    text = "Add at least one drug to track concentration. For multi-drug pills, add each component separately.",
                                    style = MaterialTheme.typography.bodySmall,
                                    color = MaterialTheme.colorScheme.onTertiaryContainer.copy(alpha = 0.8f)
                                )
                            } else {
                                drugs.forEachIndexed { index, drugState ->
                                    Spacer(modifier = Modifier.height(12.dp))
                                    DrugCard(
                                        drugState = drugState,
                                        index = index,
                                        existingTemplates = existingTemplates,
                                        currentTemplate = template,
                                        onUpdate = { updated ->
                                            drugs = drugs.toMutableList().apply {
                                                this[index] = updated
                                            }
                                        },
                                        onRemove = {
                                            drugs = drugs.toMutableList().apply {
                                                removeAt(index)
                                            }
                                        }
                                    )
                                }
                            }
                        }
                    }
                    
                    Spacer(modifier = Modifier.height(16.dp))
                }
                
                // Reminder Configuration Section
                Text(
                    text = "Reminder Settings",
                    style = MaterialTheme.typography.titleMedium,
                    fontWeight = FontWeight.Bold
                )
                
                Spacer(modifier = Modifier.height(8.dp))
                
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    Switch(
                        checked = reminderEnabled,
                        onCheckedChange = { reminderEnabled = it }
                    )
                    Spacer(modifier = Modifier.width(12.dp))
                    Text(
                        text = "Enable time-based reminders",
                        style = MaterialTheme.typography.bodyMedium
                    )
                }
                
                if (reminderEnabled) {
                    Spacer(modifier = Modifier.height(8.dp))
                    
                    OutlinedTextField(
                        value = reminderThresholdHours,
                        onValueChange = { reminderThresholdHours = it },
                        label = { Text("Reminder threshold (hours)") },
                        placeholder = { Text("24 or 0.5") },
                        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                        modifier = Modifier.fillMaxWidth(),
                        supportingText = { 
                            Text("Warn if no event logged for this many hours (0.5 = 30min)")
                        }
                    )
                }
                
                Spacer(modifier = Modifier.height(24.dp))
                
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.spacedBy(8.dp)
                ) {
                    OutlinedButton(
                        onClick = onDismiss,
                        modifier = Modifier.weight(1f)
                    ) {
                        Text("Cancel")
                    }
                    
                    Button(
                        onClick = {
                            try {
                                val drugsList = if (templateType == TemplateType.DECAYING) {
                                    // Pass all existing templates to toDrug for proper unit derivance
                                    val allTemplatesForResolution = existingTemplates.filter { it.id != template?.id }
                                    
                                    drugs.mapNotNull { it.toDrug(allTemplatesForResolution) }
                                } else emptyList()
                                
                                val finalReminderThresholdMillis = if (reminderEnabled) {
                                    reminderThresholdHours.toDoubleOrNull()?.let { hours ->
                                        (hours * 60.0 * 60.0 * 1000.0).toLong()
                                    }
                                } else null
                                
                                val newTemplate = Template(
                                    id = template?.id ?: UUID.randomUUID().toString(),
                                    name = name,
                                    templateType = templateType,
                                    metricUnit = if (templateType == TemplateType.METRIC) metricUnit else "",
                                    drugs = drugsList,
                                    extendedData = null, // No longer used
                                    reminderEnabled = reminderEnabled,
                                    reminderThresholdHours = null, // Deprecated
                                    reminderThresholdMillis = finalReminderThresholdMillis,
                                    concentrationReminderEnabled = false, // Deprecated - moved to individual drugs
                                    concentrationReminderThreshold = null, // Deprecated - moved to individual drugs
                                    visible = template?.visible ?: true,
                                    enabled = templateEnabled,
                                    requiresInput = when (templateType) {
                                        TemplateType.METRIC -> true
                                        TemplateType.GENERAL, TemplateType.DECAYING -> requiresInput
                                    }
                                )
                                
                                onSave(newTemplate)
                            } catch (e: Exception) {
                                e.printStackTrace()
                            }
                        },
                        enabled = name.isNotBlank() && (templateType != TemplateType.DECAYING || drugs.isNotEmpty()),
                        modifier = Modifier.weight(1f)
                    ) {
                        Text("Save")
                    }
                }
            }
        }
    }
}

@Composable
fun DrugCard(
    drugState: DrugEditState,
    index: Int,
    existingTemplates: List<Template>,
    currentTemplate: Template?,
    onUpdate: (DrugEditState) -> Unit,
    onRemove: () -> Unit
) {
    Card(
        modifier = Modifier.fillMaxWidth(),
        colors = CardDefaults.cardColors(
            containerColor = MaterialTheme.colorScheme.surface
        )
    ) {
        Column(
            modifier = Modifier.padding(12.dp)
        ) {
            Row(
                modifier = Modifier.fillMaxWidth(),
                horizontalArrangement = Arrangement.SpaceBetween,
                verticalAlignment = Alignment.CenterVertically
            ) {
                Text(
                    text = if (drugState.drugType.isNotBlank()) 
                        "Drug: ${drugState.drugType}" 
                    else 
                        "Drug ${index + 1}",
                    style = MaterialTheme.typography.titleSmall,
                    fontWeight = FontWeight.Bold
                )
                Row {
                    IconButton(
                        onClick = { onUpdate(drugState.copy(isExpanded = !drugState.isExpanded)) }
                    ) {
                        Icon(
                            if (drugState.isExpanded) Icons.Default.ExpandLess else Icons.Default.ExpandMore,
                            contentDescription = if (drugState.isExpanded) "Collapse" else "Expand"
                        )
                    }
                    IconButton(onClick = onRemove) {
                        Icon(Icons.Default.Close, contentDescription = "Remove drug")
                    }
                }
            }
            
            if (drugState.isExpanded) {
                Spacer(modifier = Modifier.height(8.dp))
                
                OutlinedTextField(
                    value = drugState.drugType,
                    onValueChange = { onUpdate(drugState.copy(drugType = it)) },
                    label = { Text("Drug Name") },
                    placeholder = { Text("e.g., Aspirin") },
                    modifier = Modifier.fillMaxWidth(),
                    singleLine = true
                )
                
                Spacer(modifier = Modifier.height(8.dp))
                
                Row(
                    modifier = Modifier.fillMaxWidth(),
                    horizontalArrangement = Arrangement.spacedBy(8.dp)
                ) {
                    OutlinedTextField(
                        value = drugState.factor,
                        onValueChange = { onUpdate(drugState.copy(factor = it)) },
                        label = { Text("Factor") },
                        placeholder = { Text("500") },
                        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                        modifier = Modifier.weight(1f),
                        singleLine = true,
                        supportingText = { Text("Amount per unit (e.g., mg per pill)") }
                    )
                    
                    OutlinedTextField(
                        value = drugState.unit,
                        onValueChange = { 
                            // Only allow unit changes for source drugs
                            if (drugState.decayType != null) {
                                onUpdate(drugState.copy(unit = it))
                            }
                        },
                        label = { Text("Unit") },
                        placeholder = { Text("mg") },
                        modifier = Modifier.weight(1f),
                        singleLine = true,
                        enabled = drugState.decayType != null, // Only enable for source drugs
                        supportingText = {
                            if (drugState.decayType == null) {
                                Text(
                                    "Unit derived from source template",
                                    style = MaterialTheme.typography.bodySmall,
                                    color = MaterialTheme.colorScheme.onSurfaceVariant
                                )
                            }
                        }
                    )
                }
                
                Spacer(modifier = Modifier.height(12.dp))
                
                // Fixed source/derived detection logic
                val drugTypeTrimmed = drugState.drugType.trim()
                
                // Find the actual source template for this drug type (excluding current template)
                val actualSourceTemplate = if (drugTypeTrimmed.isNotBlank()) {
                    existingTemplates
                        .filter { it.templateType == TemplateType.DECAYING && it.id != currentTemplate?.id }
                        .find { template ->
                            template.getSourceDrugs().any { drug ->
                                drug.drugType == drugTypeTrimmed
                            }
                        }
                } else null
                
                // Check if the CURRENT template contains the source drug for this drug type
                val isCurrentTemplateSource = currentTemplate?.getSourceDrugs()?.any { it.drugType == drugTypeTrimmed } == true
                
                // A drug is derived if:
                // 1. It has decayType == null (explicitly marked as derived), OR
                // 2. There's another template that already has the source drug for this type
                val isDerivedDrug = drugState.decayType == null
                val shouldPreventSourceCreation = actualSourceTemplate != null && !isCurrentTemplateSource
                val sourceTemplateName = actualSourceTemplate?.name
                
                // Auto-update logic: only convert to derived if there's a conflict and we're not already the source
                LaunchedEffect(shouldPreventSourceCreation, drugState.decayType, isCurrentTemplateSource) {
                    if (shouldPreventSourceCreation && drugState.decayType != null && !isCurrentTemplateSource) {
                        // Another template is already the source, convert this to derived
                        onUpdate(drugState.copy(decayType = null, unit = ""))
                    }
                }
                
                if (isDerivedDrug && sourceTemplateName != null) {
                    // Show informative message for derived drugs
                    Card(
                        modifier = Modifier.fillMaxWidth(),
                        colors = CardDefaults.cardColors(
                            containerColor = MaterialTheme.colorScheme.primaryContainer
                        )
                    ) {
                        Column(
                            modifier = Modifier.padding(12.dp)
                        ) {
                            Text(
                                text = "Derived Drug",
                                style = MaterialTheme.typography.titleSmall,
                                fontWeight = FontWeight.Bold,
                                color = MaterialTheme.colorScheme.onPrimaryContainer
                            )
                            Text(
                                text = "Drug data configured in template: $sourceTemplateName",
                                style = MaterialTheme.typography.bodyMedium,
                                color = MaterialTheme.colorScheme.onPrimaryContainer
                            )
                            Text(
                                text = "Decay type, half-life, and concentration reminders are derived from the source template.",
                                style = MaterialTheme.typography.bodySmall,
                                color = MaterialTheme.colorScheme.onPrimaryContainer.copy(alpha = 0.8f)
                            )
                        }
                    }
                } else if (!isDerivedDrug) {
                    // Show normal decay configuration for source drugs
                    Text(
                        text = "Decay Type",
                        style = MaterialTheme.typography.labelMedium,
                        fontWeight = FontWeight.Medium
                    )
                    
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        horizontalArrangement = Arrangement.spacedBy(8.dp)
                    ) {
                        Row(
                            modifier = Modifier
                                .weight(1f)
                                .selectable(
                                    selected = drugState.decayType == DecayType.HALF_LIFE,
                                    onClick = { onUpdate(drugState.copy(decayType = DecayType.HALF_LIFE)) },
                                    role = Role.RadioButton
                                ),
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            RadioButton(
                                selected = drugState.decayType == DecayType.HALF_LIFE,
                                onClick = null
                            )
                            Text("Half-life", style = MaterialTheme.typography.bodySmall)
                        }
                        
                        Row(
                            modifier = Modifier
                                .weight(1f)
                                .selectable(
                                    selected = drugState.decayType == DecayType.CONSTANT,
                                    onClick = { onUpdate(drugState.copy(decayType = DecayType.CONSTANT)) },
                                    role = Role.RadioButton
                                ),
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            RadioButton(
                                selected = drugState.decayType == DecayType.CONSTANT,
                                onClick = null
                            )
                            Text("Constant", style = MaterialTheme.typography.bodySmall)
                        }
                    }
                    
                    Spacer(modifier = Modifier.height(8.dp))
                    
                    if (drugState.decayType == DecayType.HALF_LIFE) {
                        OutlinedTextField(
                            value = drugState.halfLifeHours,
                            onValueChange = { onUpdate(drugState.copy(halfLifeHours = it)) },
                            label = { Text("Half-life (hours)") },
                            placeholder = { Text("24.0") },
                            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                            modifier = Modifier.fillMaxWidth(),
                            singleLine = true
                        )
                    } else {
                        OutlinedTextField(
                            value = drugState.hourlyDecayRate,
                            onValueChange = { onUpdate(drugState.copy(hourlyDecayRate = it)) },
                            label = { Text("Hourly Decay Rate") },
                            placeholder = { Text("0.1") },
                            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                            modifier = Modifier.fillMaxWidth(),
                            singleLine = true
                        )
                    }
                    
                    Spacer(modifier = Modifier.height(12.dp))
                    
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        Switch(
                            checked = drugState.concentrationReminderEnabled,
                            onCheckedChange = { 
                                onUpdate(drugState.copy(concentrationReminderEnabled = it)) 
                            }
                        )
                        Spacer(modifier = Modifier.width(8.dp))
                        Text(
                            text = "Concentration reminder",
                            style = MaterialTheme.typography.bodySmall
                        )
                    }
                    
                    if (drugState.concentrationReminderEnabled) {
                        Spacer(modifier = Modifier.height(8.dp))
                        
                        OutlinedTextField(
                            value = drugState.concentrationReminderThreshold,
                            onValueChange = { 
                                onUpdate(drugState.copy(concentrationReminderThreshold = it)) 
                            },
                            label = { Text("Threshold (${drugState.unit})") },
                            placeholder = { Text("50") },
                            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                            modifier = Modifier.fillMaxWidth(),
                            singleLine = true,
                            supportingText = { 
                                Text("Warn when ${drugState.drugType.ifBlank { "drug" }} falls below this level")
                            }
                        )
                    }
                } else {
                    // For derived drugs where we can't find the source template
                    Card(
                        modifier = Modifier.fillMaxWidth(),
                        colors = CardDefaults.cardColors(
                            containerColor = MaterialTheme.colorScheme.errorContainer
                        )
                    ) {
                        Column(
                            modifier = Modifier.padding(12.dp)
                        ) {
                            Text(
                                text = "Derived Drug (Source Not Found)",
                                style = MaterialTheme.typography.titleSmall,
                                fontWeight = FontWeight.Bold,
                                color = MaterialTheme.colorScheme.onErrorContainer
                            )
                            Text(
                                text = "This drug appears to be a derived drug, but the source template could not be found. Consider creating a source template first or converting this to a source drug.",
                                style = MaterialTheme.typography.bodySmall,
                                color = MaterialTheme.colorScheme.onErrorContainer.copy(alpha = 0.8f)
                            )
                        }
                    }
                }
            }
        }
    }
}
