package de.ntdote.medicalcalendarlog.ui.components

import androidx.compose.foundation.layout.*
import androidx.compose.foundation.shape.CircleShape
import androidx.compose.material3.*
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.data.TemplateType
import de.ntdote.medicalcalendarlog.utils.TimeDuration
import de.ntdote.medicalcalendarlog.utils.TimeFormatter
import java.util.Locale
import kotlin.math.roundToInt

@Composable
fun TemplateButton(
    template: Template,
    timeSinceLastEvent: String,
    reminderThresholdMillis: Long?,
    currentConcentration: Double?,
    peakConcentration: Double?,
    latestMetricValue: Double?,
    concentrationsPerDrug: Map<String, Double> = emptyMap(),
    peakConcentrationsPerDrug: Map<String, Double> = emptyMap(),
    onClick: () -> Unit,
    modifier: Modifier = Modifier
) {
    Card(
        modifier = modifier.fillMaxWidth(),
        onClick = onClick,
        elevation = CardDefaults.cardElevation(defaultElevation = 4.dp)
    ) {
        Box(
            modifier = Modifier
                .fillMaxWidth()
                .padding(16.dp)
        ) {
            // Main content
            Column(
                modifier = Modifier.fillMaxWidth()
            ) {
                Text(
                    text = template.name,
                    style = MaterialTheme.typography.titleMedium,
                    fontWeight = FontWeight.Bold,
                    maxLines = 2,
                    overflow = TextOverflow.Ellipsis
                )
                
                if (template.templateType == TemplateType.DECAYING && template.drugs.isNotEmpty()) {
                    Spacer(modifier = Modifier.height(8.dp))
                    
                    // DEBUG: Log concentration data received
                    android.util.Log.d("MCL", "TemplateButton: Template '${template.name}' concentrationsPerDrug=$concentrationsPerDrug, peakConcentrationsPerDrug=$peakConcentrationsPerDrug")
                    
                    // Display concentration table for all DECAYING templates (single or multi-drug)
                    // Always show all drugs, even if concentration is below 1% of peak
                    template.drugs.forEach { drug ->
                        val concentration = concentrationsPerDrug[drug.drugType] ?: 0.0
                        val peak = peakConcentrationsPerDrug[drug.drugType] ?: 0.0
                        android.util.Log.d("MCL", "TemplateButton: Drug '${drug.drugType}' - concentration=$concentration, peak=$peak")
                        
                        // Display concentration value (show "0" if below 1% of peak)
                        val displayConcentration = if (peak > 0 && concentration < (peak * 0.01)) 0.0 else concentration
                        
                        // Check if concentration is below reminder threshold
                        val isBelowThreshold = template.concentrationReminderEnabled &&
                            template.concentrationReminderThreshold != null &&
                            template.concentrationReminderThreshold > 0 &&
                            concentration < template.concentrationReminderThreshold
                        
                        Row(
                            modifier = Modifier.fillMaxWidth(),
                            horizontalArrangement = Arrangement.SpaceBetween,
                            verticalAlignment = Alignment.CenterVertically
                        ) {
                            // Drug name with optional alert emoji
                            Text(
                                text = if (isBelowThreshold) "❗${drug.drugType}" else drug.drugType,
                                style = MaterialTheme.typography.bodySmall,
                                color = MaterialTheme.colorScheme.onSurfaceVariant
                            )
                            
                            // Concentration value (always show, use "0" for very low concentrations)
                            Text(
                                text = formatConcentrationWithUnit(displayConcentration, drug.unit),
                                style = MaterialTheme.typography.bodySmall,
                                fontWeight = FontWeight.Medium,
                                color = MaterialTheme.colorScheme.onSurface
                            )
                        }
                        Spacer(modifier = Modifier.height(4.dp))
                    }
                    
                    // Fallback: if no concentration data available, show simple drug info
                    if (concentrationsPerDrug.isEmpty()) {
                        // No concentration data - show simple drug info
                        val drugInfo = template.drugs.joinToString(" • ") { drug ->
                            if (drug.unit.isNotEmpty()) {
                                "${drug.drugType} (${drug.unit})"
                            } else {
                                drug.drugType
                            }
                        }
                        Text(
                            text = drugInfo,
                            style = MaterialTheme.typography.bodySmall,
                            color = MaterialTheme.colorScheme.onSurfaceVariant
                        )
                    }
                } else if (template.templateType == TemplateType.METRIC) {
                    Spacer(modifier = Modifier.height(4.dp))
                    Text(
                        text = "Metric",
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                } else if (template.templateType == TemplateType.GENERAL) {
                    Spacer(modifier = Modifier.height(4.dp))
                    Text(
                        text = "General",
                        style = MaterialTheme.typography.bodySmall,
                        color = MaterialTheme.colorScheme.onSurfaceVariant
                    )
                }
            }

            // Time since last event badge (upper right corner)
            Badge(
                containerColor = MaterialTheme.colorScheme.primary,
                contentColor = MaterialTheme.colorScheme.onPrimary,
                modifier = Modifier
                    .align(Alignment.TopEnd)
                    .offset(x = 8.dp, y = (-8).dp)
                    .defaultMinSize(minWidth = 40.dp, minHeight = 28.dp)
            ) {
                if (reminderThresholdMillis != null) {
                    // Parse the time string properly to handle both minutes and hours
                    val isOverdue = isTimeOverdue(timeSinceLastEvent, reminderThresholdMillis)
                    
                    Row(
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        if (isOverdue) {
                            Text(
                                text = "❗",
                                color = Color.Red,
                                fontSize = 14.sp,
                                fontWeight = FontWeight.Bold
                            )
                            Spacer(modifier = Modifier.width(2.dp))
                        }
                        
                        Text(
                            text = timeSinceLastEvent,
                            style = MaterialTheme.typography.labelLarge,
                            fontSize = 14.sp,
                            fontWeight = FontWeight.Bold
                        )
                        
                        Text(
                            text = "/${formatThreshold(reminderThresholdMillis)}",
                            style = MaterialTheme.typography.labelSmall,
                            fontSize = 10.sp,
                            fontWeight = FontWeight.Normal
                        )
                    }
                } else {
                    Text(
                        text = timeSinceLastEvent,
                        style = MaterialTheme.typography.labelLarge,
                        fontSize = 14.sp,
                        fontWeight = FontWeight.Bold
                    )
                }
            }

            // Value badge (lower right corner) - only for METRIC templates
            if (template.templateType == TemplateType.METRIC && latestMetricValue != null) {
                Badge(
                    containerColor = getMetricBadgeColor(),
                    contentColor = Color.Black,
                    modifier = Modifier
                        .align(Alignment.BottomEnd)
                        .offset(x = 8.dp, y = 8.dp)
                        .defaultMinSize(minWidth = 50.dp, minHeight = 28.dp)
                ) {
                    Text(
                        text = formatMetricWithUnit(latestMetricValue, template.metricUnit),
                        fontSize = 12.sp,
                        fontWeight = FontWeight.Bold
                    )
                }
            }
        }
    }
}

@Composable
private fun getTimeBadgeColor(timeSinceLastEvent: String): Color {
    return when {
        timeSinceLastEvent == "Never" -> MaterialTheme.colorScheme.error
        timeSinceLastEvent.endsWith("m") -> MaterialTheme.colorScheme.primary
        timeSinceLastEvent.endsWith("h") -> MaterialTheme.colorScheme.secondary
        else -> MaterialTheme.colorScheme.tertiary
    }
}

@Composable
private fun getConcentrationBadgeColor(concentration: Double): Color {
    return when {
        concentration > 50 -> Color(0xFF4CAF50) // Green for high concentration
        concentration > 20 -> Color(0xFFFF9800) // Orange for medium concentration
        concentration > 5 -> Color(0xFFFFC107) // Yellow for low concentration
        else -> Color(0xFF9E9E9E) // Gray for very low concentration
    }
}

private fun formatConcentration(concentration: Double): String {
    return when {
        concentration >= 100 -> "${concentration.roundToInt()}"
        concentration >= 10 -> String.format(Locale.US, "%.1f", concentration)
        concentration >= 1 -> String.format(Locale.US, "%.2f", concentration)
        else -> String.format(Locale.US, "%.3f", concentration)
    }
}

private fun formatConcentrationWithUnit(concentration: Double, unit: String): String {
    val value = when {
        concentration >= 100 -> "${concentration.roundToInt()}"
        concentration >= 10 -> String.format(Locale.US, "%.1f", concentration)
        concentration >= 1 -> String.format(Locale.US, "%.2f", concentration)
        else -> String.format(Locale.US, "%.3f", concentration)
    }
    return if (unit.isNotEmpty()) "$value $unit" else value
}

@Composable
private fun getMetricBadgeColor(): Color {
    return Color(0xFF2196F3) // Blue for metrics
}

private fun formatMetricWithUnit(value: Double, unit: String): String {
    val formattedValue = when {
        value >= 100 -> "${value.roundToInt()}"
        value >= 10 -> String.format(Locale.US, "%.1f", value)
        value >= 1 -> String.format(Locale.US, "%.2f", value)
        else -> String.format(Locale.US, "%.3f", value)
    }
    return if (unit.isNotEmpty()) "$formattedValue $unit" else formattedValue
}

/**
 * Formats threshold milliseconds into a human-readable format
 * Examples: 90h -> "3d 18h", 48h -> "2d", 6h -> "6h", 30min -> "30m"
 */
private fun formatThreshold(thresholdMillis: Long): String {
    val totalMinutes = (thresholdMillis / (60 * 1000.0)).roundToInt()
    
    return when {
        totalMinutes < 60 -> "${totalMinutes}m"
        totalMinutes < 24 * 60 -> {
            val hours = totalMinutes / 60
            val remainingMinutes = totalMinutes % 60
            if (remainingMinutes == 0) "${hours}h" else "${hours}h ${remainingMinutes}m"
        }
        totalMinutes % (24 * 60) == 0 -> "${totalMinutes / (24 * 60)}d"
        else -> {
            val days = totalMinutes / (24 * 60)
            val remainingHours = (totalMinutes % (24 * 60)) / 60
            val remainingMinutes = totalMinutes % 60
            when {
                remainingMinutes == 0 -> "${days}d ${remainingHours}h"
                remainingHours == 0 -> "${days}d ${remainingMinutes}m"
                else -> "${days}d ${remainingHours}h" // Skip minutes in complex display
            }
        }
    }
}

/**
 * Determines if the time since last event exceeds the reminder threshold
 * Handles both minute and hour formats (e.g., "3m", "2h", "1d 4h")
 */
private fun isTimeOverdue(timeSinceLastEvent: String, reminderThresholdMillis: Long): Boolean {
    // Handle special case
    if (timeSinceLastEvent == "Never") return true
    
    // Convert the time string to minutes for comparison
    val timeInMinutes = when {
        // Combined format: "2d 4h" or "3d 22h" - check this first before simple formats
        timeSinceLastEvent.contains("d") && timeSinceLastEvent.contains("h") -> {
            val parts = timeSinceLastEvent.trim().split(Regex("\\s+"))
            var totalMinutes = 0.0
            
            for (part in parts) {
                when {
                    part.endsWith("d") -> {
                        val days = part.removeSuffix("d").toDoubleOrNull() ?: 0.0
                        totalMinutes += days * 24.0 * 60.0
                    }
                    part.endsWith("h") -> {
                        val hours = part.removeSuffix("h").toDoubleOrNull() ?: 0.0
                        totalMinutes += hours * 60.0
                    }
                }
            }
            totalMinutes
        }
        
        // Minutes format: "30m"
        timeSinceLastEvent.endsWith("m") -> {
            timeSinceLastEvent.removeSuffix("m").toDoubleOrNull() ?: 0.0
        }
        
        // Hours format: "3h"
        timeSinceLastEvent.endsWith("h") -> {
            val hours = timeSinceLastEvent.removeSuffix("h").toDoubleOrNull() ?: 0.0
            hours * 60.0 // Convert to minutes
        }
        
        // Days format: "2d"
        timeSinceLastEvent.endsWith("d") -> {
            val days = timeSinceLastEvent.removeSuffix("d").toDoubleOrNull() ?: 0.0
            days * 24.0 * 60.0 // Convert to minutes
        }
        
        // Default case
        else -> 0.0
    }
    
    // Compare with threshold
    val thresholdMinutes = reminderThresholdMillis / (60 * 1000.0)
    return timeInMinutes >= thresholdMinutes
}
