package de.ntdote.medicalcalendarlog.ui.components

import android.text.util.Linkify
import android.widget.TextView
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.unit.dp
import androidx.compose.ui.viewinterop.AndroidView
import androidx.compose.ui.window.Dialog
import androidx.compose.ui.window.DialogProperties
import io.noties.markwon.Markwon
import io.noties.markwon.html.HtmlPlugin
import io.noties.markwon.linkify.LinkifyPlugin

@Composable
fun MarkdownDialog(
    markdownContent: String,
    onDismiss: () -> Unit,
    showDontShowAgain: Boolean = false,
    dontShowAgainChecked: Boolean = false,
    onDontShowAgainChanged: (Boolean) -> Unit = {},
    title: String = "Notice"
) {
    val context = LocalContext.current
    
    // Get Material3 colors for theming
    val textColor = MaterialTheme.colorScheme.onSurface.hashCode()
    val linkColor = MaterialTheme.colorScheme.primary.hashCode()
    val backgroundColor = MaterialTheme.colorScheme.surface.hashCode()
    
    Dialog(
        onDismissRequest = onDismiss,
        properties = DialogProperties(usePlatformDefaultWidth = false)
    ) {
        Card(
            modifier = Modifier
                .fillMaxWidth(0.9f)
                .fillMaxHeight(0.7f)
        ) {
            Column(
                modifier = Modifier
                    .fillMaxSize()
                    .padding(16.dp)
            ) {
                // Title
                Text(
                    text = title,
                    style = MaterialTheme.typography.titleLarge,
                    modifier = Modifier.padding(bottom = 16.dp)
                )
                
                // Markdown content
                AndroidView(
                    factory = { ctx ->
                        TextView(ctx).apply {
                            // Configure Markwon with basic HTML support and controlled linking
                            val markwon = Markwon.builder(ctx)
                                .usePlugin(HtmlPlugin.create())
                                .usePlugin(LinkifyPlugin.create(
                                    // Only enable web URLs and email, disable phone numbers
                                    Linkify.WEB_URLS or Linkify.EMAIL_ADDRESSES
                                ))
                                .build()

                            // Set text color and link color from Material3 theme
                            setTextColor(textColor)
                            setLinkTextColor(linkColor)
                            setBackgroundColor(backgroundColor)
                            
                            // Set padding
                            setPadding(
                                (8 * resources.displayMetrics.density).toInt(),
                                (8 * resources.displayMetrics.density).toInt(),
                                (8 * resources.displayMetrics.density).toInt(),
                                (8 * resources.displayMetrics.density).toInt()
                            )

                            // Render markdown with HTML support
                            markwon.setMarkdown(this, markdownContent)
                        }
                    },
                    modifier = Modifier
                        .weight(1f)
                        .verticalScroll(rememberScrollState())
                )
                
                // Don't show again checkbox
                if (showDontShowAgain) {
                    Spacer(modifier = Modifier.height(8.dp))
                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        verticalAlignment = Alignment.CenterVertically
                    ) {
                        Checkbox(
                            checked = dontShowAgainChecked,
                            onCheckedChange = onDontShowAgainChanged
                        )
                        Text(
                            text = "Don't show again",
                            style = MaterialTheme.typography.bodyMedium
                        )
                    }
                }
                
                // Dismiss button
                Spacer(modifier = Modifier.height(8.dp))
                Button(
                    onClick = onDismiss,
                    modifier = Modifier.align(Alignment.End)
                ) {
                    Text("Got it")
                }
            }
        }
    }
}
