package de.ntdote.medicalcalendarlog.ui

import android.app.Activity
import android.appwidget.AppWidgetManager
import android.content.Intent
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.foundation.layout.*
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import androidx.lifecycle.lifecycleScope
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.repository.WidgetRepository
import de.ntdote.medicalcalendarlog.service.TemplateWidgetProvider
import de.ntdote.medicalcalendarlog.ui.theme.MedicalCalendarlogTheme
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch

/**
 * Transparent activity that shows a mini-menu when widget is clicked
 * Offers two options: Settings and Log
 */
class WidgetMenuActivity : ComponentActivity() {

    private var appWidgetId = AppWidgetManager.INVALID_APPWIDGET_ID
    private var templateId: String? = null
    private lateinit var widgetRepository: WidgetRepository
    private lateinit var preferencesRepository: PreferencesRepository

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        // Check if hourly worker needs to run due to user interaction
        de.ntdote.medicalcalendarlog.service.HourlyReminderWorker.checkAndRunWorkerIfNeeded(this, "WidgetMenuActivity")

        // Get widget ID and template ID from intent
        appWidgetId = intent?.getIntExtra(EXTRA_WIDGET_ID, AppWidgetManager.INVALID_APPWIDGET_ID) 
            ?: AppWidgetManager.INVALID_APPWIDGET_ID
        templateId = intent?.getStringExtra(EXTRA_TEMPLATE_ID)

        if (appWidgetId == AppWidgetManager.INVALID_APPWIDGET_ID || templateId == null) {
            finish()
            return
        }

        widgetRepository = WidgetRepository(this)
        preferencesRepository = PreferencesRepository(this)

        setContent {
            MedicalCalendarlogTheme {
                WidgetMenuDialog(
                    onSettingsClick = {
                        openWidgetSettings()
                    },
                    onLogClick = {
                        logTemplate()
                    },
                    onDismiss = {
                        finish()
                    }
                )
            }
        }
    }

    @Composable
    private fun WidgetMenuDialog(
        onSettingsClick: () -> Unit,
        onLogClick: () -> Unit,
        onDismiss: () -> Unit
    ) {
        // Minimal dialog with no title, no cancel button
        AlertDialog(
            onDismissRequest = onDismiss,
            text = {
                Column(
                    modifier = Modifier.wrapContentWidth(),
                    verticalArrangement = Arrangement.spacedBy(8.dp)
                ) {
                    Button(
                        onClick = onSettingsClick,
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Text("⚙️ Settings")
                    }
                    Button(
                        onClick = onLogClick,
                        modifier = Modifier.fillMaxWidth()
                    ) {
                        Text("➕ Log")
                    }
                }
            },
            confirmButton = {}
        )
    }

    private fun openWidgetSettings() {
        val intent = Intent(this, WidgetConfigurationActivity::class.java).apply {
            putExtra(AppWidgetManager.EXTRA_APPWIDGET_ID, appWidgetId)
            putExtra(EXTRA_EDIT_MODE, true)
        }
        startActivity(intent)
        finish()
    }

    private fun logTemplate() {
        lifecycleScope.launch {
            val templates = preferencesRepository.templates.first()
            val template = templates.find { it.id == templateId }

            if (template != null && template.requiresInput) {
                // Launch InputDialogActivity for input
                val dialogIntent = Intent(this@WidgetMenuActivity, InputDialogActivity::class.java).apply {
                    putExtra(InputDialogActivity.EXTRA_TEMPLATE_ID, templateId)
                    putExtra(InputDialogActivity.EXTRA_TEMPLATE_NAME, template.name)
                    addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
                }
                startActivity(dialogIntent)
            } else {
                // Log event directly via broadcast
                val intent = Intent(this@WidgetMenuActivity, TemplateWidgetProvider::class.java).apply {
                    action = TemplateWidgetProvider.ACTION_WIDGET_LOG
                    putExtra(TemplateWidgetProvider.EXTRA_TEMPLATE_ID, templateId)
                    putExtra(TemplateWidgetProvider.EXTRA_WIDGET_ID, appWidgetId)
                }
                sendBroadcast(intent)
            }
            finish()
        }
    }

    companion object {
        const val EXTRA_WIDGET_ID = "widget_id"
        const val EXTRA_TEMPLATE_ID = "template_id"
        const val EXTRA_EDIT_MODE = "edit_mode"
    }
}
