package de.ntdote.medicalcalendarlog.ui

import android.app.Activity
import android.app.NotificationManager
import android.content.Context
import android.content.Intent
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import androidx.lifecycle.lifecycleScope
import kotlinx.coroutines.delay
import de.ntdote.medicalcalendarlog.repository.CalendarRepository
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.ui.theme.MedicalCalendarlogTheme
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch

/**
 * Transparent dialog activity for inputting dosage/value from widgets
 */
class InputDialogActivity : ComponentActivity() {

    companion object {
        const val EXTRA_TEMPLATE_ID = "template_id"
        const val EXTRA_TEMPLATE_NAME = "template_name"
        const val EXTRA_NOTIFICATION_ID = "notification_id"
    }

    private lateinit var calendarRepository: CalendarRepository
    private lateinit var preferencesRepository: PreferencesRepository
    private var notificationId: Int = -1

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        calendarRepository = CalendarRepository(contentResolver, this)
        preferencesRepository = PreferencesRepository(this)

        val templateId = intent.getStringExtra(EXTRA_TEMPLATE_ID)
        val templateName = intent.getStringExtra(EXTRA_TEMPLATE_NAME)
        notificationId = intent.getIntExtra(EXTRA_NOTIFICATION_ID, -1)

        if (templateId == null || templateName == null) {
            dismissNotificationAndFinish()
            return
        }

        setContent {
            MedicalCalendarlogTheme {
                InputDialog(
                    templateName = templateName,
                    onConfirm = { value ->
                        lifecycleScope.launch {
                            createEventWithValue(templateId, templateName, value)
                        }
                    },
                    onDismiss = {
                        dismissNotificationAndFinish()
                    }
                )
            }
        }
    }

    @Composable
    private fun InputDialog(
        templateName: String,
        onConfirm: (Double) -> Unit,
        onDismiss: () -> Unit
    ) {
        var inputValue by remember { mutableStateOf("") }
        val focusRequester = remember { FocusRequester() }

        AlertDialog(
            onDismissRequest = onDismiss,
            title = { Text("Enter Value") },
            text = {
                Column {
                    Text("Enter the value for $templateName:")
                    Spacer(modifier = Modifier.height(8.dp))
                    OutlinedTextField(
                        value = inputValue,
                        onValueChange = { inputValue = it },
                        label = { Text("Value") },
                        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Decimal),
                        singleLine = true,
                        modifier = Modifier.focusRequester(focusRequester)
                    )
                }
            },
            confirmButton = {
                TextButton(
                    onClick = {
                        val value = inputValue.toDoubleOrNull()
                        if (value != null) {
                            onConfirm(value)
                        }
                    },
                    enabled = inputValue.toDoubleOrNull() != null
                ) {
                    Text("Create Event")
                }
            },
            dismissButton = {
                TextButton(onClick = onDismiss) {
                    Text("Cancel")
                }
            }
        )
        
        // Auto-focus the text field when dialog opens
        LaunchedEffect(Unit) {
            delay(100) // Small delay to ensure text field is composed
            focusRequester.requestFocus()
        }
    }

    private suspend fun createEventWithValue(templateId: String, templateName: String, value: Double) {
        try {
            val template = preferencesRepository.templates.first().find { it.id == templateId }
            if (template == null) {
                dismissNotificationAndFinish()
                return
            }

            val calendarId = preferencesRepository.selectedCalendarId.first()
            if (calendarId == null) {
                dismissNotificationAndFinish()
                return
            }

            // Use centralized event creation (handles notification cancellation and widget updates)
            calendarRepository.createEventFromTemplate(calendarId, template, value)

            // Set result and finish
            setResult(Activity.RESULT_OK)
            dismissNotificationAndFinish()
        } catch (e: Exception) {
            e.printStackTrace()
            setResult(Activity.RESULT_CANCELED)
            dismissNotificationAndFinish()
        }
    }
    
    /**
     * Dismiss the notification (if any) and finish the activity
     */
    private fun dismissNotificationAndFinish() {
        if (notificationId != -1) {
            val notificationManager = getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
            notificationManager.cancel(notificationId)
        }
        finish()
    }
}
