package de.ntdote.medicalcalendarlog.ui

import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import de.ntdote.medicalcalendarlog.BuildConfig
import de.ntdote.medicalcalendarlog.data.BackupInfo
import de.ntdote.medicalcalendarlog.data.VersionColor
import de.ntdote.medicalcalendarlog.repository.CalendarRepository
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.ui.theme.MedicalCalendarlogTheme
import de.ntdote.medicalcalendarlog.utils.ConfigExportImport
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import java.text.SimpleDateFormat
import java.util.*

class BackupSelectionActivity : ComponentActivity() {
    
    private lateinit var calendarRepository: CalendarRepository
    private lateinit var preferencesRepository: PreferencesRepository
    private lateinit var configExportImport: ConfigExportImport
    
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        
        calendarRepository = CalendarRepository(contentResolver, this)
        preferencesRepository = PreferencesRepository(this)
        configExportImport = ConfigExportImport(contentResolver)
        
        setContent {
            MedicalCalendarlogTheme {
                BackupSelectionScreen(
                    calendarRepository = calendarRepository,
                    preferencesRepository = preferencesRepository,
                    configExportImport = configExportImport,
                    onBackClick = { finish() },
                    onBackupRestored = { finish() }
                )
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun BackupSelectionScreen(
    calendarRepository: CalendarRepository,
    preferencesRepository: PreferencesRepository,
    configExportImport: ConfigExportImport,
    onBackClick: () -> Unit,
    onBackupRestored: () -> Unit
) {
    val scope = rememberCoroutineScope()
    var backups by remember { mutableStateOf<List<BackupInfo>>(emptyList()) }
    var isLoading by remember { mutableStateOf(true) }
    var errorMessage by remember { mutableStateOf<String?>(null) }
    var showDeleteDialog by remember { mutableStateOf(false) }
    var backupToDelete by remember { mutableStateOf<BackupInfo?>(null) }
    var showRestoreDialog by remember { mutableStateOf(false) }
    var backupToRestore by remember { mutableStateOf<BackupInfo?>(null) }
    
    // Load backups on composition
    LaunchedEffect(Unit) {
        scope.launch {
            try {
                val calendarId = preferencesRepository.selectedCalendarId.first()
                if (calendarId != null) {
                    backups = calendarRepository.getAllBackups(calendarId)
                } else {
                    errorMessage = "No calendar selected"
                }
            } catch (e: Exception) {
                errorMessage = "Failed to load backups: ${e.message}"
            } finally {
                isLoading = false
            }
        }
    }
    
    Scaffold(
        topBar = {
            TopAppBar(
                title = { Text("Restore Backup") },
                navigationIcon = {
                    IconButton(onClick = onBackClick) {
                        Icon(Icons.AutoMirrored.Filled.ArrowBack, contentDescription = "Back")
                    }
                }
            )
        }
    ) { paddingValues ->
        Box(
            modifier = Modifier
                .fillMaxSize()
                .padding(paddingValues)
        ) {
            when {
                isLoading -> {
                    CircularProgressIndicator(
                        modifier = Modifier.align(Alignment.Center)
                    )
                }
                errorMessage != null -> {
                    Text(
                        text = errorMessage ?: "",
                        color = MaterialTheme.colorScheme.error,
                        modifier = Modifier
                            .align(Alignment.Center)
                            .padding(16.dp)
                    )
                }
                backups.isEmpty() -> {
                    Text(
                        text = "No backups found",
                        style = MaterialTheme.typography.bodyLarge,
                        color = MaterialTheme.colorScheme.onSurfaceVariant,
                        modifier = Modifier
                            .align(Alignment.Center)
                            .padding(16.dp)
                    )
                }
                else -> {
                    LazyColumn(
                        modifier = Modifier.fillMaxSize(),
                        contentPadding = PaddingValues(16.dp),
                        verticalArrangement = Arrangement.spacedBy(12.dp)
                    ) {
                        items(backups) { backup ->
                            BackupListItem(
                                backup = backup,
                                currentVersionCode = BuildConfig.VERSION_CODE,
                                onBackupClick = {
                                    backupToRestore = backup
                                    showRestoreDialog = true
                                },
                                onDeleteClick = {
                                    backupToDelete = backup
                                    showDeleteDialog = true
                                }
                            )
                        }
                    }
                }
            }
        }
    }
    
    // Delete confirmation dialog
    if (showDeleteDialog && backupToDelete != null) {
        AlertDialog(
            onDismissRequest = { showDeleteDialog = false },
            title = { Text("Delete Backup?") },
            text = { Text("This backup will be permanently deleted.") },
            confirmButton = {
                TextButton(
                    onClick = {
                        scope.launch {
                            val success = calendarRepository.deleteBackup(backupToDelete!!.eventId)
                            if (success) {
                                // Reload backups
                                val calendarId = preferencesRepository.selectedCalendarId.first()
                                if (calendarId != null) {
                                    backups = calendarRepository.getAllBackups(calendarId)
                                }
                            }
                            showDeleteDialog = false
                            backupToDelete = null
                        }
                    }
                ) {
                    Text("Delete")
                }
            },
            dismissButton = {
                TextButton(onClick = { showDeleteDialog = false }) {
                    Text("Cancel")
                }
            }
        )
    }
    
    // Restore confirmation dialog
    if (showRestoreDialog && backupToRestore != null) {
        AlertDialog(
            onDismissRequest = { showRestoreDialog = false },
            title = { Text("Restore Backup?") },
            text = {
                Column {
                    Text("Restore configuration from:")
                    Spacer(modifier = Modifier.height(8.dp))
                    Text(
                        text = formatBackupDate(backupToRestore!!.timestamp),
                        fontWeight = FontWeight.Bold
                    )
                    Text("Version: ${backupToRestore!!.versionName} (${backupToRestore!!.versionCode})")
                    Text("Templates: ${backupToRestore!!.templateCount}")
                    Spacer(modifier = Modifier.height(8.dp))
                    Text(
                        text = "This will replace your current configuration.",
                        color = MaterialTheme.colorScheme.error
                    )
                }
            },
            confirmButton = {
                TextButton(
                    onClick = {
                        scope.launch {
                            try {
                                val result = configExportImport.importFromBackup(backupToRestore!!.jsonData)
                                result.fold(
                                    onSuccess = { templates ->
                                        preferencesRepository.replaceAllTemplates(templates)
                                        showRestoreDialog = false
                                        onBackupRestored()
                                    },
                                    onFailure = { exception ->
                                        errorMessage = "Failed to restore backup: ${exception.message}"
                                        showRestoreDialog = false
                                    }
                                )
                            } catch (e: Exception) {
                                errorMessage = "Failed to restore backup: ${e.message}"
                                showRestoreDialog = false
                            }
                        }
                    }
                ) {
                    Text("Restore")
                }
            },
            dismissButton = {
                TextButton(onClick = { showRestoreDialog = false }) {
                    Text("Cancel")
                }
            }
        )
    }
}

@Composable
fun BackupListItem(
    backup: BackupInfo,
    currentVersionCode: Int,
    onBackupClick: () -> Unit,
    onDeleteClick: () -> Unit
) {
    val versionColor = backup.getVersionColor(currentVersionCode)
    val backgroundColor = when (versionColor) {
        VersionColor.SAME -> Color(0xFF4CAF50).copy(alpha = 0.1f)    // Green tint
        VersionColor.OLDER -> Color(0xFFFF9800).copy(alpha = 0.1f)   // Orange tint
        VersionColor.NEWER -> Color(0xFFF44336).copy(alpha = 0.1f)   // Red tint
    }
    val accentColor = when (versionColor) {
        VersionColor.SAME -> Color(0xFF4CAF50)    // Green
        VersionColor.OLDER -> Color(0xFFFF9800)   // Orange
        VersionColor.NEWER -> Color(0xFFF44336)   // Red
    }
    
    Card(
        modifier = Modifier
            .fillMaxWidth()
            .clickable(onClick = onBackupClick),
        colors = CardDefaults.cardColors(
            containerColor = backgroundColor
        )
    ) {
        Row(
            modifier = Modifier
                .fillMaxWidth()
                .padding(16.dp),
            horizontalArrangement = Arrangement.SpaceBetween,
            verticalAlignment = Alignment.CenterVertically
        ) {
            Column(modifier = Modifier.weight(1f)) {
                Text(
                    text = formatBackupDate(backup.timestamp),
                    style = MaterialTheme.typography.titleMedium,
                    fontWeight = FontWeight.Bold
                )
                Spacer(modifier = Modifier.height(4.dp))
                Text(
                    text = "Version: ${backup.versionName} (${backup.versionCode})",
                    style = MaterialTheme.typography.bodyMedium,
                    color = accentColor,
                    fontWeight = FontWeight.Medium
                )
                Spacer(modifier = Modifier.height(2.dp))
                Text(
                    text = "${backup.templateCount} template${if (backup.templateCount != 1) "s" else ""}",
                    style = MaterialTheme.typography.bodySmall,
                    color = MaterialTheme.colorScheme.onSurfaceVariant
                )
            }
            
            IconButton(
                onClick = onDeleteClick,
                modifier = Modifier.size(48.dp)
            ) {
                Icon(
                    imageVector = Icons.Default.Delete,
                    contentDescription = "Delete backup",
                    tint = MaterialTheme.colorScheme.error
                )
            }
        }
    }
}

private fun formatBackupDate(date: Date): String {
    val dateFormat = SimpleDateFormat("MMM dd, yyyy HH:mm", Locale.getDefault())
    return dateFormat.format(date)
}
