package de.ntdote.medicalcalendarlog.service

import android.app.PendingIntent
import android.appwidget.AppWidgetManager
import android.appwidget.AppWidgetProvider
import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.graphics.Color
import android.view.View
import android.widget.RemoteViews
import de.ntdote.medicalcalendarlog.MainActivity
import de.ntdote.medicalcalendarlog.R
import de.ntdote.medicalcalendarlog.data.CalendarEvent
import de.ntdote.medicalcalendarlog.data.TemplateType
import de.ntdote.medicalcalendarlog.repository.CalendarRepository
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.repository.WidgetRepository
import de.ntdote.medicalcalendarlog.utils.ConcentrationCalculator
import de.ntdote.medicalcalendarlog.utils.TimeDuration
import de.ntdote.medicalcalendarlog.utils.TimeFormatter
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch
import java.util.Date
import java.util.concurrent.TimeUnit

/**
 * Widget provider for template quick-log widgets
 */
class TemplateWidgetProvider : AppWidgetProvider() {

    private val job = SupervisorJob()
    private val coroutineScope = CoroutineScope(Dispatchers.IO + job)

    override fun onReceive(context: Context, intent: Intent) {
        super.onReceive(context, intent)
        
        // Check if hourly worker needs to run due to user interaction with widget
        HourlyReminderWorker.checkAndRunWorkerIfNeeded(context, "TemplateWidgetProvider")
        
        android.util.Log.d("TemplateWidgetProvider", "MCL: onReceive called with action: ${intent.action}")
        
        if (intent.action == ACTION_REFRESH_WIDGETS) {
            // Handle manual widget refresh via ADB
            android.util.Log.d("TemplateWidgetProvider", "MCL: Manual widget refresh triggered via ADB")
            coroutineScope.launch {
                val preferencesRepository = PreferencesRepository(context)
                val debugModeEnabled = preferencesRepository.debugModeEnabled.first()
                android.util.Log.d("TemplateWidgetProvider", "MCL: Calling updateAllWidgets (debug mode: $debugModeEnabled)")
                updateAllWidgets(context, debugModeEnabled)
            }
        } else if (intent.action == ACTION_WIDGET_LOG) {
            val templateId = intent.getStringExtra(EXTRA_TEMPLATE_ID)
            val widgetId = intent.getIntExtra(EXTRA_WIDGET_ID, -1)
            
            if (templateId != null) {
                // Check if template requires input
                coroutineScope.launch {
                    val preferencesRepository = PreferencesRepository(context)
                    val templates = preferencesRepository.templates.first()
                    val template = templates.find { it.id == templateId }
                    
                    if (template == null) {
                        // Template not found - log error and do nothing
                        android.util.Log.e("MCL", "Widget clicked but template not found: templateId=$templateId, widgetId=$widgetId")
                        return@launch
                    }
                    
                    if (template.requiresInput) {
                        // Launch InputDialogActivity for input
                        android.util.Log.d("MCL", "Widget logging WITH requiresInput: template=${template.name}, templateId=$templateId, widgetId=$widgetId")
                        // Widget will be updated after event is created in InputDialogActivity
                        val dialogIntent = Intent(context, de.ntdote.medicalcalendarlog.ui.InputDialogActivity::class.java).apply {
                            putExtra(de.ntdote.medicalcalendarlog.ui.InputDialogActivity.EXTRA_TEMPLATE_ID, templateId)
                            putExtra(de.ntdote.medicalcalendarlog.ui.InputDialogActivity.EXTRA_TEMPLATE_NAME, template.name)
                            addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
                        }
                        context.startActivity(dialogIntent)
                    } else {
                        // Log event directly (no input required)
                        android.util.Log.d("MCL", "Widget logging WITHOUT requiresInput: template=${template.name}, templateId=$templateId, widgetId=$widgetId")
                        // Widget will be updated after event is created in logEventForTemplate
                        logEventForTemplate(context, templateId)
                    }
                }
            }
        }
    }

    override fun onUpdate(
        context: Context,
        appWidgetManager: AppWidgetManager,
        appWidgetIds: IntArray
    ) {
        // Update all widgets
        for (appWidgetId in appWidgetIds) {
            updateWidget(context, appWidgetManager, appWidgetId)
        }
    }

    override fun onDeleted(context: Context, appWidgetIds: IntArray) {
        super.onDeleted(context, appWidgetIds)
        
        // Clean up widget configurations
        val widgetRepository = WidgetRepository(context)
        coroutineScope.launch {
            for (appWidgetId in appWidgetIds) {
                widgetRepository.removeWidgetConfig(appWidgetId)
            }
        }
    }

    override fun onEnabled(context: Context) {
        super.onEnabled(context)
        // Hourly updates are handled by HourlyReminderWorker (scheduled at boot and app start)
        // No need to schedule separately for widgets
    }

    override fun onDisabled(context: Context) {
        super.onDisabled(context)
        // Hourly updates continue for notifications even when no widgets exist
        // No need to cancel anything
    }

    companion object {
        const val ACTION_WIDGET_LOG = "de.ntdote.medicalcalendarlog.WIDGET_LOG"
        const val ACTION_REFRESH_WIDGETS = "de.ntdote.medicalcalendarlog.REFRESH_WIDGETS"
        const val EXTRA_TEMPLATE_ID = "template_id"
        const val EXTRA_WIDGET_ID = "widget_id"

        /**
         * Apply transparency to widget badge backgrounds
         */
        private fun applyTransparencyToBadges(views: RemoteViews, transparencyPercent: Int) {
            // Convert percentage (0-100) to alpha (0.0-1.0)
            val alpha = transparencyPercent / 100.0f
            
            // Apply alpha to all badge views
            // Note: setAlpha may not be available through RemoteViews on older APIs (pre-API 29)
            // We need to check API level and skip opacity on older devices
            // The exception happens during RemoteViews inflation, not when we call setFloat
            // So we must avoid calling setAlpha entirely on unsupported APIs
            
            // Check if setAlpha is supported (available in API 29+)
            // We'll skip opacity on older APIs to avoid crashes
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.Q) {
                // API 29 (Android 10) and above support setAlpha through RemoteViews
                views.setFloat(R.id.widget_template_name, "setAlpha", alpha)
                views.setFloat(R.id.widget_time_badge, "setAlpha", alpha)
                views.setFloat(R.id.widget_value_badge, "setAlpha", alpha)
            }
            // On older APIs (API 28 and below), opacity feature is disabled
            // Widgets will show at full opacity
        }

        /**
         * Set widget to two-segment mode when value badge is hidden
         * Android automatically redistributes layout weight among visible views
         */
        private fun setTwoSegmentMode(views: RemoteViews) {
            views.setViewVisibility(R.id.widget_value_badge, View.GONE)
        }

        /**
         * Set widget to three-segment mode (30/40/30 ratio) when value badge is visible
         */
        private fun setThreeSegmentMode(views: RemoteViews) {
            views.setViewVisibility(R.id.widget_value_badge, View.VISIBLE)
        }

        /**
         * Update a single widget
         */
        fun updateWidget(context: Context, appWidgetManager: AppWidgetManager, appWidgetId: Int, debugMode: Boolean = false) {
            val widgetRepository = WidgetRepository(context)
            val calendarRepository = CalendarRepository(context.contentResolver, context)
            val preferencesRepository = PreferencesRepository(context)

            CoroutineScope(Dispatchers.IO + SupervisorJob()).launch {
                val widgetConfig = widgetRepository.getWidgetConfigSync(appWidgetId)
                val views = RemoteViews(context.packageName, R.layout.widget_template)

                if (widgetConfig == null) {
                    // Widget not configured - show placeholder
                    views.setTextViewText(R.id.widget_template_name, "Not configured")
                    views.setTextViewText(R.id.widget_template_info, "")
                    views.setTextViewText(R.id.widget_time_badge, "--")
                    setTwoSegmentMode(views)
                } else {
                    // Load template
                    val templates = preferencesRepository.templates.first()
                    val template = templates.find { it.id == widgetConfig.templateId }

                    if (template == null) {
                        // Template deleted - gray out widget
                        views.setTextViewText(R.id.widget_template_name, "Template deleted")
                        views.setTextViewText(R.id.widget_template_info, "")
                        views.setTextViewText(R.id.widget_time_badge, "--")
                        setTwoSegmentMode(views)
                        
                        // Apply transparency to deleted template indicator
                        val alpha = (widgetConfig.transparency / 100.0 * 255).toInt()
                        val grayColor = Color.argb(alpha, 200, 200, 200)
                        views.setInt(R.id.widget_container, "setBackgroundColor", grayColor)
                    } else {
                        // Normal widget display
                        
                        // Apply transparency to all badge backgrounds
                        applyTransparencyToBadges(views, widgetConfig.transparency)
                        
                        // Template name - only shown when showTemplateName is true
                        if (widgetConfig.showTemplateName) {
                            // If this is a DECAYING template with multiple drugs and we're showing concentration,
                            // display "templatename:drugname" to clarify which drug the concentration refers to
                            val displayName = if (template.templateType == TemplateType.DECAYING && 
                                                 template.drugs.size > 1 && 
                                                 widgetConfig.showValue) {
                                // Get the selected drug name
                                val selectedDrug = template.drugs.getOrNull(widgetConfig.selectedDrugIndex)
                                if (selectedDrug != null) {
                                    "${template.name}:${selectedDrug.drugType}"
                                } else {
                                    template.name
                                }
                            } else {
                                template.name
                            }
                            views.setTextViewText(R.id.widget_template_name, displayName)
                            views.setViewVisibility(R.id.widget_template_name, View.VISIBLE)
                        } else {
                            views.setViewVisibility(R.id.widget_template_name, View.GONE)
                        }

                        // Hide info line - not used in widget
                        views.setViewVisibility(R.id.widget_template_info, View.GONE)

                        // Time badge visibility and content
                        if (widgetConfig.showTime) {
                            views.setViewVisibility(R.id.widget_time_badge, View.VISIBLE)
                            
                            // Check if we should show percentage format
                            val thresholdMillis = template.getEffectiveReminderThresholdMillis()
                            
                            // Always log the percentage mode status for debugging
                            android.util.Log.d("MCL_Widget", "MCL: Widget $appWidgetId - usePercentage=${widgetConfig.usePercentage}, thresholdMillis=$thresholdMillis")
                            
                            if (widgetConfig.usePercentage && thresholdMillis != null) {
                                // Show percentage format
                                val timePercentage = calculateTimePercentage(context, calendarRepository, widgetConfig.templateId, thresholdMillis)
                                // Compact mode: no % sign, otherwise include it
                                val displayText = if (widgetConfig.compactTime) "$timePercentage" else "${timePercentage}%"
                                views.setTextViewText(R.id.widget_time_badge, displayText)
                                
                                // Color: red when >= 100%, blue otherwise
                                if (timePercentage >= 100) {
                                    views.setInt(R.id.widget_time_badge, "setBackgroundResource", R.drawable.widget_badge_red)
                                } else {
                                    views.setInt(R.id.widget_time_badge, "setBackgroundResource", R.drawable.widget_badge_blue)
                                }
                                
                                // Always log the percentage calculation for debugging
                                android.util.Log.d("MCL_Widget", "MCL: Updating widget $appWidgetId - Time percentage: ${timePercentage}%, displayText=$displayText")
                            } else {
                                // Show normal time format
                                val timeSinceLastEvent = getTimeSinceLastEvent(context, calendarRepository, widgetConfig.templateId, widgetConfig.compactTime)
                                views.setTextViewText(R.id.widget_time_badge, timeSinceLastEvent)

                                // Debug logging - show widget time badge content with precise timestamp
                                if (debugMode) {
                                    val precisTime = getTimeSinceLastEventPrecise(context, calendarRepository, widgetConfig.templateId)
                                    android.util.Log.d("MCL_Widget", "MCL: [time-tag: $precisTime] Updating widget $appWidgetId (display: $timeSinceLastEvent)")
                                }

                                // Set time badge color based on reminder threshold
                                if (thresholdMillis != null) {
                                    val isOverdue = isTimeOverdue(timeSinceLastEvent, thresholdMillis)
                                    if (isOverdue) {
                                        views.setInt(R.id.widget_time_badge, "setBackgroundResource", R.drawable.widget_badge_red)
                                    } else {
                                        views.setInt(R.id.widget_time_badge, "setBackgroundResource", R.drawable.widget_badge_blue)
                                    }
                                }
                            }
                        } else {
                            views.setViewVisibility(R.id.widget_time_badge, View.GONE)
                        }

                        // Show value badge for DECAYING and METRIC templates based on config
                        var shouldShowValueBadge = false
                        if (widgetConfig.showValue) {
                            when (template.templateType) {
                                TemplateType.DECAYING -> {
                                    // Check if we should show concentration percentage - use drug-specific threshold
                                    val selectedDrug = template.drugs.getOrNull(widgetConfig.selectedDrugIndex)
                                    val concentrationThreshold = if (selectedDrug?.concentrationReminderEnabled == true) {
                                        selectedDrug.concentrationReminderThreshold
                                    } else {
                                        null
                                    }
                                    
                                    // Always log concentration percentage mode status with drug info
                                    android.util.Log.d("MCL_Widget", "MCL: Widget $appWidgetId - concentration usePercentage=${widgetConfig.usePercentage}, selectedDrugIndex=${widgetConfig.selectedDrugIndex}, drug=${selectedDrug?.drugType}, threshold=$concentrationThreshold")
                                    
                                    if (widgetConfig.usePercentage && concentrationThreshold != null && concentrationThreshold > 0) {
                                        // Show percentage format for concentration
                                        val concentrationPercentage = calculateConcentrationPercentage(
                                            context, calendarRepository, template.id, concentrationThreshold, widgetConfig.selectedDrugIndex
                                        )
                                        
                                        if (concentrationPercentage != null) {
                                            // Compact mode: no % sign, otherwise include it
                                            val displayText = if (widgetConfig.compactTime) "$concentrationPercentage" else "${concentrationPercentage}%"
                                            views.setTextViewText(R.id.widget_value_badge, displayText)
                                            
                                            // Color: red when < 100%, normal concentration colors otherwise
                                            if (concentrationPercentage < 100) {
                                                views.setInt(R.id.widget_value_badge, "setBackgroundResource", R.drawable.widget_badge_red)
                                            } else {
                                                // Use concentration-based colors for values >= 100%
                                                val actualConcentration = getCurrentConcentration(context, calendarRepository, template.id, widgetConfig.selectedDrugIndex) ?: 0.0
                                                views.setInt(R.id.widget_value_badge, "setBackgroundResource", 
                                                    getConcentrationColorDrawable(actualConcentration))
                                            }
                                            
                                            shouldShowValueBadge = true
                                            
                                            // Always log concentration percentage calculation
                                            android.util.Log.d("MCL_Widget", "MCL: Updating widget $appWidgetId - Concentration percentage: ${concentrationPercentage}%, displayText=$displayText")
                                        } else {
                                            android.util.Log.d("MCL_Widget", "MCL: Widget $appWidgetId - Concentration percentage calculation returned null")
                                        }
                                    } else {
                                        // Show normal concentration format - use selectedDrugIndex
                                        val concentration = getCurrentConcentration(context, calendarRepository, template.id, widgetConfig.selectedDrugIndex)
                                        val peakConcentration = getPeakConcentration(context, calendarRepository, template.id)
                                        
                                        if (concentration != null && concentration > 0 && peakConcentration != null && 
                                            peakConcentration > 0 && concentration >= (peakConcentration * 0.01)) {
                                            val selectedDrug = template.drugs.getOrNull(widgetConfig.selectedDrugIndex)
                                            val unit = if (widgetConfig.showUnit && !widgetConfig.compactTime) selectedDrug?.unit ?: "" else ""
                                            // Format with unit on same line or separate line based on space
                                            val valueText = if (unit.isNotEmpty()) {
                                                formatConcentrationWithUnit(concentration, unit) + " " + unit
                                            } else {
                                                formatConcentrationWithUnit(concentration, unit)
                                            }
                                            views.setTextViewText(R.id.widget_value_badge, valueText)
                                            views.setInt(R.id.widget_value_badge, "setBackgroundResource", 
                                                getConcentrationColorDrawable(concentration))
                                            shouldShowValueBadge = true
                                        }
                                    }
                                }
                                TemplateType.METRIC -> {
                                    // Metric templates don't support percentage mode (no decay/reminder concept)
                                    val latestValue = getLatestMetricValue(context, calendarRepository, template.id)
                                    if (latestValue != null) {
                                        val unit = if (widgetConfig.showUnit && !widgetConfig.compactTime) template.metricUnit else ""
                                        // Format with unit on same line or separate line based on space
                                        val valueText = if (unit.isNotEmpty()) {
                                            formatMetricWithUnit(latestValue, unit) + " " + unit
                                        } else {
                                            formatMetricWithUnit(latestValue, unit)
                                        }
                                        views.setTextViewText(R.id.widget_value_badge, valueText)
                                        views.setInt(R.id.widget_value_badge, "setBackgroundResource", 
                                            R.drawable.widget_badge_blue)
                                        shouldShowValueBadge = true
                                    }
                                }
                                else -> { }
                            }
                        }
                        
                        if (shouldShowValueBadge) {
                            setThreeSegmentMode(views)
                        } else {
                            setTwoSegmentMode(views)
                        }

                        // Set click intents based on nameClickQuickLog setting
                        if (widgetConfig.nameClickQuickLog && widgetConfig.showTemplateName) {
                            // Name area: Direct log
                            val nameLogIntent = Intent(context, TemplateWidgetProvider::class.java).apply {
                                action = ACTION_WIDGET_LOG
                                putExtra(EXTRA_TEMPLATE_ID, widgetConfig.templateId)
                                putExtra(EXTRA_WIDGET_ID, appWidgetId)
                            }
                            val nameLogPendingIntent = PendingIntent.getBroadcast(
                                context,
                                appWidgetId * 1000 + 1, // Unique request code
                                nameLogIntent,
                                PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
                            )
                            views.setOnClickPendingIntent(R.id.widget_template_name, nameLogPendingIntent)
                            
                            // Other areas (time badge, value badge): Menu
                            val menuIntent = Intent(context, de.ntdote.medicalcalendarlog.ui.WidgetMenuActivity::class.java).apply {
                                putExtra(de.ntdote.medicalcalendarlog.ui.WidgetMenuActivity.EXTRA_WIDGET_ID, appWidgetId)
                                putExtra(de.ntdote.medicalcalendarlog.ui.WidgetMenuActivity.EXTRA_TEMPLATE_ID, widgetConfig.templateId)
                                addFlags(Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK)
                            }
                            val menuPendingIntent = PendingIntent.getActivity(
                                context,
                                appWidgetId * 1000 + 2, // Unique request code
                                menuIntent,
                                PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
                            )
                            views.setOnClickPendingIntent(R.id.widget_time_badge, menuPendingIntent)
                            views.setOnClickPendingIntent(R.id.widget_value_badge, menuPendingIntent)
                        } else {
                            // All areas go to menu (name disabled or nameClickQuickLog disabled)
                            val menuIntent = Intent(context, de.ntdote.medicalcalendarlog.ui.WidgetMenuActivity::class.java).apply {
                                putExtra(de.ntdote.medicalcalendarlog.ui.WidgetMenuActivity.EXTRA_WIDGET_ID, appWidgetId)
                                putExtra(de.ntdote.medicalcalendarlog.ui.WidgetMenuActivity.EXTRA_TEMPLATE_ID, widgetConfig.templateId)
                                addFlags(Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK)
                            }
                            val menuPendingIntent = PendingIntent.getActivity(
                                context,
                                appWidgetId, // Unique request code per widget
                                menuIntent,
                                PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
                            )
                            views.setOnClickPendingIntent(R.id.widget_template_name, menuPendingIntent)
                            views.setOnClickPendingIntent(R.id.widget_time_badge, menuPendingIntent)
                            views.setOnClickPendingIntent(R.id.widget_value_badge, menuPendingIntent)
                        }
                    }
                }

                appWidgetManager.updateAppWidget(appWidgetId, views)
            }
        }

        /**
         * Update all widgets
         */
        fun updateAllWidgets(context: Context, debugMode: Boolean = false) {
            val appWidgetManager = AppWidgetManager.getInstance(context)
            val componentName = ComponentName(context, TemplateWidgetProvider::class.java)
            val appWidgetIds = appWidgetManager.getAppWidgetIds(componentName)
            
            if (debugMode) {
                android.util.Log.d("MCL_Widget", "MCL: Updating ${appWidgetIds.size} widgets")
            }
            
            for (appWidgetId in appWidgetIds) {
                updateWidget(context, appWidgetManager, appWidgetId, debugMode)
            }
        }

        private suspend fun getEventsForTemplate(
            context: Context,
            calendarRepository: CalendarRepository,
            preferencesRepository: PreferencesRepository,
            templateId: String
        ): List<CalendarEvent> {
            val templates = preferencesRepository.templates.first()
            val template = templates.find { it.id == templateId } ?: return emptyList()
            
            val calendarId = preferencesRepository.selectedCalendarId.first() ?: return emptyList()
            val allEvents = calendarRepository.getEventsForPeriod(calendarId, 30)
            
            // Filter events by template name - handle requiresInput templates
            return allEvents.filter { event ->
                if (template.requiresInput) {
                    event.title.startsWith(template.name) && 
                    (event.title == template.name || event.title.substring(template.name.length).all { it.isDigit() || it == '.' })
                } else {
                    event.title.startsWith(template.name)
                }
            }
        }

        private suspend fun getTimeSinceLastEvent(
            context: Context,
            calendarRepository: CalendarRepository,
            templateId: String,
            compactFormat: Boolean = false
        ): String {
            val preferencesRepository = PreferencesRepository(context)
            val events = getEventsForTemplate(context, calendarRepository, preferencesRepository, templateId)
            if (events.isEmpty()) return "Never"

            val lastEvent = events.maxByOrNull { it.startTime.time } ?: return "Never"
            val duration = TimeDuration.since(lastEvent.startTime)
            
            // Widget mode: never show minutes, minimum is 0h
            return TimeFormatter.format(duration, compact = compactFormat, widgetMode = true)
        }

        /**
         * Get precise time since last event with hours, minutes, and seconds (for debugging only)
         */
        private suspend fun getTimeSinceLastEventPrecise(
            context: Context,
            calendarRepository: CalendarRepository,
            templateId: String
        ): String {
            val preferencesRepository = PreferencesRepository(context)
            val events = getEventsForTemplate(context, calendarRepository, preferencesRepository, templateId)
            if (events.isEmpty()) return "Never"

            val lastEvent = events.maxByOrNull { it.startTime.time } ?: return "Never"
            val currentTime = System.currentTimeMillis()
            val timeDiff = currentTime - lastEvent.startTime.time

            val hours = TimeUnit.MILLISECONDS.toHours(timeDiff)
            val minutes = TimeUnit.MILLISECONDS.toMinutes(timeDiff) % 60
            val seconds = TimeUnit.MILLISECONDS.toSeconds(timeDiff) % 60
            val days = TimeUnit.MILLISECONDS.toDays(timeDiff)

            return when {
                days > 0 -> {
                    val remainingHours = hours % 24
                    "${days}d ${remainingHours}h ${minutes}m ${seconds}s"
                }
                hours > 0 -> "${hours}h ${minutes}m ${seconds}s"
                minutes > 0 -> "${minutes}m ${seconds}s"
                else -> "${seconds}s"
            }
        }

        private suspend fun getCurrentConcentration(
            context: Context,
            calendarRepository: CalendarRepository,
            templateId: String,
            drugIndex: Int = 0
        ): Double? {
            val preferencesRepository = PreferencesRepository(context)
            val templates = preferencesRepository.templates.first()
            val template = templates.find { it.id == templateId } ?: return null
            
            if (template.templateType != TemplateType.DECAYING || template.drugs.isEmpty()) {
                return null
            }

            // Get the specific drug from the template's drug list
            val drug = template.drugs.getOrNull(drugIndex) ?: return 0.0
            
            // For decaying templates, calculate cross-template drug concentrations
            val calendarId = preferencesRepository.selectedCalendarId.first() ?: return null
            val daysBack = preferencesRepository.daysBack.first()
            val events = calendarRepository.getEventsForPeriod(calendarId, daysBack)
            
            // Use cross-template concentration calculation for consistency with graphs
            return ConcentrationCalculator.calculateCurrentDrugConcentration(
                drug.drugType, templates, events
            )
        }

        private suspend fun getPeakConcentration(
            context: Context,
            calendarRepository: CalendarRepository,
            templateId: String
        ): Double? {
            val preferencesRepository = PreferencesRepository(context)
            val events = getEventsForTemplate(context, calendarRepository, preferencesRepository, templateId)
            return events.maxOfOrNull { it.dosage }
        }

        private suspend fun getLatestMetricValue(
            context: Context,
            calendarRepository: CalendarRepository,
            templateId: String
        ): Double? {
            val preferencesRepository = PreferencesRepository(context)
            val events = getEventsForTemplate(context, calendarRepository, preferencesRepository, templateId)
            if (events.isEmpty()) return null
            
            val latestEvent = events.maxByOrNull { it.startTime.time }
            return latestEvent?.dosage
        }

        private fun formatConcentrationWithUnit(concentration: Double, unit: String): String {
            return when {
                concentration >= 100 -> "${concentration.toInt()}"
                concentration >= 10 -> String.format("%.1f", concentration)
                concentration >= 1 -> String.format("%.2f", concentration)
                else -> String.format("%.3f", concentration)
            }
        }

        private fun formatMetricWithUnit(value: Double, unit: String): String {
            return when {
                value >= 100 -> "${value.toInt()}"
                value >= 10 -> String.format("%.1f", value)
                value >= 1 -> String.format("%.2f", value)
                else -> String.format("%.3f", value)
            }
        }

        /**
         * Calculate time percentage based on time since last event vs reminder threshold
         * Returns percentage as integer (0-9999+)
         */
        private suspend fun calculateTimePercentage(
            context: Context,
            calendarRepository: CalendarRepository,
            templateId: String,
            reminderThresholdMillis: Long
        ): Int {
            val preferencesRepository = PreferencesRepository(context)
            val events = getEventsForTemplate(context, calendarRepository, preferencesRepository, templateId)
            if (events.isEmpty()) return 100 // Show 100% if never logged
            
            val lastEvent = events.maxByOrNull { it.startTime.time } ?: return 100
            val timeSinceLastEvent = System.currentTimeMillis() - lastEvent.startTime.time
            
            val percentage = (timeSinceLastEvent.toDouble() / reminderThresholdMillis.toDouble() * 100.0).toInt()
            return percentage.coerceAtLeast(0) // Never negative
        }

        /**
         * Calculate concentration percentage based on current concentration vs alert threshold
         * Returns percentage as integer (0-9999+)
         */
        private suspend fun calculateConcentrationPercentage(
            context: Context,
            calendarRepository: CalendarRepository,
            templateId: String,
            alertThreshold: Double,
            selectedDrugIndex: Int = 0
        ): Int? {
            val currentConcentration = getCurrentConcentration(context, calendarRepository, templateId, selectedDrugIndex) ?: return null
            if (alertThreshold <= 0) return null
            
            val percentage = (currentConcentration / alertThreshold * 100.0).toInt()
            return percentage.coerceAtLeast(0) // Never negative
        }

        private fun getConcentrationColorDrawable(concentration: Double): Int {
            return when {
                concentration > 50 -> R.drawable.widget_badge_green
                concentration > 20 -> R.drawable.widget_badge_orange
                concentration > 5 -> R.drawable.widget_badge_yellow
                else -> R.drawable.widget_badge_gray
            }
        }

        private fun isTimeOverdue(timeSinceLastEvent: String, reminderThresholdMillis: Long): Boolean {
            if (timeSinceLastEvent == "Never") return true

            val timeInMinutes = when {
                // Handle compact format "3d12" (no spaces, no trailing "h" for hours)
                timeSinceLastEvent.contains("d") && !timeSinceLastEvent.contains(" ") && !timeSinceLastEvent.contains("h") -> {
                    val daysPart = timeSinceLastEvent.substringBefore("d")
                    val hoursPart = timeSinceLastEvent.substringAfter("d")
                    val days = daysPart.toDoubleOrNull() ?: 0.0
                    val hours = if (hoursPart.isNotEmpty()) hoursPart.toDoubleOrNull() ?: 0.0 else 0.0
                    ((days * 24.0) + hours) * 60.0 // Convert to minutes
                }
                // Handle normal format "3d 12h" (with spaces)
                timeSinceLastEvent.contains("d") && timeSinceLastEvent.contains("h") -> {
                    val parts = timeSinceLastEvent.trim().split(Regex("\\s+"))
                    var totalMinutes = 0.0
                    for (part in parts) {
                        when {
                            part.endsWith("d") -> {
                                val days = part.removeSuffix("d").toDoubleOrNull() ?: 0.0
                                totalMinutes += days * 24.0 * 60.0
                            }
                            part.endsWith("h") -> {
                                val hours = part.removeSuffix("h").toDoubleOrNull() ?: 0.0
                                totalMinutes += hours * 60.0
                            }
                        }
                    }
                    totalMinutes
                }
                timeSinceLastEvent.endsWith("m") -> {
                    timeSinceLastEvent.removeSuffix("m").toDoubleOrNull() ?: 0.0
                }
                timeSinceLastEvent.endsWith("h") -> {
                    val hours = timeSinceLastEvent.removeSuffix("h").toDoubleOrNull() ?: 0.0
                    hours * 60.0 // Convert to minutes
                }
                timeSinceLastEvent.endsWith("d") -> {
                    val days = timeSinceLastEvent.removeSuffix("d").toDoubleOrNull() ?: 0.0
                    days * 24.0 * 60.0 // Convert to minutes
                }
                else -> 0.0
            }

            val thresholdMinutes = reminderThresholdMillis / (60 * 1000.0)
            return timeInMinutes >= thresholdMinutes
        }

        /**
         * Log an event for the given template
         */
        private suspend fun logEventForTemplate(context: Context, templateId: String) {
            val preferencesRepository = PreferencesRepository(context)
            val calendarRepository = CalendarRepository(context.contentResolver, context)
            
            val templates = preferencesRepository.templates.first()
            val template = templates.find { it.id == templateId } ?: return
            
            val calendarId = preferencesRepository.selectedCalendarId.first() ?: return
            
            // Use centralized event creation (handles notification cancellation and widget updates)
            calendarRepository.createEventFromTemplate(calendarId, template)
        }
    }
}
