package de.ntdote.medicalcalendarlog.service

import android.Manifest
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.os.Build
import android.util.Log
import androidx.core.content.ContextCompat
import androidx.core.app.NotificationCompat
import androidx.work.CoroutineWorker
import androidx.work.WorkerParameters
import de.ntdote.medicalcalendarlog.MainActivity
import de.ntdote.medicalcalendarlog.R
import de.ntdote.medicalcalendarlog.data.CalendarEvent
import de.ntdote.medicalcalendarlog.data.Template
import de.ntdote.medicalcalendarlog.repository.CalendarRepository
import de.ntdote.medicalcalendarlog.repository.PreferencesRepository
import de.ntdote.medicalcalendarlog.utils.AdvancedCalculationEngine
import de.ntdote.medicalcalendarlog.utils.ConcentrationCalculator
import de.ntdote.medicalcalendarlog.utils.EventMatching
import de.ntdote.medicalcalendarlog.utils.NotificationBuilder
import kotlinx.coroutines.flow.first
import java.util.Date
import java.util.concurrent.TimeUnit
import kotlin.math.abs

/**
 * Worker that runs every hour to:
 * 1. Update all widgets with current time/concentration data
 * 2. Renew notifications that are still current
 * 3. Create notifications for items that were overlooked due to device being powered down
 * 4. Refresh the information in notifications (e.g., update overdue time)
 * 5. Update notification content to reflect current overdue status (e.g., 2h overdue after 2 hours)
 */
class HourlyReminderWorker(
    context: Context,
    params: WorkerParameters
) : CoroutineWorker(context, params) {

    companion object {
        private const val TAG = "MCL"
        
        const val ACTION_EVALUATE_NOTIFICATIONS = "de.ntdote.medicalcalendarlog.EVALUATE_NOTIFICATIONS"
        
        // Key for storing the last execution time in SharedPreferences
        private const val PREFS_KEY_LAST_EXECUTION = "hourly_worker_last_execution"
        
        // Key for storing the overlooked notifications in SharedPreferences
        private const val PREFS_KEY_OVERLOOKED_NOTIFICATIONS = "overlooked_notifications_"
        
        // Key for storing the original notification times
        private const val PREFS_KEY_ORIGINAL_NOTIFICATION_TIME = "original_notification_time_"
        
        // Key for storing notification IDs
        private const val PREFS_KEY_NOTIFICATION_ID = "notification_id_"
        
        /**
         * Manually trigger notification evaluation (for debugging via ADB)
         */
        fun evaluateNotifications(context: Context) {
            android.util.Log.d(TAG, "Manual notification evaluation triggered via ADB")
            
            // Create a one-time work request to run immediately with manual tag
            val workRequest = androidx.work.OneTimeWorkRequestBuilder<HourlyReminderWorker>()
                .setInputData(
                    androidx.work.workDataOf("trigger_type" to "manual")
                )
                .build()
            
            androidx.work.WorkManager.getInstance(context).enqueue(workRequest)
        }
        
        /**
         * Cancel notification for a template if it exists
         */
        fun cancelNotificationIfExists(context: Context, templateId: String, isConcentration: Boolean = false) {
            val prefs = context.getSharedPreferences("reminder_prefs", Context.MODE_PRIVATE)
            val key = if (isConcentration) {
                "${PREFS_KEY_NOTIFICATION_ID}${templateId}_concentration"
            } else {
                "${PREFS_KEY_NOTIFICATION_ID}${templateId}"
            }
            
            val notificationId = prefs.getInt(key, -1)
            if (notificationId != -1) {
                val notificationManager = context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
                notificationManager.cancel(notificationId)
                Log.d(TAG, "Cancelled notification with ID $notificationId for template $templateId")
                
                // Remove the notification ID from preferences
                prefs.edit().remove(key).apply()
                
                // Also remove any original notification time
                val originalTimeKey = if (isConcentration) {
                    "${PREFS_KEY_ORIGINAL_NOTIFICATION_TIME}${templateId}_concentration"
                } else {
                    "${PREFS_KEY_ORIGINAL_NOTIFICATION_TIME}${templateId}"
                }
                prefs.edit().remove(originalTimeKey).apply()
            }
        }
        
        /**
         * Check if the worker needs to be run due to user interaction.
         * If last run was more than 75 minutes ago, trigger the worker immediately.
         */
        fun checkAndRunWorkerIfNeeded(context: Context, triggerSource: String) {
 //           return // TODO: reenable once worker/scheduling is fixed/completed - or delete?
            val prefs = context.getSharedPreferences("reminder_prefs", Context.MODE_PRIVATE)
            val lastExecutionTime = prefs.getLong(PREFS_KEY_LAST_EXECUTION, 0L)
            val currentTime = System.currentTimeMillis()
            
            // Grace period: 1 hour + 15 minutes = 75 minutes = 4,500,000 milliseconds
            val gracePeriodMs = 75 * 60 * 1000L
            
            if (lastExecutionTime == 0L || (currentTime - lastExecutionTime) > gracePeriodMs) {
                val minutesOverdue = (currentTime - lastExecutionTime) / (60 * 1000L)
                Log.d(TAG, "Hourly worker overdue by ${minutesOverdue}min, running due to user wake-up from: $triggerSource")
                
                // Update the last execution time immediately to prevent duplicate runs
                prefs.edit().putLong(PREFS_KEY_LAST_EXECUTION, currentTime).apply()
                
                // Create and enqueue a one-time work request
                val workRequest = androidx.work.OneTimeWorkRequestBuilder<HourlyReminderWorker>()
                    .addTag("user-wakeup")
                    .build()
                
                androidx.work.WorkManager.getInstance(context).enqueue(workRequest)
                
            } else {
                val nextRunDue = gracePeriodMs - (currentTime - lastExecutionTime)
                val minutesTillNext = nextRunDue / (60 * 1000L)
                Log.d(TAG, "Hourly worker on schedule, next run due in ${minutesTillNext}min (source: $triggerSource)")
            }
        }
    }

    /**
     * Check if calendar permissions are granted
     */
    private fun hasCalendarPermissions(): Boolean {
        val readCalendarPermission = ContextCompat.checkSelfPermission(
            applicationContext, 
            Manifest.permission.READ_CALENDAR
        )
        
        val writeCalendarPermission = ContextCompat.checkSelfPermission(
            applicationContext, 
            Manifest.permission.WRITE_CALENDAR
        )
        
        return readCalendarPermission == PackageManager.PERMISSION_GRANTED &&
               writeCalendarPermission == PackageManager.PERMISSION_GRANTED
    }
    
    override suspend fun doWork(): Result {
        val startTime = System.currentTimeMillis()
        val timeFormatter = java.text.SimpleDateFormat("HH:mm:ss", java.util.Locale.getDefault())
        val currentTimeStr = timeFormatter.format(java.util.Date(startTime))
        
        Log.d(TAG, "Hourly worker started (notifications + widgets)")
        
        // Check if calendar permissions are granted
        if (!hasCalendarPermissions()) {
            Log.w(TAG, "Cannot run hourly worker - calendar permissions not granted")
            return Result.retry() // Retry later in case permissions are granted
        }
        
        try {
            // Check if debug mode is enabled
            val preferencesRepository = PreferencesRepository(applicationContext)
            val debugModeEnabled = preferencesRepository.debugModeEnabled.first()
            
            // First, update all widgets
            TemplateWidgetProvider.updateAllWidgets(applicationContext, debugModeEnabled)
            Log.d(TAG, "Widgets updated (debug: $debugModeEnabled)")
            
            // Pass context to CalendarRepository so it can check permissions
            val calendarRepository = CalendarRepository(applicationContext.contentResolver, applicationContext)
            
            val templates = preferencesRepository.templates.first()
            val selectedCalendarId = preferencesRepository.selectedCalendarId.first() ?: return Result.failure()
            
            // Get current time
            val currentTime = Date()
            
            // Get the last execution time
            val prefs = applicationContext.getSharedPreferences("reminder_prefs", Context.MODE_PRIVATE)
            val lastExecutionTime = prefs.getLong(PREFS_KEY_LAST_EXECUTION, 0L)
            
            // Check if this is the first run or if the device was powered down
            val deviceWasPoweredDown = lastExecutionTime > 0 && 
                (currentTime.time - lastExecutionTime) > (2 * 60 * 60 * 1000) // More than 2 hours since last run
            
            if (deviceWasPoweredDown) {
                Log.d(TAG, "Device appears to have been powered down. Checking for overlooked notifications.")
            }
            
            // Store current execution time
            prefs.edit().putLong(PREFS_KEY_LAST_EXECUTION, currentTime.time).apply()
            
            // Check if this is a manual run based on input data
            val triggerType = inputData.getString("trigger_type") ?: "hourly"
            
            // Record worker run with appropriate type in PreferencesRepository for UI display
            preferencesRepository.recordWorkerRun(currentTime.time, triggerType)
            
            // Get events for analysis
            val events = calendarRepository.getEventsForPeriod(selectedCalendarId, 24 * 7) // 7 days
            
            // Process time-based reminders
            processTimeBasedReminders(templates, events, currentTime, lastExecutionTime, deviceWasPoweredDown)
            
            // Process concentration-based reminders
            processConcentrationReminders(templates, events, currentTime, lastExecutionTime, deviceWasPoweredDown)
            
            // Trigger recalculation of all reminders to ensure future notifications are scheduled
            ReminderService.recalculateAllReminders(applicationContext, "hourly_worker")
            
            return Result.success()
        } catch (e: Exception) {
            Log.e(TAG, "Error in hourly reminder worker", e)
            return Result.retry()
        }
    }
    
    /**
     * Process time-based reminders
     */
    private suspend fun processTimeBasedReminders(
        templates: List<Template>,
        events: List<CalendarEvent>,
        currentTime: Date,
        lastExecutionTime: Long,
        deviceWasPoweredDown: Boolean = false
    ) {
        // Get shared preferences for tracking notification times
        val prefs = applicationContext.getSharedPreferences("reminder_prefs", Context.MODE_PRIVATE)
        val preferencesRepository = PreferencesRepository(applicationContext)
        val calendarRepository = CalendarRepository(applicationContext.contentResolver, applicationContext)
        
        // Filter templates with time-based reminders enabled (only process enabled templates)
        templates.filter { 
            it.reminderEnabled && 
            it.reminderThresholdMillis != null &&
            it.enabled // Only process enabled templates
        }.forEach { template ->
            // Check if template is muted
            if (NotificationActionReceiver.isTemplateMuted(applicationContext, template.id, false)) {
                // Check if we should clear the mute (new event logged)
                NotificationActionReceiver.checkAndClearMuteOnNewEvent(
                    applicationContext, template.id, false, calendarRepository, preferencesRepository
                )
                // If still muted after check, skip this template
                if (NotificationActionReceiver.isTemplateMuted(applicationContext, template.id, false)) {
                    logMuteStatus(template.id, template.name, false)
                    return@forEach
                }
            }
            try {
                // Find relevant events for this template
                val relevantEvents = findRelevantEvents(template, events)
                
                // Find the last event for this template
                val lastEvent = relevantEvents.maxByOrNull { it.startTime }
                
                if (lastEvent != null) {
                    // Calculate when the reminder should trigger
                    val thresholdMillis = template.getEffectiveReminderThresholdMillis()!!
                    val reminderTime = Date(lastEvent.startTime.time + thresholdMillis)
                    
                    // Check if the reminder should be triggered now
                    if (reminderTime.before(currentTime)) {
                        // Get the original notification time if it exists
                        val originalNotificationKey = PREFS_KEY_ORIGINAL_NOTIFICATION_TIME + template.id
                        val originalNotificationTime = prefs.getLong(originalNotificationKey, 0L)
                        
                        // Calculate how long it's been since the reminder should have triggered
                        val hoursSinceReminder = TimeUnit.MILLISECONDS.toHours(
                            currentTime.time - reminderTime.time
                        )
                        
                        // Calculate how long it's been since the original notification
                        // If this is the first time we're notifying (originalNotificationTime == 0), then hoursSinceOriginalNotification = 0
                        val hoursSinceOriginalNotification = if (originalNotificationTime == 0L) {
                            // First notification - store current time and return 0
                            prefs.edit().putLong(originalNotificationKey, currentTime.time).apply()
                            0L
                        } else {
                            // Subsequent notification - calculate hours since original
                            TimeUnit.MILLISECONDS.toHours(currentTime.time - originalNotificationTime)
                        }
                        
                        // If device was off during the last scheduled check, this might be an overlooked notification
                        val isOverlooked = deviceWasPoweredDown && reminderTime.time < lastExecutionTime
                        
                        // Get the drug type for decaying templates
                        val drugType = if (template.templateType == de.ntdote.medicalcalendarlog.data.TemplateType.DECAYING && template.drugs.isNotEmpty()) {
                            template.getDrugTypesDisplay()
                        } else {
                            template.name
                        }
                        
                        // Send notification with updated status
                        sendTimeBasedNotification(
                            template, 
                            drugType, 
                            hoursSinceReminder, 
                            isOverlooked,
                            hoursSinceOriginalNotification
                        )
                    } else {
                        // If the reminder time is in the future, clear any stored original notification time
                        val originalNotificationKey = PREFS_KEY_ORIGINAL_NOTIFICATION_TIME + template.id
                        if (prefs.contains(originalNotificationKey)) {
                            prefs.edit().remove(originalNotificationKey).apply()
                        }
                    }
                }
            } catch (e: Exception) {
                Log.e(TAG, "Error processing time-based reminder for template ${template.id}", e)
            }
        }
    }
    
    /**
     * Process concentration-based reminders
     */
    private suspend fun processConcentrationReminders(
        templates: List<Template>,
        events: List<CalendarEvent>,
        currentTime: Date,
        lastExecutionTime: Long,
        deviceWasPoweredDown: Boolean = false
    ) {
        // Get shared preferences for tracking notification times
        val prefs = applicationContext.getSharedPreferences("reminder_prefs", Context.MODE_PRIVATE)
        val preferencesRepository = PreferencesRepository(applicationContext)
        val calendarRepository = CalendarRepository(applicationContext.contentResolver, applicationContext)
        
        // Filter templates with concentration-based reminders enabled (only process enabled templates)
        templates.filter { 
            it.templateType == de.ntdote.medicalcalendarlog.data.TemplateType.DECAYING &&
            it.drugs.any { drug -> drug.concentrationReminderEnabled && drug.concentrationReminderThreshold != null } &&
            it.enabled // Only process enabled templates
        }.forEach { template ->
            // Process each drug in the template that has concentration reminders enabled
            template.drugs.filter { drug -> 
                drug.concentrationReminderEnabled && drug.concentrationReminderThreshold != null 
            }.forEach { drug ->
                // Check if template is muted for this drug (use drug-specific mute key)
                if (NotificationActionReceiver.isTemplateMuted(applicationContext, template.id, true)) {
                    // Check if we should clear the mute (new event logged)
                    NotificationActionReceiver.checkAndClearMuteOnNewEvent(
                        applicationContext, template.id, true, calendarRepository, preferencesRepository
                    )
                    // If still muted after check, skip this drug
                    if (NotificationActionReceiver.isTemplateMuted(applicationContext, template.id, true)) {
                        logMuteStatus(template.id, "${template.name} (${drug.drugType})", true)
                        return@forEach
                    }
                }
                try {
                    val threshold = drug.concentrationReminderThreshold!!
                    val drugType = drug.drugType
                    
                    // Calculate current concentration
                    val currentConcentration = ConcentrationCalculator.calculateCurrentDrugConcentration(
                        drugType, templates, events, currentTime
                    )
                    
                    // Check if concentration is below threshold
                    if (currentConcentration < threshold) {
                        // Get the last notification time for this drug's concentration (use drug-specific key)
                        val lastNotifiedKey = "concentration_notified_${template.id}_${drugType}"
                        val lastNotifiedTime = prefs.getLong(lastNotifiedKey, 0L)
                        
                        // Get or set the original notification time (use drug-specific key)
                        val originalNotificationKey = PREFS_KEY_ORIGINAL_NOTIFICATION_TIME + template.id + "_" + drugType + "_concentration"
                        val originalNotificationTime = prefs.getLong(originalNotificationKey, 0L)
                        
                        // Calculate how long it's been since the original notification
                        // If this is the first time we're notifying (originalNotificationTime == 0), then hoursSinceOriginalNotification = 0
                        val hoursSinceOriginalNotification = if (originalNotificationTime == 0L) {
                            // First notification - store current time and return 0
                            prefs.edit().putLong(originalNotificationKey, currentTime.time).apply()
                            0L
                        } else {
                            // Subsequent notification - calculate hours since original
                            TimeUnit.MILLISECONDS.toHours(currentTime.time - originalNotificationTime)
                        }
                        
                        // Determine if we should send a notification
                        val shouldNotify = lastNotifiedTime == 0L || 
                            deviceWasPoweredDown || // Always notify if device was powered down
                            (currentTime.time - lastNotifiedTime) > (4 * 60 * 60 * 1000L) // 4 hours since last notification
                        
                        if (shouldNotify) {
                            // Mark as notified
                            prefs.edit().putLong(lastNotifiedKey, currentTime.time).apply()
                            
                            // Send concentration notification
                            sendConcentrationNotification(
                                template, 
                                drugType, 
                                currentConcentration,
                                hoursSinceOriginalNotification
                            )
                        }
                    } else {
                        // If concentration is above threshold, clear any stored original notification time (use drug-specific key)
                        val originalNotificationKey = PREFS_KEY_ORIGINAL_NOTIFICATION_TIME + template.id + "_" + drugType + "_concentration"
                        if (prefs.contains(originalNotificationKey)) {
                            prefs.edit().remove(originalNotificationKey).apply()
                        }
                    }
                } catch (e: Exception) {
                    Log.e(TAG, "Error processing concentration reminder for template ${template.id}, drug ${drug.drugType}", e)
                }
            }
        }
    }
    
    /**
     * Find events relevant to a specific template
     */
    private fun findRelevantEvents(template: Template, events: List<CalendarEvent>): List<CalendarEvent> {
        // Use unified matching logic
        return EventMatching.filterEventsForTemplate(events, template)
    }
    
    /**
     * Send a time-based notification using unified builder
     */
    private fun sendTimeBasedNotification(
        template: Template,
        drugName: String,
        hoursOverdue: Long,
        isOverlooked: Boolean,
        hoursSinceOriginalNotification: Long = 0
    ) {
        val notificationManager = applicationContext.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        NotificationBuilder.createNotificationChannels(notificationManager)
        
        // Use unified notification builder
        val (notification, notificationId) = NotificationBuilder.buildTimeBasedNotification(
            context = applicationContext,
            template = template,
            drugName = drugName,
            hoursOverdue = hoursOverdue,
            isOverlooked = isOverlooked,
            hoursSinceOriginalNotification = hoursSinceOriginalNotification
        )
        
        // Store the notification ID in shared preferences for later reference
        val prefs = applicationContext.getSharedPreferences("reminder_prefs", Context.MODE_PRIVATE)
        prefs.edit().putInt("notification_id_${template.id}", notificationId).apply()
        
        notificationManager.notify(notificationId, notification)
        
        Log.d(TAG, "Sent/updated time-based notification for $drugName, ${hoursOverdue}h overdue, " +
              "overlooked: $isOverlooked, hours since original: $hoursSinceOriginalNotification")
    }
    
    /**
     * Send a concentration-based notification using unified builder
     */
    private fun sendConcentrationNotification(
        template: Template,
        drugName: String,
        currentConcentration: Double,
        hoursSinceOriginalNotification: Long = 0
    ) {
        val notificationManager = applicationContext.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager
        NotificationBuilder.createNotificationChannels(notificationManager)
        
        // Use unified notification builder
        val (notification, notificationId) = NotificationBuilder.buildConcentrationNotification(
            context = applicationContext,
            template = template,
            drugName = drugName,
            currentConcentration = currentConcentration,
            hoursSinceOriginalNotification = hoursSinceOriginalNotification
        )
        
        // Store the notification ID in shared preferences for later reference
        val prefs = applicationContext.getSharedPreferences("reminder_prefs", Context.MODE_PRIVATE)
        prefs.edit().putInt("notification_id_${template.id}_concentration", notificationId).apply()
        
        notificationManager.notify(notificationId, notification)
        
        Log.d(TAG, "Sent/updated concentration notification for $drugName, current: $currentConcentration, hours since original: $hoursSinceOriginalNotification")
    }
    
    
    /**
     * Log mute status for debugging
     */
    private fun logMuteStatus(templateId: String, templateName: String, isConcentration: Boolean) {
        val prefs = applicationContext.getSharedPreferences("notification_mute_prefs", Context.MODE_PRIVATE)
        val currentTime = System.currentTimeMillis()
        
        val muteKey = if (isConcentration) "${templateId}_concentration" else templateId
        
        // Check 24h mute
        val mute24hUntil = prefs.getLong("mute_24h_until_$muteKey", 0L)
        if (mute24hUntil > currentTime) {
            val remainingMillis = mute24hUntil - currentTime
            val remainingHours = TimeUnit.MILLISECONDS.toHours(remainingMillis)
            val remainingMinutes = TimeUnit.MILLISECONDS.toMinutes(remainingMillis) % 60
            
            Log.d(TAG, "MCL: no notification for ${templateName}, muted till ${remainingHours}h ${remainingMinutes}m remaining")
            return
        }
        
        // Check mute until relog
        val muteUntilRelog = prefs.getBoolean("mute_until_relog_$muteKey", false)
        if (muteUntilRelog) {
            Log.d(TAG, "MCL: no notification for ${templateName}, muted till next-log")
        }
    }
}
